import argparse
import configparser
import os
import sys

import rdaf.component
import rdaf.component.pseudo_platform as pseudo_platform
import rdaf.component.platform as comp_platform
import rdaf.component.oia as comp_app_oia
import rdaf.component.ssh as comp_ssh
import rdaf.component.haproxy as haproxy
import rdaf.component.nginx as nginx
import rdaf.component.keepalived as keepalived
import rdaf.component.log_monitoring as comp_log_monitoring
import rdaf.component.bulk_stats as comp_bulkstats
import rdaf.component.file_object as comp_fileobject
import rdaf.component.opensearch_external as comp_opensearch_external
import rdaf.component.proxy as proxy
import rdaf.component.cert as comp_cert
import rdaf.contextual
import rdaf.component.dockerregistry as docker
import rdaf.component.worker as comp_worker
import rdaf.component.nats as nats
import rdaf.component.graphdb as graphdb
import rdaf.component.minio as minio
import rdaf.component.mariadb as mariadb
import rdaf.component.opensearch as opensearch
import rdaf.component.kafka as kafka
from rdaf.component.event_gateway import EventGateway
import rdaf.component.self_monitoring as comp_self_monitoring


class CliCmdHandler:

    def configure_parser(self, parser):
        r"""
        Configures the passed `parser` and returns an instance of `CliCmdHandler`
        :param  parser: Parser that will be configured for the particular command
                        handled by this `CliCmdHandler` type
        :type   parser: ArgParser
        :rtype:  CliCmdHandler
        """
        return

    def before_handle(self, cmd_args: argparse.Namespace,
                      config_parser: configparser.ConfigParser):
        _read_configs(config_parser)
        self._load_all_components(config_parser)

    def handle(self, cmd_args, config_parser: configparser.ConfigParser):
        r"""
        Handle the command invocation
        :param  cmd_args: The arguments passed to the command
        :type   cmd_args: argparse.ArgumentParser.Namespace
        """
        return

    def _load_all_components(self, config_parser: configparser.ConfigParser):
        components = [comp_ssh.SSHKeyManager(), pseudo_platform.PseudoComponent(), comp_cert.CertManager(),
                      nats.Nats(), docker.DockerRegistry(), proxy.Proxy(), minio.Minio(),
                      mariadb.MariaDB(), opensearch.Opensearch(), kafka.Kafka(), graphdb.GraphDB(),
                      haproxy.HAProxy(), nginx.Nginx(), keepalived.Keepalived(), comp_platform.RDAPlatform(), comp_bulkstats.BulkStats(), comp_fileobject.FileObject(),
                      comp_self_monitoring.SelfMonitoring(), comp_worker.Worker(), comp_opensearch_external.OpensearchExternal(),
                      comp_log_monitoring.LogMonitoring(), comp_app_oia.OIA(), EventGateway()]
        # register each of these components
        for component in components:
            rdaf.contextual.COMPONENT_REGISTRY.register(component)
        # let all the components load their configs
        for component in components:
            component.load_config(config_parser)


class ExitCmdHandler(CliCmdHandler):
    def handle(self, cmd_args, config_parser: configparser.ConfigParser):
        sys.exit(0)


def _read_configs(config_parser: configparser.ConfigParser, cfg_file: os.path = None):
    if cfg_file is None:
        config_file_path = os.path.expanduser(os.path.join('/opt', 'rdaf', 'rdaf.cfg'))
    else:
        config_file_path = cfg_file
    if not os.path.isfile(config_file_path):
        return config_parser
    with open(config_file_path, 'r') as f:
        config_parser.read_file(f)
    return config_parser


def is_rdaf_setup_done() -> bool:
    config_file_path = os.path.expanduser(os.path.join('/opt', 'rdaf', 'rdaf.cfg'))
    if not os.path.isfile(config_file_path):
        return False
    throw_away_config_parser = configparser.ConfigParser(allow_no_value=True)
    with open(config_file_path, 'r') as f:
        throw_away_config_parser.read_file(f)
    # TODO: this is a hackish way to verify if rdaf setup command was run
    return throw_away_config_parser.has_section('common')

def is_rdaf_telegraf_setup_done() -> bool:
    config_file_path = os.path.expanduser(os.path.join('/opt', 'rdaf-telegraf', 'rdaf-telegraf.cfg'))
    if not os.path.isfile(config_file_path):
        return False
    throw_away_config_parser = configparser.ConfigParser(allow_no_value=True)
    with open(config_file_path, 'r') as f:
        throw_away_config_parser.read_file(f)
    # TODO: this is a hackish way to verify if rdaf setup command was run
    return throw_away_config_parser.has_section('common')