import argparse
import configparser
import logging
import rdaf
import rdaf.rdafutils as rdafutils
from rdaf.component import Component
from rdaf import InvalidCmdUsageException
from rdaf.cmd import CliCmdHandler, status
from rdaf.contextual import COMPONENT_REGISTRY

logger = logging.getLogger(__name__)


class ApplicationCmdHandler(CliCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = False

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)
        tag_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the app components')

        choice_parser = argparse.ArgumentParser(add_help=False)
        choice_parser.add_argument('app_name',
                                   nargs='?',
                                   action='store',
                                   default='OIA',
                                   help='Select the APP to act on')
        choice_parser.add_argument('--service',
                                   dest="services",
                                   action='append',
                                   default=None,
                                   help='Restrict the scope of the command to specific service')

        app_commands_parser = parser.add_subparsers(dest='app_op', metavar='{}',
                                                    help='commands')

        app_commands_parser.add_parser('status', help='Status of the App')

        up_parser = app_commands_parser.add_parser('up',
                                                      parents=[choice_parser],
                                                      help='Create the App service'
                                                           'Containers')

        down_parser = app_commands_parser.add_parser('down',
                                                     parents=[choice_parser],
                                                     help='Delete the App service Containers')
        down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')

        install_parser = app_commands_parser.add_parser('install',
                                                        parents=[tag_parser, choice_parser],
                                                        help='Install the App service containers')
        
        upgrade_parser = app_commands_parser.add_parser('upgrade',
                                                        parents=[tag_parser, choice_parser],
                                                        help='Upgrade the App service containers')

        upgrade_parser.add_argument('--rolling-upgrade', dest='rolling', action='store_true',
                                    help='Performs a zero downtime upgrade')

        upgrade_parser.add_argument('--timeout', type=int, default=120,
                                    help='Timeout value for maintenance command (default: 120)')

        update_config_parser = app_commands_parser.add_parser('update-config',
                                                              parents=[choice_parser],
                                                              help='Updated configurations of one or more components')
        
        start_parser = app_commands_parser.add_parser('start',
                                                      parents=[choice_parser],
                                                      help='Start the App service'
                                                           'Containers')

        stop_parser = app_commands_parser.add_parser('stop',
                                                     parents=[choice_parser],
                                                     help='Stop the App service Containers')
        stop_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')


    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.app_op
        if cmd == 'status':
            components = COMPONENT_REGISTRY.get_by_category(category='apps')
            return status.StatusCmdHandler().handle(cmd_args, config_parser, components=components)

        component = rdaf.contextual.COMPONENT_REGISTRY.get(str(cmd_args.app_name).casefold())
        if cmd == 'install':
            component.pull_images(cmd_args, config_parser)
            component.install(cmd_args, config_parser)
            return
        if cmd == 'upgrade':
            component.pull_images(cmd_args, config_parser)
            component.upgrade(cmd_args, config_parser)
            return
        if cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF app container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the app container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('app down operation has been cancelled')
                    return
            component.down(cmd_args, config_parser)
            return
        if cmd == 'up':
            component.up(cmd_args, config_parser)
            return
        if cmd == 'update-config':
            component.update_config(config_parser)
            return
        if cmd == 'start':
            component.start(cmd_args, config_parser)
            return
        if cmd == 'stop':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Stopping of RDAF app container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to stop the app container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('app stop operation has been cancelled')
                    return
            component.stop(cmd_args, config_parser)
            return
        else:
            raise InvalidCmdUsageException()


class K8SApplicationCmdHandler(CliCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = True

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)
        tag_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the app components')

        choice_parser = argparse.ArgumentParser(add_help=False)
        choice_parser.add_argument('app_name',
                                   nargs='?',
                                   action='store',
                                   default='OIA',
                                   help='Select the APP to act on')
        choice_parser.add_argument('--service',
                                   dest="services",
                                   action='append',
                                   default=None,
                                   help='Restrict the scope of the command to specific service')

        app_commands_parser = parser.add_subparsers(dest='app_op', metavar='{}',
                                                    help='commands')

        app_commands_parser.add_parser('status', help='Status of the App')

        install_parser = app_commands_parser.add_parser('install',
                                                        parents=[tag_parser, choice_parser],
                                                        help='Install the App service containers')

        upgrade_parser = app_commands_parser.add_parser('upgrade',
                                                        parents=[tag_parser, choice_parser],
                                                        help='Upgrade the App service containers')

        update_config_parser = app_commands_parser.add_parser('update-config',
                                                              parents=[choice_parser],
                                                              help='Updated configurations of one or more components')

        app_up_parser = app_commands_parser.add_parser('up',
                                                        parents=[choice_parser],
                                                        help='Create the RDAF Application Containers')

        app_down_parser = app_commands_parser.add_parser('down',
                                                        parents=[choice_parser],
                                                        help='Delete the RDAF'
                                                            ' Application Containers')
        
        app_down_parser.add_argument('--force', action='store_true',
                                    help='Delete the RDAF Application Containers Forefully')
        app_down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')


    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.app_op
        if cmd == 'status':
            components = COMPONENT_REGISTRY.get_by_category(category='apps')
            return status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, components=components)

        component = rdaf.contextual.COMPONENT_REGISTRY.get(str(cmd_args.app_name).casefold())
        namespace = Component.get_namespace(config_parser)
        if cmd == 'install':
            component.k8s_pull_images(cmd_args, config_parser)
            component.k8s_install(cmd_args, config_parser)
            status_cmd = '\033[1;4m kubectl get pods -n {} -l app_name={} \033[0m'\
                .format(namespace, str(cmd_args.app_name).casefold())
            print('\n')
            logger.info('Please check ' + str(cmd_args.app_name).casefold() + ' pods status using - ' + status_cmd)
            return
        if cmd == 'upgrade':
            component.k8s_pull_images(cmd_args, config_parser)
            component.k8s_upgrade(cmd_args, config_parser)
            status_cmd = '\033[1;4m kubectl get pods -n {} -l app_name={} \033[0m' \
                .format(namespace, str(cmd_args.app_name).casefold())
            print('\n')
            logger.info('Please check ' + str(cmd_args.app_name).casefold() + ' pods status using - ' + status_cmd)
            return
        if cmd == 'update-config':
            component.update_k8s_config(config_parser)
            return
        if cmd == 'up':
            component.k8s_up(cmd_args, config_parser)
            return
        if cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF app container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the app container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('app down operation has been cancelled')
                    return
            component.k8s_down(cmd_args, config_parser)
            return
        else:
            raise InvalidCmdUsageException()

