import argparse
import configparser
import datetime
import logging
import os
import shutil
import subprocess

import rdaf.component.dockerregistry
import rdaf.component.keepalived
from rdaf import rdafutils
from rdaf.cmd import CliCmdHandler
from rdaf.contextual import COMPONENT_REGISTRY

logger = logging.getLogger(__name__)


class BackupCmdHandler(CliCmdHandler):

    def configure_parser(self, parser: argparse.ArgumentParser):
        parser.add_argument('--dest-dir',
                            dest="backup_dest_dir",
                            action='store',
                            required=True,
                            help='Directory into which the backup will be stored')
        parser.add_argument('--create-tar',
                            dest="create_tar",
                            action='store_true',
                            required=False,
                            default=False,
                            help='Creates a tar file for the backed up data')
        parser.add_argument('--service',
                            dest="services",
                            action='append',
                            default=None,
                            help='Backup only the specified components')

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        components = []
        components.extend(COMPONENT_REGISTRY.get_by_category(category='platform'))
        components.extend(COMPONENT_REGISTRY.get_by_category(category='infra'))
        components.extend(COMPONENT_REGISTRY.get_by_category(category='other'))
        ignored_components = [rdaf.component.dockerregistry.COMPONENT_NAME]

        component_names = ['config']
        for component in components:
            component_name = component.get_name()
            if component_name in ignored_components:
                continue
            component_names.append(component_name)

        specific_services = [] if cmd_args.services is None else cmd_args.services
        if len(specific_services) > 0:
            for entry in specific_services:
                if entry not in component_names:
                    rdafutils.cli_err_exit("Unknown component specified: " + entry)

        backup_dest_dir = os.path.abspath(cmd_args.backup_dest_dir)
        now = datetime.datetime.now()
        current_time = str(now.date()) + '-' + str(now.timestamp())
        if cmd_args.services is not None:
            service = '-'.join(cmd_args.services)
            current_run_name = service + '-backup-' + current_time
        else:
            current_run_name = 'rdaf-backup-' + current_time
        current_run_backup_root_dir = os.path.join(backup_dest_dir, current_run_name)
        os.makedirs(current_run_backup_root_dir)
        os.makedirs(os.path.join(current_run_backup_root_dir, 'cert'))
        os.makedirs(os.path.join(current_run_backup_root_dir, 'data'))
        os.makedirs(os.path.join(current_run_backup_root_dir, 'config'))
        os.makedirs(os.path.join(current_run_backup_root_dir, 'deployment-scripts'))

        # copy cert
        rdaf.component._copytree(os.path.join('/opt', 'rdaf', 'cert'),
                                 os.path.join(current_run_backup_root_dir, 'cert'))

        # copy the deployment-scripts
        rdaf.component._copytree(os.path.join('/opt', 'rdaf', 'deployment-scripts'),
                                 os.path.join(current_run_backup_root_dir, 'deployment-scripts'))

        # copy the rdaf.cfg
        cfg_file = os.path.expanduser(os.path.join('/opt/rdaf/', 'rdaf.cfg'))
        shutil.copyfile(os.path.abspath(cfg_file),
                        os.path.join(current_run_backup_root_dir, 'rdaf.cfg'))

        backup_state_recorder = configparser.ConfigParser(allow_no_value=True)
        backup_state_recorder.add_section('cli')
        backup_state_recorder.set('cli', 'version', rdafutils.get_cli_version())
        backup_state_recorder.add_section('components')

        if 'config' in specific_services:
            for component in components:
                component_name = component.get_name()
                if component_name in ignored_components:
                    continue

                logger.info('Backing up config of ' + component.get_name())
                component.backup_conf(cmd_args, config_parser, current_run_backup_root_dir)
                backup_state_recorder.set('components', component.get_name())
        else:
            # Now initiate config and data backup for each component
            for component in components:
                component_name = component.get_name()
                if component_name in ignored_components:
                    continue
                if len(specific_services) > 0 and component_name not in specific_services:
                    logger.debug('Skipping backup of ' + component_name)
                    continue
                logger.info('Backing up config of ' + component.get_name())
                component.backup_conf(cmd_args, config_parser, current_run_backup_root_dir)
                logger.info('Backing up data of ' + component.get_name())
                component.backup_data(cmd_args, config_parser, backup_state_recorder, current_run_backup_root_dir)

        with open(os.path.join(current_run_backup_root_dir, 'rdaf-backup.cfg'), 'w') as f:
            backup_state_recorder.write(f)

        # create a tar
        if cmd_args.create_tar:
            tar_name = current_run_name + '.tar.gz'
            backup_tar = os.path.join(current_run_backup_root_dir, tar_name)
            logger.info('Creating a tar archive at ' + backup_tar + ' for the backed up content')
            paths_to_tar = ['rdaf-backup.cfg', 'rdaf.cfg', 'config', 'cert', 'data',
                            'deployment-scripts']
            tar_command = 'touch ' + backup_tar \
                          + ' && sudo tar --exclude=' + tar_name \
                          + ' -czf ' + backup_tar \
                          + ' -C ' + current_run_backup_root_dir + ' ' + ' '.join(paths_to_tar)
            completed_process = subprocess.run([tar_command], shell=True, text=True,
                                               cwd=current_run_backup_root_dir)
            paths_to_remove = ['rdaf-backup.cfg', 'rdaf.cfg', 'config', 'cert', 'data',
                            'deployment-scripts']
            for path in paths_to_remove:
                src_path = os.path.join(current_run_backup_root_dir, path)
                if os.path.isfile(src_path):
                    os.remove(src_path)
                else:
                    shutil.rmtree(src_path, ignore_errors=True)
            if completed_process.returncode != 0:
                rdafutils.cli_err_exit('Failed to create backup archive')
                return
        logger.info('Backup of installation completed successfully to location '
                    + current_run_backup_root_dir)


class K8SBackupCmdHandler(CliCmdHandler):

    def configure_parser(self, parser: argparse.ArgumentParser):
        parser.add_argument('--dest-dir',
                            dest="backup_dest_dir",
                            action='store',
                            required=True,
                            help='Directory into which the backup will be stored')
        parser.add_argument('--create-tar',
                            dest="create_tar",
                            action='store_true',
                            required=False,
                            default=False,
                            help='Creates a tar file for the backed up data')
        parser.add_argument('--service',
                            dest="services",
                            action='append',
                            default=None,
                            help='Backup only the specified components')

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        components = []
        components.extend(COMPONENT_REGISTRY.get_by_category(category='infra'))
        components.extend(COMPONENT_REGISTRY.get_by_category(category='other'))
        ignored_components = [rdaf.component.dockerregistry.COMPONENT_NAME, rdaf.component.keepalived.COMPONENT_NAME]

        component_names = ['config']
        for component in components:
            component_name = component.get_name()
            if component_name in ignored_components:
                continue
            component_names.append(component_name)

        specific_services = [] if cmd_args.services is None else cmd_args.services
        if len(specific_services) > 0:
            for entry in specific_services:
                if entry not in component_names:
                    rdafutils.cli_err_exit("Unknown component specified: " + entry)
        
        backup_dest_dir = os.path.abspath(cmd_args.backup_dest_dir)
        now = datetime.datetime.now()
        current_time = str(now.date()) + '-' + str(now.timestamp())
        if cmd_args.services is not None:
            service = '-'.join(cmd_args.services)
            current_run_name = service + '-backup-' + current_time
        else:
            current_run_name = 'rdaf-backup-' + current_time
        current_run_backup_root_dir = os.path.join(backup_dest_dir, current_run_name)
        os.makedirs(current_run_backup_root_dir)
        os.makedirs(os.path.join(current_run_backup_root_dir, 'cert'))
        os.makedirs(os.path.join(current_run_backup_root_dir, 'data'))
        os.makedirs(os.path.join(current_run_backup_root_dir, 'config'))
        os.makedirs(os.path.join(current_run_backup_root_dir, 'deployment-scripts'))

        # copy cert
        rdaf.component._copytree(os.path.join('/opt', 'rdaf', 'cert'),
                                 os.path.join(current_run_backup_root_dir, 'cert'))

        # copy the deployment-scripts
        rdaf.component._copytree(os.path.join('/opt', 'rdaf', 'deployment-scripts'),
                                 os.path.join(current_run_backup_root_dir, 'deployment-scripts'))

        # copy the rdaf.cfg
        cfg_file = os.path.expanduser(os.path.join('/opt/rdaf/', 'rdaf.cfg'))
        shutil.copyfile(os.path.abspath(cfg_file),
                        os.path.join(current_run_backup_root_dir, 'rdaf.cfg'))

        backup_state_recorder = configparser.ConfigParser(allow_no_value=True)
        backup_state_recorder.add_section('cli')
        backup_state_recorder.set('cli', 'version', rdafutils.get_cli_version())
        backup_state_recorder.add_section('components')

        if 'config' in specific_services:
            for component in components:
                component_name = component.get_name()
                if component_name in ignored_components:
                    continue
                logger.info('Backing up config of ' + component.get_name())
                component.k8s_backup_conf(cmd_args, config_parser, current_run_backup_root_dir)
                backup_state_recorder.set('components', component.get_name())
        else:
            # Now initiate config and data backup for each component
            for component in components:
                component_name = component.get_name()
                if component_name in ignored_components:
                    continue
                if len(specific_services) > 0 and component_name not in specific_services:
                    logger.debug('Skipping backup of ' + component_name)
                    continue
                logger.info('Backing up config of ' + component.get_name())
                component.k8s_backup_conf(cmd_args, config_parser, current_run_backup_root_dir)
                logger.info('Backing up data of ' + component.get_name())
                component.k8s_backup_data(cmd_args, config_parser, backup_state_recorder, current_run_backup_root_dir)

        with open(os.path.join(current_run_backup_root_dir, 'rdaf-backup.cfg'), 'w') as f:
            backup_state_recorder.write(f)

        # create a tar
        if cmd_args.create_tar:
            tar_name = current_run_name + '.tar.gz'
            backup_tar = os.path.join(current_run_backup_root_dir, tar_name)
            logger.info('Creating a tar archive at ' + backup_tar + ' for the backed up content')
            paths_to_tar = ['rdaf-backup.cfg', 'rdaf.cfg', 'config', 'cert', 'data',
                            'deployment-scripts']
            tar_command = 'touch ' + backup_tar \
                          + ' && sudo tar --exclude=' + tar_name \
                          + ' -czf ' + backup_tar \
                          + ' -C ' + current_run_backup_root_dir + ' ' + ' '.join(paths_to_tar)
            completed_process = subprocess.run([tar_command], shell=True, text=True,
                                               cwd=current_run_backup_root_dir)
            paths_to_remove = ['rdaf-backup.cfg', 'rdaf.cfg', 'config', 'cert', 'data',
                               'deployment-scripts']
            for path in paths_to_remove:
                src_path = os.path.join(current_run_backup_root_dir, path)
                if os.path.isfile(src_path):
                    os.remove(src_path)
                else:
                    shutil.rmtree(src_path, ignore_errors=True)
            if completed_process.returncode != 0:
                rdafutils.cli_err_exit('Failed to create backup archive')
                return
        logger.info('Backup of installation completed successfully to location '
                    + current_run_backup_root_dir)
