import argparse
import configparser
import logging
import os
import shutil
import rdaf.contextual
import rdaf.component.platform
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.cmd import CliCmdHandler
from rdaf.component import Component
from rdaf.component.ssh import SSHKeyManager
from rdaf.rdafutils import cli_err_exit

logger = logging.getLogger(__name__)

class GeoDRCmdHandler(CliCmdHandler):

    def configure_parser(self, parser):
        service_parser = argparse.ArgumentParser(add_help=False)
        service_parser.add_argument('--service',
                                    dest="services",
                                    action='append',
                                    default=None,
                                    help='Restrict the scope of the command to a specific service')
        
        index_parser = argparse.ArgumentParser(add_help=False)
        index_parser.add_argument('--index',
                                    dest="index",
                                    action='append',
                                    default=None,
                                    help='Restrict the scope of the command to a specific index for opensearch service')
        
        geodr_commands_parser = parser.add_subparsers(dest='geodr_op', metavar='{}',
                                                      help='GeoDR commands')

        replication_start_parser = geodr_commands_parser.add_parser('start-replication',
                                                        parents=[service_parser, index_parser],
                                                        help='Start GeoDR replication')

        replication_prepare_parser = geodr_commands_parser.add_parser('prepare-replication',
                                                                    parents=[service_parser],
                                                                    help='Prepare GeoDR replication')
        replication_status_parser = geodr_commands_parser.add_parser('replication-status',
                                                        parents=[service_parser, index_parser],
                                                        help='GeoDR replication status')
        
        apply_metadata_parser = geodr_commands_parser.add_parser('apply-metadata',
                                                        parents=[service_parser],
                                                        help='Apply metadata for the service')

        configure_parser = geodr_commands_parser.add_parser('configure',
                                                        help='Start GeoDR replication')

        configure_parser.add_argument('--secondary-cfg',
                                      dest='secondary_cfg',
                                      action='store',
                                      default=None,
                                      help='Specify the path to the file where secondary rdaf.cfg file is present')

        configure_parser.add_argument('--ssh-password',
                                      dest="ssh_password",
                                      action='store',
                                      default=None,
                                      help='Password to use to do an initial SSH login '
                                           'while setting up password-less SSH between hosts')

        switch_primary_parser = geodr_commands_parser.add_parser('switch-primary',
                                                                    parents=[service_parser],
                                                                    help='Switch secondary setup as primary')

        replication_stop_parser = geodr_commands_parser.add_parser('stop-replication',
                                                                    parents=[service_parser, index_parser],
                                                                    help='Stop GeoDR replication')
        
        replication_stop_parser.add_argument('--delete_index',
                                         dest="delete_index",
                                         action='store_true',
                                         default=False,
                                         help='Delete index after stopping replication')
        
        replication_stop_parser.add_argument('--delete_rule',
                                         dest="delete_rule",
                                         action='store_true',
                                         default=False,
                                         help='Delete rule before stopping replication')
        
    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if cmd_args.geodr_op == 'configure':
            if not Component.is_geodr_deployment(config_parser):
                cli_err_exit("This is not a replication setup")
            logger.info("Configuring primary with secondary config")
            secondary_cfg = cmd_args.secondary_cfg
            destination_path = os.path.join('/opt', 'rdaf', 'rdaf-peer.cfg')
            shutil.copyfile(secondary_cfg, destination_path)
            logger.info(f"Copying the secondary setup cfg file to {destination_path}")
            peer_hosts = set()
            config = configparser.ConfigParser(allow_no_value=True)
            with open(destination_path, 'r') as f:
                config.read_file(f)
            for section in config.sections():
                if section == 'docker' or 'host' not in config[section]:  # Skip the docker section
                    continue
                host_entries = set(config.get(section, 'host').split(','))
                peer_hosts.update(host_entries)
            ssh_manager = COMPONENT_REGISTRY.require(rdaf.component.ssh.SSHKeyManager.COMPONENT_NAME)
            ssh_password = cmd_args.ssh_password
            for host in peer_hosts:
                ssh_manager.setup_keys_for_host(host, ssh_password)
            platform = COMPONENT_REGISTRY.require(rdaf.component.platform.COMPONENT_NAME)
            platform._generate_geodr_config(config_parser)
        elif cmd_args.geodr_op == 'prepare-replication':
            if not Component.is_geodr_deployment(config_parser):
                cli_err_exit("This is not a replication setup")
            destination_path = os.path.join('/opt', 'rdaf', 'rdaf-peer.cfg')
            if not os.path.exists(destination_path):
                cli_err_exit(f"{destination_path} is missing or peer config is not configured.")
            preferred_primary = config_parser.getboolean('rdaf-cli', 'primary', fallback=True)
            if not preferred_primary:
                cli_err_exit("Replication can only be prepared from primary instance")
            peer_configs = configparser.ConfigParser(allow_no_value=True)
            with open(destination_path, 'r') as f:
                peer_configs.read_file(f)

            if not cmd_args.services:
                logger.info("Use --service service-name to prepare component specific replication.")
            # TODO see if multiple services need to be allowed
            service = cmd_args.services[0]
            if service not in ['mariadb']:
                logger.info(f"service name {service} not a valid component name.. ['mariadb'] is allowed")
                return

            component = rdaf.contextual.COMPONENT_REGISTRY.get(service)
            logger.info(f"preparing replication for {service}")
            component.geodr_prepare_replication(cmd_args, config_parser, peer_configs)
        elif cmd_args.geodr_op == 'start-replication':
            if not Component.is_geodr_deployment(config_parser):
                cli_err_exit("This is not a replication setup")
            destination_path = os.path.join('/opt', 'rdaf', 'rdaf-peer.cfg')
            if not os.path.exists(destination_path):
                cli_err_exit(f"{destination_path} is missing or peer config is not configured.")
            preferred_primary = config_parser.getboolean('rdaf-cli', 'primary', fallback=True)
            if not preferred_primary:
                cli_err_exit("Replication can only be started from primary instance")
            peer_configs = configparser.ConfigParser(allow_no_value=True)
            with open(destination_path, 'r') as f:
                peer_configs.read_file(f)

            if not cmd_args.services:
                cli_err_exit("Use --service service-name to start component specific replication.")
            # TODO see if multiple services need to be allowed
            service = cmd_args.services[0]
            if service not in ['minio', 'mariadb', 'opensearch', 'graphdb', 'opensearch_external']:
                logger.info(f"service name {service} not a valid replication component name")
                return

            component = rdaf.contextual.COMPONENT_REGISTRY.get(service)
            logger.info(f"Starting replication for {service}")
            component.geodr_start_replication(cmd_args, config_parser, peer_configs)
        elif cmd_args.geodr_op == 'replication-status':
            if not Component.is_geodr_deployment(config_parser):
                cli_err_exit("This is not a replication setup")
            destination_path = os.path.join('/opt', 'rdaf', 'rdaf-peer.cfg')
            if not os.path.exists(destination_path):
                cli_err_exit(f"{destination_path} is missing or peer config is not configured.")
            preferred_primary = config_parser.getboolean('rdaf-cli', 'primary', fallback=True)
            if not preferred_primary:
                cli_err_exit("Replication status can only be obtained from the primary instance.")
            peer_configs = configparser.ConfigParser(allow_no_value=True)
            with open(destination_path, 'r') as f:
                peer_configs.read_file(f)
            if not cmd_args.services:
                cli_err_exit("Use --service service-name to start component specific replication.")
            # TODO see if multiple services need to be allowed
            service = cmd_args.services[0]
            if service not in ['minio', 'mariadb', 'opensearch', 'graphdb', 'opensearch_external']:
                logger.info(f"service name {service} not a valid replication component name")
                return

            component = rdaf.contextual.COMPONENT_REGISTRY.get(service)
            component.geodr_status(cmd_args, config_parser, peer_configs)
        elif cmd_args.geodr_op == 'switch-primary':
            if not Component.is_geodr_deployment(config_parser):
                cli_err_exit("This is not a replication setup")
            #TODO add checks here to see if it is a secondary setup etc..
            destination_path = os.path.join('/opt', 'rdaf', 'rdaf-peer.cfg')
            if not os.path.exists(destination_path):
                cli_err_exit(f"{destination_path} is missing or peer config is not configured.")
            peer_configs = configparser.ConfigParser(allow_no_value=True)
            with open(destination_path, 'r') as f:
                peer_configs.read_file(f)

            services = cmd_args.services if cmd_args.services else ['mariadb', 'opensearch', 'opensearch_external']

            for service in services:
                if service not in ['mariadb', 'opensearch', 'opensearch_external']:
                    logger.warning(f"{service} is not a valid component to swtich to primary.")
                    continue
                component = rdaf.contextual.COMPONENT_REGISTRY.get(service)
                component.switch_primary(cmd_args, config_parser, peer_configs)
        elif cmd_args.geodr_op == 'stop-replication':
            if not Component.is_geodr_deployment(config_parser):
                cli_err_exit("This is not a replication setup")
            #TODO add checks here to see if it is a secondary setup etc..
            destination_path = os.path.join('/opt', 'rdaf', 'rdaf-peer.cfg')
            if not os.path.exists(destination_path):
                cli_err_exit(f"{destination_path} is missing or peer config is not configured.")
            peer_configs = configparser.ConfigParser(allow_no_value=True)
            with open(destination_path, 'r') as f:
                peer_configs.read_file(f)

            services = cmd_args.services if cmd_args.services else ['minio', 'opensearch', 'opensearch_external']
            for service in services:
                if service not in ['minio', 'opensearch', 'opensearch_external']:
                    logger.warning(f"{service} is not a valid component to stop replication.")
                    continue
                component = rdaf.contextual.COMPONENT_REGISTRY.get(service)
                component.geodr_stop_replication(cmd_args, config_parser, peer_configs)
        elif cmd_args.geodr_op == 'apply-metadata':
            if not Component.is_geodr_deployment(config_parser):
                cli_err_exit("This is not a replication setup")
            destination_path = os.path.join('/opt', 'rdaf', 'rdaf-peer.cfg')
            if not os.path.exists(destination_path):
                cli_err_exit(f"{destination_path} is missing or peer config is not configured.")
            peer_configs = configparser.ConfigParser(allow_no_value=True)
            with open(destination_path, 'r') as f:
                peer_configs.read_file(f)
            if not cmd_args.services:
                cli_err_exit("Use --service service-name to apply metadata for the service.")
            # TODO see if multiple services need to be allowed
            service = cmd_args.services[0]
            if service not in ['opensearch', 'opensearch_external']:
                logger.info(f"service name {service} not a valid component name for applying metadata")
                return
            component = rdaf.contextual.COMPONENT_REGISTRY.get(service)
            component.geodr_apply_metadata(cmd_args, config_parser, peer_configs)


