import argparse
import configparser
import logging

from rdaf import InvalidCmdUsageException, rdafutils
from rdaf.cmd import CliCmdHandler, status
from rdaf.contextual import COMPONENT_REGISTRY

logger = logging.getLogger(__name__)


class LogMonitoringCmdHandler(CliCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = False

    def configure_parser(self, parser):
        log_monitoring_commands_parser = parser.add_subparsers(dest='log_monitoring_op', help='commands')

        upgrade_parser = log_monitoring_commands_parser.add_parser('upgrade',
                                                                   help='Upgrade log monitoring components')
        upgrade_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='Docker image tag')

        install_parser = log_monitoring_commands_parser.add_parser('install',
                                                                   help='Install log monitoring components')
        install_parser.add_argument('--log-monitoring-host',
                                    dest='log_monitoring_host',
                                    action='store',
                                    required=True,
                                    help='Log monitoring host')
        install_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='Log monitoring image tag')
        install_parser.add_argument('--no-prompt',
                                    dest='no_prompt',
                                    action='store_true',
                                    default=False,
                                    help='Don\'t prompt for inputs')

        log_monitoring_commands_parser.add_parser('status', help='Status of the RDAF log monitoring')

        up_parser = log_monitoring_commands_parser.add_parser('up',
                                                              help='Create the RDAF log monitoring Containers')

        down_parser = log_monitoring_commands_parser.add_parser('down',
                                                                help='Delete the RDAF log monitoring Containers')

        start_parser = log_monitoring_commands_parser.add_parser('start',
                                                                 help='Start the RDAF log monitoring Containers')

        stop_parser = log_monitoring_commands_parser.add_parser('stop',
                                                                help='Stop the RDAF log monitoring Containers')

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.log_monitoring_op
        component = COMPONENT_REGISTRY.get('rda_log_monitoring')
        if cmd == 'install':
            if self.k8s:
                component.k8s_pull_images(cmd_args, config_parser)
                component.k8s_install(cmd_args, config_parser)
            else:
                component.pull_images(cmd_args, config_parser)
                component.install(cmd_args, config_parser)
        elif cmd == 'upgrade':
            if self.k8s:
                component.k8s_pull_images(cmd_args, config_parser)
                component.k8s_upgrade(cmd_args, config_parser)
            else:
                component.pull_images(cmd_args, config_parser)
                component.upgrade(cmd_args, config_parser)
        elif cmd == 'down':
            if self.k8s:
                component.k8s_down(cmd_args, config_parser)
            else:
                component.down(cmd_args, config_parser)
        elif cmd == 'up':
            if self.k8s:
               component.k8s_up(cmd_args, config_parser)
            else: 
                component.up(cmd_args, config_parser)
        elif cmd == 'status':
            return status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, components=[component])
        elif cmd == 'stop':
            component.stop(cmd_args, config_parser)
        elif cmd == 'start':
            component.start(cmd_args, config_parser)
        else:
            raise InvalidCmdUsageException()


class K8SLogMonitoringCmdHandler(LogMonitoringCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = True

    def configure_parser(self, parser):
        log_monitoring_commands_parser = parser.add_subparsers(dest='log_monitoring_op', help='commands')

        upgrade_parser = log_monitoring_commands_parser.add_parser('upgrade',
                                                                   help='Upgrade log monitoring components')
        upgrade_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='Docker image tag')

        install_parser = log_monitoring_commands_parser.add_parser('install',
                                                                   help='Install log monitoring components')
        install_parser.add_argument('--log-monitoring-host',
                                    dest='log_monitoring_host',
                                    action='store',
                                    required=True,
                                    help='Log monitoring host')
        install_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='Log monitoring image tag')

        log_monitoring_commands_parser.add_parser('status', help='Status of the RDAF log monitoring')

        up_parser = log_monitoring_commands_parser.add_parser('up',
                                                              help='Create the RDAF log monitoring Containers')
        
        down_parser = log_monitoring_commands_parser.add_parser('down',
                                                                help='Delete the RDAF log monitoring Containers')
        
        down_parser.add_argument('--force', action='store_true',
                                help='Delete the RDAF log monitoring Containers Forefully')