import argparse
import configparser
import json
import logging
import sys
import time
from datetime import datetime

from rdaf import InvalidCmdUsageException
from rdaf import rdafutils
from rdaf.cmd import CliCmdHandler
from rdaf.component import Component
from rdaf.component import run_command, execute_command

logger = logging.getLogger(__name__)


class MaintenanceCmdHandler(CliCmdHandler):

    @staticmethod
    def start_maintenance(cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = Component.get_namespace(config_parser)
        terminating_pods = f'kubectl get pods -n {namespace} | grep Terminating'
        return_code, stdout, stderr = execute_command(terminating_pods)
        if return_code != 0:
            logger.info(f"No pods in terminating mode. Exiting")
            return

        k8d_pods_command = f"kubectl -n {namespace} get pods | grep Terminating | awk '{{print $1}}'"
        return_code, stdout, stderr = execute_command(k8d_pods_command)
        k8s_pods = set(stdout.split())
        logger.info("Following k8s pods are in terminating state")
        run_command(terminating_pods)
        logger.info(f"{len(k8s_pods)} k8s pods are in terminating mode")

        # Command to list RDAC pods in JSON format
        return_code, stdout, stderr = execute_command('rdac pods --json')
        sanitized_output = rdafutils.cleanup_stdout(stdout)
        rdac_pods = json.loads(sanitized_output)
        maint_pods = {}
        for pod in rdac_pods:
            # here hostname will be representing pod's unique short uuid
            if pod.get("hostname") in k8s_pods:
                maint_pods[pod.get("hostname")] = pod

        if len(k8s_pods) != len(maint_pods):
            logger.info(f"WARNING!!! following pods are not found in rdac")
            missing = k8s_pods - set(maint_pods.keys())
            print(missing)
            if not rdafutils.query_yes_no("Continue with missing pods?"):
               return

        table = [["Pod ID", "Pod Type", "Version", "Age", "Hostname", "Maintenance", "Pod Status"]]
        for pod in maint_pods.values():
            age = datetime.fromisoformat(pod.get("now")) - datetime.fromisoformat(pod.get("started_at"))
            age = str(age).rsplit(".", 1)[0]
            table.append([pod.get("pod_id", ""), pod.get("pod_type", ""), pod.get("build_tag", ""), str(age),
                          pod.get("hostname", ""), pod.get("is_maintenance"), pod.get("pod_ready")])

        column_headers = table[0]
        rows = table[1:]
        rdafutils.print_tabular(column_headers, rows)

        if not rdafutils.query_yes_no("Continue moving above pods to maintenance mode?"):
           return

        # Writing the RDAC pod information to a file
        with open("/tmp/terminating_pods.txt", "w") as f:
            for row in table:
                f.write('| {:15} | {:20} | {:10} | {:20} | {:60}\n'.format(*row))
        logger.debug(f"Wrote the rdac pods to be put in maintenance mode into file /tmp/terminating_pods.txt")

        # Prepare a comma-separated list of pod IDs for the maintenance command
        pod_ids = ",".join([pod.get("pod_id") for pod in maint_pods.values()])

        # Access the timeout value from cmd_args (default to 120)
        timeout = cmd_args.timeout
        maint_command = f"rdac maintenance start --ids {pod_ids}"
        logger.info('Initiating Maintenance Mode...')
        execute_command(maint_command)
        time.sleep(20)

        logger.info("Following pods are in maintenance mode")
        maint_pods_command = 'rdac pods --show_maintenance --json'
        return_code, stdout, stderr = execute_command(maint_pods_command)
        sanitized_output = rdafutils.cleanup_stdout(stdout)
        rdac_pods = json.loads(sanitized_output)
        maintenance_pods = {}
        for pod in rdac_pods:
            if pod.get("hostname") in k8s_pods:
                maintenance_pods[pod.get("hostname")] = pod

        if maintenance_pods:
            table = [["Pod ID", "Pod Type", "Version", "Age", "Hostname", "Maintenance", "Pod Status"]]
            for pod in maintenance_pods.values():
                  age = datetime.fromisoformat(pod.get("now")) - datetime.fromisoformat(pod.get("started_at"))
                  age = str(age).rsplit(".", 1)[0]
                  table.append([pod.get("pod_id", ""), pod.get("pod_type", ""), pod.get("build_tag", ""),
                                str(age), pod.get("hostname", ""), pod.get("is_maintenance"),
                                pod.get("pod_ready")])
            column_headers = table[0]
            rows = table[1:]
            rdafutils.print_tabular(column_headers, rows)
        logger.info(f"Waiting for timeout of {timeout} seconds...")
        time.sleep(timeout)

        if not rdafutils.query_yes_no("Do you want to initiate forceful deletion of above pods?"):
           return
        logger.info('Initiating forceful termination...')
        for pod_hostname in maintenance_pods.keys():
            terminate_command = f"kubectl delete pod {pod_hostname} -n {namespace} --force --grace-period=0"
            try:
                run_command(terminate_command)
            except Exception:
                continue

        logger.info("Pods deleted successfully.")
        scheduler_leader = False
        # First, check for the "scheduler" leader
        while not scheduler_leader:
            return_code, stdout, stderr = execute_command('rdac pods --json')
            sanitized_output = rdafutils.cleanup_stdout(stdout)
            rdac_pods = json.loads(sanitized_output)
            pod_types = [pod.get("pod_type") for pod in rdac_pods]
            if 'scheduler' not in pod_types:
                break
            for pod in rdac_pods:
                if pod.get("pod_leader") and pod.get("pod_type") == "scheduler":
                    scheduler_leader = True
                    logger.info("Scheduler leader elected for {} {}".format(pod.get("pod_type"), pod.get("hostname")))
                    break

            if not scheduler_leader:
                logger.info("No Scheduler leader elected. Waiting for leader election...")
                time.sleep(10)

        asset_ingester_leader = False
        # Now that we have the "scheduler" leader, check for "asset_ingester" leader
        while not asset_ingester_leader:
            return_code, stdout, stderr = execute_command('rdac pods --json')
            sanitized_output = rdafutils.cleanup_stdout(stdout)
            rdac_pods = json.loads(sanitized_output)
            pod_types = [pod.get("pod_type") for pod in rdac_pods]
            # if pod is not present we consider ignoring the leader election
            if 'cfxdimensions-app-assets-ingester' not in pod_types:
                asset_ingester_leader = True

            for pod in rdac_pods:
                if pod.get("pod_leader"):
                    if pod.get("pod_type") == "cfxdimensions-app-assets-ingester":
                        asset_ingester_leader = True
                        logger.info("Asset Ingester leader elected for {} {}".format(pod.get("pod_type"), pod.get("hostname")))

            if asset_ingester_leader:
                break
            if not asset_ingester_leader:
                logger.info("No asset ingester service leader elected. Waiting for leader election...")
            time.sleep(10)


    def configure_parser(self, parser):
        maintenance_commands_parser = parser.add_subparsers(dest='maintenance_op', metavar='{}',
                                                            help='Maintenance commands')
        start_parser = maintenance_commands_parser.add_parser('start', help='Start maintenance mode')

        # Add the --timeout option with a default value of 120
        start_parser.add_argument('--timeout', type=int, default=120, help='Timeout value for maintenance command (default: 120)')

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.maintenance_op
        if cmd == 'start':
            logger.info("Starting maintenance mode.")
            self.start_maintenance(cmd_args, config_parser)
        else:
            raise InvalidCmdUsageException()
