import argparse
import configparser
import json
import logging

import rdaf
import rdaf.component
from rdaf import InvalidCmdUsageException, rdafutils
from rdaf.cmd import CliCmdHandler, status
from rdaf.component import Component
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.component import dockerregistry, opensearch_external, ssh
from rdaf.rdafutils import query_yes_no, cli_err_exit

logger = logging.getLogger(__name__)


class opensearch_externalCmdHandler(CliCmdHandler):

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)

        opensearch_external_commands_parser = parser.add_subparsers(dest='opensearch_external_op', metavar='{}',
                                                       help='commands')

        setup_parser = opensearch_external_commands_parser.add_parser('setup',
                                                              parents=[tag_parser],
                                                              help='Setup Opensearch External')
        setup_parser.add_argument('--no-prompt',
                                  dest='no_prompt',
                                  action='store_true',
                                  default=False,
                                  help='Don\'t prompt for inputs')

        setup_parser.add_argument('--ssh-password',
                                  dest="ssh_password",
                                  action='store',
                                  default=None,
                                  help='Password to use to do an initial SSH login'
                                       ' while setting up password-less SSH between hosts')
        setup_parser.add_argument('--data-host',
                            dest='data_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the data hosts')
        setup_parser.add_argument('--cluster-manager-host',
                                    dest='cluster_manager_host',
                                    action='append',
                                    default=None,
                                    help='Host name or IP address of the cluster manger hosts')
        setup_parser.add_argument('--client-host',
                                    dest='client_host',
                                    action='append',
                                    default=None,
                                    help='Host name or IP address of the client hosts')
        setup_parser.add_argument('--os-zoning',
                                    dest='os_zoning',
                                    action='store_true',
                                    default=False,
                                    help='Enable cluster zoning for OpenSearch')
        add_opensearch_external_parser = opensearch_external_commands_parser.add_parser('add-opensearch-external-host',
                                                                   help='Add extra opensearch external vm')
        add_opensearch_external_parser.add_argument('host', help='Host name or IP address of the '
                                                         'host which needs to be '
                                                         'added as a opensearch external host')
        add_opensearch_external_parser.add_argument('--ssh-password', dest='ssh_password', required=True,
                                            action='store',
                                            help='Password for the SSH user')
        setup_parser.add_argument('--os-external-admin-user',
                            dest='os_external_admin_user',
                            action='store',
                            default=None,
                            help='Admin user name to be used for the cluster')
        setup_parser.add_argument('--os-external-admin-password',
                            dest='os_external_admin_password',
                            action='store',
                            default=None,
                            help='Admin user password to be used for the cluster')
        setup_parser.add_argument('--config-file',
                            dest='config_file',
                            action='store',
                            default=None,
                            help='Specify the path to the file where all the input configs are present')
        install_parser = opensearch_external_commands_parser.add_parser('install',
                                                              help='Install the RDAF opensearch_external '
                                                                   'containers')
        install_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the opensearch_external components')

        status_parser = opensearch_external_commands_parser.add_parser('status',
                                                          help='Status of the RDAF opensearch_external Component')

        upgrade_parser = opensearch_external_commands_parser.add_parser('upgrade',
                                                            help='Upgrade the RDAF opensearch_external '
                                                                 'Component')
        upgrade_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the opensearch_external components')
        start_parser = opensearch_external_commands_parser.add_parser('start',
                                                          help='Start the RDAF opensearch_external Component')

        stop_parser = opensearch_external_commands_parser.add_parser('stop',
                                                         help='Stop the RDAF opensearch_external Component')

        stop_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                        help='Don\'t prompt for inputs')

        down_parser = opensearch_external_commands_parser.add_parser('down',
                                                          help='Start the RDAF opensearch_external Component')

        down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                        help='Don\'t prompt for inputs')

        up_parser = opensearch_external_commands_parser.add_parser('up',
                                                         help='Stop the RDAF opensearch_external Component')
        
        purge_audit_logs = opensearch_external_commands_parser.add_parser(
                               'purge_audit_logs', help='OpenSearch to delete security audit log indices (default: 15 days)')

        purge_audit_logs.add_argument('--days', type=int, default=15, 
                               help='Number of days after which security audit log indices should be deleted (default: 15 days)')

        reset_parser = opensearch_external_commands_parser.add_parser('reset', help='Reset the Opensearch External Component')
        reset_parser.add_argument('--no-prompt',
                                  dest='no_prompt',
                                  action='store_true',
                                  default=False,
                                  help='Don\'t prompt for inputs')


    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.opensearch_external_op
        component = COMPONENT_REGISTRY.get('opensearch_external')
        if cmd == 'setup':
            # Handle config file if provided
            if cmd_args.config_file:
                self.populate_from_json(cmd_args.config_file, cmd_args)
            component.do_setup(cmd_args, config_parser)
        elif cmd == 'install':
            component.pull_images(cmd_args, config_parser)
            component.install(cmd_args, config_parser)
        elif cmd == 'upgrade':
            component.pull_images(cmd_args, config_parser)
            component.upgrade(cmd_args, config_parser)
        elif cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF opensearch_external container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the opensearch_external container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('opensearch_external down operation has been cancelled')
                    return
            component.down(cmd_args, config_parser)
        elif cmd == 'up':
            component.up(cmd_args, config_parser)
        elif cmd == 'status':
            return status.StatusCmdHandler().handle(cmd_args, config_parser, components=[component])
        elif cmd == 'add-opensearch-external-host':
            self.add_opensearch_external_host(cmd_args, config_parser)
        elif cmd == 'stop':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Stopping of RDAF opensearch_external container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to stop the opensearch_external container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('opensearch_external stop operation has been cancelled')
                    return
            component.stop(cmd_args, config_parser)
        elif cmd == 'start':
            component.start(cmd_args, config_parser)
        elif cmd == 'purge_audit_logs':
            component._apply_security_audit_policy(cmd_args, config_parser)
        elif cmd == 'reset':
            self.reset(cmd_args, config_parser)
        else:
            raise InvalidCmdUsageException()

    @staticmethod
    def populate_from_json(config_file, cmd_args):
        """Populate command arguments from JSON config file"""
        try:
            with open(config_file, 'r') as f:
                config = json.load(f)
            
            # Check if os-zoning is enabled and validate zones
            os_zoning = config.get('os-zoning', False)
            if os_zoning:
                if 'zones' not in config:
                    cli_err_exit("Zone configuration is required when 'os-zoning' is true. "
                               "Please provide 'zones' in the configuration file with format: "
                               '{"zones": {"zone-0": ["host1"], "zone-1": ["host2"]}}')
                
                zones = config.get('zones')
                if not zones or not isinstance(zones, dict):
                    cli_err_exit("Invalid zones configuration. Expected format: "
                               '{"zones": {"zone-0": ["host1"], "zone-1": ["host2"]}}')
                
                # Validate that zones contain host lists and store zone config
                for zone_name, hosts in zones.items():
                    if not isinstance(hosts, list) or not hosts:
                        cli_err_exit(f"Zone '{zone_name}' must contain a non-empty list of hosts")
                
                setattr(cmd_args, 'zones_config', zones)
            
            for key, value in config.items():
                # Skip zones as it's handled separately above
                if key == 'zones':
                    continue
                attr_key = key.replace('-', '_')
                if "host" in attr_key.lower() and isinstance(value, str):
                    # Convert delimited string to list for host attributes
                    value_list = rdafutils.delimited_to_list(value)
                    setattr(cmd_args, attr_key, value_list if value_list else value)
                else:
                    setattr(cmd_args, attr_key, value)
            # explicitly set noprompt true
            if not 'no_prompt' in config:
                setattr(cmd_args, 'no_prompt', True)

        except json.JSONDecodeError:
            cli_err_exit(f"The file '{config_file}' is not a valid JSON file.")
        except FileNotFoundError:
            cli_err_exit(f"Error: The file '{config_file}' was not found.")
        except Exception as e:
            cli_err_exit(f"An error occurred while loading the config: {str(e)}")

    @staticmethod
    def reset(cmd_args, config_parser):
        # ask for confirmation
        if not cmd_args.no_prompt:
            warn_message = rdafutils.center_text_on_terminal(
                'Do you want to delete all the Opensearch external components and data\n')
            print(warn_message)
            cancelled = True
            if query_yes_no("Are you sure you want to reset the Opensearch external?"):
                if query_yes_no("Please confirm again?"):
                    cancelled = False
            if cancelled:
                logger.info('Opensearch external reset has been cancelled')
                return

        component = COMPONENT_REGISTRY.get('opensearch_external')
        logger.info('Initiating reset on Opensearch external components')
        component.reset(cmd_args, config_parser, is_component=True)

    def opensearch_external_hosts(self, config_parser: configparser.ConfigParser):
        if not config_parser.has_option('os_external', 'host'):
            # default to platform host
            return [Component.get_default_host()]
        hosts = config_parser.get('os_external', 'host')
        return hosts.split(',')

    def add_opensearch_external_host(self, cmd_args: argparse.Namespace,
                        config_parser: configparser.ConfigParser):
        host = cmd_args.host
        rdafutils.validate_host_name(host)
        existing_opensearch_external_hosts = self.opensearch_external_hosts(config_parser)
        if host in existing_opensearch_external_hosts:
            rdafutils.cli_err_exit(host + ' is already configured to be a opensearch_external host')

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        if host not in all_known_hosts:
            ssh_manager = COMPONENT_REGISTRY.require(
                rdaf.component.ssh.SSHKeyManager.COMPONENT_NAME)
            # setup ssh keys for this new host
            ssh_manager.setup_keys_for_host(host, cmd_args.ssh_password)
        opensearch_external_comp = COMPONENT_REGISTRY.require(rdaf.component.opensearch_external.COMPONENT_NAME)
        opensearch_external_comp.add_opensearch_external_host(host, config_parser)
        # TODO create config data
        # finally, write out the configuration changes to rdaf.cfg
        Component.write_configs(config_parser)
        logger.info('Successfully added ' + host + ' as a new opensearch_external host')

class K8Sopensearch_externalCmdHandler(CliCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = True

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)

        opensearch_external_commands_parser = parser.add_subparsers(dest='opensearch_external_op', metavar='{}',
                                                       help='commands')

        setup_parser = opensearch_external_commands_parser.add_parser('setup',
                                                              parents=[tag_parser],
                                                              help='Setup Opensearch External')
        setup_parser.add_argument('--no-prompt',
                                  dest='no_prompt',
                                  action='store_true',
                                  default=False,
                                  help='Don\'t prompt for inputs')

        setup_parser.add_argument('--ssh-password',
                                  dest="ssh_password",
                                  action='store',
                                  default=None,
                                  help='Password to use to do an initial SSH login'
                                       ' while setting up password-less SSH between hosts')

        setup_parser.add_argument('--data-host',
                            dest='data_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the data hosts')
        setup_parser.add_argument('--cluster-manager-host',
                                    dest='cluster_manager_host',
                                    action='append',
                                    default=None,
                                    help='Host name or IP address of the cluster manger hosts')
        setup_parser.add_argument('--client-host',
                                    dest='client_host',
                                    action='append',
                                    default=None,
                                    help='Host name or IP address of the client hosts')
        setup_parser.add_argument('--os-zoning',
                                    dest='os_zoning',
                                    action='store_true',
                                    default=False,
                                    help='Enable cluster zoning for OpenSearch')
        setup_parser.add_argument('--os-external-admin-user',
                            dest='os_external_admin_user',
                            action='store',
                            default=None,
                            help='Admin user name to be used for the cluster')
        setup_parser.add_argument('--os-external-admin-password',
                            dest='os_external_admin_password',
                            action='store',
                            default=None,
                            help='Admin user password to be used for the cluster')
        setup_parser.add_argument('--config-file',
                            dest='config_file',
                            action='store',
                            default=None,
                            help='Specify the path to the file where all the input configs are present')
        install_parser = opensearch_external_commands_parser.add_parser('install',
                                                              help='Install the RDAF opensearch_external '
                                                                   'Containers')
        install_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the opensearch_external components')

        install_parser.add_argument('--no-prompt',
                                   dest='no_prompt',
                                   action='store_true',
                                   default=False,
                                   help='Don\'t prompt for inputs')
        
        upgrade_parser = opensearch_external_commands_parser.add_parser('upgrade',
                                                            help='Upgrade the RDAF opensearch_external '
                                                                 'Containers')
        upgrade_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the opensearch_external components')

        status_parser = opensearch_external_commands_parser.add_parser('status',
                                                          help='Status of the RDAF opensearch_external Containers')

        down_parser = opensearch_external_commands_parser.add_parser('down',
                                                          help='Start the RDAF opensearch_external Containers')

        down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                        help='Don\'t prompt for inputs')
        down_parser.add_argument('--force', action='store_true',
                                        help='Delete the RDAF Worker Containers Forefully')

        up_parser = opensearch_external_commands_parser.add_parser('up',
                                                         help='Stop the RDAF opensearch_external Containers')
        
        purge_audit_logs = opensearch_external_commands_parser.add_parser(
                               'purge_audit_logs', help='OpenSearch to delete security audit log indices (default: 15 days)')

        purge_audit_logs.add_argument('--days', type=int, default=15, 
                               help='Number of days after which security audit log indices should be deleted (default: 15 days)')
        
        reset_parser = opensearch_external_commands_parser.add_parser('reset', help='Reset the Opensearch External Component')
        reset_parser.add_argument('--no-prompt',
                                  dest='no_prompt',
                                  action='store_true',
                                  default=False,
                                  help='Don\'t prompt for inputs')
        
    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = Component.get_namespace(config_parser)
        cmd = cmd_args.opensearch_external_op
        status_cmd = '\033[1;4m kubectl get pods -n {} -l app_component=opensearch-external \033[0m'.format(namespace)
        component = COMPONENT_REGISTRY.get('opensearch_external')
        if cmd == 'setup':
            # Handle config file if provided
            if cmd_args.config_file:
                opensearch_externalCmdHandler.populate_from_json(cmd_args.config_file, cmd_args)
            component.do_k8s_setup(cmd_args, config_parser)
            return
        if cmd == 'install':
            component.k8s_pull_images(cmd_args, config_parser)
            component.k8s_install(cmd_args, config_parser)
            print('\n')
            logger.info('Please check openseach_external pods status using - ' + status_cmd)
            return
        if cmd == 'upgrade':
            component.k8s_pull_images(cmd_args, config_parser)
            component.k8s_upgrade(cmd_args, config_parser)
            print('\n')
            logger.info('Please check opensearch_external pods status using - ' + status_cmd)
            return
        if cmd == 'status':
            return status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, components=[component])
        if cmd == 'up':
            component.k8s_up(cmd_args, config_parser)
            return
        elif cmd == 'purge_audit_logs':
            component._apply_security_audit_policy(cmd_args, config_parser)
            return
        if cmd == 'reset':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF opensearch external container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to reset the opensearch external pod(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('opensearch external reset operation has been cancelled')
                    return
            component.k8s_reset(cmd_args, config_parser, is_component=True)
            return
        if cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF opensearch_external container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the opensearch_external container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('opensearch_external down operation has been cancelled')
                    return
            component.k8s_down(cmd_args, config_parser)
            return
        else:
            raise InvalidCmdUsageException()
