import argparse
import configparser
import logging
import rdaf
import rdaf.rdafutils as rdafutils
from rdaf import InvalidCmdUsageException
from rdaf.cmd import CliCmdHandler, status
from rdaf.component import Component
from rdaf.component import dockerregistry, pseudo_platform, ssh, cert
from rdaf.contextual import COMPONENT_REGISTRY

logger = logging.getLogger(__name__)


class PlatformCmdHandler(CliCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = False

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)
        tag_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the platform components')

        service_parser = argparse.ArgumentParser(add_help=False)
        service_parser.add_argument('--service',
                                    dest="services",
                                    action='append',
                                    default=None,
                                    help='Restrict the scope of the command to specific service')

        platform_commands_parser = parser.add_subparsers(dest='platform_op', metavar='{}',
                                                         help='commands')

        add_service_host_parser = platform_commands_parser.add_parser('add-service-host',
                                                                      help='Add extra service vm')
        add_service_host_parser.add_argument('host', help='Host name or IP address of the '
                                                          'host which needs to be '
                                                          'added as a service host')
        add_service_host_parser.add_argument('--ssh-password', dest='ssh_password', required=True,
                                             action='store',
                                             help='Password for the SSH user for initial'
                                                  ' login to this host')

        platform_commands_parser.add_parser('status',
                                            help='Status of the RDAF Platform')

        up_parser = platform_commands_parser.add_parser('up',
                                                           parents=[service_parser],
                                                           help='Create the RDAF Platform '
                                                                'Containers')

        down_parser = platform_commands_parser.add_parser('down',
                                                          parents=[service_parser],
                                                          help='Deleting the RDAF Platform '
                                                               'Containers')
        down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                 help='Don\'t prompt for inputs')

        install_parser = platform_commands_parser.add_parser('install',
                                                             parents=[tag_parser, service_parser],
                                                             help='Install the RDAF platform '
                                                                  'containers')

        upgrade_parser = platform_commands_parser.add_parser('upgrade',
                                                             parents=[tag_parser, service_parser],
                                                             help='Upgrade the RDAF platform '
                                                                  'containers')

        upgrade_parser.add_argument('--rolling-upgrade', dest='rolling', action='store_true',
                                    help='Performs a zero downtime upgrade')

        upgrade_parser.add_argument('--timeout', type=int, default=120,
                                    help='Timeout value for maintenance command (default: 120)')

        generate_certs_parser = platform_commands_parser.add_parser(
            'generate-certs',
            help='Generate certificates for hosts belonging to this installation')
        generate_certs_parser.add_argument('--dir',
                                           dest="cert_root_dir",
                                           required=False,
                                           action='store',
                                           help='The directory where the certificates will be '
                                                ' generated')
        generate_certs_parser.add_argument('--overwrite',
                                           dest="overwrite",
                                           required=False,
                                           action='store_true',
                                           help='Overwrite the certificates if they already exist')
        reset_admin_user_parser = platform_commands_parser.add_parser(
                                        'reset-admin-user',
                                        help='reset the password of user')
        start_parser = platform_commands_parser.add_parser('start',
                                                           parents=[service_parser],
                                                           help='Start the RDAF Platform '
                                                                'Containers')

        stop_parser = platform_commands_parser.add_parser('stop',
                                                          parents=[service_parser],
                                                          help='Stop the RDAF Platform '
                                                               'Containers')

        stop_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                 help='Don\'t prompt for inputs')
        
        ssh_keys_parser = platform_commands_parser.add_parser('update-ssh-keys',
                                 help='Update SSH keys for all hosts')
        
        ssh_keys_parser.add_argument('--ssh-password',
                                 dest='ssh_password',
                                 required=True,
                                 action='store',
                                 help='Password for the SSH user to update keys')

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = Component.get_namespace(config_parser)
        cmd = cmd_args.platform_op
        if cmd == 'install':
            components = COMPONENT_REGISTRY.get_by_category('platform')
            for component in components:
                if self.k8s:
                    component.k8s_pull_images(cmd_args, config_parser)
                    component.k8s_install(cmd_args, config_parser)
                else:
                    component.pull_images(cmd_args, config_parser)
                    component.install(cmd_args, config_parser)

            if self.k8s:
                status_cmd = '\033[1;4m kubectl get pods -n {} -l app_category=rdaf-platform \033[0m'.format(namespace)
                print('\n')
                logger.info('Please check platform pods status using - ' + status_cmd)
            else:
                status.StatusCmdHandler().handle(cmd_args, config_parser, category='platform')
            return
        if cmd == 'upgrade':
            components = COMPONENT_REGISTRY.get_by_category('platform')
            for component in components:
                if self.k8s:
                    component.k8s_pull_images(cmd_args, config_parser)
                    component.k8s_upgrade(cmd_args, config_parser)
                else:
                    component.pull_images(cmd_args, config_parser)
                    component.upgrade(cmd_args, config_parser)
            if self.k8s:
                status_cmd = '\033[1;4m kubectl get pods -n {} -l app_category=rdaf-platform \033[0m'.format(namespace)
                print('\n')
                logger.info('Please check platform pods status using - ' + status_cmd)
            else:
                status.StatusCmdHandler().handle(cmd_args, config_parser, category='platform')
            return
        if cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF platform container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the platform container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('platform down operation has been cancelled')
                    return
            self._down_platform_components(cmd_args, config_parser)
            return
        if cmd == 'up':
            platform_components = COMPONENT_REGISTRY.get_by_category('platform')
            for component in platform_components:
                if self.k8s:
                    component.k8s_up(cmd_args, config_parser)
                else:
                    component.up(cmd_args, config_parser)
                    status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, category='platform')
            return
        if cmd == 'stop':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Stopping of RDAF platform container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to stop the platform container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('platform stop operation has been cancelled')
                    return
            self._stop_platform_components(cmd_args, config_parser)
            return
        if cmd == 'start':
            platform_components = COMPONENT_REGISTRY.get_by_category('platform')
            for component in platform_components:
                component.start(cmd_args, config_parser)
            status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, category='platform')
            return
        if cmd == 'status':
            return status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, category='platform')
        if cmd == 'add-service-host':
            self.add_service_host(cmd_args, config_parser)
            return
        if cmd == 'generate-certs':
            cert_manager = COMPONENT_REGISTRY.require(
                rdaf.component.cert.CertManager.COMPONENT_NAME)
            cert_manager.generate_certs(cmd_args, config_parser, self.k8s)
            cert_manager.copy_certs()
            if self.k8s:
                components = rdaf.contextual.COMPONENT_REGISTRY.get_all_components()
                for component in components:
                    component.create_cert_configs(config_parser)
            return
        if cmd == 'reset-admin-user':
            platform_component = COMPONENT_REGISTRY.get('platform')
            platform_component.reset_admin_user(config_parser, self.k8s)
            return
        if cmd == 'update-ssh-keys':
            all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
            for host in all_known_hosts:
                ssh_manager = COMPONENT_REGISTRY.require(
                    rdaf.component.ssh.SSHKeyManager.COMPONENT_NAME)
                # setup ssh keys for this new host
                ssh_manager.setup_keys_for_host(host, cmd_args.ssh_password)
        else:
            raise InvalidCmdUsageException()

    def add_service_host(self, cmd_args: argparse.Namespace,
                         config_parser: configparser.ConfigParser):
        host = cmd_args.host
        rdafutils.validate_host_name(host)
        existing_service_hosts = Component.get_service_hosts(config_parser)
        if host in existing_service_hosts:
            rdafutils.cli_err_exit(host + ' is already configured to be a service host')
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        if host not in all_known_hosts:
            ssh_manager = COMPONENT_REGISTRY.require(
                rdaf.component.ssh.SSHKeyManager.COMPONENT_NAME)
            # setup ssh keys for this new host
            ssh_manager.setup_keys_for_host(host, cmd_args.ssh_password)
            docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
            docker_registry.docker_login(host, config_parser)

        pseudo_platform = COMPONENT_REGISTRY.require(
            rdaf.component.pseudo_platform.PseudoComponent.COMPONENT_NAME)
        pseudo_platform.add_service_host(host, config_parser)
        if host not in all_known_hosts:
            # copy certs
            cert_manager = COMPONENT_REGISTRY.require(
                rdaf.component.cert.CertManager.COMPONENT_NAME)
            cert_manager.copy_certs([host])

        # finally, write out the configuration changes to rdaf.cfg
        Component.write_configs(config_parser)
        logger.info('Successfully added ' + host + ' as a new service host')

    def _down_platform_components(self, cmd_args: argparse.Namespace,
                                  config_parser: configparser.ConfigParser):
        components = COMPONENT_REGISTRY.get_by_category('platform')
        for component in components:
            if self.k8s:
                component.k8s_down(cmd_args, config_parser)
            else:
                component.down(cmd_args, config_parser)

    def _stop_platform_components(self, cmd_args: argparse.Namespace,
                                  config_parser: configparser.ConfigParser):
        components = COMPONENT_REGISTRY.get_by_category('platform')
        for component in components:
            component.stop(cmd_args, config_parser)


class K8SPlatformCmdHandler(PlatformCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = True

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)
        tag_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the platform components')

        service_parser = argparse.ArgumentParser(add_help=False)
        service_parser.add_argument('--service',
                                    dest="services",
                                    action='append',
                                    default=None,
                                    help='Restrict the scope of the command to specific service')

        platform_commands_parser = parser.add_subparsers(dest='platform_op', metavar='{}',
                                                         help='commands')

        platform_commands_parser.add_parser('status',
                                            help='Status of the RDAF Platform')

        install_parser = platform_commands_parser.add_parser('install',
                                                             parents=[tag_parser, service_parser],
                                                             help='Install the RDAF platform '
                                                                  'containers')

        upgrade_parser = platform_commands_parser.add_parser('upgrade',
                                                             parents=[tag_parser, service_parser],
                                                             help='Upgrade the RDAF platform '
                                                                  'containers')

        platform_up_parser = platform_commands_parser.add_parser('up',
                                                            parents=[service_parser],
                                                            help='Create the RDAF Platform Containers')

        platform_down_parser = platform_commands_parser.add_parser('down',
                                                             parents=[service_parser],
                                                             help='Delete the RDAF Platform Containers')
        platform_down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                          help='Don\'t prompt for inputs')
        
        platform_down_parser.add_argument('--force', action='store_true',
                                          help='Delete the RDAF Platform Containers Forefully')

        generate_certs_parser = platform_commands_parser.add_parser(
            'generate-certs',
            help='Generate certificates for hosts belonging to this installation')
        generate_certs_parser.add_argument('--dir',
                                           dest="cert_root_dir",
                                           required=False,
                                           action='store',
                                           help='The directory where the certificates will be '
                                                ' generated')

        generate_certs_parser.add_argument('--overwrite',
                                           dest="overwrite",
                                           required=False,
                                           action='store_true',
                                           help='Overwrite the certificates if they already exist')

        reset_admin_user_parser = platform_commands_parser.add_parser('reset-admin-user',
                                                                      help='reset the password of user')
        
        ssh_keys_parser = platform_commands_parser.add_parser('update-ssh-keys',
                                 help='Update SSH keys for all hosts')
        
        ssh_keys_parser.add_argument('--ssh-password',
                                 dest='ssh_password',
                                 required=True,
                                 action='store',
                                 help='Password for the SSH user to update keys')

        