import argparse
import configparser
import logging

from rdaf import InvalidCmdUsageException
from rdaf.cmd import CliCmdHandler, status
from rdaf.contextual import COMPONENT_REGISTRY

logger = logging.getLogger(__name__)

class SelfMonitoringCmdHandler(CliCmdHandler):

    def __init__(self, k8s=False):
        super().__init__()
        self.k8s = k8s
        
    def configure_parser(self, parser):
        self_monitoring_commands_parser = parser.add_subparsers(dest='self_monitoring_op', help='commands')

        setup_parser = self_monitoring_commands_parser.add_parser('setup',
                                                                   help='setup self_monitoring component')
        
        install_parser = self_monitoring_commands_parser.add_parser('install',
                                                                   help='install the self_monitoring container')
        
        upgrade_parser = self_monitoring_commands_parser.add_parser('upgrade',
                                                                   help='upgrade the self_monitoring container')

        up_parser = self_monitoring_commands_parser.add_parser('up',
                                                                   help='create the self_monitoring container')

        down_parser = self_monitoring_commands_parser.add_parser('down',
                                                                   help='delete the self_monitoring container')

        start_parser = self_monitoring_commands_parser.add_parser('start',
                                                                   help='start the self_monitoring container')

        stop_parser = self_monitoring_commands_parser.add_parser('stop',
                                                                   help='stop the self_monitoring container')
        
        status_parser = self_monitoring_commands_parser.add_parser('status',
                                                          help='status of the self_monitoring container')

        install_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    help='self monitoring image tag')
        
        upgrade_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    help='self monitoring image tag')

        setup_parser.add_argument('--deployment-name',
                            dest='deployment_name',
                            action='store',
                            default=None,
                            help='deployment name')

        setup_parser.add_argument('--self-monitoring-host',
                            dest='self_monitoring_host',
                            action='store',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 'self_monitoring will be installed')

        setup_parser.add_argument('--no-prompt',
                                  dest='no_prompt',
                                  action='store_true',
                                  default=False,
                                  help='Don\'t prompt for inputs')

        setup_parser.add_argument('--portal-uname',
                                    dest='portal_uname',
                                    action='store',
                                    help='portal username of the self monitoring'
                                    )
        setup_parser.add_argument('--portal-password',
                                    dest='portal_password',
                                    action='store',
                                    help='portal password of the self monitoring'
                                    )
        setup_parser.add_argument('--post-to-slack',
                                    dest='post_to_slack',
                                    action='store_true',
                                    default=False,
                                    help='post message to slack'
                                    )
        setup_parser.add_argument('--post-to-webex',
                                    dest='post_to_webex',
                                    action='store_true',
                                    default=False,
                                    help='post message to webex'
                                    )
        setup_parser.add_argument('--post-to-email',
                                    dest='post_to_email',
                                    action='store_true',
                                    default=False,
                                    help='post message to email'
                                    )
        setup_parser.add_argument('--slack-token',
                                    dest='slack_token',
                                    action='store',
                                    help='token used to connect to slack'
                                    )

        setup_parser.add_argument('--slack-channel-name',
                                    dest='slack_channel_name',
                                    action='store',
                                    help='name of the slack channel to send messages'
                                    )

        setup_parser.add_argument('--slack-channel-id',
                                    dest='slack_channel_id',
                                    action='store',
                                    help='id of the slack channel'
                                    )

        setup_parser.add_argument('--slack-members-id',
                                    dest='slack_members_ids',
                                    action='store',
                                    help='id of the slack members'
                                    )

        setup_parser.add_argument('--slack-members-names',
                                    dest='slack_member_names',
                                    action='store',
                                    type=str,
                                    nargs='+',
                                    help='name of the slack members'
                                    )
        setup_parser.add_argument('--webex-token',
                                    dest='webex_token',
                                    action='store',
                                    help='token used to connect to webex'
                                    )
        setup_parser.add_argument('--webex-room-id',
                                    dest='webex_room_id',
                                    action='store',
                                    help='room_id used to connect to webex'
                                    )
        setup_parser.add_argument('--smtp-server',
                                    dest='smtp_server',
                                    action='store',
                                    help='SMTP server  used to send emails'
                                    )
        setup_parser.add_argument('--smtp-port',
                                    dest='smtp_port',
                                    action='store',
                                    help='SMTP port  used to send emails'
                                    )
        setup_parser.add_argument('--sender-email',
                                    dest='sender_email',
                                    action='store',
                                    help='Email address to send notifications from'
                                    )
        setup_parser.add_argument('--sender-password',
                                    dest='sender_password',
                                    action='store',
                                    help='Password for the sender email'
                                    )
        setup_parser.add_argument('--recipient-email',
                                    dest='recipient_emails',
                                    action='store',
                                    help='Comma-separated list of recipient emails'
                                    )

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.self_monitoring_op
        component = COMPONENT_REGISTRY.get('cfx_self_monitor')
        if cmd == 'setup':
            component.do_setup(cmd_args, config_parser)
        elif cmd == 'install':
            component.install(cmd_args, config_parser)
        elif cmd == 'upgrade':
            component.upgrade(cmd_args, config_parser)
        elif cmd == 'up':
            component.up(cmd_args, config_parser)
        elif cmd == 'down':
            component.down(cmd_args, config_parser)
        elif cmd == 'start':
            component.start(cmd_args, config_parser)
        elif cmd == 'stop':
            component.stop(cmd_args, config_parser)
        elif cmd == 'status':
            status.StatusCmdHandler().handle(cmd_args, config_parser, components=[component])
        else:
            raise InvalidCmdUsageException()

class K8sSelfMonitoringCmdHandler(SelfMonitoringCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = True
        
    