import argparse
import configparser
import json
import logging
import socket
from collections import OrderedDict

import termcolor
import getpass
from rdaf import rdafutils
from rdaf.component import execute_command
from rdaf.cmd import CliCmdHandler
import rdaf
import yaml
import os
import rdaf.component as comp
import paramiko
import distro
import rdaf.component.dockerregistry
from rdaf.contextual import COMPONENT_REGISTRY


logger = logging.getLogger(__name__)


class ValidateCmdHandler(CliCmdHandler):
  
    def configure_parser(self, parser):
        validate_commands_parser = parser.add_subparsers(dest='validate_op', metavar='{}',
                                                         help='commands')
        validate_commands_parser.add_parser('setup',
                                            help='Verifying the pre-requisites before setup')
        
        validate_commands_parser.add_parser('values-yaml',
                                            help='Validate the user inputs values.yaml file')

        validate_commands_parser.add_parser('configs',
                                            help='Validates setup level generic configurations')

    
    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.validate_op
        if cmd == 'values-yaml':
            self._validate_values_yaml(config_parser)
        elif cmd == 'configs':
            self._validate_configs(config_parser)
        elif cmd == 'setup':
            self.validate_setup(config_parser)

    @staticmethod
    def validate_setup(config_parser):
        results = []
        config_file_path = os.path.expanduser(os.path.join('/opt', 'rdaf', 'rdaf.cfg'))
        ssh_password = None
        ssh_key_file = None
        if not os.path.exists(config_file_path):
            infra_hosts, other_hosts, ssh_user, ssh_password = get_user_input()
            involved_hosts = set(infra_hosts + other_hosts)
        else:
            ssh_dir, ssh_user, ssh_key_file = fetch_ssh_creds(config_parser)
            involved_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
                          skip_components=[comp.dockerregistry.COMPONENT_NAME])
            infra_components = COMPONENT_REGISTRY.get_by_category('infra')
            component_hosts = []
            for component in infra_components:
                component_hosts += component.get_hosts()

            # Combine all infra hosts into one list
            infra_hosts = list(set(component_hosts))

        registry_host, registry_user, registry_password = get_docker_registry_details()

        # Check if Java is installed
        logger.info("Checking Java installation...")
        try:
           java_check_command = 'java -version'
           java_exit_code, java_out, java_err = execute_command(java_check_command)
           if java_exit_code == 0:
               logger.info("Java is installed")
           else:
               raise RuntimeError("Java not found")
        except Exception as e:
           logger.error(f"Error checking Java: {e}")
           return

        # Check if Keytool esists
        logger.info("Checking keytool...")
        try:
           keytool_check_command = 'command -v keytool'
           keytool_exit_code, keytool_out, keytool_err = execute_command(keytool_check_command)
           if keytool_exit_code == 0:
               logger.info("Keytool exists")
           else:
               raise RuntimeError("Keytool not found")
        except Exception as e:
           logger.error(f"Error checking Keytool: {e}")
           return
        
        for host in involved_hosts:
            ssh = None
            try:
                logger.info(f"Connecting to {host}...")
                ssh = paramiko.SSHClient()
                ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())
                ssh.load_system_host_keys()
                if not os.path.exists(config_file_path):
                    ssh.connect(host, username=ssh_user, password=ssh_password)
                else:
                    ssh.connect(host, username=ssh_user, key_filename=ssh_key_file)

                # SSH Connection Check
                logger.info(f"Checking SSH connectivity to {host}...")
                results.append([host, 'SSH_Connection_Check', 'OK'])

                # Sudo Access to ssh user
                logger.info(f"Checking sudo access for {ssh_user} on {host}...")
                sudo_access_command = f"sudo -l -U {ssh_user}"
                stdin, stdout, stderr = ssh.exec_command(sudo_access_command)
                sudo_access_exit_code = stdout.channel.recv_exit_status()
                results.append([host, 'Sudo_Access_Check', "OK" if sudo_access_exit_code == 0 else "Failed"])

                # Docker Installation Check
                with comp.Component.new_docker_client(socket.gethostname(), config_parser, timeout=60) as docker_client:
                    logger.info(f"Checking Docker Client Connectivity to {host}...")
                    docker_version = docker_client.getDockerVersion()
                    if docker_version is None:
                        raise ValueError("Error connecting Docker client..")
                    else:
                        results.append([host, 'Docker_Client', "OK"])

                # Docker-Compose Installation Check
                logger.info(f"Checking Docker Compose version on {host}...")
                docker_compose_version_command = 'docker-compose --version'
                stdin, stdout, stderr = ssh.exec_command(docker_compose_version_command)
                compose_exit_code = stdout.channel.recv_exit_status()
                results.append([host, 'Docker_Compose_Installation', "OK" if compose_exit_code == 0 else "Failed"])

                # Docker-ImagePull Check
                logger.info(f"Checking docker-login and Image-Pull on {host}...")
                command = 'docker login -u={user} -p={password} {host} && docker pull {host}/rda-platform-busybox:1.0.4' \
                    .format(user=registry_user, password=registry_password, host=registry_host)
                stdin, stdout, stderr = ssh.exec_command(command)
                pull_exit_code = stdout.channel.recv_exit_status()
                results.append([host, 'Docker_ImagePull', "OK" if pull_exit_code == 0 else "Failed"])

                # Delete Image Check
                logger.info(f"Removing Image on {host}...")
                delete_image_command = 'docker rmi {}/rda-platform-busybox:1.0.4'.format(registry_host)
                stdin, stdout, stderr = ssh.exec_command(delete_image_command)
                del_exit_code = stdout.channel.recv_exit_status()
                results.append([host, 'Image_Deletion', "OK" if del_exit_code == 0 else "Failed"])

                if host in infra_hosts:
                    # Checking Mount-Points of infra-components
                    paths_to_check = ['/minio-data', '/var/mysql', '/graphdb', '/opensearch', '/kafka-logs']
                    existing_paths = []

                    for path in paths_to_check:
                        try:
                            command = f'test -e {path} && echo "{path}" || echo ""'
                            stdin, stdout, stderr = ssh.exec_command(command)
                            output = stdout.read().decode('utf-8').strip()
                            if output:
                                existing_paths.append(output)
                            path_exit_code = stdout.channel.recv_exit_status()
                            if path_exit_code != 0:
                                logger.error(f"Error validating path")
                        except Exception as e:
                            logger.exception(f"Error validating path {path} on host {host}: {e}")
                    if existing_paths:
                        logger.info(f"Paths {', '.join(existing_paths)} exist on host {host}")
                    else:
                        logger.error(f"No paths found on host {host}")

                    results.append([host, 'Mount_Points', "OK" if path_exit_code == 0 else "Failed"])

                logger.info(f"Command execution on {host} completed.")
            except Exception as e:
                # Handle exceptions or log errors
                logger.error(f"Error executing command on {host}: {str(e)}")
                results.append([host, 'SSH_Connection', 'Failed'])
                results.append([host, 'Sudo_Access', 'Failed'])
                results.append([host, 'Docker_Installation', 'Failed'])
                results.append([host, 'Docker_Compose_Installation', 'Failed'])
                results.append([host, 'Docker_ImagePull', 'Failed'])
                results.append([host, 'Image_Deletion', 'Failed'])
                results.append([host, 'Mount_Points', 'Failed'])
            finally:
                if ssh is not None:
                    logger.info(f"Closing SSH connection to {host}.")
                    ssh.close()

        # Print the results table
        column_headers = ["Host", "Check Type", "Result"]
        rdafutils.print_tabular(column_headers, results)

    def _validate_configs(self, config_parser):
        # check ssh connection for all hosts.
        ssh_dir = os.path.expanduser("~/.ssh")
        ssh_key_file = os.path.join(ssh_dir, 'id_rsa.pub')
        ssh_user = config_parser.get('ssh-key-manager', 'ssh_user')
        ssh = paramiko.SSHClient()
        ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        ssh.load_system_host_keys()
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[comp.dockerregistry.COMPONENT_NAME])
        for host in all_known_hosts:
            if not comp.Component.is_local_host(host):
                try:
                    logger.info('checking connection for the host ' + host)
                    ssh.connect(host, username=ssh_user, key_filename=ssh_key_file)
                    logger.info('ssh check for host ' + host + ' successful')
                except Exception:
                    logger.exception('ssh config-check has failed on host' + host)

        # check docker is up
        docker_cmd = "docker version"
        for host in all_known_hosts:
            try:
                comp.execute_command_ssh(docker_cmd, host, config_parser)
                logger.info('Docker is installed on host ' + host)
            except Exception:
                logger.exception('Docker config-check has failed on host ' + host)

        # check docker-compose
        docker_compose_cmd = 'docker-compose --version'
        for host in all_known_hosts:
            try:
                comp.execute_command_ssh(docker_compose_cmd, host, config_parser)
                logger.info('Docker-compose is installed on host ' + host)
            except Exception:
                logger.exception('Docker-compose config-check has failed on host ' + host)

        self._ports_check(config_parser)
        self._duplicate_host_check()

    @staticmethod
    def _validate_values_yaml(config_parser):
        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if not os.path.exists(values_file):
            return
        with open(values_file) as f:
            try:
                data = yaml.safe_load(f)
            except yaml.YAMLError as e:
                logger.exception(e) 
                
        service_hosts = comp.Component.get_service_hosts(config_parser)
        oia_yaml = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        services = comp.Component.get_involved_services(oia_yaml)
        for service in (set(services)):
            if service not in data['services'] or 'hosts' not in data['services'][service]:
                continue
            if type(data['services'][service]) is not dict:
                logger.error("invalid yaml format,':' is missing for the hosts in a service {0}".format(service))
                continue
            if type(data['services'][service]['hosts']) is str:
                logger.error("Invalid yaml format,'-' is missing for the arguments in a service {0}".format(service))
                continue
            for host in data['services'][service]['hosts']:
                if host not in service_hosts:
                    logger.error("{0} specified for {1} is not a service host".format(host, service))
    
    @staticmethod
    def _ports_check(config_parser: configparser.ConfigParser):
        infra_components = COMPONENT_REGISTRY.get_by_category(category='infra')
        if distro.id() == 'ubuntu':
            ports_list_cmd = 'sudo ufw status'
        else:
            ports_list_cmd = 'sudo firewall-cmd --list-ports'
        for component in infra_components:
            port_host_tup = component.get_ports()
            if port_host_tup != ():
                for host in port_host_tup[0]:
                    output = comp.execute_command_ssh(ports_list_cmd, host, config_parser)
                    output1 = output[1].strip()
                    if 'not running' not in output1 and 'inactive' not in output1:
                        for port in port_host_tup[1]:
                            port_tcp = port + '/tcp'
                            if port_tcp in output1:
                                logger.info("port is open " + str(port) + " on host "
                                            + host + ' of component '
                                            + component.get_name())
                            else:
                                logger.info("\033[38;2;{};{};{}m{} \033[38;2;255;255;255m".format
                                            (255, 0, 0, "port is not open " + str(port)
                                             + " on host " + host
                                             + ' of component ' + component.get_name()))

    @staticmethod
    def check_ports_on_host(host, config_parser: configparser.ConfigParser, component_name=None):
        if component_name:
            infra_components = [COMPONENT_REGISTRY.get(component_name)]
        else:
            infra_components = COMPONENT_REGISTRY.get_by_category(category='infra')
        statuses = []
        if distro.id() == 'ubuntu':
            ports_list_cmd = 'sudo ufw status'
        else:
            ports_list_cmd = 'sudo firewall-cmd --list-ports'
        for component in infra_components:
            port_host_tup = component.get_ports()
            if port_host_tup != ():
                inaccessible_ports = []
                output = comp.execute_command_ssh(ports_list_cmd, host, config_parser)
                output1 = output[1].strip()
                if 'not running' not in output1 and 'inactive' not in output1:
                    for port in port_host_tup[1]:
                        port_tcp = port + '/tcp'
                        if port_tcp in output1:
                            logger.debug("port is open " + str(port) + " on host "
                                        + host + ' of component '
                                        + component.get_name())
                        else:
                            logger.debug("\033[38;2;{};{};{}m{} \033[38;2;255;255;255m".format
                                        (255, 0, 0, "port is not open " + str(port)
                                            + " on host " + host
                                            + ' of component ' + component.get_name()))
                            inaccessible_ports.append(str(port))
                
                if not inaccessible_ports:
                    statuses.append([component.get_name(),
                                    "Firewall Port",
                                    "OK",
                                    "N/A",
                                    host])
                else:
                    statuses.append([component.get_name(),
                                    "Firewall Port",
                                    termcolor.colored("Failed", color='red'),
                                    "{} ports are not open".format(','.join(inaccessible_ports)),
                                    host])
        return statuses
    
    @staticmethod
    def check_tcp(component_name, hostname, ports, timeout=3.0, is_cluster=False):
        '''
        @param hostname: Host name or IP Address of the target
        @param port: TCP Port number
        @param timeout: Timeout in seconds
        '''
        failed_ports = []
        if component_name == "haproxy":
            if not os.path.exists("/opt/rdaf/deployment-scripts/oia.yaml"):
                ports.remove("25")
        if component_name == "mariadb":
            if "4444" in ports:
                ports.remove("4444")
            if "4568" in ports:
                ports.remove("4568")
        for port in ports:
            try:
                sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
                sock.settimeout(timeout)
                result = sock.connect_ex((hostname, int(port)))
                if result != 0:
                    failed_ports.append(str(port))
            except Exception as e:
                failed_ports.append(str(port))
            finally:
                try:
                    sock.close()
                except:
                    pass
        if failed_ports:
            return [component_name,    
                     "Port Connection", 
                     termcolor.colored("Failed", color='red'),
                     "TCP Check Failed on ports {}".format(",".join(failed_ports)),
                     hostname]
        else:
            return [component_name,    
                     "Port Connection", 
                     "OK",
                     "N/A",
                     hostname]
    @staticmethod
    def check_all_ports_on_host(host, ports, config_parser: configparser.ConfigParser, component_name):
        statuses = []
        if distro.id() == 'ubuntu':
            ports_list_cmd = 'sudo ufw status'
        else:
            ports_list_cmd = 'sudo firewall-cmd --list-ports'
        
        inaccessible_ports = []
        output = comp.execute_command_ssh(ports_list_cmd, host, config_parser)
        output1 = output[1].strip()
        if 'not running' not in output1 and 'inactive' not in output1:
            for port in ports:
                port_tcp = str(port) + '/tcp'
                if port_tcp in output1:
                        logger.debug("port is open " + str(port) + " on host "
                                    + host + ' of component '
                                    + component_name)
                else:
                    logger.debug("\033[38;2;{};{};{}m{} \033[38;2;255;255;255m".format
                                (255, 0, 0, "port is not open " + str(port)
                                    + " on host " + host
                                    + ' of component ' + component_name))
                    inaccessible_ports.append(str(port))
        if not inaccessible_ports:
            statuses.append([component_name,
                            "Firewall Port",
                            "OK",
                            "N/A",
                            host])
        else:
            statuses.append([component_name,
                            "Firewall Port",
                            termcolor.colored("Failed", color='red'),
                            "{} ports are not open".format(','.join(inaccessible_ports)),
                            host])
        return statuses

    @staticmethod
    def _duplicate_host_check():
        infra_components = COMPONENT_REGISTRY.get_by_category(category='infra')
        for component in infra_components:
            comp_name = component.get_name()
            hosts = component.get_hosts()
            if len(hosts) != len(set(hosts)):
                logger.info("Duplicate hosts given for the component " + comp_name)


class K8sValidateCmdHandler(CliCmdHandler):

    def configure_parser(self, parser):
        validate_commands_parser = parser.add_subparsers(dest='validate_op', metavar='{}',
                                                         help='commands')

        validate_commands_parser.add_parser('setup',
                                            help='Verifying the pre-requisites for setup')

        validate_commands_parser.add_parser('configs',
                                            help='Validates setup level generic configurations')

        validate_commands_parser.add_parser('values-yaml',
                                            help='Validate the user inputs values.yaml file')

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.validate_op
        if cmd == 'values-yaml':
            self._validate_values_yaml(config_parser)
        elif cmd == 'setup':
            self.validate_setup(config_parser)

    @staticmethod
    def _validate_values_yaml(config_parser):
        # TODO - we should add validations to all known components
        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if not os.path.exists(values_file):
            return
        with open(values_file) as f:
            try:
                data = yaml.safe_load(f)
            except yaml.YAMLError as e:
                logger.exception(e)

        oia_yaml = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        services = comp.Component.get_involved_services(oia_yaml)
        for service in (set(services)):
            if service not in data['services'] or 'hosts' not in data['services'][service]:
                continue
            if type(data['services'][service]) is not dict:
                logger.error("invalid yaml format,':' is missing for the hosts in a service {0}".format(service))
                continue

    @staticmethod
    def validate_setup(config_parser):
        # Check if kubectl is installed
        results = []
        try:
            kubectl_version = f'kubectl version'
            execute_command(kubectl_version)
            logger.info("kubectl is installed.")
        except:
            logger.error("kubectl is not installed. Please install kubectl.")
            return

        # Fetch internal IPs of nodes 
        try:
            kubectl_get_nodes_cmd = f'kubectl get nodes -o wide'
            return_code, stdout, stderr = execute_command(kubectl_get_nodes_cmd)
            lines = stdout.split("\n")[1:]
            hosts = [line.split()[5] for line in lines if line.strip() and "master" not in line.lower()]
            node_status_dict = {line.split()[0]: line.split()[1] for line in lines if line.strip()}
            all_nodes_ready = all(status.lower() == "ready" for status in node_status_dict.values())

            if all_nodes_ready:
                logger.info("All master and worker nodes are in Ready state.")
            else:
                for node, status in node_status_dict.items():
                    node_type = "master" if "master" in node.lower() else "worker"
                    if status.lower() != "ready":
                        logger.error(f"{node_type.capitalize()} node {node} is not in Ready state.")
        except:
            logger.exception(f"Failed to run kubectl get nodes")
            return
        
        config_file_path = os.path.expanduser(os.path.join('/opt', 'rdaf', 'rdaf.cfg'))
        ssh_password = None
        ssh_key_file = None
        if not os.path.exists(config_file_path):
            infra_hosts, other_hosts, ssh_user, ssh_password = get_user_input()
            involved_hosts = list(set(infra_hosts + other_hosts))
        else:
            ssh_dir, ssh_user, ssh_key_file = fetch_ssh_creds(config_parser)
            involved_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
                          skip_components=[comp.dockerregistry.COMPONENT_NAME])
            infra_components = COMPONENT_REGISTRY.get_by_category('infra')
            component_hosts = []
            for component in infra_components:
                component_hosts += component.get_hosts()

            # Combine all infra hosts into one list
            infra_hosts = list(set(component_hosts))

        registry_host, registry_user, registry_password = get_docker_registry_details()

        # Check if Java is installed
        logger.info("Checking Java installation...")
        try:
           java_check_command = 'java -version'
           java_exit_code, java_out, java_err = execute_command(java_check_command)
           if java_exit_code == 0:
               logger.info("Java is installed")
           else:
               raise RuntimeError("Java not found")
        except Exception as e:
           logger.error(f"Error checking Java: {e}")
           return

        # Check if Keytool exists
        logger.info("Checking keytool...")
        try:
           keytool_check_command = 'command -v keytool'
           keytool_exit_code, keytool_out, keytool_err = execute_command(keytool_check_command)
           if keytool_exit_code == 0:
               logger.info("Keytool exists")
           else:
               raise RuntimeError("Keytool not found")
        except Exception as e:
           logger.error(f"Error checking Keytool: {e}")
           return

        for host in involved_hosts:
            ssh = None
            try:
                logger.info(f"Connecting to {host}...")
                ssh = paramiko.SSHClient()
                ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())
                ssh.load_system_host_keys()
                if not os.path.exists(config_file_path):
                    ssh.connect(host, username=ssh_user, password=ssh_password)
                else:
                    ssh.connect(host, username=ssh_user, key_filename=ssh_key_file)
                logger.info(f"Checking ssh connectivity to {host}...")

                # SSH Connection Check
                results.append([host, 'SSH_Connection_Check', 'OK'])
                # Sudo Access to Users
                logger.info(f"Checking sudo access for {ssh_user} on {host}...")
                sudo_access_command = f"sudo -l -U {ssh_user}"
                stdin, stdout, stderr = ssh.exec_command(sudo_access_command)
                sudo_access_exit_code = stdout.channel.recv_exit_status()
                results.append([host, 'Sudo_Access_Check', "OK" if sudo_access_exit_code == 0 else "Failed"])

                # Docker Installation Check
                with comp.Component.new_docker_client(socket.gethostname(), config_parser, timeout=60) as docker_client:
                    logger.info(f"Checking Docker Client connectivity to {host}...")
                    docker_version = docker_client.getDockerVersion()
                    if docker_version is None:
                        raise ValueError("Error connecting Docker client..")
                    else:
                        results.append([host, 'Docker_Client', "OK"])

                # Docker-Compose Installation Check
                logger.info(f"Checking Docker Compose version on {host}...")
                docker_compose_version_command = 'docker-compose --version'
                stdin, stdout, stderr = ssh.exec_command(docker_compose_version_command)
                compose_exit_code = stdout.channel.recv_exit_status()
                results.append([host, 'Docker_Compose_Installation', "OK" if compose_exit_code == 0 else "Failed"])

                # Docker-ImagePull Check
                logger.info(f"Checking docker-login and Image-Pull on {host}...")
                command = 'docker login -u={user} -p={password} {host} && docker pull {host}/rda-platform-busybox:1.0.4'\
                           .format(user=registry_user, password=registry_password, host=registry_host)
                stdin, stdout, stderr = ssh.exec_command(command)
                pull_exit_code = stdout.channel.recv_exit_status()
                results.append([host, 'Docker_ImagePull', "OK" if pull_exit_code == 0 else "Failed"])

                # Delete Image Check
                logger.info(f"Removing Image on {host}...")
                delete_image_command = 'docker rmi {}/rda-platform-busybox:1.0.4'.format(registry_host)
                stdin, stdout, stderr = ssh.exec_command(delete_image_command)
                del_exit_code = stdout.channel.recv_exit_status()
                results.append([host, 'Image_Deletion', "OK" if del_exit_code == 0 else "Failed"])


                if host in infra_hosts:
                    path_exit_code = -1
                    # Checking Mount-Points of infra-components
                    paths_to_check = ['/minio-data', '/var/mysql', '/graphdb', '/opensearch', '/kafka-logs']
                    existing_paths = []
                    missing_paths = []
                    for path in paths_to_check:
                        try:
                            command = f'test -e {path} && echo "{path}" || echo ""'
                            stdin, stdout, stderr = ssh.exec_command(command)
                            output = stdout.read().decode('utf-8').strip()
                            if output:
                                existing_paths.append(output)
                            else:
                                missing_paths.append(path)
                            path_exit_code = stdout.channel.recv_exit_status()
                            if path_exit_code != 0:
                                logger.error(f"Error validating path")
                        except Exception as e:
                            logger.exception(f"Error validating path {path} on host {host}: {e}")
                    if existing_paths:
                        logger.info(f"Paths {', '.join(existing_paths)} exist on host {host}")
                        if missing_paths:
                           logger.error(f"Missing paths on host {host}: {', '.join(missing_paths)}")
                           results.append([host, 'Mount_Points', "Failed"])
                        else:
                           results.append([host, 'Mount_Points', "OK"])
                    else:
                        logger.error(f"No paths found on host {host}")
                        results.append([host, 'Mount_Points', "OK" if path_exit_code == 0 else "Failed"])
    
                    logger.info(f"Command execution on {host} completed.")

            except Exception as e:
                # Handle exceptions or log errors
                logger.error(f"Error executing command on {host}: {str(e)}")
                results.append([host, 'SSH_Connection', 'Failed'])
                results.append([host, 'Sudo_Access', 'Failed'])
                results.append([host, 'Docker_Client', 'Failed'])
                results.append([host, 'Docker_Compose_Installation', 'Failed'])
                results.append([host, 'Docker_ImagePull', 'Failed'])
                results.append([host, 'Image_Deletion', 'Failed'])
                results.append([host, 'Mount_Points', 'Failed'])

            finally:
                if ssh is not None:
                    logger.info(f"Closing SSH connection to {host}.")
                    ssh.close()

        # Print the results table
        column_headers = ["Host", "Check Type", "Result"]
        rdafutils.print_tabular(column_headers, results)

def get_user_input():
    infra_hosts = input("Enter infra hosts: ").split(',')
    other_hosts = input("Enter other involved hosts: ").split(',')
    ssh_user = input("Enter SSH username: ")
    ssh_password = getpass.getpass("Enter SSH password: ")
    return infra_hosts, other_hosts, ssh_user, ssh_password

def fetch_ssh_creds(config_parser):
    ssh_dir = os.path.expanduser("~/.ssh")
    ssh_key_file = os.path.join(ssh_dir, 'id_rsa.pub')
    ssh_user = config_parser.get('ssh-key-manager', 'ssh_user')
    return ssh_dir, ssh_user, ssh_key_file

def get_docker_registry_details():
    from rdaf.component.dockerregistry import COMPONENT_NAME
    from rdaf.contextual import COMPONENT_REGISTRY
    docker_registry = COMPONENT_REGISTRY.require(COMPONENT_NAME)

    docker_configs = docker_registry.registry_config_as_json()
    if not docker_configs.get('username', None):
        docker_configs = dict()
        repository_templates_root = rdaf.get_repository_templates_dir_root()
        docker_repo_template = os.path.join(repository_templates_root, 'docker.repo')
        if not os.path.isfile(docker_repo_template):
            raise IOError(docker_repo_template + ' is either missing or is not a file')
        with open(docker_repo_template, 'r') as f:
            template = json.loads(f.read(), object_pairs_hook=OrderedDict)
        entries = {}
        for k, v in template.items():
            entries[k] = v
        docker_registry_template = template['CloudFabrix'][0]['docker-registry']
        for attr in docker_registry_template:
            docker_configs[attr] = docker_registry_template[attr]

    host = docker_configs['host']
    if docker_configs['port']:
        host += ':' + str(docker_configs['port'])
    if docker_configs.get('project', None):
        host += f"/{docker_configs.get('project')}"
    return host, docker_configs['username'], docker_configs['password']

