import argparse
import configparser
import logging

import rdaf
from rdaf import InvalidCmdUsageException, rdafutils
from rdaf.cmd import CliCmdHandler, status
from rdaf.component import Component
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.component import dockerregistry, worker, ssh

logger = logging.getLogger(__name__)


class WorkerCmdHandler(CliCmdHandler):

    def __init__(self, k8s=False):
        super().__init__()
        self.k8s = k8s

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)
        tag_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the worker components')

        worker_commands_parser = parser.add_subparsers(dest='worker_op', metavar='{}',
                                                       help='commands')

        add_worker_host_parser = worker_commands_parser.add_parser('add-worker-host',
                                                                   help='Add extra worker vm')
        add_worker_host_parser.add_argument('--site-group',
                                            dest = 'site_group',
                                            type = str,
                                            required = False,
                                            default='rda-site-01',
                                            help='Optional site-group name for the extra worker VM'
                                            )
        add_worker_host_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')
        
        add_worker_host_parser.add_argument('host', help='Host name or IP address of the ' 
                                                         'host which needs to be ' 
                                                         'added as a worker host')
        add_worker_host_parser.add_argument('--ssh-password', dest='ssh_password', required=True,
                                            action='store',
                                            help='Password for the SSH user for initial' 
                                                 ' login to this host')

        worker_commands_parser.add_parser('status', help='Status of the RDAF Worker')

        up_parser = worker_commands_parser.add_parser('up',
                                                         help='Create the RDAF Worker Containers')

        down_parser = worker_commands_parser.add_parser('down',
                                                        help='Delete the RDAF Worker Containers')
        
        down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')

        install_parser = worker_commands_parser.add_parser('install',
                                                             parents=[tag_parser],
                                                             help='Install the RDAF Worker '
                                                                  'containers')
        upgrade_parser = worker_commands_parser.add_parser('upgrade',
                                                           parents=[tag_parser],
                                                           help='Upgrade the RDAF Worker '
                                                                'containers')
        upgrade_parser.add_argument('--rolling-upgrade', dest='rolling', action='store_true',
                                    help='Performs a zero downtime upgrade')

        upgrade_parser.add_argument('--timeout', type=int, default=120,
                                    help='Timeout value for maintenance command (default: 120)')

        start_parser = worker_commands_parser.add_parser('start',
                                                         help='Start the RDAF Worker Containers')

        stop_parser = worker_commands_parser.add_parser('stop',
                                                        help='Stop the RDAF Worker Containers')
        
        stop_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')


    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.worker_op
        if cmd == 'install':
            component = COMPONENT_REGISTRY.get('rda_worker')
            component.pull_images(cmd_args, config_parser)
            component.install(cmd_args, config_parser)
            return
        if cmd == 'upgrade':
            component = COMPONENT_REGISTRY.get('rda_worker')
            component.pull_images(cmd_args, config_parser)
            component.upgrade(cmd_args, config_parser)
            return
        if cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF worker container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the worker container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('worker down operation has been cancelled')
                    return
            component = COMPONENT_REGISTRY.get('rda_worker')
            component.down(cmd_args, config_parser)
            return
        if cmd == 'up':
            component = COMPONENT_REGISTRY.get('rda_worker')
            component.up(cmd_args, config_parser)
            return
        if cmd == 'status':
            component = COMPONENT_REGISTRY.get('rda_worker')
            return status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, components=[component])
        if cmd == 'add-worker-host':
            self.add_worker_host(cmd_args, config_parser)
            return
        if cmd == 'stop':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Stopping of RDAF worker container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to stop the worker container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('worker stop operation has been cancelled')
                    return
            component = COMPONENT_REGISTRY.get('rda_worker')
            component.stop(cmd_args, config_parser)
            return
        if cmd == 'start':
            component = COMPONENT_REGISTRY.get('rda_worker')
            component.start(cmd_args, config_parser)
            return
        else:
            raise InvalidCmdUsageException()

    def add_worker_host(self, cmd_args: argparse.Namespace,
                        config_parser: configparser.ConfigParser):
        host = cmd_args.host
        rdafutils.validate_host_name(host)
        existing_worker_hosts = Component.get_worker_hosts(config_parser)
        if host in existing_worker_hosts:
            rdafutils.cli_err_exit(host + ' is already configured to be a worker host')
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        if host not in all_known_hosts:
            ssh_manager = COMPONENT_REGISTRY.require(
                rdaf.component.ssh.SSHKeyManager.COMPONENT_NAME)
            # setup ssh keys for this new host
            ssh_manager.setup_keys_for_host(host, cmd_args.ssh_password)
        worker_comp = COMPONENT_REGISTRY.require(rdaf.component.worker.COMPONENT_NAME)
        worker_comp.add_worker_host(cmd_args, host, config_parser)
        # TODO create config data
        # finally, write out the configuration changes to rdaf.cfg
        Component.write_configs(config_parser)
        logger.info('Successfully added ' + host + ' as a new worker host')

class K8SWorkerCmdHandler(CliCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = True

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)
        tag_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the worker components')

        worker_commands_parser = parser.add_subparsers(dest='worker_op', metavar='{}',
                                                       help='commands')

        worker_commands_parser.add_parser('status', help='Status of the RDAF Worker')

        install_parser = worker_commands_parser.add_parser('install',
                                                           parents=[tag_parser],
                                                           help='Install the RDAF Worker '
                                                                'containers')
        upgrade_parser = worker_commands_parser.add_parser('upgrade',
                                                           parents=[tag_parser],
                                                           help='Upgrade the RDAF Worker '
                                                                'containers')

        worker_up_parser = worker_commands_parser.add_parser('up',
                                                            help='Create the RDAF Worker Containers')

        worker_down_parser = worker_commands_parser.add_parser('down',
                                                             help='Delete the RDAF'
                                                                  ' Worker Containers')
        
        worker_down_parser.add_argument('--force', action='store_true',
                                        help='Delete the RDAF Worker Containers Forefully')
        
        worker_down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')


    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = Component.get_namespace(config_parser)
        cmd = cmd_args.worker_op
        status_cmd = '\033[1;4m kubectl get pods -n {} -l app_component=rda-worker \033[0m'.format(namespace)
        if cmd == 'install':
            component = COMPONENT_REGISTRY.get('rda_worker')
            component.k8s_pull_images(cmd_args, config_parser)
            component.k8s_install(cmd_args, config_parser)
            print('\n')
            logger.info('Please check worker pods status using - ' + status_cmd)
            return
        if cmd == 'upgrade':
            component = COMPONENT_REGISTRY.get('rda_worker')
            component.k8s_pull_images(cmd_args, config_parser)
            component.k8s_upgrade(cmd_args, config_parser)
            print('\n')
            logger.info('Please check worker pods status using - ' + status_cmd)
            return
        if cmd == 'status':
            component = COMPONENT_REGISTRY.get('rda_worker')
            return status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, components=[component])
        if cmd == 'up':
            component = COMPONENT_REGISTRY.get('rda_worker')
            component.k8s_up(cmd_args, config_parser)
            return
        if cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF worker container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the worker container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('worker down operation has been cancelled')
                    return
            component = COMPONENT_REGISTRY.get('rda_worker')
            component.k8s_down(cmd_args, config_parser)
            return
        else:
            raise InvalidCmdUsageException()

