import argparse
import configparser
import datetime
import json
import logging
import os
import socket
import string
from typing import Callable, Any, List

import yaml

import rdaf
from rdaf.component import Component, OtherCategoryOrder, _comma_delimited_to_list, _list_to_comma_delimited, \
    run_potential_ssh_command, execute_command_ssh, do_potential_scp, run_command, execute_command
from rdaf.rdafutils import cli_err_exit

COMPONENT_NAME = 'rda_event_gateway'
logger = logging.getLogger(__name__)


class EventGateway(Component):
    _option_host = 'host'

    def __init__(self):
        super().__init__(COMPONENT_NAME, COMPONENT_NAME, category='other',
                         category_order=OtherCategoryOrder.EVENT_GATEWAY.value)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host:
            return _comma_delimited_to_list
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host:
            return _list_to_comma_delimited
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_host] = []
        return default_configs

    def gather_minimal_setup_inputs(self, cmd_args, config_parser):
        event_gateway_configs = self._init_default_configs()
        event_gateway_configs[self._option_host] = [self.get_default_host()]
        self._mark_configured(event_gateway_configs, config_parser)

    def gather_setup_inputs(self, cmd_args: argparse.Namespace,
                            config_parser: configparser.ConfigParser):
        event_gateway_configs = self._init_default_configs()
        default_host_name = Component.get_default_host()
        err_msg = 'No Event Gateway host specified. Use --event-gateway-host ' \
                  + 'to specify one'
        desc = 'What are the host(s) on which you want the Event Gateway to be installed?'
        event_gateway_hosts = self._parse_or_prompt_hosts(cmd_args.rda_event_gateway_host,
                                                          default_host_name,
                                                          err_msg, desc,
                                                          'Event Gateway host(s)',
                                                          cmd_args.no_prompt)

        event_gateway_configs[self._option_host] = event_gateway_hosts
        self._mark_configured(event_gateway_configs, config_parser)
    
    def do_setup(self, cmd_args, config_parser):
        trap_conf = os.path.join(rdaf.get_templates_dir_root(), 'trap_to_alert_go.yaml')
        with open(trap_conf, 'r') as f:
            template_content = f.read()
        for host in self.get_hosts():
            dest_location = os.path.join(os.path.join(self.get_conf_dir(), 'config'),
                                      'snmptrap', 'trap_to_alert_go.yaml')
            created_location = rdaf.component.create_file(host, template_content.encode(encoding='UTF-8'), dest_location)
            logger.info('Created snmp trap alert configuration at ' + created_location + ' on host ' + host)

    def get_k8s_component_name(self):
        return 'rda-event-gateway'
    
    def get_hosts(self) -> list:
        return self.configs[self._option_host]

    def get_deployment_file_name(self) -> os.path:
        return 'event-gateway.yaml'

    def get_deployment_file_path(self) -> os.path:
        return os.path.join('/opt', 'rdaf', 'deployment-scripts', self.get_deployment_file_name())

    def get_conf_dir(self) -> os.path:
        return os.path.join(self.get_rdaf_install_root(), 'event_gateway')

    def get_deployment_replacements(self, cmd_args: argparse.Namespace) -> dict:
        replacements = self._get_docker_repo()
        replacements['TAG'] = cmd_args.tag
        return replacements

    def create_compose_file_event_gateway(self, cmd_args: argparse.Namespace):
        compose_file = self.get_deployment_file_path()
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(),
                                 os.path.basename(self.get_deployment_file_path()))
        replacements = self.get_deployment_replacements(cmd_args)
        
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        original_values = yaml.safe_load(original_content)
        self.apply_user_inputs('rda_event_gateway', original_values['services']['rda_event_gateway'])
        with open(compose_file, 'w') as f:
            yaml.safe_dump(original_values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        for host in self.get_hosts():
            if not Component.is_local_host(host):
                do_potential_scp(host, compose_file, compose_file)

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        network_config = os.path.join(os.path.join(self.get_rdaf_install_root(), 'config'),
                                      'network_config', 'config.json')
        if not os.path.exists(network_config):
            cli_err_exit("Please install platform before installing Event Gateway...")

        for host in self.get_hosts():
            if not Component.is_local_host(host):
                logger.info('Creating directory ' + os.path.dirname(network_config) + ' on host ' + host)
                do_potential_scp(host, network_config, network_config)

        self.create_compose_file_event_gateway(cmd_args)
        deployment_file = self.get_deployment_file_path()
        command = '/usr/local/bin/docker-compose -f {file} up -d '.format(file=deployment_file)
        for host in self.get_hosts():
            logger.info("Installing event_gateway on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.create_compose_file_event_gateway(cmd_args)
        deployment_file = self.get_deployment_file_path()
        command = '/usr/local/bin/docker-compose -f {file} up -d '.format(file=deployment_file)
        for host in self.get_hosts():
            logger.info("Upgrading event_gateway on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        command = '/usr/local/bin/docker-compose -f ' + deployment_file + ' up -d '
        for host in self.get_hosts():
            logger.info("Creating event_gateway on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        for host in reversed(self.get_hosts()):
            logger.info("Deleting event_gateway on host {}".format(host))
            command = '/usr/local/bin/docker-compose -f ' + deployment_file + ' rm -fs rda_event_gateway'
            run_potential_ssh_command(host, command, config_parser)

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        command = '/usr/local/bin/docker-compose -f ' + deployment_file + ' start '
        for host in self.get_hosts():
            logger.info("Starting event_gateway on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        for host in reversed(self.get_hosts()):
            logger.info("stopping event_gateway on host {}".format(host))
            command = '/usr/local/bin/docker-compose -f ' + deployment_file + ' stop rda_event_gateway'
            run_potential_ssh_command(host, command, config_parser)

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        statuses = []
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return statuses
        for host in self.get_hosts():
            component_status = dict()
            statuses.append(component_status)
            component_status['component_name'] = self.get_name()
            component_status['host'] = host
            try:
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = self.find_component_container_on_host(docker_client, all_states=True)
                    if len(containers) == 0:
                        logger.debug(
                            'No container found for ' + self.get_name() + ' on host ' + host)
                        component_status['containers'] = []
                    else:
                        component_status['containers'] = containers
            except Exception:
                logger.debug('Failed to get status of ' + self.get_name() + ' on host ' + host,
                             exc_info=1)
                # set the status as error
                component_status['status'] = {'error': True, 'message': 'Unknown'}

        return statuses

    def get_k8s_install_args(self, cmd_args):
        args = '--set tag={} '.format(cmd_args.tag)
        # adding the timestamp for daily tags which will work for upgrade scenarios to pull the
        # latest image, with a change to the chart templates.
        if cmd_args.tag == 'daily':
            timestamp = datetime.datetime.now().strftime("%Y-%m-%d-%H:%M:%S")
            args += ' --set {service}.extraEnvs[0].name=timestamp,{service}.extraEnvs[0].value={value}' \
                .format(service='rda_event_gateway', value=str(timestamp))
        return args
    
    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != "k8s":
            return
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        image = 'ubuntu-rda-event-gateway'
        for host in self.get_hosts():
            logger.info(f"Pulling image {image} for service event gateway on host {host}")
            docker_pull_command = f'docker pull {docker_repo}/{image}:{cmd_args.tag}'
            run_potential_ssh_command(host, docker_pull_command, config_parser)

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        install_command = 'helm install --create-namespace -n {} -f {} {} {} {} '\
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(install_command)

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        upgrade_command = 'helm upgrade --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(upgrade_command)

    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
        replicas = data.get('rda_event_gateway', {}).get('replicas')
        logger.info("Creating service: {} ".format(self.get_k8s_component_name()))
        command = 'kubectl scale deployment.apps/{} -n {} --replicas={}'\
            .format(self.get_k8s_component_name(), namespace, replicas)
        run_command(command)
    
    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        logger.info("Deleting service: {} ".format(self.get_k8s_component_name()))
        command = 'kubectl scale deployment.apps/{} -n {} --replicas=0'.format(self.get_k8s_component_name(), namespace)
        run_command(command)
        if cmd_args.force:
                delete_pods_command = f'kubectl delete pods -n {namespace} -l app_component=rda-event-gateway --grace-period=0 --force'
                run_command(delete_pods_command)

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        namespace = self.get_namespace(config_parser)
        statuses = []
        status_command = 'kubectl get pods -n {} -l app_component={} -o json'\
            .format(namespace, self.get_k8s_component_name())
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of event-gateway, due to: {}.".format(str(stderr)))
        result = json.loads(str(stdout))
        items = result['items']
        hosts = self.get_hosts()
        for host in hosts:
            deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
            if not items and os.path.exists(deployment_path):
                    statuses.append({
                        'component_name': self.get_k8s_component_name(),
                        'host': host,
                        'containers': [{
                            'Id': 'N/A',
                            'Image': 'N/A',
                            'State': 'N/A',
                            'Status': 'Not Provisioned'
                        }]
                    })
        else:
            for item in items:
                pod = dict()
                statuses.append(pod)
                pod['component_name'] = self.get_k8s_component_name()
                pod['host'] = item['status'].get('hostIP', 'Unknown')
                pod['containers'] = []
                if 'containerStatuses' in item['status']:
                    for entry in item['status']['containerStatuses']:
                        if 'containerID' in entry.keys():
                            container = dict()
                            container['Id'] = entry['containerID']
                            container['Image'] = entry['image']
                            for key in entry['state'].keys():
                                container['State'] = key
                                if key == 'running':
                                    container['Status'] = self.get_container_age(entry['state'][key]['startedAt'])
                                else:
                                    container['Status'] = key
                                break
                            pod['containers'].append(container)
            return statuses

