import argparse
import configparser
import logging
import os
import json
import socket
import string
from typing import Callable, Any, List
import termcolor
import yaml
import time
import rdaf.component.cert
import rdaf.component.mariadb
import rdaf.component.pseudo_platform
from rdaf import get_templates_dir_root
from rdaf import rdafutils
from rdaf.component import Component, InfraCategoryOrder, create_file, run_potential_ssh_command, run_command, execute_command, cli_err_exit
from rdaf.component import _comma_delimited_to_list, do_potential_scp, run_command_exitcode
from rdaf.component import _list_to_comma_delimited
from rdaf.component.minio import COMPONENT_NAME as MINIO_COMPONENT
from rdaf.contextual import COMPONENT_REGISTRY

COMPONENT_NAME = 'haproxy'
logger = logging.getLogger(__name__)
_docker_image_name = 'haproxy'


class HAProxy(Component):
    _option_host = 'host'
    _option_advertised_ext_host = 'advertised_external_host'
    _option_advertised_ext_interface = 'advertised_external_interface'
    _option_advertised_int_host = 'advertised_internal_host'
    _option_advertised_int_interface = 'advertised_internal_interface'

    _haproxy_stats_port = 7222

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'haproxy', 'infra', InfraCategoryOrder.HAPROXY.value)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host:
            return _comma_delimited_to_list
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host:
            return _list_to_comma_delimited
        return None

    def get_ports(self) -> tuple:
        ports = ['7222', '9443', '3307', '80', '443', '25', '8808']
        hosts = self.get_hosts()
        return hosts, ports
    
    def get_k8s_component_name(self):
        return 'rda-haproxy'
    
    def get_k8s_component_label(self):
        return 'app_component={}'.format(self.get_k8s_component_name())

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_host] = None
        default_configs[self._option_advertised_ext_host] = None
        default_configs[self._option_advertised_ext_interface] = None
        default_configs[self._option_advertised_int_host] = None
        default_configs[self._option_advertised_int_interface] = None
        return default_configs

    def gather_minimal_setup_inputs(self, cmd_args, config_parser):
        haproxy_configs = self._init_default_configs()
        haproxy_configs[self._option_host] = [self.get_default_host()]
        self._mark_configured(haproxy_configs, config_parser)

    def gather_k8s_setup_inputs(self, cmd_args, config_parser):
        haproxy_configs = self._init_default_configs()
        default_host = Component.get_default_host()
        no_prompt_err_msg = 'No HAProxy host specified.'
        host_desc = 'What is the host on which you want HAProxy to be provisioned?'
        haproxy_configs[self._option_host] = Component._parse_or_prompt_hosts(
            cmd_args.haproxy_host,
            default_host,
            no_prompt_err_msg, host_desc,
            'HAProxy host',
            cmd_args.no_prompt)

        adv_configs = self._gather_advertised_hosts_interfaces(
            haproxy_configs[self._option_host],
            cmd_args, config_parser)
        if adv_configs is not None:
            haproxy_configs.update(adv_configs)
        self._mark_configured(haproxy_configs, config_parser)

    def gather_setup_inputs(self, cmd_args, config_parser):
        haproxy_configs = self._init_default_configs()
        default_host = Component.get_default_host()
        no_prompt_err_msg = 'No HAProxy host specified.'
        host_desc = 'What is the host on which you want HAProxy to be provisioned?'
        haproxy_configs[self._option_host] = Component._parse_or_prompt_hosts(
            cmd_args.haproxy_host,
            default_host,
            no_prompt_err_msg, host_desc,
            'HAProxy host',
            cmd_args.no_prompt)
        # we gather advertised hosts and interfaces only if there are more than one
        # haproxy hosts configured. otherwise, we just use the single haproxy host
        # as the advertised host

        if len(haproxy_configs[self._option_host]) > 1:
            adv_configs = self._gather_advertised_hosts_interfaces(
                haproxy_configs[self._option_host],
                cmd_args, config_parser)
            if adv_configs is not None:
                haproxy_configs.update(adv_configs)
        else:
            # throw an error if advertised hosts are configured and only one haproxy host
            # is configured. we don't support advertised hosts when haproxy host is just one
            if cmd_args.advertised_int_host:
                rdafutils.cli_err_exit('Advertised internal host '
                                       + cmd_args.advertised_int_host + ' must not be set'
                                       + ' when only one haproxy host '
                                       + str(haproxy_configs[self._option_host])
                                       + ' is present')
            if cmd_args.advertised_int_interface:
                rdafutils.cli_err_exit('Advertised internal interface '
                                       + cmd_args.advertised_int_interface + ' must not be set'
                                       + ' when only one haproxy host '
                                       + str(haproxy_configs[self._option_host])
                                       + ' is present')
            if cmd_args.advertised_ext_host:
                rdafutils.cli_err_exit('Advertised external host '
                                       + cmd_args.advertised_ext_host + ' must not be set'
                                       + ' when only one haproxy host '
                                       + str(haproxy_configs[self._option_host])
                                       + ' is present')
            if cmd_args.advertised_ext_interface:
                rdafutils.cli_err_exit('Advertised external interface '
                                       + cmd_args.advertised_ext_interface + ' must not be set'
                                       + ' when only one haproxy host '
                                       + str(haproxy_configs[self._option_host])
                                       + ' is present')
        self._mark_configured(haproxy_configs, config_parser)

    def validate_setup_inputs(self, cmd_args, config_parser):
        # if any advertised internal/external IPs are configured, then those
        # shouldn't be part of the components' hosts
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts()
        adv_external_host = self.get_advertised_external_host()
        adv_internal_host = self.get_advertised_internal_host()
        if adv_external_host and adv_external_host in all_known_hosts:
            rdafutils.cli_err_exit('Invalid advertised external host ' + adv_external_host
                                   + ', since it already hosts one of the components')
        if adv_internal_host and adv_internal_host in all_known_hosts:
            rdafutils.cli_err_exit('Invalid advertised internal host ' + adv_internal_host
                                   + ', since it already hosts one of the components')

    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != 'k8s':
            return
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        for host in self.get_hosts():
            logger.info(f'Pulling rda-platform-k8s-haproxy image on host {host}')
            docker_pull_command = f'docker pull {docker_repo}/rda-platform-k8s-haproxy:{cmd_args.tag}'
            run_potential_ssh_command(host, docker_pull_command, config_parser)
            
    def create_cert_configs(self, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        logger.info("Creating TLS secrets for HAProxy and SSL..")
        cert_secret = 'kubectl create secret generic -n {} haproxy-pem ' \
                      '--from-file=haproxy.pem=/opt/rdaf/cert/rdaf/rdaf.pem ' \
                      '--save-config --dry-run=client -o yaml | kubectl apply -f -'.format(namespace)
        key_secret = 'kubectl create secret generic -n {} ssl-key ' \
                     '--from-file=rdaf.key=/opt/rdaf/cert/rdaf/rdaf.key ' \
                     '--save-config --dry-run=client -o yaml | kubectl apply -f -'.format(namespace)
    
        cert_secret_2 = 'kubectl create secret generic -n {} ssl-cert ' \
                        '--from-file=rdaf.cert=/opt/rdaf/cert/rdaf/rdaf.cert ' \
                        '--save-config --dry-run=client -o yaml | kubectl apply -f -'.format(namespace)
        run_command(cert_secret)
        run_command(key_secret)
        run_command(cert_secret_2)

    
    def do_k8s_setup(self, cmd_args, config_parser):
        self.create_cert_configs(config_parser)
        namespace = self.get_namespace(config_parser)
        replacements = self._get_docker_repo()
        replacements['REPLICAS'] = 1 if len(self.get_hosts()) == 1 else 2
        replacements['IP'] = self.get_advertised_external_host()
        replacements['NAMESPACE'] = namespace

        haproxy_template_path = os.path.join(get_templates_dir_root(), 'k8s-local', 'haproxy-values.yaml')
        haproxy_dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'haproxy-values.yaml')
        with open(haproxy_template_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        with open(haproxy_dest_path, 'w') as f:
                f.write(original_content) 

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        if self.get_deployment_type(config_parser) == 'aws':
            return
        namespace = self.get_namespace(config_parser)
        statuses = []
        service_host_map = {}
        status_command = f'kubectl get pods -n {namespace} -l "app_component==rda-haproxy" -o json'
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of component haproxy, due to: {}.".format(str(stderr)))

        result = json.loads(str(stdout))
        items = result['items']
        hosts = self.get_hosts()
        for host in hosts:     
            if not items or all(item['metadata']['labels'].get('app_component') != 'rda-haproxy' for item in items):
                    statuses.append({
                        'component_name': self.get_k8s_component_name(),
                        'host': host,
                        'containers': [{
                            'Id': 'N/A',
                            'Image': 'N/A',
                            'State': 'N/A',
                            'Status': 'Not Provisioned'
                        }]
                    })
            else:
                for item in items:
                    pod = dict()
                    pod['component_name'] = item['metadata']['labels']['app_component']
                    pod['host'] = item['status'].get('hostIP', 'Unknown')
                    pod['containers'] = []
                    if 'containerStatuses' in item['status']:
                        for entry in item['status']['containerStatuses']:
                            if 'containerID' in entry.keys():
                                container = dict()
                                container['Id'] = entry['containerID']
                                container['Image'] = entry['image']
                                for key in entry['state'].keys():
                                    container['State'] = key
                                    if key == 'running':
                                        container['Status'] = self.get_container_age(entry['state'][key]['startedAt'])
                                    else:
                                        container['Status'] = key
                                    break
                                pod['containers'].append(container)
                    service_host_map.setdefault(pod['component_name'], {})
                    service_host_map[pod['component_name']][pod['host']] = pod
        for component_pods in service_host_map.values():
            statuses.extend(component_pods.values())

        return statuses

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        if self.get_deployment_type(config_parser) != "aws":
            nginx_svc = 'kubectl apply -f {} -n {}'\
                .format(os.path.join(get_templates_dir_root(), 'k8s-local', 'nginx-service.yaml'), namespace)
            run_command(nginx_svc)
            haproxy_chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), 'rda-haproxy')
            haproxy_deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', 'rda-haproxy')
            self.copy_helm_chart(haproxy_chart_template_path, haproxy_deployment_path)

            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'haproxy-values.yaml')
            haproxy_install_command = 'helm install --create-namespace -n {} -f {} {} {} {} ' \
                .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), 'rda-haproxy', haproxy_deployment_path)
            run_command(haproxy_install_command)

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        if self.get_deployment_type(config_parser) != "aws":
            haproxy_chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), 'rda-haproxy')
            haproxy_deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', 'rda-haproxy')
            self.copy_helm_chart(haproxy_chart_template_path, haproxy_deployment_path)
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'haproxy-values.yaml')
            upgrade_command = 'helm upgrade --install --create-namespace -n {} -f {} {} {} {} ' \
                 .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), 'rda-haproxy', haproxy_deployment_path)
            run_command(upgrade_command)

    def k8s_reset(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.reset(cmd_args, config_parser)

    def _gather_advertised_hosts_interfaces(self, haproxy_hosts: List[str],
                                            cmd_args: argparse.Namespace,
                                            config_parser: configparser.ConfigParser) -> dict:
        haproxy_configs = dict()
        desc = 'What is the network interface on which you want the rdaf to ' \
               'be accessible externally?'
        err_msg = 'No interface specified for binding the rdaf externally. Please use' \
                  ' --advertised-external-interface to specify one'
        intf = Component._parse_or_prompt_value(cmd_args.advertised_ext_interface,
                                                'eth0',
                                                err_msg,
                                                desc,
                                                'Advertised external interface',
                                                cmd_args.no_prompt)
        # TODO: verify that the interface is indeed present
        haproxy_configs[self._option_advertised_ext_interface] = intf

        no_prompt_err_msg = 'No advertised external host specified. Please use ' \
                            '--advertised-external-host to specify one'
        host_desc = 'What is the host on which you want the platform to be externally accessible?'
        adv_hosts = Component._parse_or_prompt_hosts(cmd_args.advertised_ext_host,
                                                     haproxy_hosts[0],
                                                     no_prompt_err_msg,
                                                     host_desc, 'Advertised external host',
                                                     cmd_args.no_prompt)
        if len(adv_hosts) != 1:
            rdafutils.cli_err_exit('Exactly one host is allowed for --advertised-external-host. '
                                   'Invalid value ' + str(adv_hosts))
        haproxy_configs[self._option_advertised_ext_host] = adv_hosts[0]

        # now gather app service deployment mode
        desc = 'Do you want to specify an internal advertised host?'
        if cmd_args.no_prompt:
            bind_internal_inf = cmd_args.bind_internal_inf
        else:
            bind_internal_inf = rdafutils.query_yes_no(desc, default='no')
        if bind_internal_inf:
            internal_intf = Component._parse_or_prompt_value(
                cmd_args.advertised_int_interface,
                haproxy_configs[self._option_advertised_ext_interface],
                'No interface specified for binding the platform internally. Please use'
                ' --advertised-internal-interface to specify one',
                'What is the network interface on which you want the platform to '
                'be accessible internally?',
                'Advertised internal interface',
                cmd_args.no_prompt)
            # TODO: verify that the interface is indeed present
            haproxy_configs[self._option_advertised_int_interface] = internal_intf
            internal_adv_hosts = Component._parse_or_prompt_hosts(
                cmd_args.advertised_int_host,
                haproxy_configs[self._option_advertised_ext_host],
                'No advertised internal host specified. Please use --advertised-internal-host'
                ' to specify one',
                'What is the host on which you want the platform to be internally accessible?',
                'Advertised internal host',
                cmd_args.no_prompt)
            if len(internal_adv_hosts) != 1:
                rdafutils.cli_err_exit('Exactly one host is allowed for --advertised-internal-host.'
                                       'Invalid value ' + str(internal_adv_hosts))
            haproxy_configs[self._option_advertised_int_host] = internal_adv_hosts[0]
        return haproxy_configs

    def do_setup(self, cmd_args, config_parser):
        for haproxy_host in self.get_hosts():
            content = self._create_haproxy_cfg_content()
            dest_location = os.path.join(self.get_conf_dir(), 'haproxy.cfg')
            created_location = create_file(haproxy_host, content.encode(encoding='UTF-8'), dest_location)
            logger.info('Created HAProxy configuration at ' + created_location + ' on ' + haproxy_host)

            self._create_haproxy_pem(haproxy_host)

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        return super().status(cmd_args, config_parser)

    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        cmd = f'helm list -n {namespace} -q'
        ret, stdout, stderr = execute_command(cmd)

        if 'rda-haproxy' not in stdout:
            logger.warning(f"Helm release 'rda-haproxy' not found in namespace {namespace}")
        else:
            logger.info(f"Uninstalling helm chart: rda-haproxy")
            cmd = f'helm uninstall -n {namespace} rda-haproxy'
            run_command(cmd)

    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'haproxy-values.yaml')
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), 'rda-haproxy')
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', 'rda-haproxy')
        self.copy_helm_chart(chart_template_path, deployment_path)
        install_command = 'helm upgrade --install  --create-namespace -n {} -f {} {} {} ' \
            .format(namespace, values_yaml, 'rda-haproxy', deployment_path)
        run_command(install_command)

    def get_hosts(self) -> list:
        return self.configs[self._option_host]

    def get_advertised_external_host(self) -> str:
        return self.configs[self._option_advertised_ext_host]

    def get_advertised_external_interface(self) -> str:
        return self.configs[self._option_advertised_ext_interface]

    def get_advertised_internal_host(self) -> str:
        return self.configs[self._option_advertised_int_host]

    def get_advertised_internal_interface(self) -> str:
        return self.configs[self._option_advertised_int_interface]

    def is_keepalived_necessary(self) -> bool:
        if self.get_hosts() and len(self.get_hosts()) > 1:
            # if more than 1 haproxy is involved, we manage the high availability
            # of HAProxy through keepalived
            return True
        return False

    def get_external_access_host(self) -> str:
        haproxy_hosts = self.get_hosts()
        if len(haproxy_hosts) == 1:
            # when there's only one HA proxy host involved, we use that as the
            # host on which we access it for internal service(s) communication
            return haproxy_hosts[0]
        adv_external_host = self.get_advertised_external_host()
        if adv_external_host is not None:
            return adv_external_host
        return haproxy_hosts[0]

    def get_internal_access_host(self) -> str:
        haproxy_hosts = self.get_hosts()
        config_file_path = os.path.expanduser(os.path.join('/opt', 'rdaf', 'rdaf.cfg'))
        if not os.path.isfile(config_file_path):
            return None
        config_parser = configparser.ConfigParser(allow_no_value=True)
        with open(config_file_path, 'r') as f:
            config_parser.read_file(f)

        if not config_parser.has_section('rdaf-cli'):
            return None
        # Check the 'deployment' key
        deployment_type = config_parser.get('rdaf-cli', 'deployment', fallback=None)
        if deployment_type == "non-k8s" and len(haproxy_hosts) == 1:
              #if len(haproxy_hosts) == 1:
              # when there's only one HA proxy host involved, we use that as the
              # host on which we access it for internal service(s) communication
              return haproxy_hosts[0]
        # more than one haproxy host involved, check if the advertised internal host
        # is configured. If yes, return that, else return the first haproxy host
        adv_internal_host = self.get_advertised_internal_host()
        if adv_internal_host is not None:
            return adv_internal_host
        # more than one haproxy host involved, check if the advertised external host
        # is configured. If yes, return that, else return the first haproxy host
        adv_external_host = self.get_advertised_external_host()
        if adv_external_host is not None:
            return adv_external_host
        return haproxy_hosts[0]

    def get_deployment_env(self, host: str):
        haproxy_env_vars = dict()
        haproxy_env_vars['HAPROXY_MOUNT'] = os.path.join(self.get_conf_dir())
        haproxy_env_vars['HAPROXY_CERT'] = os.path.join('/opt/rdaf/cert/rdaf/')
        haproxy_env_vars['HOST_IP'] = self.get_external_access_host()
        return haproxy_env_vars


    def _create_haproxy_cfg_content(self) -> str:
        template_root = get_templates_dir_root()
        haproxy_cfg_template = os.path.join(template_root, 'haproxy.cfg')
        with open(haproxy_cfg_template, 'r') as f:
            template_content = f.read()
        template_replacement_values = dict()
        # TODO: review this
        template_replacement_values['HAPROXY_BIND_ADDR'] = '*'
        # create the server directive for Minio
        template_replacement_values.update(self._get_minio_template_replacements())
        # setup mariadb rules
        template_replacement_values.update(self._get_mariadb_template_replacements())
        # rda api server rules
        template_replacement_values.update(self._get_rda_api_server_template_replacements())

        # ui portal rules
        template_replacement_values.update(self._get_portal_template_replacements())
        # replace the variables and write out the file
        return string.Template(template_content).substitute(template_replacement_values)

    def _get_rda_api_server_template_replacements(self) -> dict:
        replacements = dict()
        platform_component = COMPONENT_REGISTRY \
            .require(rdaf.component.pseudo_platform.PseudoComponent.COMPONENT_NAME)
        platform_hosts = platform_component.get_api_server_exposed_hosts_ports()
        server_id = 0
        portal_server_directives = []
        for host, port in platform_hosts:
            server_id += 1
            server_name = 'rda-api-server-' + host
            portal_server_directives.append('server ' + server_name +
                                            ' ' + host + ':' + str(port))

        replacements['BACKEND_RDA_SERVER_DIRECTIVE'] = '\n    '.join(portal_server_directives)
        return replacements

    def _get_portal_template_replacements(self) -> dict:
        replacements = dict()
        platform_component = COMPONENT_REGISTRY \
            .require(rdaf.component.pseudo_platform.PseudoComponent.COMPONENT_NAME)
        platform_hosts = platform_component.get_portal_host_ports()
        server_id = 0
        portal_server_directives = []
        for portal_host, portal_port in platform_hosts:
            server_id += 1
            server_name = 'portal-' + portal_host
            cookie_val = 'rdaf-portal-' + str(server_id)
            portal_server_directives.append('server ' + server_name +
                                            ' ' + portal_host + ':' + str(portal_port) +
                                            ' ' + 'check' + ' ' + 'cookie ' + cookie_val)

        replacements['BACKEND_PORTAL_SERVER_DIRECTIVE'] = '\n    '.join(portal_server_directives)
        return replacements

    def _get_minio_template_replacements(self) -> dict:
        replacements = dict()
        minio_component = COMPONENT_REGISTRY.require(MINIO_COMPONENT)
        minio_host_ports = minio_component.get_minio_host_ports()
        server_id = 0
        minio_server_directives = []
        for minio_host, minio_port in minio_host_ports:
            server_id += 1
            server_name = 'minio-' + minio_host
            cookie_val = 'rdaf-objstr-' + str(server_id)
            minio_server_directives.append('server ' + server_name +
                                           ' ' + minio_host + ':' + str(minio_port) +
                                           ' ' + 'check' + ' ' + 'cookie ' + cookie_val)

        replacements['BACKEND_MINIO_SERVER_DIRECTIVE'] = '\n    '.join(minio_server_directives)
        return replacements

    def _get_mariadb_template_replacements(self) -> dict:
        replacements = dict()
        # TODO: fix this hardcoded port used by mariadb frontend
        replacements['MARIADB_HAPROXY_BIND_PORT'] = str(3307)
        mariadb = COMPONENT_REGISTRY.require(rdaf.component.mariadb.COMPONENT_NAME)
        mariadb_hosts = mariadb.get_hosts()
        # TODO: fix this hardcoding of user name
        if len(mariadb_hosts) == 1:
            replacements['BACKEND_MARIADB_HEALTH_CHECK_DIRECTIVE'] = \
                'option mysql-check user health_check_user'
        else:
            replacements['BACKEND_MARIADB_HEALTH_CHECK_DIRECTIVE'] = \
            'option external-check\n    external-check command /maria_cluster_check'

        mariadb_server_directives = []
        for index, host in enumerate(mariadb_hosts):
            server_name = 'mariadb-' + host
            if index != 1 and len(mariadb_hosts) > 1:
                mariadb_server_directives.append('server ' + server_name +
                                                 ' ' + host + ':3306' + ' ' + 'check backup')
            else:
                mariadb_server_directives.append('server ' + server_name +
                                                 ' ' + host + ':3306' + ' ' + 'check')
        replacements['BACKEND_MARIADB_SERVER_DIRECTIVE'] = '\n    '.join(mariadb_server_directives)
        return replacements

    def append_config(self, app_content, config_parser: configparser.ConfigParser):
        existing_content = self._create_haproxy_cfg_content()
        if app_content:
            existing_content = self._append_webhook_acl(existing_content)

        content = existing_content + app_content
        dest_location = os.path.join(self.get_conf_dir(), 'haproxy.cfg')
        for host in self.get_hosts():
            created_location = create_file(host, content.encode(encoding='UTF-8'), dest_location)
            logger.info('Updated HAProxy configuration at ' + created_location + ' on ' + host)
        self._restart(config_parser)

    def append_k8s_config(self, app_content, config_parser: configparser.ConfigParser):
        existing_content = self._create_haproxy_k8s_cfg_content(config_parser)
        if app_content:
            existing_content = self._append_webhook_acl(existing_content)

        content = existing_content + app_content
        dest_location = os.path.join(self.get_conf_dir(), 'haproxy.cfg')
        for host in self.get_hosts():
            created_location = create_file(host, content.encode(encoding='UTF-8'), dest_location)
            logger.info('Updated HAProxy configuration at ' + created_location + ' on ' + host)
        self._restart(config_parser)

    def _restart(self, config_parser: configparser.ConfigParser):
        # we restart in reverse order
        for host in reversed(self.get_hosts()):
            compose_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', host,
                                        self.get_deployment_file_name())
            if not os.path.exists(compose_file):
                continue
            if self.component_name not in self.get_involved_services(compose_file):
                continue
            logger.info("Restarting Haproxy on host: " + host)
            command = '/usr/local/bin/docker-compose --project-name infra -f ' + compose_file \
                      + ' restart ' + self.component_name
            run_potential_ssh_command(host, command, config_parser)

    def healthcheck(self, component_name, host, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        command = "nc -z {} 7222 && echo $?".format(host)
        ret, stdout, stderr = run_command_exitcode(command, socket.gethostname(), config_parser)

        if ret != 0:
            return [component_name, "Service Status", termcolor.colored('Failed', color='red'),
                    "Unable to connect to port 7222"]

        return [component_name, "Service Status", "OK", "N/A"]

    @staticmethod
    def _create_haproxy_pem(haproxy_host):
        # the haproxy docker image is hardcoded to expect a haproxy.pem file, instead
        # of accepting a configurable cert file name. So until the docker image is
        # fixed/enhanced, we create a file copy named haproxy.pem
        cert_manager = COMPONENT_REGISTRY.require(rdaf.component.cert.CertManager.COMPONENT_NAME)
        cert_dir = os.path.join(cert_manager.get_cert_root_dir(), 'rdaf')
        cert_file = os.path.join(cert_dir, 'rdaf.pem')
        haproxy_pem_file = os.path.join(cert_dir, 'haproxy.pem')
        do_potential_scp(haproxy_host, cert_file, haproxy_pem_file)
        logger.debug('Copied ' + cert_file + ' to ' + haproxy_pem_file
                     + ' for haproxy component\'s use')

    def _do_docker_compose_checks(self, config_parser):
        hosts = self.get_hosts()
        docker_compose_file = os.path.join(rdaf.get_scripts_dir_root(), 'docker-compose')
        docker_compose_file_path = os.path.join('/usr', 'local', 'bin', 'docker-compose')
        remote_tmp_path = '/tmp/docker-compose'
        chmod_cmd = 'sudo chmod +x /usr/local/bin/docker-compose'
        copy_command = 'sudo cp ' + docker_compose_file + ' ' + docker_compose_file_path
        ssh_copy_command = 'sudo cp ' + remote_tmp_path + ' ' + docker_compose_file_path

        from rdaf.component.dockerregistry import COMPONENT_NAME
        from rdaf.contextual import COMPONENT_REGISTRY
        docker_registry = COMPONENT_REGISTRY.require(COMPONENT_NAME)

        for host in hosts:
            if Component.is_local_host(host):
                run_command(copy_command)
                run_command(chmod_cmd)
            else:
                do_potential_scp(host, docker_compose_file, remote_tmp_path, sudo=True)
                run_potential_ssh_command(host, ssh_copy_command, config_parser)
                run_potential_ssh_command(host, chmod_cmd, config_parser)

            docker_registry.docker_login(host, config_parser)

    def _create_haproxy_k8s_cfg_content(self, config_parser: configparser.ConfigParser) -> str:
        template_root = get_templates_dir_root()
        haproxy_cfg_template = os.path.join(template_root, 'haproxy.cfg')
        with open(haproxy_cfg_template, 'r') as f:
            template_content = f.read()
        template_replacement_values = dict()

        template_replacement_values['HAPROXY_BIND_ADDR'] = '*'
        # create the server directive for Minio
        minio_component = COMPONENT_REGISTRY.require(MINIO_COMPONENT)
        minio_host_ports = minio_component.get_minio_host_ports()
        server_id = 0
        minio_server_directives = []
        for minio_host, minio_port in minio_host_ports:
            server_id += 1
            server_name = 'minio-' + minio_host
            cookie_val = 'rdaf-objstr-' + str(server_id)
            minio_server_directives.append('server ' + server_name +
                                           ' ' + minio_host + ':' + '30443' +
                                           ' ' + 'check' + ' ' + 'cookie ' + cookie_val)

        template_replacement_values['BACKEND_MINIO_SERVER_DIRECTIVE'] = '\n    '.join(minio_server_directives)

        # setup mariadb rule
        mariadb_port = self.get_service_node_port('rdaf-mariadb-public', config_parser)
        template_replacement_values['MARIADB_HAPROXY_BIND_PORT'] = str(3307)
        mariadb = COMPONENT_REGISTRY.require(rdaf.component.mariadb.COMPONENT_NAME)
        mariadb_hosts = mariadb.get_hosts()
        if len(mariadb_hosts) == 1:
            template_replacement_values['BACKEND_MARIADB_HEALTH_CHECK_DIRECTIVE'] = \
                'option mysql-check user health_check_user'
        else:
            template_replacement_values['BACKEND_MARIADB_HEALTH_CHECK_DIRECTIVE'] = \
            'option external-check\n    external-check command /maria_cluster_check'

        mariadb_server_directives = []
        for index, host in enumerate(mariadb_hosts):
            server_name = 'mariadb-' + host
            if index != 1 and len(mariadb_hosts) > 1:
                mariadb_server_directives.append('server ' + server_name +
                                                 ' ' + host + ':' + mariadb_port + ' ' + 'check backup')
            else:
                mariadb_server_directives.append('server ' + server_name +
                                                 ' ' + host + ':' + mariadb_port + ' ' + 'check')
        template_replacement_values['BACKEND_MARIADB_SERVER_DIRECTIVE'] = '\n    '.join(mariadb_server_directives)

        # rda api server rules
        platform_component = COMPONENT_REGISTRY \
            .require(rdaf.component.pseudo_platform.PseudoComponent.COMPONENT_NAME)
        platform_hosts = platform_component.get_api_server_exposed_hosts_ports()
        api_server_port = self.get_service_node_port('rdaf-api-server', config_parser)
        server_id = 0
        portal_server_directives = []
        for host, port in platform_hosts:
            server_id += 1
            server_name = 'rda-api-server-' + host
            portal_server_directives.append('server ' + server_name +
                                            ' ' + host + ':' + api_server_port)

        template_replacement_values['BACKEND_RDA_SERVER_DIRECTIVE'] = '\n    '.join(portal_server_directives)

        # ui portal rules
        portal_hosts = platform_component.get_portal_host_ports()
        server_id = 0
        portal_server_directives = []
        portal_node_port = self.get_service_node_port('rdaf-portal', config_parser)
        for portal_host, portal_port in portal_hosts:
            server_id += 1
            server_name = 'portal-' + portal_host
            cookie_val = 'rdaf-portal-' + str(server_id)
            portal_server_directives.append('server ' + server_name +
                                            ' ' + portal_host + ':' + portal_node_port +
                                            ' ' + 'check' + ' ' + 'cookie ' + cookie_val)

        template_replacement_values['BACKEND_PORTAL_SERVER_DIRECTIVE'] = '\n'.join(portal_server_directives)
        return string.Template(template_content).substitute(template_replacement_values)

    def restore_conf(self, config_parser: configparser.ConfigParser,
                     backup_content_root_dir: os.path):
        for host in self.get_hosts():
            run_potential_ssh_command(host, 'sudo chown -R ' + str(os.getuid()) + ' '
                                      + self.get_conf_dir(), config_parser)
        super().restore_conf(config_parser, backup_content_root_dir)

    def k8s_backup_conf(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_dir_root: os.path):
        self.backup_conf(cmd_args, config_parser, backup_dir_root)

    def k8s_restore_conf(self, config_parser: configparser.ConfigParser,
                         backup_content_root_dir: os.path):
        deployments_dir = os.path.join('/opt', 'rdaf', 'deployment-scripts')
        # restoring infra.yaml
        for host in self.get_hosts():
            file_path = os.path.join(deployments_dir, host, 'infra.yaml')
            if not os.path.exists(file_path) or Component.is_local_host(host):
                continue
            do_potential_scp(host, file_path, file_path)
    
    @staticmethod
    def _append_webhook_acl(existing_content):
        lines = existing_content.split('\n')
        for index, line in enumerate(lines):
            if 'bind *:443 ssl crt /opt/certificates/haproxy.pem' in line:
                lines.insert(index + 1, '    acl WEBHOOK_PATH path_beg -i /webhooks/')
                lines.insert(index + 2, '    use_backend webhook if WEBHOOK_PATH')
                break
        existing_content = """
{}
""".format("\n".join(lines[1:]))
        return existing_content
