import argparse
import configparser
import json
import logging
import os
import string

import yaml
from kafka import KafkaAdminClient
from typing import Callable, Any, List, Tuple

import rdaf
from rdaf import get_templates_dir_root, rdafutils
from rdaf.rdafutils import cli_err_exit
import rdaf.component.cert as cert
from rdaf.component import Component, InfraCategoryOrder, \
    _host_dir_loader, _host_dir_storer, _apply_data_dir_defaults, execute_command, run_potential_ssh_command, run_command, \
    do_potential_scp
from rdaf.component import _comma_delimited_to_list
from rdaf.component import _list_to_comma_delimited
from rdaf.contextual import COMPONENT_REGISTRY
import termcolor

COMPONENT_NAME = 'kafka'
logger = logging.getLogger(__name__)


class Kafka(Component):
    _option_data_dir = 'datadir'
    _option_host = 'host'
    _option_kraft_cluster_id = 'kraft_cluster_id'
    _ext_user= 'external_user'
    _ext_password = 'external_password'

    _default_data_dirs = ['/kafka-logs']


    def __init__(self):
        super().__init__(COMPONENT_NAME, 'kafka', 'infra', InfraCategoryOrder.KAFKA.value)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host:
            return _comma_delimited_to_list
        if config_name == self._option_data_dir:
            # convert the comma separated value to list of tuples
            return _host_dir_loader
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host:
            return _list_to_comma_delimited
        if config_name == self._option_data_dir:
            # convert the list of tuple to a comma separate value
            return _host_dir_storer
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_data_dir] = None
        default_configs[self._option_host] = []
        default_configs[self._option_kraft_cluster_id] = None
        return default_configs

    def gather_setup_inputs(self, cmd_args, config_parser):
        kafka_configs = self._init_default_configs()
        default_host_name = Component.get_default_host()
        no_prompt_err_msg = 'No Kafka server host specified. Use --kafka_server_host ' \
                            'to specify a Kafka server host'
        kafka_host_desc = 'What is the "host/path-on-host" on which you want the Kafka ' \
                          'server to be provisioned?'
        host_dirs = Component._parse_or_prompt_host_dirs(
            cmd_args.kafka_server_host,
            default_host_name,
            no_prompt_err_msg, kafka_host_desc,
            'Kafka server host/path',
            cmd_args.no_prompt)
        kafka_configs[self._option_data_dir] = _apply_data_dir_defaults(host_dirs,
                                                                        self._default_data_dirs)
        kafka_hosts = []
        for host, data_dirs in kafka_configs[self._option_data_dir]:
            kafka_hosts.append(host)
        kafka_configs[self._option_host] = kafka_hosts
        kafka_configs[self._option_kraft_cluster_id] = execute_command("uuidgen --time | tr -d '-' | base64 | cut -b 1-22")[1].strip()
        self._mark_configured(kafka_configs, config_parser)

    def get_hosts(self) -> list:
        return self.configs[self._option_host]

    def get_ports(self) -> tuple:
        ports = ['9092', '9093', '9094']
        hosts = self.get_hosts()
        return hosts, ports

    def _get_host_data_dirs(self) -> List[Tuple[str, List[str]]]:
        return self.configs[self._option_data_dir]

    def do_setup(self, cmd_args, config_parser):
        # owning the data dir
        for host, data_dirs in self.configs[self._option_data_dir]:
            data_dir = data_dirs[0]
            command = 'sudo mkdir -p ' + data_dir + ' && sudo chown -R 1001 ' + data_dir \
                      + ' && sudo chgrp -R 1001 ' + data_dir
            run_potential_ssh_command(host, command, config_parser)

        kafka_log_dir = self.get_logs_dir()
        for host in self.get_hosts():
            command = 'sudo mkdir -p ' + kafka_log_dir + ' && sudo chown -R 1001 ' + kafka_log_dir \
                      + ' && sudo chgrp -R 1001 ' + kafka_log_dir
            run_potential_ssh_command(host, command, config_parser)

        self._generate_conf_file(config_parser)

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if self.get_deployment_type(config_parser) == 'rda-edge':
            return
        super().install(cmd_args, config_parser)

    def get_k8s_component_name(self):
        return 'rda-kafka'

    def get_k8s_chart_name(self):
        return 'rda_kafka'

    def get_k8s_component_label(self):
        return 'app.kubernetes.io/name=kafka'

    def do_k8s_setup(self, cmd_args, config_parser):
        cert_manager: cert.CertManager = COMPONENT_REGISTRY.require(
            cert.CertManager.COMPONENT_NAME)
        keystore_pass = cert_manager.get_keystore_pass()
        self.create_cert_configs(config_parser)
        replacements = self._get_docker_repo()
        replacements['REPLICAS'] = len(self.get_hosts())
        replacements['REPLICATION_FACTOR'] = 1 if len(self.get_hosts()) == 1 else 3
        replacements['OFFSETS_TOPIC_REPLICATION_FACTOR'] = 1 if len(self.get_hosts()) == 1 else 3
        replacements['TRANSACTION_STATE_LOG_REPLICATION_FACTOR'] = 1 if len(self.get_hosts()) == 1 else 3
        replacements['TRANSACTION_STATE_LOG_MIN_ISR'] = 1 if len(self.get_hosts()) == 1 else 2
        replacements['MIN_INSYNC_REPLICAS'] = 1 if len(self.get_hosts()) == 1 else 2
        replacements['TRUSTSTORE_PASSWORD'] = keystore_pass

        template_path = os.path.join(get_templates_dir_root(), 'k8s-local', 'kafka-values.yaml')
        dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'kafka-values.yaml')
        with open(template_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        values = yaml.safe_load(original_content)
        with open(dest_path, 'w') as f:
            yaml.safe_dump(values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)


    def create_cert_configs(self, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        # create the kubectl secrets
        logger.debug("Creating tls kafka secret..")
        cert_secret = 'kubectl create secret generic -n {} kafka-secret ' \
                      '--from-file=kafka.keystore.jks=/opt/rdaf/cert/rdaf/rdaf.jks ' \
                      '--from-file=kafka.truststore.jks=/opt/rdaf/cert/truststore/ca_truststore ' \
                      '--save-config --dry-run=client -o yaml |  kubectl apply -f -'.format(namespace)
        run_command(cert_secret)

    def get_k8s_install_args(self, cmd_args):
        args = '--set image.tag={}'.format(cmd_args.tag, cmd_args.tag)
        return args

    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != 'k8s':
            return
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            for image in ['rda-platform-kafka','rda-platform-kubectl']:
                logger.info(f'Pulling {image} on host {host}')
                docker_pull_command = f'docker pull {docker_repo}/{image}:{cmd_args.tag}'
                run_potential_ssh_command(host, docker_pull_command, config_parser)

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        if self.get_deployment_type(config_parser) != "aws":
            kafka_pv_file = 'kafka-pv-cluster.yaml' if len(self.get_hosts()) > 1 else 'kafka-pv.yaml'
            kafka_pvs = 'kubectl apply -f {} -n {}'.format(
                os.path.join(get_templates_dir_root(), 'k8s-local', kafka_pv_file), namespace)
            run_command(kafka_pvs)

        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'kafka-values.yaml')
        install_command = 'helm install --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(), deployment_path)
        run_command(install_command)

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'kafka-values.yaml')
        upgrade_command = 'helm upgrade --install --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(upgrade_command)

    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        comp_name = self.get_k8s_component_name()
        label = f"app.kubernetes.io/instance={comp_name}"
        cmd = f'kubectl get all -l {label} -n {namespace} -o json'
        ret, stdout, stderr = execute_command(cmd)
        component = json.loads(stdout)
        for comp in component["items"]:
            if comp["kind"] == 'StatefulSet':
                command = 'kubectl scale statefulset.apps/{} -n {} --replicas={}'\
                    .format(comp['metadata']['name'], namespace, len(self.get_hosts()))
                run_command(command) 

    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):  
        namespace = self.get_namespace(config_parser)
        run_command(f'kubectl scale statefulset.apps/rda-kafka-controller -n {namespace} --replicas=0')
 
    def get_deployment_env(self, host: str):
        cert_manager: cert.CertManager = COMPONENT_REGISTRY.require(
            cert.CertManager.COMPONENT_NAME)

        env = dict()
        env['KAFKA_ID'] = str(self.get_hosts().index(host) + 1)
        env['KAFKA_ENDPOINT'] = host
        env['TRUSTSTORE_PASSWORD'] = cert_manager.get_keystore_pass()
        env['QUORUM_VOTERS'] = ','.join(['{}@{}:9094'.format(i+1, host) for i, host in enumerate(self.get_hosts())])
        for host_entry, data_dirs in self.configs[self._option_data_dir]:
            if host == host_entry:
                env['KAFKA_DATA_MOUNT'] = data_dirs[0]

        env['KAFKA_REPLICATION_FACTOR'] = 1 if len(self.get_hosts()) < 3 else 3
        env['KAFKA_OFFSETS_TOPIC_REPLICATION_FACTOR'] = 1 if len(self.get_hosts()) < 3 else 3
        env['KAFKA_TRANSACTION_STATE_LOG_REPLICATION_FACTOR'] = 1 if len(self.get_hosts()) < 3 else 3
        env['KAFKA_MIN_INSYNC_REPLICAS'] = 1 if len(self.get_hosts()) < 3 else 2
        env['KAFKA_TRANSACTION_STATE_LOG_MIN_ISR'] = 1 if len(self.get_hosts()) < 3 else 2
        env['KRAFT_CLUSTER_ID'] = self.configs[self._option_kraft_cluster_id]
        return env

    def _generate_conf_file(self, config_parser):
        log4j_template = os.path.join(rdaf.get_templates_dir_root(), 'kafka-log4j.properties')
        log4j_dest = os.path.join(self.get_conf_dir(), 'log4j.properties')
        server_config = os.path.join(self.get_conf_dir(), 'kafka-server.conf')
        command = 'sudo chown -R 1001 ' + self.get_conf_dir() + ' && sudo chgrp -R 1001 ' \
                  + self.get_conf_dir()
        for host in self.get_hosts():
            output = """KafkaServer {
                        org.apache.kafka.common.security.scram.ScramLoginModule required
                        username="admin"
                        password="abcd123$"
                        user_admin="abcd123$";
                        };"""

            rdaf.component.create_file(host, output.encode(encoding='UTF-8'), server_config)
            do_potential_scp(host, log4j_template, log4j_dest)
            run_potential_ssh_command(host, command, config_parser)

    def setup_tenant(self, tenant_id: str, config_parser: configparser.ConfigParser):
        logger.info("Creating kafka tenant user...")
        kafka_host = self.get_hosts()[0]
        password = tenant_id+'.secret'
        user = tenant_id + '.user'
        ext_password = rdafutils.gen_password_with_uuid(tenant_id)
        ext_user = tenant_id + '.external'
        try:
            with Component.new_docker_client(kafka_host, config_parser, timeout=60) as docker_client:
                containers = self.find_component_container_on_host(docker_client,
                                                                   all_states=True)
                if len(containers) == 0:
                    cli_err_exit('No container found for ' + self.get_name()
                                 + ' on host ' + kafka_host)
                container = containers[0]
                tenant_user = "/opt/bitnami/kafka/bin/kafka-configs.sh --bootstrap-server {0}:9092 " \
                              "--alter --add-config 'SCRAM-SHA-256=[iterations=8192,password={1}]' " \
                               "--entity-type users --entity-name {2}".format(kafka_host, password, user)
                output, error = docker_client.docker_exec(container['Id'], tenant_user)
                logger.debug(output)
                tenant_user = "/opt/bitnami/kafka/bin/kafka-configs.sh --bootstrap-server {0}:9092 " \
                              "--alter --add-config 'SCRAM-SHA-512=[password={1}]' " \
                              "--entity-type users --entity-name {2}".format(kafka_host, password, user)
                output, error = docker_client.docker_exec(container['Id'], tenant_user)
                logger.debug(output)
                topics_acl = '/opt/bitnami/kafka/bin/kafka-acls.sh --bootstrap-server {0}:9092 --add --allow-principal ' \
                             'User:{1} --operation All --resource-pattern-type ' \
                             'prefixed --topic {2}'.format(kafka_host, user, tenant_id)
                output, error = docker_client.docker_exec(container['Id'], topics_acl)
                logger.debug(output)
                ext_user_creation = "/opt/bitnami/kafka/bin/kafka-configs.sh --bootstrap-server {0}:9092 " \
                                    "--alter --add-config 'SCRAM-SHA-256=[iterations=8192,password={1}]' --entity-type users --entity-name {2}" \
                    .format(kafka_host, ext_password, ext_user)
                output, error = docker_client.docker_exec(container['Id'], ext_user_creation)
                logger.debug(output)
                ext_user_creation = "/opt/bitnami/kafka/bin/kafka-configs.sh --bootstrap-server {0}:9092 " \
                                    "--alter --add-config 'SCRAM-SHA-256=[iterations=8192,password={1}]' --entity-type users --entity-name {2}" \
                    .format(kafka_host, ext_password, ext_user)
                output, error = docker_client.docker_exec(container['Id'], ext_user_creation)
                logger.debug(output)

                ext_topics_acl = '/opt/bitnami/kafka/bin/kafka-acls.sh --bootstrap-server {0}:9092 --add --allow-principal ' \
                             'User:{1} --operation All --resource-pattern-type ' \
                             'prefixed --topic {2}.external'.format(kafka_host, ext_user, tenant_id)
                output, error = docker_client.docker_exec(container['Id'], ext_topics_acl)
                logger.debug(output)

                self.configs[self._ext_user] = ext_user
                self.configs[self._ext_password] = rdafutils.str_base64_encode(ext_password)
                self._mark_configured(self.configs, config_parser)
                self.write_configs(config_parser)
        except Exception:
            logger.debug('Failed to configure tenant of ' + self.get_name() + ' on host '
                         + kafka_host, exc_info=1)
        return ext_user, ext_password

    def setup_k8s_tenant(self, config_parser: configparser.ConfigParser, tenant_id: str):
        logger.info("Creating kafka tenant user...")
        namespace = self.get_namespace(config_parser)
        pod_name = 'rda-kafka-controller-0'
        password = tenant_id+'.secret'
        user = tenant_id + '.user'
        ext_password = rdafutils.gen_password_with_uuid(tenant_id)
        ext_user = tenant_id + '.external'

        # entity user creation
        user_creation = 'kubectl exec {pod} -n {namespace} -- /opt/bitnami/kafka/bin/kafka-configs.sh ' \
                        '--bootstrap-server {pod}.rda-kafka-controller-headless.{namespace}.svc.cluster.local:9092 ' \
                        '--alter --add-config SCRAM-SHA-256=[iterations=8192,password={password}]' \
                        ' --entity-type users --entity-name {user}'\
            .format(user=user, password=password, pod=pod_name, namespace=namespace)
        run_command(user_creation)

        user_creation = 'kubectl exec {pod} -n {namespace} -- /opt/bitnami/kafka/bin/kafka-configs.sh ' \
                        '--bootstrap-server {pod}.rda-kafka-controller-headless.{namespace}.svc.cluster.local:9092 ' \
                        '--alter --add-config SCRAM-SHA-512=[password={password}] --entity-type users --entity-name {user}' \
            .format(user=user, password=password, pod=pod_name, namespace=namespace)
        run_command(user_creation)

        # principal
        principal = 'kubectl exec {pod} -n {namespace} -- /opt/bitnami/kafka/bin/kafka-acls.sh ' \
                    '--bootstrap-server {pod}.rda-kafka-controller-headless.{namespace}.svc.cluster.local:9092 --add ' \
                    '--allow-principal User:{user} --operation All --resource-pattern-type prefixed --topic {tenant_id}'\
            .format(pod=pod_name, user=user, tenant_id=tenant_id, namespace=namespace)
        run_command(principal)

        # external user creation
        ext_user_creation = 'kubectl exec {pod} -n {namespace} -- /opt/bitnami/kafka/bin/kafka-configs.sh ' \
                            '--bootstrap-server {pod}.rda-kafka-controller-headless.{namespace}.svc.cluster.local:9092 ' \
                            '--alter --add-config SCRAM-SHA-256=[iterations=8192,password={password}]' \
                            ' --entity-type users --entity-name {user}' \
            .format(user=ext_user, password=ext_password, pod=pod_name, namespace=namespace)
        run_command(ext_user_creation)

        ext_user_creation = 'kubectl exec {pod} -n {namespace} -- /opt/bitnami/kafka/bin/kafka-configs.sh ' \
                            '--bootstrap-server {pod}.rda-kafka-controller-headless.{namespace}.svc.cluster.local:9092 ' \
                            '--alter --add-config SCRAM-SHA-512=[password={password}] --entity-type users --entity-name {user}' \
            .format(user=ext_user, password=ext_password, pod=pod_name, namespace=namespace)
        run_command(ext_user_creation)

        ext_principal = 'kubectl exec {pod} -n {namespace} -- /opt/bitnami/kafka/bin/kafka-acls.sh ' \
                    '--bootstrap-server {pod}.rda-kafka-controller-headless.{namespace}.svc.cluster.local:9092 --add ' \
                    '--allow-principal User:{user} --operation All --resource-pattern-type prefixed --topic {tenant_id}.external' \
            .format(pod=pod_name, user=ext_user, tenant_id=tenant_id, namespace=namespace)
        run_command(ext_principal)

        self.configs[self._ext_user] = ext_user
        self.configs[self._ext_password] = rdafutils.str_base64_encode(ext_password)
        self._mark_configured(self.configs, config_parser)
        self.write_configs(config_parser)

        return ext_user, ext_password

    def backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        # we don't backup kafka data
        return

    def k8s_restore_conf(self, config_parser: configparser.ConfigParser,
                         backup_content_root_dir: os.path):
        self.create_cert_configs(config_parser)

    def healthcheck(self, component_name, host, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        kafka_host = host
        try:
            admin_client = KafkaAdminClient(bootstrap_servers=['{0}:9092'.format(kafka_host)])
            admin_client.list_topics()
            return [component_name, "Service Status", "OK", "N/A"]
        except Exception as e:
            logger.debug('Failed to configure tenant of ' + self.get_name() + ' on host '
                         + kafka_host, exc_info=1)
            return [component_name, "Service Status", termcolor.colored("Failed", color='red'), str(e)]

    def get_k8s_public_endpoint(self, config_parser):
        kafka_endpoints = []
        if self.get_deployment_type(config_parser) != "aws":
            for i in range(len(self.get_hosts())):
                port = self.get_service_node_port('rda-kafka-controller-{}-external'.format(str(i)), config_parser)
                kafka_endpoints.append('{}:{}'.format(self.get_hosts()[i], port))
        else:
            namespace = self.get_namespace(config_parser)
            ret, stdout, stderr = execute_command(f"kubectl get svc/rda-kafka-lb -n {namespace} -o json")
            if ret != 0:
                cli_err_exit("Failed to get status of kafka loadbalancer service, due to: {}.".format(str(stderr)))
            result = json.loads(str(stdout))
            if 'hostname' in result['status']['loadBalancer']['ingress'][0]:
                host = result['status']['loadBalancer']['ingress'][0]['hostname']
            else:
                host = result['status']['loadBalancer']['ingress'][0]['ip']

            for i in range(len(self.get_hosts())):
                port = result['spec']['ports'][i]['port']
                kafka_endpoints.append('{}:{}'.format(host, str(port)))

        return ','.join(kafka_endpoints)