import argparse
import configparser
import logging
import os
import random
import distro
import string
from typing import List
import termcolor

import rdaf.component.haproxy as haproxy
from rdaf import get_templates_dir_root
from rdaf.component import Component, InfraCategoryOrder, run_potential_ssh_command, \
    create_file, run_command_exitcode, check_potential_remote_file_exists
from rdaf.contextual import COMPONENT_REGISTRY

COMPONENT_NAME = 'keepalived'
logger = logging.getLogger(__name__)


class Keepalived(Component):
    def __init__(self):
        super().__init__(COMPONENT_NAME, 'keepalived', 'infra', InfraCategoryOrder.KEEPALIVED.value)

    def get_hosts(self) -> list:
        haproxy_component: haproxy.HAProxy = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
        if haproxy_component.get_hosts() is None:
            return []
        return haproxy_component.get_hosts() if len(haproxy_component.get_hosts()) > 1 else []

    def do_setup(self, cmd_args, config_parser):
        haproxy_component: haproxy.HAProxy = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
        if not haproxy_component.is_keepalived_necessary():
            logger.info('Skipping setup of keepalived since it isn\'t needed')
            return
        # we configure keepalived on each haproxy host
        haproxy_hosts = haproxy_component.get_hosts()
        advertised_ext_interface = haproxy_component.get_advertised_external_interface()
        advertised_ext_host = haproxy_component.get_advertised_external_host()
        advertised_int_interface = haproxy_component.get_advertised_internal_interface()
        advertised_int_host = haproxy_component.get_advertised_internal_host()
        first_host = True
        # virtual router id should be a number between 0 and 255
        ext_virtual_router_id = random.randint(0, 255)
        int_virtual_router_id = random.randint(151, 255)
        # only first 8 chars of auth_pass value are used by keepalived
        eight_char_password = '1a2b3c4d'
        template_root = get_templates_dir_root()
        keepalived_conf_template = os.path.join(template_root, 'keepalived-multi.conf') \
            if advertised_int_host else os.path.join(template_root, 'keepalived.conf')
        with open(keepalived_conf_template, 'r') as f:
            template_content = f.read()
        for host in haproxy_hosts:
            replacement_values = dict()
            replacement_values['HEALTH_CHECK_SCRIPT_USER'] = 'root'
            replacement_values['HEALTH_CHECK_SCRIPT_COMMAND'] = '"/usr/bin/pgrep haproxy"'
            replacement_values['VRRP_EXT_INSTANCE_INTERFACE'] = advertised_ext_interface
            replacement_values['VRRP_INT_INSTANCE_INTERFACE'] = advertised_int_interface
            replacement_values['VRRP_INSTANCE_STATE'] = 'MASTER' if first_host else 'BACKUP'
            replacement_values['VRRP_EXT_INSTANCE_ROUTER_ID'] = ext_virtual_router_id
            replacement_values['VRRP_INT_INSTANCE_ROUTER_ID'] = int_virtual_router_id
            # as per spec, 255 is the highest priority and should be assigned to master.
            # any other priority, like the 100 here, is just a random value.
            replacement_values['VRRP_INSTANCE_PRIORITY'] = '255' if first_host else '100'
            replacement_values['VRRP_EXT_INSTANCE_IPADDR'] = \
                advertised_ext_host + self.get_netmask_cidr(advertised_ext_interface, config_parser)
            if advertised_int_host:
                replacement_values['VRRP_INT_INSTANCE_IPADDR'] = \
                    advertised_int_host + self.get_netmask_cidr(advertised_int_interface, config_parser)
            replacement_values['AUTH_PASS'] = eight_char_password
            substituted_content = string.Template(template_content).substitute(replacement_values)
            dest_location = os.path.join(self.get_conf_dir(), 'keepalived.conf')
            created_location = create_file(host, substituted_content.encode(encoding='UTF-8'),
                                           dest_location)
            first_host = False
            logger.info('Created keepalived configuration at ' + created_location
                        + ' on host ' + host)

            if distro.id() == 'ubuntu':
                cmd = 'sudo ufw allow from 224.0.0.18; sudo ufw allow to 224.0.0.18'
                run_potential_ssh_command(host, cmd, config_parser)

    def pull_images(self, cmd_args, config_parser):
        pass

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        haproxy_component = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
        if not haproxy_component.is_keepalived_necessary():
            logger.info('Skipping start of keepalived since it isn\'t installed')
            return
        # we start keepalived on each haproxy host
        haproxy_hosts = haproxy_component.get_hosts()
        for host in haproxy_hosts:
            try:
                conf_file_path = os.path.join(self.get_conf_dir(), 'keepalived.conf')
                # write out the conf file contents to /etc/keepalived/keepalived.conf file
                # which is used by the keepalived systemd service
                command = 'cat ' + conf_file_path + ' | sudo tee /etc/keepalived/keepalived.conf'
                run_potential_ssh_command(host, command, config_parser)
                # now enable and start the service
                run_potential_ssh_command(host, 'sudo systemctl enable keepalived'
                                                ' && sudo systemctl restart keepalived',
                                          config_parser)
                logger.info('Started keepalived on ' + host)
            except Exception:
                # log and move on
                logger.debug('Failed to start keepalived on host ' + host, exc_info=1)
                logger.warning('Failed to start keepalived on host ' + host)
        
    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        # we start keepalived on each haproxy host
        self.install(cmd_args, config_parser)

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        haproxy_component = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
        if not haproxy_component.is_keepalived_necessary():
            logger.info('Skipping start of keepalived since it isn\'t installed')
            return
        # we start keepalived on each haproxy host
        haproxy_hosts = haproxy_component.get_hosts()
        for host in haproxy_hosts:
            try:
                if not check_potential_remote_file_exists(host, '/etc/keepalived/keepalived.conf'):
                    conf_file_path = os.path.join(self.get_conf_dir(), 'keepalived.conf')
                    # write out the conf file contents to /etc/keepalived/keepalived.conf file
                    # which is used by the keepalived systemd service
                    command = 'cat ' + conf_file_path + ' | sudo tee /etc/keepalived/keepalived.conf'
                    run_potential_ssh_command(host, command, config_parser)
                # now enable and start the service
                run_potential_ssh_command(host, 'sudo systemctl enable keepalived'
                                                ' && sudo systemctl restart keepalived',
                                          config_parser)
                logger.info('Started keepalived on ' + host)
            except Exception:
                # log and move on
                logger.debug('Failed to start keepalived on host ' + host, exc_info=1)
                logger.warning('Failed to start keepalived on host ' + host)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        haproxy_component = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
        if not haproxy_component.is_keepalived_necessary():
            logger.info('Skipping stop of keepalived since it isn\'t installed')
            return
        # we stop keepalived on each haproxy host
        haproxy_hosts = haproxy_component.get_hosts()
        for host in haproxy_hosts:
            try:
                run_potential_ssh_command(host, 'sudo systemctl stop keepalived', config_parser)
                logger.info('Stopped keepalived on ' + host)
            except Exception:
                # log and move on
                logger.debug('Failed to stop keepalived on host ' + host, exc_info=1)
                logger.warning('Failed to stop keepalived on host ' + host)

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
               k8s=False) -> List[dict]:
        statuses = []
        haproxy_component = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
        if not haproxy_component.is_keepalived_necessary():
            return []
        haproxy_hosts = haproxy_component.get_hosts()
        for host in haproxy_hosts:
            component_status = dict()
            statuses.append(component_status)
            component_status['component_name'] = self.get_name()
            component_status['host'] = host
            component_status['containers'] = []
            try:
                exit_code, output, stderr = run_command_exitcode("systemctl is-active keepalived",
                                                                 host, config_parser)
            except Exception:
                component_status['containers'] = [{'Id': 'N/A',
                                                   'State': 'Unknown',
                                                   'Status': 'Unknown',
                                                   'Image': 'N/A'}]
                continue

            if exit_code != 0:
                logger.debug('Status check for ' + self.get_name() + ' on host ' + host
                             + ' returned ' + str(exit_code))
                component_status['containers'] = [{'Id': 'N/A',
                                                   'State': 'Not Running',
                                                   'Status': 'Not Running',
                                                   'Image': 'N/A'}]
            else:
                component_status['containers'] = [{'Id': 'N/A',
                                                   'State': 'running',
                                                   'Status': output.strip(),
                                                   'Image': 'N/A'}]
        return statuses

    def healthcheck(self, component_name, host, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        try:
            exit_code, output, stderr = run_command_exitcode("systemctl is-active keepalived",
                                                             host, config_parser)
        except Exception as e:
            return [component_name, "Service Status",
                               termcolor.colored("Failed", color='red'), str(e), host]

        if exit_code != 0:
            logger.debug('Status check for ' + self.get_name() + ' on host ' + host
                         + ' returned ' + str(exit_code))
            return [component_name, "Service Status",
                               termcolor.colored("Failed", color='red'), "Not Running", host]

        return [component_name, "Service Status", "OK", "N/A", host]

    def restore_conf(self, config_parser: configparser.ConfigParser, backup_content_root_dir: os.path):
        for host in self.get_hosts():
            run_potential_ssh_command(host, 'sudo chown -R ' + str(os.getuid()) + ' '
                                      + self.get_conf_dir(), config_parser)
        super().restore_conf(config_parser, backup_content_root_dir)

    def get_netmask_cidr(self, interface_name, config_parser):
        haproxy_component: haproxy.HAProxy = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
        command = f"ip -o -4 addr show {interface_name} | awk '{{print $4}}'"
        exit_code, stdout, stderr = run_command_exitcode(command, haproxy_component.get_hosts()[0], config_parser)
        if exit_code != 0 and not stdout:
            return '/24'
        return '/' + stdout.strip().split("/")[-1]

