import argparse
import json
import os
import logging
import shutil
import string
import tempfile
from time import sleep
from typing import Callable, Any, List
import configparser
import rdaf
import yaml
from rdaf import rdafutils
from rdaf.component import Component, OtherCategoryOrder, _comma_delimited_to_list, _list_to_comma_delimited, \
    do_potential_scp, execute_command, remove_dir_contents, run_command, run_potential_ssh_command, \
    do_potential_scp_fetch, check_potential_remote_file_exists
from rdaf.contextual import COMPONENT_REGISTRY
import rdaf.component.opensearch as opensearch_comp
import rdaf.component.minio as minio_comp
from rdaf.dockerutils import cliDockerSession
from rdaf.rdafutils import str_base64_encode, str_base64_decode, cli_err_exit

logger = logging.getLogger(__name__)
COMPONENT_NAME = 'rda_log_monitoring'


class LogMonitoring(Component):
    _option_host = 'host'
    _opensearch_user = 'opensearch_user'
    _opensearch_password = 'opensearch_password'

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'log_monitoring',  category='other',
                         category_order=OtherCategoryOrder.LOG_MONITORING.value)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host:
            return _comma_delimited_to_list
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host:
            return _list_to_comma_delimited
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_host] = None
        default_configs[self._opensearch_user] = None
        default_configs[self._opensearch_password] = None
        return default_configs

    def get_hosts(self) -> list:
        if not self.configs[self._option_host]:
            return []
        return self.configs[self._option_host]

    def get_ports(self) -> tuple:
        ports = ['5045', '5043']
        hosts = self.get_hosts()
        return hosts, ports

    def get_k8s_component_name(self):
        return 'rda-logstash'

    def get_k8s_chart_name(self):
        return 'rda_logstash'

    @staticmethod
    def logstash_install_root():
        return os.path.join('/opt', 'rdaf', 'logstash')

    @staticmethod
    def filebeat_install_root():
        return os.path.join('/opt', 'rdaf', 'file-beat')

    def get_logsatsh_deployment_file_path(self) -> os.path:
        return os.path.join("/opt/rdaf/deployment-scripts", 'logstash.yaml')

    def get_filebeat_deployment_file_path(self) -> os.path:
        return os.path.join("/opt/rdaf/deployment-scripts", 'filebeat.yaml')

    def get_logsatsh_deployment_replacements(self, cmd_args: argparse.Namespace) -> dict:
        replacements = self._get_docker_repo()
        replacements['TAG'] = cmd_args.tag
        replacements['LOGSTASH_CONFIG'] = os.path.join(self.logstash_install_root(), 'config')
        replacements['LOGSTASH_PIPELINE'] = os.path.join(self.logstash_install_root(), 'pipeline')
        replacements['LOGSTASH_TEMPLATES'] = os.path.join(self.logstash_install_root(), 'templates')
        replacements['LOGSTASH_DATA'] = os.path.join(self.logstash_install_root(), 'data')
        replacements['LOGSTASH_LOGS'] = os.path.join(self.logstash_install_root(), 'logs')
        return replacements

    def get_filebeat_deployment_replacements(self, cmd_args: argparse.Namespace) -> dict:
        replacements = self._get_docker_repo()
        replacements['TAG'] = cmd_args.tag
        replacements['FILEBEAT_CONFIG'] = os.path.join(self.filebeat_install_root(), 'config')
        replacements['FILEBEAT_DATA'] = os.path.join(self.filebeat_install_root(), 'data')
        replacements['FILEBEAT_LOGS'] = os.path.join(self.filebeat_install_root(), 'logs')
        replacements['FILEBEAT_LOG_CONFIG'] = os.path.join(self.filebeat_install_root(), 'log_config')
        replacements['FILEBEAT_CERTS'] = os.path.join(self.filebeat_install_root(), 'certs')
        replacements['FILEBEAT_TEMPLATES'] = os.path.join(self.filebeat_install_root(), 'templates')
        return replacements

    def create_compose_file_logstash(self, cmd_args: argparse.Namespace):
        compose_file = self.get_logsatsh_deployment_file_path()
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), os.path.basename(compose_file))
        replacements = self.get_logsatsh_deployment_replacements(cmd_args)
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        original_values = yaml.safe_load(original_content)
        self.apply_user_inputs('logstash', original_values['services']['logstash'])
        with open(compose_file, 'w') as f:
            yaml.safe_dump(original_values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        for host in self.get_hosts():
            if not Component.is_local_host(host):
                do_potential_scp(host, compose_file, compose_file)

    def create_compose_file_file_beat(self, cmd_args: argparse.Namespace):
        compose_file = self.get_filebeat_deployment_file_path()
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), os.path.basename(compose_file))
        replacements = self.get_filebeat_deployment_replacements(cmd_args)
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        original_values = yaml.safe_load(original_content)
        self.apply_user_inputs('filebeat', original_values['services']['filebeat'])
        with open(compose_file, 'w') as f:
            yaml.safe_dump(original_values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        for host in all_known_hosts:
            if not Component.is_local_host(host):
                do_potential_scp(host, compose_file, compose_file)

    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != 'k8s':
            return
        self.pull_images(cmd_args, config_parser)

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        network_json = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'config.json')
        if not os.path.exists(network_json):
            rdafutils.cli_err_exit('Platform services are not installed...')
        with open(network_json) as f:
            data = json.load(f)
        tenant_id = data['tenant_id']

        log_monitoring_hosts = cmd_args.log_monitoring_host
        self.configs[self._option_host] = _comma_delimited_to_list(log_monitoring_hosts)
        self._mark_configured(self.configs, config_parser)
        self.label_nodes()
        # creating a log_streaming user for rdaf_service_logs index
        # -stream-rdaf-services-logs
        index = '{}-stream-rdaf-services-logs'.format(tenant_id)
        if not self.configs[self._opensearch_user]:
            es_port = '9200'
            if self.get_deployment_type(config_parser) == "k8s":
                es_port = self.get_service_node_port('opensearch-cluster-master', config_parser)
            es = COMPONENT_REGISTRY.require(opensearch_comp.COMPONENT_NAME)
            username, password = es.setup_log_streaming_user(config_parser, index, es_port)
            self.configs[self._opensearch_user] = str_base64_encode(username)
            self.configs[self._opensearch_password] = str_base64_encode(password)
            self._mark_configured(self.configs, config_parser)
            self.write_configs(config_parser)

        username = str_base64_decode(self.configs[self._opensearch_user])
        password = str_base64_decode(self.configs[self._opensearch_password])

        data = {"number_of_shards": 3, "number_of_replicas": 1, "refresh_interval": "60s"}
        with tempfile.NamedTemporaryFile(mode='w') as f:
            f.write(json.dumps(data))
            f.flush()
            index_settings = os.path.join("/tmp", os.path.basename(f.name))
            registry_pod = self.get_pods_names(config_parser, 'app_component=rda-registry')[0]
            copy_setting = 'kubectl -n {} cp {} {}:{}'\
                .format(namespace, f.name, registry_pod.split('/')[-1], index_settings)
            run_command(copy_setting)
            logger.info("Creating rdaf_services_logs pstream...")
            pstream_creation = 'kubectl -n {} exec {} -- bash -c "rdac pstream add ' \
                               '--name rdaf_services_logs --index {} --retention_days 15 ' \
                               '--timestamp @timestamp --os_index_settings {}"'\
                .format(namespace, registry_pod, index, index_settings)
            run_command(pstream_creation)

        minio = COMPONENT_REGISTRY.require(minio_comp.COMPONENT_NAME)
        minio.enable_logstash_minio_logs(config_parser, port='30443')

        # generate values.yaml and do an install of charts
        replacements = self._get_docker_repo()
        replacements['INDEX'] = index
        replacements['NAMESPACE'] = namespace
        replacements['OS_USER'] = username
        replacements['OS_PASS'] = password
        replacements['REPLICAS'] = len(self.get_hosts())
        template_path = os.path.join(rdaf.get_templates_dir_root(), 'k8s-local', 'log-monitoring-values.yaml')
        dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'log-monitoring-values.yaml')
        with open(template_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        with open(dest_path, 'w') as f:
            f.write(original_content)

        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'log-monitoring-values.yaml')
        install_command = 'helm install  --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(), deployment_path)
        run_command(install_command)

        # file beat
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), 'rda-filebeat')
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', 'rda-filebeat')
        self.copy_helm_chart(chart_template_path, deployment_path)
        install_command = 'helm install  --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), 'rda-filebeat', deployment_path)
        run_command(install_command)
        logger.info('Successfully installed and configured rdaf log monitoring...')

    def get_k8s_install_args(self, cmd_args):
        args = '--set tag={} '.format(cmd_args.tag)
        return args

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'log-monitoring-values.yaml')
        install_command = 'helm upgrade --install  --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(install_command)

        # file beat
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), 'rda-filebeat')
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', 'rda-filebeat')
        self.copy_helm_chart(chart_template_path, deployment_path)
        install_command = 'helm upgrade --install  --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), 'rda-filebeat', deployment_path)
        run_command(install_command)

    def pull_images(self, cmd_args, config_parser):
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        if self.configs.get(self._option_host):
            hosts = self.get_hosts()
        else:
            hosts = _comma_delimited_to_list(cmd_args.log_monitoring_host)
        for host in hosts:
            logger.info(f'Pulling logstash image on host {host}')
            docker_pull_command = f'docker pull {docker_repo}/rda-platform-logstash:{cmd_args.tag}'
            run_potential_ssh_command(host, docker_pull_command, config_parser)
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        for host in all_known_hosts:
            logger.info(f'Pulling filebeat image on host {host}')
            docker_pull_command = f'docker pull {docker_repo}/rda-platform-filebeat:{cmd_args.tag}'
            run_potential_ssh_command(host, docker_pull_command, config_parser)

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        network_json = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'config.json')
        if not os.path.exists(network_json):
            rdafutils.cli_err_exit('config.json is missing.. please install rdaf platform')
        with open(network_json) as f:
            data = json.load(f)
        tenant_id = data['tenant_id']

        if not shutil.which('rdac'):
            rdafutils.cli_err_exit('rdac is not installed...Please install rdac')

        self.open_ports(config_parser)
        log_monitoring_hosts = cmd_args.log_monitoring_host
        self.configs[self._option_host] = _comma_delimited_to_list(log_monitoring_hosts)
        self._mark_configured(self.configs, config_parser)
        self.write_configs(config_parser)
        # doing a docker login
        docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
        for host in self.get_hosts():
            docker_registry.docker_login(host, config_parser)
        self._setup_install_root_dir_hierarchy(config_parser)
        self.copy_cert()
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        # creating a log_streaming user for rdaf_service_logs index
        # -stream-rdaf-services-logs
        index = '{}-stream-rdaf-services-logs'.format(tenant_id)
        if not self.configs[self._opensearch_user]:
            es = COMPONENT_REGISTRY.require(opensearch_comp.COMPONENT_NAME)
            username, password = es.setup_log_streaming_user(config_parser, index)
            self.configs[self._opensearch_user] = str_base64_encode(username)
            self.configs[self._opensearch_password] = str_base64_encode(password)
            self._mark_configured(self.configs, config_parser)
            self.write_configs(config_parser)

        username = str_base64_decode(self.configs[self._opensearch_user])
        password = str_base64_decode(self.configs[self._opensearch_password])

        data = {"number_of_shards": 3, "number_of_replicas": 1, "refresh_interval": "60s"}
        user_data_path = os.path.expanduser('~/rdac_data/')
        with tempfile.NamedTemporaryFile(mode='w', dir=user_data_path) as f:
            f.write(json.dumps(data))
            f.flush()
            index_settings = os.path.join("/data", os.path.basename(f.name))
            cmd = 'rdac pstream add --name rdaf_services_logs --index {} ' \
                  '--retention_days 15 --timestamp @timestamp --os_index_settings {}'.format(index, index_settings)
            run_command(cmd)

        self.create_compose_file_logstash(cmd_args)
        self.create_compose_file_file_beat(cmd_args)

        command = '/usr/local/bin/docker-compose -f {file} pull && ' \
                  '/usr/local/bin/docker-compose --project-name logstash -f {file} up -d '\
            .format(file=self.get_logsatsh_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Installing logstash on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

        command = '/usr/local/bin/docker-compose -f {file} pull && ' \
                  '/usr/local/bin/docker-compose --project-name file-beat -f {file} up -d '\
            .format(file=self.get_filebeat_deployment_file_path())
        for host in all_known_hosts:
            logger.info("Installing file-beat on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)
        sleep(20)
        self.file_beat_config(config_parser)
        self.logstash_config(index, username, password, config_parser)
        self.restart(config_parser)

        minio = COMPONENT_REGISTRY.require(minio_comp.COMPONENT_NAME)
        minio.enable_logstash_minio_logs(config_parser)
        logger.info('Successfully installed and configured rdaf log streaming')

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        # TODO: we will remove this in coming releases
        self.open_ports(config_parser)
        self.create_compose_file_logstash(cmd_args)
        self.create_compose_file_file_beat(cmd_args)
        command = '/usr/local/bin/docker-compose -f {file} pull && ' \
                  '/usr/local/bin/docker-compose --project-name logstash -f {file} up -d ' \
            .format(file=self.get_logsatsh_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Upgrading logstash on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        command = '/usr/local/bin/docker-compose -f {file} pull && ' \
                  '/usr/local/bin/docker-compose --project-name file-beat -f {file} up -d ' \
            .format(file=self.get_filebeat_deployment_file_path())
        # TODO: we will address this in coming releases
        for host in all_known_hosts:
            logger.info("Upgrading file-beat on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)
        sleep(20)
        self.file_beat_config(config_parser)
        self.restart(config_parser)

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        namespace = self.get_namespace(config_parser)
        statuses = []
        service_host_map = {}
        status_command = f'kubectl get pods -n {namespace} -l "app_component in (rda-logstash,rda-filebeat)" -o json'
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of component log monitoring, due to: {}.".format(str(stderr)))

        result = json.loads(str(stdout))
        items = result['items']
        hosts = self.get_hosts()
        for host in hosts:
            deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
            if not items and os.path.exists(deployment_path):
                    statuses.append({
                        'component_name': self.get_k8s_component_name(),
                        'host': host,
                        'containers': [{
                            'Id': 'N/A',
                            'Image': 'N/A',
                            'State': 'N/A',
                            'Status': 'Not Provisioned'
                        }]
                    })
            else:
                for item in items:
                    pod = dict()
                    pod['component_name'] = item['metadata']['labels']['app_component']
                    pod['host'] = item['status'].get('hostIP', 'Unknown')
                    pod['containers'] = []
                    if 'containers' in item['spec']:
                        for entry in item['spec']['containers']:
                            if 'image' in entry.keys():
                                container = dict()
                                image = entry['image']
                    if 'containerStatuses' in item['status']:
                        for entry in item['status']['containerStatuses']:
                            if 'containerID' in entry.keys():
                                container = dict()
                                container['Id'] = entry['containerID']
                                container['Image'] = image
                                for key in entry['state'].keys():
                                    container['State'] = key
                                    if key == 'running':
                                        container['Status'] = self.get_container_age(entry['state'][key]['startedAt'])
                                    else:
                                        container['Status'] = key
                                    break
                                pod['containers'].append(container)
                    service_host_map.setdefault(pod['component_name'], {})
                    service_host_map[pod['component_name']][pod['host']] = pod
        
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])

        for key, value in service_host_map.items():
            if key == 'rda-filebeat':
                for host in all_known_hosts:
                    if host in value.keys():
                        statuses.append(value.get(host))
                    else:
                        entry = dict()
                        statuses.append(entry)
                        entry['component_name'] = key
                        entry['host'] = host
                        entry['containers'] = []
            else:
                for host in value.keys():
                        statuses.append(value.get(host))
        return statuses

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
               k8s=False) -> List[dict]:
        statuses = []
        if not os.path.exists(self.get_logsatsh_deployment_file_path()):
            return statuses

        # logstash service status
        for host in self.get_hosts():
            component_status = dict()
            statuses.append(component_status)
            component_status['component_name'] = 'logstash'
            component_status['host'] = host
            try:
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = self._find_component_container_on_host(docker_client,
                                                                        'logstash', all_states=True)
                    if len(containers) == 0:
                        logger.debug('No container found for logstash on host ' + host)
                        component_status['containers'] = []
                    else:
                        component_status['containers'] = containers
            except Exception:
                logger.debug('Failed to get status of logstash on host ' + host, exc_info=1)
                # set the status as error
                component_status['status'] = {'error': True, 'message': 'Unknown'}

        # file-beat service status
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        for host in all_known_hosts:
            component_status = dict()
            statuses.append(component_status)
            component_status['component_name'] = 'filebeat'
            component_status['host'] = host
            try:
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = self._find_component_container_on_host(docker_client,
                                                                        'filebeat', all_states=True)
                    if len(containers) == 0:
                        logger.debug(
                            'No container found for filebeat on host ' + host)
                        component_status['containers'] = []
                    else:
                        component_status['containers'] = containers
            except Exception:
                logger.debug('Failed to get status of filebeat on host ' + host, exc_info=1)
                # set the status as error
                component_status['status'] = {'error': True, 'message': 'Unknown'}
        return statuses

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_logsatsh_deployment_file_path()):
            return

        command = '/usr/local/bin/docker-compose --project-name logstash -f {} up -d ' \
            .format(self.get_logsatsh_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Creating logstash services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        command = '/usr/local/bin/docker-compose --project-name file-beat -f {} up -d ' \
            .format(self.get_filebeat_deployment_file_path())
        for host in all_known_hosts:
            logger.info("Creating file-beat services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        # logstash down
        cmd = f'kubectl get pods -n {namespace} -l app_component=rda-logstash -o json'
        ret, stdout, stderr = execute_command(cmd)
        components = json.loads(stdout)
        if not components["items"]:
            logger.warning(f"No Pods found for logstash in namespace {namespace}")
        else:
            for component in components["items"]:
                comp = component["metadata"]["labels"]["app_component"]
                if comp == 'rda-logstash':
                    logger.info(f"Deleting service: {comp} ")
                    command = f'kubectl scale deployments.apps/{comp} -n {namespace} --replicas=0'
                    run_command(command)
                    if cmd_args.force:
                        delete_pods_command = f'kubectl delete pods -n {namespace} -l app_component={comp} --grace-period=0 --force'
                        run_command(delete_pods_command)

        # Check if rda-filebeat exists in Helm list
        cmd = f'helm list -n {namespace} -q'
        ret, stdout, stderr = execute_command(cmd)

        if 'rda-filebeat' not in stdout:
            logger.warning(f"Helm release 'rda-filebeat' not found in namespace {namespace}")
        else:
            logger.info(f"Uninstalling helm chart: rda-filebeat")
            cmd = f'helm uninstall -n {namespace} rda-filebeat'
            run_command(cmd)
            if cmd_args.force:
                delete_pods_command = f'kubectl delete pods -n {namespace} -l app_component=rda-filebeat --grace-period=0 --force'
                run_command(delete_pods_command)

    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        # logstash up
        cmd = f'kubectl get deployments -n {namespace} -l app_component=rda-logstash -o json'
        ret, stdout, stderr = execute_command(cmd)
        components = json.loads(stdout)

        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'log-monitoring-values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
        service_map = {'rda-logstash': 'rda_logstash'}
        for component in components["items"]:
            comp = component["metadata"]["labels"]["app_component"]
            replicas = data[service_map[comp]]['replicas']
            command = f'kubectl scale deployment.apps/{comp} -n {namespace} --replicas={replicas}'
            run_command(command)
            logger.info(f"Creating service: {comp} ")

        # file-beat up
        image_tag_cmd = (
            "kubectl get deployment -n rda-fabric -l app_component=rda-logstash -o json "
            "| jq -r '.items[].spec.template.spec.containers[].image'")
        ret, stdout, stderr = execute_command(image_tag_cmd)
        tag = stdout.split(":")[-1]
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'log-monitoring-values.yaml')
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), 'rda-filebeat')
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', 'rda-filebeat')
        self.copy_helm_chart(chart_template_path, deployment_path)
        install_command = ( 'helm upgrade --install {} {} '
            '--create-namespace -n {} -f {} --set tag={}').format('rda-filebeat', deployment_path, namespace, values_yaml, tag)
        run_command(install_command)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_logsatsh_deployment_file_path()):
            return

        command = '/usr/local/bin/docker-compose --project-name logstash -f {file} rm -fsv'\
            .format(file=self.get_logsatsh_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Deleting logstash service on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

        command = '/usr/local/bin/docker-compose --project-name file-beat -f {file} rm -fsv'\
            .format(file=self.get_filebeat_deployment_file_path())
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        for host in all_known_hosts:
            logger.info("Deleting file-beat service on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_logsatsh_deployment_file_path()):
            return

        command = '/usr/local/bin/docker-compose --project-name logstash -f {} start ' \
            .format(self.get_logsatsh_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Starting logstash services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        command = '/usr/local/bin/docker-compose --project-name file-beat -f {} start '\
            .format(self.get_filebeat_deployment_file_path())
        for host in all_known_hosts:
            logger.info("Starting filebeat services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_logsatsh_deployment_file_path()):
            return
        command = '/usr/local/bin/docker-compose --project-name logstash -f {} stop ' \
            .format(self.get_logsatsh_deployment_file_path())
        for host in reversed(self.get_hosts()):
            logger.info("Stopping logstash services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        command = '/usr/local/bin/docker-compose --project-name file-beat -f {} stop ' \
            .format(self.get_filebeat_deployment_file_path())
        for host in all_known_hosts:
            logger.info("Stopping filebeat services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def restart(self, config_parser: configparser.ConfigParser):
        command = '/usr/local/bin/docker-compose --project-name logstash -f {} restart ' \
            .format(self.get_logsatsh_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Restarting logstash services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        command = '/usr/local/bin/docker-compose --project-name file-beat -f {} restart ' \
            .format(self.get_filebeat_deployment_file_path())
        for host in all_known_hosts:
            logger.info("Restarting filebeat services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def reset(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_logsatsh_deployment_file_path()):
            return
        self.down(cmd_args, config_parser)
        logstash_install_root = self.logstash_install_root()
        for host in self.get_hosts():
            if os.path.exists(logstash_install_root):
                try:
                    remove_dir_contents(host, logstash_install_root, config_parser, use_sudo=True)
                except Exception:
                    logger.debug('Failed to delete ' + logstash_install_root + ' on host ' + host)

        filebeat_install_root = self.filebeat_install_root()
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        for host in all_known_hosts:
            if os.path.exists(filebeat_install_root):
                try:
                    remove_dir_contents(host, filebeat_install_root, config_parser, use_sudo=True)
                except Exception:
                    logger.debug('Failed to delete ' + filebeat_install_root + ' on host ' + host)

    def logstash_config(self, index, opensearch_user, opensearch_password,
                        config_parser: configparser.ConfigParser):
        uid = os.getuid()
        gid = os.getgid()
        logstash_host = self.get_hosts()[0]
        rdaf_service_conf = os.path.join('/opt/rdaf/logstash/pipeline/rda_services.conf')
        rdaf_minio_conf = os.path.join('/opt/rdaf/logstash/pipeline/rda_minio.conf')
        filebeats_docker_conf = os.path.join('/opt/rdaf/logstash/pipeline/filebeats_docker.conf')
        cmd = "sudo chmod -R 666 " + rdaf_minio_conf + ' ' + rdaf_service_conf + ' ' + filebeats_docker_conf + ' && sudo chown -R ' + str(
                    uid) + ' ' + rdaf_minio_conf + ' ' + rdaf_service_conf + ' ' + filebeats_docker_conf + ' && sudo chgrp -R ' + str(
                    gid) + ' ' + rdaf_minio_conf + ' ' + rdaf_service_conf + ' ' + filebeats_docker_conf
        run_potential_ssh_command(logstash_host, cmd, config_parser)
        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            do_potential_scp_fetch(logstash_host, rdaf_service_conf, os.path.join(tmp, 'rda_services.conf'), is_dir=False)
            do_potential_scp_fetch(logstash_host, rdaf_minio_conf, os.path.join(tmp, 'rda_minio.conf'), is_dir=False)
            do_potential_scp_fetch(logstash_host, filebeats_docker_conf, os.path.join(tmp, 'filebeats_docker.conf'), is_dir=False)

            opensearch = COMPONENT_REGISTRY.require(opensearch_comp.COMPONENT_NAME)
            opensearch_endpoint = ','.join(['"https://{}:9200"'.format(host) for host in opensearch.get_hosts()])
            with open(os.path.join(tmp, 'rda_services.conf')) as service_config:
                data = service_config.read()
                data = data.replace('INDEX_NAME', index)
                data = data.replace('OS_HOSTS', opensearch_endpoint)
                data = data.replace('OS_USERNAME', opensearch_user)
                data = data.replace('OS_PASSWORD', opensearch_password)
            with open(os.path.join(tmp, 'rda_services.conf'), 'w') as file:
                file.write(data)

            with open(os.path.join(tmp, 'rda_minio.conf')) as minio_config:
                data = minio_config.read()
                data = data.replace('INDEX_NAME', index)
                data = data.replace('OS_HOSTS', opensearch_endpoint)
                data = data.replace('OS_USERNAME', opensearch_user)
                data = data.replace('OS_PASSWORD', opensearch_password)
            with open(os.path.join(tmp, 'rda_minio.conf'), 'w') as file:
                file.write(data)

            with open(os.path.join(tmp, 'filebeats_docker.conf')) as filebeats_config:
                data = filebeats_config.read()
                data = data.replace('INDEX_NAME', index)
                data = data.replace('OS_HOSTS', opensearch_endpoint)
                data = data.replace('OS_USERNAME', opensearch_user)
                data = data.replace('OS_PASSWORD', opensearch_password)
            with open(os.path.join(tmp, 'filebeats_docker.conf'), 'w') as file:
                file.write(data)

            for host in self.get_hosts():
                run_potential_ssh_command(host, cmd, config_parser)
                do_potential_scp(host, os.path.join(tmp, 'rda_services.conf'), rdaf_service_conf)
                do_potential_scp(host, os.path.join(tmp, 'rda_minio.conf'), rdaf_minio_conf)
                do_potential_scp(host, os.path.join(tmp, 'filebeats_docker.conf'), filebeats_docker_conf)

    def file_beat_config(self, config_parser):
        file_beat_output_conf = os.path.join('/opt/rdaf/file-beat/config/filebeat.yml')
        cmd = "sudo chmod -R 666 " + file_beat_output_conf
        run_command(cmd)
        # TODO need to handle for upcoming release
        data_directory = '/opt/rdaf/file-beat/data'
        data_cmd = "sudo chmod -R 777 " + data_directory
        run_command(data_cmd)
        logstash_endpoint = [i + ':5043' for i in self.get_hosts()]
        # Load the Filebeat configuration file
        with open(file_beat_output_conf, 'r') as config_file:
            lines = config_file.readlines()

        # Modify the Logstash hosts line with the desired host
        for i, line in enumerate(lines):
            if line.strip().startswith('hosts:'):
                lines[i] = f'  hosts: {logstash_endpoint}\n'
                #lines[i] = '  hosts: [ "new_host:5043"]\n'
                break

        with open(file_beat_output_conf, 'w', encoding='utf-8') as file:
            file.writelines(lines)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        log_config = '/opt/rdaf/file-beat/log_config'
        source_path = os.path.join(rdaf.get_templates_dir_root(), "rdaf_infra_svc.yml")
        log_command = ('sudo mkdir -p ' + log_config + ' && sudo chmod -R 777 ' + log_config
                       + ' && sudo touch /opt/rdaf/file-beat/log_config/.log_config-folder-created')
        dest_path = os.path.join(log_config, 'rdaf_infra_svc.yml')
        for host in all_known_hosts:
            run_potential_ssh_command(host, log_command, config_parser)
            if not Component.is_local_host(host):
                do_potential_scp(host, source_path, dest_path, sudo=True)
                run_potential_ssh_command(host, cmd, config_parser)
                run_potential_ssh_command(host, data_cmd, config_parser)
                do_potential_scp(host, file_beat_output_conf, file_beat_output_conf, sudo=True)
            else:
                run_command(f"sudo cp -f {source_path} {dest_path}")

    def copy_cert(self):
        src_path = os.path.join('/opt', 'rdaf', 'cert', 'ca', 'ca.crt')
        dest_path = os.path.join('/opt', 'rdaf', 'logstash', 'config', 'cert', 'ca.crt')
        for host in self.get_hosts():
            do_potential_scp(host, src_path, dest_path)

    def _setup_install_root_dir_hierarchy(self, config_parser):
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        uid = os.getuid()
        gid = os.getgid()
        logstash_dirs = [os.path.join(self.logstash_install_root(), 'config', 'cert'),
                         os.path.join(self.logstash_install_root(), 'data'),
                         os.path.join(self.logstash_install_root(), 'logs'),
                         os.path.join(self.logstash_install_root(), 'templates'),
                         os.path.join(self.logstash_install_root(), 'pipeline')]
        file_beat_dirs = [os.path.join(self.filebeat_install_root(), 'config'),
                           os.path.join(self.filebeat_install_root(), 'data'),
                           os.path.join(self.filebeat_install_root(), 'logs')]
        for host in self.get_hosts():
            for path in logstash_dirs:
                logger.info('Creating directory on host ' + host + path + ' and setting ownership to user '
                            + str(uid) + ' and group to group ' + str(gid))
                command = 'sudo mkdir -p ' + path + ' && sudo chown -R ' + str(
                    uid) + ' ' + path + ' && sudo chgrp -R ' + str(
                    gid) + ' ' + path
                run_potential_ssh_command(host, command, config_parser)
        for host in all_known_hosts:
            for path in file_beat_dirs:
                logger.info('Creating directory on host ' + host + path + ' and setting ownership to user '
                            + str(uid) + ' and group to group ' + str(gid))
                command = 'sudo mkdir -p ' + path + ' && sudo chown -R ' + str(
                    uid) + ' ' + path + ' && sudo chgrp -R ' + str(
                    gid) + ' ' + path
                run_potential_ssh_command(host, command, config_parser)

    def _find_component_container_on_host(self, docker_client: cliDockerSession, service,
                                          all_states: bool = False) -> List:
        labels = {'com.docker.compose.service': service}
        component_filter_labels = []
        for k, v in labels.items():
            component_filter_labels.append(str(k + '=' + v))
        return docker_client.client.containers(all=all_states,
                                               filters={'label': component_filter_labels})

    def label_nodes(self):
        # label the nodes
        logger.info("Labelling the monitoring nodes")
        # <ip, node>
        nodes = {}
        ret, stdout, stderr = execute_command('kubectl get nodes -o json')
        if ret != 0:
            cli_err_exit("Failed to get nodes of kubernetes cluster, due to: {}."
                         .format(self.get_k8s_component_name(), str(stderr)))

        result = json.loads(str(stdout))
        for node in result['items']:
            # we don't use master nodes for deploying
            if 'node-role.kubernetes.io/control-plane' in node['metadata']['labels']:
                continue

            for i in node['status']['addresses']:
                if i['type'] == "InternalIP":
                    nodes[i['address']] = node['metadata']['name']
                    break

        for host in self.get_hosts():
            if not nodes[host]:
                rdafutils.cli_err_exit("unknown monitoring host {}, Please use one of the "
                                       "kubernetes worker nodes".format(host))

            logger.info('applying node label "rdaf_monitoring_services=allow" for monitoring host: ' + host)
            run_command('kubectl label nodes {} {} --overwrite'.format(nodes[host], 'rdaf_monitoring_services=allow'))

    def backup_conf(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_dir_root: os.path):
        # copy/backup from each component specific host
        for host in self.get_hosts():
            if not check_potential_remote_file_exists(host, self.logstash_install_root()):
                return
            dest_config_dir = os.path.join(backup_dir_root, 'config', host, 'logstash')
            logger.info('Backing up logstash from host ' + host + ' to ' + dest_config_dir)
            os.makedirs(dest_config_dir, exist_ok=True)
            do_potential_scp_fetch(host, self.logstash_install_root(), dest_config_dir)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        for host in all_known_hosts:
            if not check_potential_remote_file_exists(host, self.filebeat_install_root()):
                return
            data_directory = '/opt/rdaf/file-beat/data'
            data_cmd = "sudo chmod -R 777 " + data_directory
            run_potential_ssh_command(host, data_cmd, config_parser)
            dest_config_dir = os.path.join(backup_dir_root, 'config', host, 'file-beat')
            logger.info('Backing up file-beat from host ' + host + ' to ' + dest_config_dir)
            os.makedirs(dest_config_dir, exist_ok=True)
            do_potential_scp_fetch(host, self.filebeat_install_root(), dest_config_dir)

    def restore_conf(self, config_parser: configparser.ConfigParser, backup_content_root_dir: os.path):
        for host in self.get_hosts():
            config_backup_dir = os.path.join(backup_content_root_dir, 'config', host, 'logstash')
            if not os.path.exists(config_backup_dir):
                # no backed up content exists
                return

            run_potential_ssh_command(host, 'mkdir -p ' + self.logstash_install_root(), config_parser)
            logger.info('Restoring ' + config_backup_dir + ' to ' + self.logstash_install_root() + ' on host ' + host)
            do_potential_scp(host, config_backup_dir, self.logstash_install_root(), sudo=True)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        for host in all_known_hosts:
            config_backup_dir = os.path.join(backup_content_root_dir, 'config', host, 'file-beat')
            if not os.path.exists(config_backup_dir):
                # no backed up content exists
                return
            run_potential_ssh_command(host, 'mkdir -p ' + self.filebeat_install_root(), config_parser)
            logger.info('Restoring ' + config_backup_dir + ' to '
                        + self.filebeat_install_root() + ' on host ' + host)
            do_potential_scp(host, config_backup_dir, self.filebeat_install_root(), sudo=True)
