import argparse
import json
import os
import logging
import socket
import string
import subprocess
import tempfile
import time
from typing import Callable, Any, List, Tuple
import configparser
import rdaf

from rdaf import get_templates_dir_root
from rdaf.component import Component, InfraCategoryOrder, execute_command, run_command_exitcode, \
    run_potential_ssh_command
from rdaf.component import _list_to_comma_delimited, run_command, _comma_delimited_to_list,\
    _host_dir_loader, _host_dir_storer, _apply_data_dir_defaults
from rdaf.contextual import COMPONENT_REGISTRY
import rdaf.component.haproxy as haproxy_comp
from rdaf.rdafutils import str_base64_encode, gen_password, str_base64_decode, cli_err_exit
import termcolor

logger = logging.getLogger(__name__)
COMPONENT_NAME = 'minio'


class Minio(Component):
    _option_host = 'host'
    _option_data_dir = 'datadir'
    _option_user = 'user'
    _option_password = 'password'

    _default_data_dirs = ['/minio-data']

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'minio', 'infra', InfraCategoryOrder.MINIO.value)

    def get_k8s_component_name(self):
        return 'rda-minio'

    def get_k8s_chart_name(self):
        return 'rda_minio'

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_data_dir] = None
        default_configs[self._option_host] = None
        default_configs[self._option_user] = None
        default_configs[self._option_password] = None
        return default_configs

    def get_hosts(self) -> list:
        return self.configs[self._option_host]

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host:
            return _list_to_comma_delimited
        if config_name == self._option_data_dir:
            # convert the list of tuple to a comma separate value
            return _host_dir_storer
        return None

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host:
            return _comma_delimited_to_list
        if config_name == self._option_data_dir:
            # convert the comma separated value to list of tuples
            return _host_dir_loader
        return None

    def get_user(self) -> str:
        return str_base64_decode(self.configs[self._option_user])

    def get_password(self) -> str:
        return str_base64_decode(self.configs[self._option_password])

    def get_escaped_password(self) -> str:
        return self.get_password().replace("$", "\\$").replace("&", "/&")

    def get_minio_host_ports(self) -> List[Tuple]:
        host_ports = []
        for host in self.configs[self._option_host]:
            host_ports.append((host, 9000))
        return host_ports

    def get_ports(self) -> tuple:
        ports = ['9000']
        hosts = self.get_hosts()
        return hosts, ports

    def _get_host_data_dirs(self) -> List[Tuple[str, List[str]]]:
        return self.configs[self._option_data_dir]

    def gather_minimal_setup_inputs(self, cmd_args, config_parser):
        minio_configs = self._init_default_configs()
        minio_configs[self._option_host] = [self.get_default_host()]
        minio_configs[self._option_data_dir] = [(self.get_default_host(), self._default_data_dirs)]
        minio_configs[self._option_user] = str_base64_encode('rdafadmin')
        minio_configs[self._option_password] = str_base64_encode(gen_password(num_chars=8))
        self._mark_configured(minio_configs, config_parser)

    def gather_setup_inputs(self, cmd_args, config_parser):
        minio_configs = self._init_default_configs()
        default_host_name = Component.get_default_host()
        no_prompt_err_msg = 'No minio server host specified. Use ' \
                            '--minio-server-host to specify one'
        host_desc = 'What is the "host/path-on-host" where you want Minio to be provisioned?'
        host_dirs = Component._parse_or_prompt_host_dirs(cmd_args.minio_server_host,
                                                         default_host_name,
                                                         no_prompt_err_msg, host_desc,
                                                         'Minio server host/path',
                                                         cmd_args.no_prompt)
        minio_configs[self._option_data_dir] = _apply_data_dir_defaults(host_dirs,
                                                                        self._default_data_dirs)
        hosts = []
        for host, data_dirs in host_dirs:
            hosts.append(host)
        minio_configs[self._option_host] = hosts

        minio_user_desc = 'What is the user name you want to give for Minio root user that ' \
                          'will be created and used by the RDAF platform?'
        user_no_prompt_err_msg = 'No Minio user specified. Use --minio-user to specify one'
        user = rdaf.component.Component._parse_or_prompt_value(cmd_args.minio_user,
                                                               'rdafadmin',
                                                               user_no_prompt_err_msg,
                                                               minio_user_desc,
                                                               'Minio user',
                                                               cmd_args.no_prompt)
        minio_configs[self._option_user] = str_base64_encode(user)

        pass_desc = 'What is the password you want to use for the newly created ' \
                    'Minio root user?'
        pass_no_prompt_err_msg = 'No Minio password specified. Use --minio-password' \
                                 ' to specify one'
        default_autogen_password = gen_password(num_chars=8)
        passwd = rdaf.component.Component._parse_or_prompt_value(cmd_args.minio_password,
                                                                 default_autogen_password,
                                                                 pass_no_prompt_err_msg,
                                                                 pass_desc, 'Minio password',
                                                                 cmd_args.no_prompt,
                                                                 password=True, password_min_length=8)
        minio_configs[self._option_password] = str_base64_encode(passwd)
        self._mark_configured(minio_configs, config_parser)

    def do_setup(self, cmd_args, config_parser):
        # mc tool
        mc_path = os.path.join('/usr', 'local', 'bin', 'mc')
        mc_file = os.path.join(rdaf.get_scripts_dir_root(), 'mc')
        run_command('sudo cp ' + mc_file + ' ' + mc_path + ' && '
                    + 'sudo chmod +x ' + mc_path)

    def do_k8s_setup(self, cmd_args, config_parser):
        namespace = self.get_namespace(config_parser)
        self.do_setup(cmd_args, config_parser)
        replacements = self._get_docker_repo()
        replacements['REPLICAS'] = len(self.get_hosts())
        replacements['DEPLOYMENT_MODE'] = 'standalone' if len(self.get_hosts()) == 1 else 'distributed'
        replacements['MINIO_USER'] = str_base64_decode(self.configs[self._option_user])
        replacements['MINIO_PASSWORD'] = str_base64_decode(self.configs[self._option_password])

        template_dir = 'k8s-local'
        if self.get_deployment_type(config_parser) == "aws":
            template_dir = 'k8s-aws'
        template_path = os.path.join(get_templates_dir_root(), template_dir, 'minio-values.yaml')
        dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'minio-values.yaml')
        with open(template_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        with open(dest_path, 'w') as f:
            f.write(original_content)
        pv_template_path = os.path.join(get_templates_dir_root(), 'k8s-local', 'minio-pv-cluster.yaml')
        pv_dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'minio-pv-cluster.yaml')
        replacements['NAMESPACE'] = namespace
        with open(pv_template_path, 'r') as f:
            pv_template_content = f.read()
        pv_original_content = string.Template(pv_template_content).safe_substitute(replacements)
        with open(pv_dest_path, 'w') as f:
            f.write(pv_original_content)

    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != 'k8s':
            return
        registry = self._get_docker_repo()['DOCKER_REPO']
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            for image in ['minio', 'mc']:
                tag = cmd_args.tag
                if tag == '1.0.4':
                    tag = 'RELEASE.2024-12-18T13-15-44Z' if image == 'minio' else 'RELEASE.2024-11-21T17-21-54Z'
                logger.info(f'Pulling {image} on host {host}')
                docker_pull_command = f'docker pull {registry}/{image}:{tag}'
                run_potential_ssh_command(host, docker_pull_command, config_parser)

    def pull_images(self, cmd_args, config_parser):
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            docker_pull_command = f'docker pull {docker_repo}/minio:RELEASE.2024-12-18T13-15-44Z'
            run_potential_ssh_command(host, docker_pull_command, config_parser)

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        if self.get_deployment_type(config_parser) != "aws":
            minio_pvs = (
                os.path.join('/opt', 'rdaf', 'deployment-scripts', 'minio-pv-cluster.yaml')
                if len(self.get_hosts()) > 1
                else os.path.join(get_templates_dir_root(), 'k8s-local', 'minio-pv.yaml'))
            run_command('kubectl apply -f {} -n {}'.format(minio_pvs, namespace))

        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'minio-values.yaml')
        install_command = 'helm install --create-namespace -n {} -f {} {} {}' \
            .format(namespace, values_yaml, self.get_k8s_component_name(), deployment_path)
        run_command(install_command)

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'minio-values.yaml')
        upgrade_command = 'helm upgrade --install --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(upgrade_command)

    def get_k8s_component_label(self):
        return 'release={}'.format(self.get_k8s_component_name())

    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        label = self.get_k8s_component_label()
        cmd = f'kubectl get pods -l {label} -n {namespace} -o json'
        ret, stdout, stderr = execute_command(cmd)
        component = json.loads(stdout)
        for items in component["items"]:
            metadata = items["metadata"]["ownerReferences"]
            for pod in metadata:
                if pod["kind"] == "StatefulSet":
                    command = 'kubectl scale statefulset.apps/{} -n {} --replicas=0'\
                        .format(self.get_k8s_component_name(), namespace)
                    run_command(command)
                    return
                if pod["kind"] == "ReplicaSet":
                    command = 'kubectl scale deployment.apps/{} -n {} --replicas=0'\
                        .format(self.get_k8s_component_name(), namespace)
                    run_command(command)
                    return

    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        label = self.get_k8s_component_label()
        component_name = self.get_k8s_component_name()
        cmd = f'kubectl get all -l {label} -n {namespace} -o json'
        ret, stdout, stderr = execute_command(cmd)
        component = json.loads(stdout)
        replicas = len(self.get_hosts())
        for pod in component["items"]:
            if pod["kind"] == 'ReplicaSet':
                command = f'kubectl scale deployment.apps/{component_name} -n {namespace} --replicas={replicas}'
                run_command(command)
            if pod["kind"] == "StatefulSet":
                command = f'kubectl scale statefulset.apps/{component_name} -n {namespace} --replicas={replicas}'
                run_command(command)

    def healthcheck(self, component_name, host, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        root_user = str_base64_decode(self.configs[self._option_user])
        root_password = str_base64_decode(self.configs[self._option_password])
        val = 'http://{user}:{password}@{host}:9000'.format(user=root_user,
                                                            password=root_password, host=host)
        env = dict(os.environ, MC_HOST_myminio=val)
        minio_info = 'mc --insecure admin info myminio/'
        ret, stdout, stderr = run_command_exitcode(minio_info, socket.gethostname(), config_parser, env=env)
        if ret != 0:
            return [component_name, "Service Status", termcolor.colored("Failed", color='red'),
                    'Failed to connect to minio']

        return [component_name, "Service Status", "OK", "N/A"]

    def backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        minio_backup_dir = os.path.join(backup_dir_root, 'data', self.get_name())
        os.makedirs(minio_backup_dir, exist_ok=True)
        root_user = str_base64_decode(self.configs[self._option_user])
        root_password = str_base64_decode(self.configs[self._option_password])
        host = self.get_hosts()[0]
        val = 'http://{user}:{password}@{host}:9000'.format(user=root_user,
                                                            password=root_password, host=host)

        env = dict(os.environ, MC_HOST_myminio=val)
        buckets_list = 'mc ls --insecure myminio/'
        ret, stdout, stderr = run_command_exitcode(buckets_list, socket.gethostname(),
                                                   config_parser, env=env)
        if ret != 0:
            cli_err_exit('Failed to list buckets on minio' + str(stderr))
        buckets = []
        for entry in stdout.split('\n'):
            bucket_name = entry.strip().split(' ')[-1]
            if bucket_name:
                buckets.append(bucket_name)

        for entry in buckets:
            logger.info("backing up bucket: " + entry)
            os.makedirs(os.path.join(minio_backup_dir, entry))
            mirror = 'mc mirror --insecure --overwrite --preserve myminio/{entry} ' \
                     '{dest}/{entry}'.format(entry=entry, dest=minio_backup_dir)
            run_command(mirror, env=env)

    def k8s_backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                        backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        namespace = self.get_namespace(config_parser)
        minio_backup_dir = os.path.join(backup_dir_root, 'data', self.get_name())
        os.makedirs(minio_backup_dir, exist_ok=True)
        root_user = str_base64_decode(self.configs[self._option_user])
        root_password = str_base64_decode(self.configs[self._option_password])
        pod = self.get_pods_names(config_parser, 'app_component=rda-minio')[0]
        pf_proc = subprocess.Popen(f'exec kubectl port-forward {pod} -n {namespace} 9000:9000', shell=True)
        time.sleep(10)
        host = '127.0.0.1'
        port = '9000'
        val = 'http://{user}:{password}@{host}:{port}'.format(user=root_user, port=port,
                                                              password=root_password, host=host)
        try:
            env = dict(os.environ, MC_HOST_myminio=val)
            buckets_list = 'mc ls --insecure myminio/'
            ret, stdout, stderr = run_command_exitcode(buckets_list, socket.gethostname(),
                                                       config_parser, env=env)
            if ret != 0:
                cli_err_exit('Failed to list buckets on minio' + str(stderr))
            buckets = []
            for entry in stdout.split('\n'):
                bucket_name = entry.strip().split(' ')[-1]
                if bucket_name:
                    buckets.append(bucket_name)

            for entry in buckets:
                logger.info("backing up bucket: " + entry)
                os.makedirs(os.path.join(minio_backup_dir, entry))
                mirror = 'mc mirror --insecure --overwrite --preserve myminio/{entry} ' \
                         '{dest}/{entry}'.format(entry=entry, dest=minio_backup_dir)

                run_command(mirror, env=env)
        finally:
            pf_proc.kill()

    def _get_docker_repo(self) -> dict:
        env = dict()
        from rdaf.component.dockerregistry import COMPONENT_NAME
        from rdaf.contextual import COMPONENT_REGISTRY
        docker_registry = COMPONENT_REGISTRY.require(COMPONENT_NAME)
        docker_registry_host_port = docker_registry.get_docker_registry_host_port()
        if (docker_registry_host_port[0].startswith('docker1.cloudfabrix.io') or
                docker_registry_host_port[0] in ('cfx-docker-01.engr.cloudfabrix.com',
                                                'cfxregistry.cloudfabrix.io')):
            env['DOCKER_REPO'] = 'quay.io/minio'
        else:
            env['DOCKER_REPO'] = docker_registry.get_docker_registry_url()
        return env

    def get_deployment_env(self, host: str):
        env = dict()
        # TODO need to handle for cluster
        minio_host, data_dirs = self.configs[self._option_data_dir][0]
        env['MINIO_MOUNT'] = data_dirs[0]
        env['MINIO_ROOT_USER'] = str_base64_decode(self.configs[self._option_user]).replace("$", "$$")
        env['MINIO_ROOT_PASSWORD'] = str_base64_decode(self.configs[self._option_password]).replace("$", "$$")

        # data dirs within the container
        data_dir_urls = []
        single_drive_fs_mode_installation = False
        if len(self.configs[self._option_data_dir]) == 1:
            minio_host, data_dirs = self.configs[self._option_data_dir][0]
            if len(data_dirs) == 1:
                single_drive_fs_mode_installation = True
                data_dir = data_dirs[0]
                # single filesystem backed minio installation
                data_dir_urls.append(data_dir)
        if not single_drive_fs_mode_installation:
            for minio_host, data_dirs in self.configs[self._option_data_dir]:
                for data_dir in data_dirs:
                    data_dir_urls.append('http://' + minio_host + data_dir)
        env['MINIO_DATA_DRIVES'] = ' '.join(data_dir_urls)
        return env

    def setup_tenant_user(self, uid, user, password, config_parser: configparser.ConfigParser, port='9000'):
        logger.info("Setting up Minio tenant access...")
        namespace = self.get_namespace(config_parser)
        self.load_config(config_parser)
        root_user = str_base64_decode(self.configs[self._option_user])
        root_password = str_base64_decode(self.configs[self._option_password])
        host = self.get_hosts()[0]
        pf_proc = None
        if self.get_deployment_type(config_parser) == 'aws':
            pod = self.get_pods_names(config_parser, 'app_component=rda-minio')[0]
            pf_proc = subprocess.Popen(f'exec kubectl port-forward {pod} -n {namespace} 9000:9000', shell=True)
            time.sleep(10)
            host = '127.0.0.1'
            port = '9000'
        val = 'http://{user}:{password}@{host}:{port}'.format(user=root_user, port=port,
                                                              password=root_password, host=host)

        env = dict(os.environ, MC_HOST_myminio=val)
        mc = os.path.join('/usr', 'local', 'bin', 'mc')
        user_add = '{mc} admin user add --insecure myminio/ {user} {password}'\
            .format(mc=mc, user=user, password=password)
        enable_user = '{mc} admin user enable --insecure myminio/ {user}'.format(mc=mc, user=user)
        make_bucket = '{mc} mb --insecure --ignore-existing myminio/tenants.{UUID}'.format(mc=mc, UUID=uid)
        run_command(user_add + ' && ' + enable_user + ' && ' + make_bucket, env=env)

        policy_template = self._create_minio_user_policy_content(uid)

        with tempfile.NamedTemporaryFile(mode='w') as tf:
            tf.write(policy_template)
            tf.flush()
            add_policy = '{mc} admin policy create  --insecure myminio/ {user}-readwrite {template}'\
                .format(mc=mc, user=user, template=tf.name)
            set_policy = '{mc} admin policy attach  --insecure myminio/ {user}-readwrite --user={user}'\
                .format(mc=mc, user=user)

            run_command(add_policy + ' && ' + set_policy, env=env)

        if pf_proc:
            pf_proc.kill()

    @staticmethod
    def _create_minio_user_policy_content(uid) -> str:
        template_root = get_templates_dir_root()
        user_policy = os.path.join(template_root, 'minio-user-policy.template')
        with open(user_policy, 'r') as f:
            template_content = f.read()

        replacements = dict()
        replacements['RDABUCKETNAME'] = 'tenants.{UUID}'.format(UUID=uid)
        substituted = string.Template(template_content).substitute(replacements)
        conf_content = '\n\n' + substituted

        return conf_content

    def required_container_state_before_restore(self):
        return 'running'

    def restore_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                     backup_content_root_dir: os.path,
                     backup_cfg_parser: configparser.ConfigParser):
        # TODO: run health check before issuing restore command
        # for restoring minio we require minio to be up and running
        minio_backup_dir = os.path.join(backup_content_root_dir, 'data', self.get_name())
        if not os.path.isdir(minio_backup_dir):
            logger.warning('Minio data backup directory ' + minio_backup_dir
                           + ' is missing, skipping restoration of minio data')
            return

        root_user = str_base64_decode(self.configs[self._option_user])
        root_password = str_base64_decode(self.configs[self._option_password])
        host = self.get_hosts()[0]
        val = 'http://{user}:{password}@{host}:9000'.format(user=root_user,
                                                            password=root_password, host=host)
        env = dict(os.environ, MC_HOST_myminio=val)
        buckets_list = 'mc ls --insecure myminio/'
        ret, stdout, stderr = run_command_exitcode(buckets_list, socket.gethostname(),
                                                   config_parser, env=env)
        if ret != 0:
            cli_err_exit('Failed to list buckets on minio' + str(stderr))
        buckets = []
        for entry in stdout.split('\n'):
            bucket_name = entry.strip().split(' ')[-1]
            if bucket_name:
                buckets.append(bucket_name)

        # clean up data dir(s)
        logger.info('Initiating a data cleanup before restoration of ' + self.get_name())
        # mc --insecure rm --recursive --force myminio/tenants.xxxxxxx/
        for bucket in buckets:
            cleanup = 'mc --insecure rb --force myminio/{bucket}'.format(bucket=bucket)
            ret, stdout, stderr = run_command_exitcode(cleanup, socket.gethostname(),
                                                       config_parser, env=env)
            if ret != 0:
                cli_err_exit('Failed to remove bucket ' + str(bucket) + ' ' + str(stderr))

        # sleep to recover from cleanup
        time.sleep(10)
        for entry in os.listdir(minio_backup_dir):
            command = 'mc --insecure mb -p myminio/{}'.format(entry)
            ret, stdout, stderr = run_command_exitcode(command, socket.gethostname(),
                                                       config_parser, env=env)
            if ret != 0:
                cli_err_exit('Failed to create bucket' + str(entry) + ' ' + str(stderr))

            mirror = 'mc --insecure mirror --remove --overwrite --preserve {src}/{entry} ' \
                     'myminio/{entry}'.format(entry=entry, src=minio_backup_dir)

            ret, stdout, stderr = run_command_exitcode(mirror, socket.gethostname(),
                                                       config_parser, env=env)
            if ret != 0:
                cli_err_exit('Failed to restore bucket' + str(entry) + ' ' + str(stderr))

    def check_pod_state_for_restore(self, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = 'kubectl get pods -n ' + namespace + ' --selector=app_component=rda-minio ' \
                  '-o jsonpath="{.items[*].status.phase}"'
        exit_code, stdout, stderr = execute_command(command)
        if 'Running' not in str(stdout):
            cli_err_exit('Minio pod should be Running for restore')

    def k8s_restore_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                         backup_content_root_dir: os.path, backup_cfg_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        minio_backup_dir = os.path.join(backup_content_root_dir, 'data', self.get_name())
        if not os.path.isdir(minio_backup_dir):
            logger.warning('Minio data backup directory ' + minio_backup_dir
                           + ' is missing, skipping restoration of minio data')
            return

        pod = self.get_pods_names(config_parser, 'app_component=rda-minio')[0]
        pf_proc = subprocess.Popen('exec kubectl port-forward {} -n {} 9000:9000'.format(pod, namespace),
                                   shell=True)
        time.sleep(10)

        root_user = str_base64_decode(self.configs[self._option_user])
        root_password = str_base64_decode(self.configs[self._option_password])
        val = 'http://{user}:{password}@{host}:{port}'.format(user=root_user, port='9000',
                                                              password=root_password, host='127.0.0.1')

        env = dict(os.environ, MC_HOST_myminio=val)
        buckets_list = 'mc ls --insecure myminio/'
        ret, stdout, stderr = run_command_exitcode(buckets_list, socket.gethostname(),
                                                   config_parser, env=env)
        if ret != 0:
            pf_proc.kill()
            cli_err_exit('Failed to list buckets on minio' + str(stderr))
        buckets = []
        for entry in stdout.split('\n'):
            bucket_name = entry.strip().split(' ')[-1]
            if bucket_name:
                buckets.append(bucket_name)

        # clean up data dir(s)
        logger.info('Initiating a data cleanup before restoration of ' + self.get_name())
        # mc --insecure rm --recursive --force myminio/tenants.xxxxxxx/
        for bucket in buckets:
            cleanup = 'mc --insecure rb --force myminio/{bucket}'.format(bucket=bucket)
            ret, stdout, stderr = run_command_exitcode(cleanup, socket.gethostname(),
                                                       config_parser, env=env)
            if ret != 0:
                pf_proc.kill()
                cli_err_exit('Failed to remove bucket ' + str(bucket) + ' ' + str(stderr))

        # sleep to recover from cleanup
        time.sleep(10)
        for entry in os.listdir(minio_backup_dir):
            command = 'mc --insecure mb -p myminio/{}'.format(entry)
            ret, stdout, stderr = run_command_exitcode(command, socket.gethostname(),
                                                       config_parser, env=env)
            if ret != 0:
                pf_proc.kill()
                cli_err_exit('Failed to create bucket' + str(entry) + ' ' + str(stderr))

            mirror = 'mc --insecure mirror --remove --overwrite --preserve {src}/{entry} ' \
                     'myminio/{entry}'.format(entry=entry, src=minio_backup_dir)

            ret, stdout, stderr = run_command_exitcode(mirror, socket.gethostname(),
                                                       config_parser, env=env)
            if ret != 0:
                pf_proc.kill()
                cli_err_exit('Failed to restore bucket' + str(entry) + ' ' + str(stderr))

        pf_proc.kill()

    def enable_logstash_minio_logs(self, config_parser, port='9000'):
        namespace = self.get_namespace(config_parser)
        logger.info('configuring minio services logs')
        if port == '9000':
            host = COMPONENT_REGISTRY.require(haproxy_comp.COMPONENT_NAME).get_internal_access_host() + ":5049"
        else:
            host = 'rdaf-logstash.{}.svc.cluster.local:5046'.format(namespace)

        minio_host = self.get_hosts()[0]
        pf_proc = None
        if self.get_deployment_type(config_parser) == 'aws':
            pod = self.get_pods_names(config_parser, 'app_component=rda-minio')[0]
            pf_proc = subprocess.Popen('exec kubectl port-forward {} -n {} 9000:9000'.format(pod,namespace),
                                       shell=True)
            time.sleep(10)
            minio_host = '127.0.0.1'
            port = '9000'
        minio_user = str_base64_decode(self.configs[self._option_user])
        minio_pass = str_base64_decode(self.configs[self._option_password])

        val = 'http://{user}:{password}@{host}:{port}'.format(user=minio_user, port=port,
                                                              password=minio_pass, host=minio_host)
        env = dict(os.environ, MC_HOST_myminio=val)
        cmd = 'mc admin config set myminio/ logger_webhook:"rdaf_log_streaming" endpoint="http://{}"'.format(host)
        run_command(cmd, env=env)
        cmd = 'mc admin config set myminio/ audit_webhook:"rdaf_log_streaming" endpoint="http://{}"'.format(host)
        run_command(cmd, env=env)

        if pf_proc:
            pf_proc.kill()

    def geodr_start_replication(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        minio_rep_user = 'rdafrepuser'
        minio_rep_password = 'rdafAbcd1234'
        root_user = str_base64_decode(self.configs[self._option_user])
        root_password = str_base64_decode(self.configs[self._option_password])
        primary_host = self.get_hosts()[0]
        val = f'http://{root_user}:{root_password}@{primary_host}:9000'
        env = dict(os.environ, MC_HOST_myminio=val)

        user_add = f'mc admin user add --insecure myminio/ {minio_rep_user} {minio_rep_password}'
        enable_user = f'mc admin user enable --insecure myminio/ {minio_rep_user}'
        run_command(user_add + ' && ' + enable_user, env=env)

        logger.info("Creating replication admin policy.")
        rep_admin_policy='''
{
    "Version": "2012-10-17",
    "Statement":
    [
        {
            "Action":
            [
                "admin:SetBucketTarget",
                "admin:GetBucketTarget"
            ],
            "Effect": "Allow",
            "Sid": "EnableRemoteBucketConfiguration"
        },
        {
            "Effect": "Allow",
            "Action":
            [
                "s3:GetReplicationConfiguration",
                "s3:ListBucket",
                "s3:ListBucketMultipartUploads",
                "s3:GetBucketLocation",
                "s3:GetBucketVersioning",
                "s3:GetObjectRetention",
                "s3:GetObjectLegalHold",
                "s3:PutReplicationConfiguration"
            ],
            "Resource":
            [
                "arn:aws:s3:::*"
            ],
            "Sid": "EnableReplicationRuleConfiguration"
        }
    ]
}'''
        with tempfile.NamedTemporaryFile(mode='w') as tf:
            tf.write(rep_admin_policy)
            tf.flush()
            create_policy = f'mc admin policy create --insecure myminio/ rep-admin-policy {tf.name}'
            run_command(create_policy, env=env)
            attach_policy = f'mc admin policy attach  --insecure myminio/ rep-admin-policy --user={minio_rep_user}'
            run_command(attach_policy, env=env)
        logger.info("Creating replication remote user policy.")
        rep_remote_user_policy = '''
        {
            "Version": "2012-10-17",
            "Statement":
            [
                {
                    "Effect": "Allow",
                    "Action":
                    [
                        "s3:GetReplicationConfiguration",
                        "s3:ListBucket",
                        "s3:ListBucketMultipartUploads",
                        "s3:GetBucketLocation",
                        "s3:GetBucketVersioning",
                        "s3:GetBucketObjectLockConfiguration",
                        "s3:GetEncryptionConfiguration"
                    ],
                    "Resource":
                    [
                        "arn:aws:s3:::*"
                    ],
                    "Sid": "EnableReplicationOnBucket"
                },
                {
                    "Effect": "Allow",
                    "Action":
                    [
                        "s3:GetReplicationConfiguration",
                        "s3:ReplicateTags",
                        "s3:AbortMultipartUpload",
                        "s3:GetObject",
                        "s3:GetObjectVersion",
                        "s3:GetObjectVersionTagging",
                        "s3:PutObject",
                        "s3:PutObjectRetention",
                        "s3:PutBucketObjectLockConfiguration",
                        "s3:PutObjectLegalHold",
                        "s3:DeleteObject",
                        "s3:ReplicateObject",
                        "s3:ReplicateDelete"
                    ],
                    "Resource":
                    [
                        "arn:aws:s3:::*"
                    ],
                    "Sid": "EnableReplicatingDataIntoBucket"
                }
            ]
        }
                '''
        with tempfile.NamedTemporaryFile(mode='w') as tf:
            tf.write(rep_remote_user_policy)
            tf.flush()
            create_policy = f'mc admin policy create --insecure myminio/ rep-remote-user-policy {tf.name}'
            run_command(create_policy, env=env)
            attach_policy = f'mc admin policy attach  --insecure myminio/ rep-remote-user-policy --user={minio_rep_user}'
            run_command(attach_policy, env=env)

        logger.info(f"Enabling versioning on the tenant bucket on {primary_host}")
        # using the replication user
        tenant_id = config_parser.get('common', 'tenant_id')
        if not tenant_id:
            cli_err_exit("Tenant Id is missing from rdaf.cfg.")
        versioning = f'mc version enable --excluded-prefixes "lep/,persistent-streams/ingest/" --insecure myminio/tenants.{tenant_id}'
        run_command(versioning, env=env)
        list_cmd = f'mc ls --insecure myminio/tenants.{tenant_id}'
        run_command(list_cmd, env=env)
        ilm_rule_cmd = f'mc ilm rule add myminio/tenants.{tenant_id} --noncurrent-expire-days 1 --expire-delete-marker'
        run_command(ilm_rule_cmd, env=env)

        logger.info("Configuring secondary minio.")
        secondary_host = peer_configs.get('minio', 'host').split(',')[0]
        val = f'http://{root_user}:{root_password}@{secondary_host}:9000'
        env = dict(os.environ, MC_HOST_myminio=val)

        user_add = f'mc admin user add --insecure myminio/ {minio_rep_user} {minio_rep_password}'
        enable_user = f'mc admin user enable --insecure myminio/ {minio_rep_user}'
        run_command(user_add + ' && ' + enable_user, env=env)

        logger.info("Creating replication admin policy on secondary")
        with tempfile.NamedTemporaryFile(mode='w') as tf:
            tf.write(rep_admin_policy)
            tf.flush()
            create_policy = f'mc admin policy create --insecure myminio/ rep-admin-policy {tf.name}'
            run_command(create_policy, env=env)
            attach_policy = f'mc admin policy attach  --insecure myminio/ rep-admin-policy --user={minio_rep_user}'
            run_command(attach_policy, env=env)
        logger.info("Creating replication remote user policy on secondary")
        with tempfile.NamedTemporaryFile(mode='w') as tf:
            tf.write(rep_remote_user_policy)
            tf.flush()
            create_policy = f'mc admin policy create --insecure myminio/ rep-remote-user-policy {tf.name}'
            run_command(create_policy, env=env)
            attach_policy = f'mc admin policy attach  --insecure myminio/ rep-remote-user-policy --user={minio_rep_user}'
            run_command(attach_policy, env=env)

        logger.info("Enabling versioning on the tenant bucket..")
        make_bucket = f'mc mb --insecure --ignore-existing myminio/tenants.{tenant_id}'
        run_command(make_bucket, env=env)
        versioning = f'mc version enable --excluded-prefixes "lep/,persistent-streams/ingest/" --insecure myminio/tenants.{tenant_id}'
        run_command(versioning, env=env)
        list_cmd = f'mc ls --insecure myminio/tenants.{tenant_id}'
        run_command(list_cmd, env=env)
        ilm_rule_cmd = f'mc ilm rule add myminio/tenants.{tenant_id} --noncurrent-expire-days 1 --expire-delete-marker'
        run_command(ilm_rule_cmd, env=env)
        logger.info(f"Enabling the replication from primary {primary_host} to secondary {secondary_host}")
        val = f'http://{root_user}:{root_password}@{primary_host}:9000'
        env = dict(os.environ, MC_HOST_myminio=val)
        replicate_cmd = (f'mc replicate add --remote-bucket http://{minio_rep_user}:{minio_rep_password}@{secondary_host}:9000/tenants.{tenant_id} '
                         f'--replicate "delete,delete-marker,existing-objects" myminio/tenants.{tenant_id}')
        run_command(replicate_cmd, env=env)


    def geodr_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                 peer_configs: configparser.ConfigParser):

        tenant_id = config_parser.get('common', 'tenant_id')
        root_user = str_base64_decode(self.configs[self._option_user])
        root_password = str_base64_decode(self.configs[self._option_password])
        primary_host = self.get_hosts()[0]

        replicate_status_command = f'mc replicate status myminio/tenants.{tenant_id}'
        val = f'http://{root_user}:{root_password}@{primary_host}:9000'
        env = dict(os.environ, MC_HOST_myminio=val)

        logger.info(f"Checking replication status for minio")
        run_command(replicate_status_command, env=env)

    def geodr_stop_replication(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        tenant_id = config_parser.get('common', 'tenant_id')
        root_user = str_base64_decode(self.configs[self._option_user])
        root_password = str_base64_decode(self.configs[self._option_password])
        primary_host = self.get_hosts()[0]

        replicate_status_command = f'mc replicate remove --all --force myminio/tenants.{tenant_id}'
        val = f'http://{root_user}:{root_password}@{primary_host}:9000'
        env = dict(os.environ, MC_HOST_myminio=val)

        run_command(replicate_status_command, env=env)
        logger.info(f"Stopping minio replication on {primary_host}.")

        ilm_rule_delete_command = f'mc ilm rule remove --all --force myminio/tenants.{tenant_id}'
        run_command(ilm_rule_delete_command, env=env)
        logger.info(f"Deleted ilm rule on {primary_host}.")