import argparse
import configparser
import logging
import os
import string
import json
import yaml
from typing import List
import rdaf.component
import rdaf.component.haproxy as haproxy
import rdaf
from rdaf import get_templates_dir_root
from rdaf.component import (Component, InfraCategoryOrder, create_file,
                            run_potential_ssh_command, execute_command, run_command, do_potential_scp)
import rdaf.component.haproxy
from rdaf.rdafutils import cli_err_exit
from rdaf.contextual import COMPONENT_REGISTRY

logger = logging.getLogger(__name__)
COMPONENT_NAME = 'nginx'


class Nginx(Component):
    
    def __init__(self):
        super().__init__(COMPONENT_NAME, 'nginx', 'infra', InfraCategoryOrder.NGINX.value)

    def get_k8s_component_name(self):
        return 'rda-nginx'

    def get_k8s_chart_name(self):
        return 'rda_nginx'
    
    def get_deployment_file_path(self) -> os.path:
        return os.path.join('/opt', 'rdaf', 'deployment-scripts', 'nginx.yaml')

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        namespace = self.get_namespace(config_parser)
        statuses = []
        status_command = 'kubectl get pods -n {} -l app_component=rda-nginx -o json'\
            .format(namespace)
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of external opensearch, due to: {}.".format(str(stderr)))
        result = json.loads(str(stdout))
        items = result['items']
        hosts = self.get_hosts()
        for host in hosts:
            deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'nginx-values.yaml')
            if not items and os.path.exists(deployment_path):
                    statuses.append({
                        'component_name': self.get_k8s_component_name(),
                        'host': host,
                        'containers': [{
                            'Id': 'N/A',
                            'Image': 'N/A',
                            'State': 'N/A',
                            'Status': 'Not Provisioned'
                        }]
                    })
        else:
            for item in items:
                pod = dict()
                statuses.append(pod)
                pod['component_name'] = self.get_k8s_component_name()
                pod['host'] = item['status'].get('hostIP', 'Unknown')
                pod['containers'] = []
                if 'containerStatuses' in item['status']:
                    for entry in item['status']['containerStatuses']:
                        if 'containerID' in entry.keys():
                            container = dict()
                            container['Id'] = entry['containerID']
                            container['Image'] = entry['image']
                            for key in entry['state'].keys():
                                container['State'] = key
                                if key == 'running':
                                    container['Status'] = self.get_container_age(entry['state'][key]['startedAt'])
                                else:
                                    container['Status'] = key
                                break
                            pod['containers'].append(container)
            return statuses

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not getattr(cmd_args, 'external_url', None):
            cli_err_exit("The --external-url argument is required.")
        self.create_cert_configs(config_parser)
        replacements = self._get_docker_repo()
        haproxy_component: haproxy.HAProxy = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
        advertised_ext_host = haproxy_component.get_advertised_external_host()
        haproxy_host = haproxy_component.get_hosts()
        replacements['TAG'] = cmd_args.tag
        replacements['REPLICAS'] = 1 if len(haproxy_host) == 1 else 2
        replacements['IP'] = advertised_ext_host
        replacements['EXTERNAL_IP']  = cmd_args.external_url
        nginx_template_path = os.path.join(get_templates_dir_root(), 'k8s-local', 'nginx-values.yaml')
        nginx_dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'nginx-values.yaml')
        with open(nginx_template_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        with open(nginx_dest_path, 'w') as f:
                f.write(original_content) 
                
        namespace = self.get_namespace(config_parser)
        if self.get_deployment_type(config_parser) != "aws":
            cert_secret_2 = 'kubectl create secret generic -n {} nginx-tls ' \
                        '--from-file=rdaf.cert=/opt/rdaf/cert/rdaf/rdaf.cert ' \
                        '--from-file=rdaf.key=/opt/rdaf/cert/rdaf/rdaf.key ' \
                        '--save-config --dry-run=client -o yaml | kubectl apply -f -'.format(namespace)
            run_command(cert_secret_2)
            nginx_chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), 'rda-nginx')
            nginx_deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', 'rda-nginx')
            self.copy_helm_chart(nginx_chart_template_path, nginx_deployment_path)

            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'nginx-values.yaml')
            nginx_install_command = 'helm install --create-namespace -n {} -f {} {} {} {} ' \
                .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), 'rda-nginx', nginx_deployment_path)
            run_command(nginx_install_command)
            logger.info("Waiting for nginx pod to be up and running...")
            nginx_status = 'kubectl wait --for=condition=Ready pod --timeout=600s --all -n {} ' \
                                '-l app_component=rda-nginx'.format(namespace)
            run_command(nginx_status)
    
    def k8s_upgrade(self, cmd_args, config_parser):
        namespace = self.get_namespace(config_parser)
        nginx_chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), 'rda-nginx')
        nginx_deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', 'rda-nginx')
        self.copy_helm_chart(nginx_chart_template_path, nginx_deployment_path)

        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'nginx-values.yaml')
        nginx_install_command = 'helm upgrade --install --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), 'rda-nginx', nginx_deployment_path)
        run_command(nginx_install_command)
    
    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = f'kubectl get deployment -n {namespace} -l app_component=rda-nginx -o json'
        ret, stdout, stderr = execute_command(command)
        components = json.loads(stdout)
        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'nginx-values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
        for component in components["items"]:
            comp = component["metadata"]["labels"]["app_component"]
            replicas = data['controller']['replicaCount']
            logger.info(f"Creating service: {comp} ")
            command = f'kubectl scale deployment.apps/rda-nginx -n {namespace} --replicas={replicas}'
            run_command(command)
            return

    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = f'kubectl get deployment -n {namespace} -l app_component=rda-nginx -o json'
        ret, stdout, stderr = execute_command(command)
        components = json.loads(stdout)
        for component in components["items"]:
            comp = component["metadata"]["labels"]["app_component"]
            logger.info(f"Deleting service: {comp} ")
            command = f'kubectl scale deployment.apps/rda-nginx -n {namespace} --replicas=0'
            run_command(command)

    def create_compose_file(self, cmd_args: argparse.Namespace):
        haproxy = COMPONENT_REGISTRY.require(rdaf.component.haproxy.COMPONENT_NAME)
        haproxy_host = haproxy.get_hosts()
        compose_file = self.get_deployment_file_path()
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), os.path.basename(compose_file))
        replacements = self.get_deployment_replacements(cmd_args)
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        original_values = yaml.safe_load(original_content)
        with open(compose_file, 'w') as f:
            yaml.safe_dump(original_values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        for host in haproxy_host:
            if not Component.is_local_host(host):
                do_potential_scp(host, compose_file, compose_file)

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not getattr(cmd_args, 'external_url', None):
            cli_err_exit("The --external-url argument is required.")
        haproxy = COMPONENT_REGISTRY.require(rdaf.component.haproxy.COMPONENT_NAME)
        haproxy_host = haproxy.get_hosts()

        logger.info("Generating nginx_config...")
        template_root = get_templates_dir_root()
        nginx_template = os.path.join(template_root, 'nginx.conf')
        with open(nginx_template, 'r') as f:
            template_content = f.read()

        haproxy_advertised_host = haproxy.get_internal_access_host()
        
        replacements= dict()
        replacements['IP'] = haproxy_advertised_host
        replacements['EXTERNAL_PORTAL_URL']  = cmd_args.external_url

        substituted_content = string.Template(template_content).safe_substitute(replacements)

        settings = os.path.join('/opt', 'rdaf', 'config', 'nginx_config', 'nginx.conf')
        uid = os.getuid()
        gid = os.getgid()
        settings_dir_command = (
                    'sudo mkdir -p ' + os.path.join('/opt', 'rdaf', 'config', 'nginx_config') + ' && sudo chown -R '
                    + str(uid) + ' ' + os.path.join('/opt', 'rdaf', 'config', 'nginx_config')
                    + ' && sudo chgrp -R ' + str(gid) + ' ' + os.path.join('/opt', 'rdaf', 'config', 'nginx_config'))
        for host in haproxy_host:
            run_potential_ssh_command(host, settings_dir_command, config_parser)
            created_location = create_file(host, substituted_content.encode(encoding='UTF-8'), settings)
            logger.info(f"Created {created_location} on {host} ")
            self.create_compose_file(cmd_args)
            deployment_file = self.get_deployment_file_path()
            command = '/usr/local/bin/docker-compose -f {file} pull && ' \
                      '/usr/local/bin/docker-compose -f {file} --project-name infra up -d '.format(file=deployment_file)
            logger.info(f"Installing nginx on host {host}")
            run_potential_ssh_command(host, command, config_parser)
    
    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        haproxy = COMPONENT_REGISTRY.require(rdaf.component.haproxy.COMPONENT_NAME)
        for host in haproxy.get_hosts():
           self.create_compose_file(cmd_args)
           deployment_file = self.get_deployment_file_path()
           command = '/usr/local/bin/docker-compose -f {file} pull && ' \
                     '/usr/local/bin/docker-compose -f {file} --project-name infra up -d '.format(file=deployment_file)
           logger.info(f"Upgrading nginx on host {host}")
           run_potential_ssh_command(host, command, config_parser)

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        statuses = []
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return statuses
        haproxy = COMPONENT_REGISTRY.require(rdaf.component.haproxy.COMPONENT_NAME)
        haproxy_host = haproxy.get_hosts()
        for host in haproxy_host:
            component_status = dict()
            statuses.append(component_status)
            component_status['component_name'] = self.get_name()
            component_status['host'] = host
            try:
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = self.find_component_container_on_host(docker_client, all_states=True)
                    if len(containers) == 0:
                        logger.debug(
                            'No container found for ' + self.get_name() + ' on host ' + host)
                        component_status['containers'] = []
                    else:
                        component_status['containers'] = containers
            except Exception:
                logger.debug('Failed to get status of ' + self.get_name() + ' on host ' + host,
                             exc_info=1)
                # set the status as error
                component_status['status'] = {'error': True, 'message': 'Unknown'}

        return statuses
    
    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        command = f'/usr/local/bin/docker-compose -f {deployment_file} --project-name infra up -d '
        haproxy = COMPONENT_REGISTRY.require(rdaf.component.haproxy.COMPONENT_NAME)
        haproxy_host = haproxy.get_hosts()
        for host in haproxy_host:
            logger.info(f"Creating nginx on host {host}")
            run_potential_ssh_command(host, command, config_parser)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        haproxy = COMPONENT_REGISTRY.require(rdaf.component.haproxy.COMPONENT_NAME)
        haproxy_host = haproxy.get_hosts()
        for host in haproxy_host:
            logger.info(f"Deleting nginx on host {host}")
            command = f'/usr/local/bin/docker-compose -f {deployment_file} --project-name infra rm -fs nginx'
            run_potential_ssh_command(host, command, config_parser)

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        command = f'/usr/local/bin/docker-compose -f {deployment_file} --project-name infra start '
        haproxy = COMPONENT_REGISTRY.require(rdaf.component.haproxy.COMPONENT_NAME)
        haproxy_host = haproxy.get_hosts()
        for host in haproxy_host:
            logger.info(f"Starting nginx on host {host}")
            run_potential_ssh_command(host, command, config_parser)

    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        haproxy = COMPONENT_REGISTRY.require(rdaf.component.haproxy.COMPONENT_NAME)
        haproxy_host = haproxy.get_hosts()
        for host in haproxy_host:
            logger.info(f"stopping nginx on host {host}")
            command = f'/usr/local/bin/docker-compose -f {deployment_file} --project-name infra stop nginx'
            run_potential_ssh_command(host, command, config_parser)
            