import argparse
import configparser
import datetime
import json
import os
import logging
import string
import subprocess
import tempfile
import time

import termcolor
import urllib3
import requests

import rdaf.component
import yaml
from docker.models.containers import ContainerCollection
from rdaf.component import cert, check_potential_remote_file_exists
from rdaf import rdafutils, get_templates_dir_root
from typing import Callable, Any, List, Tuple
from rdaf.component import Component, InfraCategoryOrder, _apply_data_dir_defaults, \
    _host_dir_storer, _host_dir_loader, run_potential_ssh_command, execute_command
from rdaf.component import _comma_delimited_to_list, run_command, do_potential_scp
from rdaf.contextual import COMPONENT_REGISTRY
import rdaf.component.platform as platform_comp
from rdaf.component import _list_to_comma_delimited
from rdaf.rdafutils import str_base64_encode, str_base64_decode, cli_err_exit
from rdaf.util import requestsutil

COMPONENT_NAME = 'opensearch'
logger = logging.getLogger(__name__)
_docker_image_name = 'opensearch'


class Opensearch(Component):
    _option_user = 'user'
    _option_password = 'password'
    _option_data_dir = 'datadir'
    _option_host = 'host'
    _default_data_dirs = ['/opensearch']

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'opensearch', 'infra',
                         InfraCategoryOrder.OPENSEARCH.value)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host:
            return _comma_delimited_to_list
        if config_name == self._option_data_dir:
            # convert the comma separated value to list of tuples
            return _host_dir_loader
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host:
            return _list_to_comma_delimited
        if config_name == self._option_data_dir:
            # convert the list of tuple to a comma separate value
            return _host_dir_storer
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_data_dir] = None
        default_configs[self._option_user] = None
        default_configs[self._option_password] = None
        default_configs[self._option_host] = None
        return default_configs

    def gather_minimal_setup_inputs(self, cmd_args, config_parser):
        opensearch_configs = self._init_default_configs()
        opensearch_configs[self._option_host] = [self.get_default_host()]
        opensearch_configs[self._option_data_dir] = [(self.get_default_host(), self._default_data_dirs)]
        opensearch_configs[self._option_user] = str_base64_encode('rdafadmin')
        opensearch_configs[self._option_password] = str_base64_encode(rdafutils.gen_password())
        self._mark_configured(opensearch_configs, config_parser)

    def gather_setup_inputs(self, cmd_args, config_parser):
        opensearch_configs = self._init_default_configs()
        default_host_name = Component.get_default_host()
        no_prompt_err_msg = 'No opensearch server host specified.'
        opensearch_host_desc = 'What is the "host/path-on-host" on ' \
                               'which you want the opensearch ' \
                               'server to be provisioned?'
        host_dirs = Component._parse_or_prompt_host_dirs(
            cmd_args.opensearch_server_host,
            default_host_name,
            no_prompt_err_msg, opensearch_host_desc,
            'opensearch server host/path',
            cmd_args.no_prompt)
        opensearch_configs[self._option_data_dir] = \
            _apply_data_dir_defaults(host_dirs, self._default_data_dirs)
        opensearch_hosts = []
        for host, data_dirs in opensearch_configs[self._option_data_dir]:
            opensearch_hosts.append(host)
        opensearch_configs[self._option_host] = opensearch_hosts

        opensearch_user_desc = 'What is the user name you want to give ' \
                               'for Opensearch admin user that ' \
                               'will be created and used by the RDAF platform?'
        user_no_prompt_err_msg = 'No Opensearch user specified. Use ' \
                                 '--opensearch-user to specify one'
        user = rdaf.component.Component._parse_or_prompt_value(cmd_args.opensearch_user,
                                                               'rdafadmin',
                                                               user_no_prompt_err_msg,
                                                               opensearch_user_desc,
                                                               'Opensearch user',
                                                               cmd_args.no_prompt)
        opensearch_configs[self._option_user] = str_base64_encode(user)

        pass_desc = 'What is the password you want to use for the newly created' \
                    ' Opensearch admin user?'
        pass_no_prompt_err_msg = 'No Opensearch password specified. Use --opensearch-password' \
                                 ' to specify one'
        default_autogen_password = rdafutils.gen_password()
        passwd = rdaf.component.Component._parse_or_prompt_value(cmd_args.opensearch_password,
                                                                 default_autogen_password,
                                                                 pass_no_prompt_err_msg,
                                                                 pass_desc, 'Opensearch password',
                                                                 cmd_args.no_prompt,
                                                                 password=True)
        opensearch_configs[self._option_password] = str_base64_encode(passwd)
        self._mark_configured(opensearch_configs, config_parser)

    def get_hosts(self) -> list:
        return self.configs[self._option_host]

    def get_ports(self) -> tuple:
        ports = ['9200', '9300'] if len(self.get_hosts()) == 1 else ['9200', '9300']
        hosts = self.get_hosts()
        return hosts, ports

    def _get_host_data_dirs(self) -> List[Tuple[str, List[str]]]:
        return self.configs[self._option_data_dir]

    def get_user(self) -> str:
        return rdafutils.str_base64_decode(self.configs[self._option_user])

    def get_password(self) -> str:
        return rdafutils.str_base64_decode(self.configs[self._option_password])

    def get_escaped_password(self) -> str:
        return str_base64_decode(self.configs[self._option_password]).replace("$", "\\$").replace("&", "/&")
        
    def do_setup(self, cmd_args, config_parser):
        content = self._create_opensearch_yaml_content(config_parser)
        for host in self.get_hosts():
            opensearch_yaml = os.path.join(self.get_conf_dir(), 'opensearch.yaml')
            created_location = rdaf.component.create_file(host, content.encode(encoding='UTF-8'),
                                                          opensearch_yaml)
            logger.info('Created Opensearch configuration at ' + created_location + ' on ' + host)

        # owning the data dir
        for host, data_dirs in self.configs[self._option_data_dir]:
            data_dir = data_dirs[0]
            command = 'sudo mkdir -p ' + data_dir + ' && sudo chown -R 1000 ' + data_dir \
                      + ' && sudo chgrp -R 1000 ' + data_dir
            run_potential_ssh_command(host, command, config_parser)
            # backup dir
            backup_dir = os.path.join("/opt", "opensearch-backup")
            command = 'sudo mkdir -p ' + backup_dir + ' && sudo chown -R 1000 ' + backup_dir \
                      + ' && sudo chgrp -R 1000 ' + backup_dir
            run_potential_ssh_command(host, command, config_parser)

        opensearch_log_dir = self.get_logs_dir()
        for host in self.get_hosts():
            command = 'sudo mkdir -p ' + opensearch_log_dir + ' && sudo chmod -R 777 ' + opensearch_log_dir
            run_potential_ssh_command(host, command, config_parser)

        self.setup_user(config_parser)

    def get_k8s_component_name(self):
        return 'rda-opensearch'

    def get_k8s_chart_name(self):
        return 'rda_opensearch'

    def do_k8s_setup(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != "aws":
            # for local deployments ensuring the folder is present
            command = 'sudo mkdir -p /opt/backup/opensearch'
            for host in self.get_hosts():
                run_potential_ssh_command(host, command, config_parser)

        logger.info("Generating hash for opensearch user.")
        hash_pass = self._generate_hash_k8s(config_parser)
        cert_manager: cert.CertManager = COMPONENT_REGISTRY.require(cert.CertManager.COMPONENT_NAME)
        self.create_cert_configs(config_parser)

        replacements = self._get_docker_repo()
        replacements['IS_SINGLE_NODE'] = "true" if len(self.get_hosts()) == 1 else "false"
        replacements['REPLICAS'] = len(self.get_hosts())
        # replacements['DISCOVERY_TYPE'] = 'single-node' if len(self.get_hosts()) == 1 else 'zen'
        replacements['CERTS_PASSWORD'] = cert_manager.get_keystore_pass()
        replacements['OPENSEARCH_USER'] = self.get_user()
        replacements['OPENSEARCH_HASH'] = hash_pass

        template_dir = 'k8s-local'
        if self.get_deployment_type(config_parser) == "aws":
            template_dir = 'k8s-aws'
        template_path = os.path.join(get_templates_dir_root(), template_dir, 'opensearch-values.yaml')
        dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'opensearch-values.yaml')
        with open(template_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        with open(dest_path, 'w') as f:
            f.write(original_content)

        # creating secret
        logger.info("Creating internal users secret.")
        template_path = os.path.join(get_templates_dir_root(), 'k8s-local', 'internal-users.yaml')
        dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'internal-users.yaml')
        replacements = dict()
        replacements['OPENSEARCH_USER'] = self.get_user()
        replacements['OPENSEARCH_HASH'] = hash_pass
        replacements['NAMESPACE'] = self.get_namespace(config_parser)
        with open(template_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        with open(dest_path, 'w') as f:
            f.write(original_content)

        run_command('kubectl apply -f ' + dest_path)

    def create_cert_configs(self, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        # create the kubectl secrets
        logger.info("Creating opensearch tls secret..")
        jks_secret = 'kubectl create secret generic -n {} jks ' \
                     '--from-file=opensearch.jks=/opt/rdaf/cert/rdaf/rdaf.jks ' \
                     '--save-config --dry-run=client -o yaml |  kubectl apply -f -'.format(namespace)
        run_command(jks_secret)
        ca_secret = 'kubectl create secret generic -n {} truststore ' \
                    '--from-file=truststore=/opt/rdaf/cert/truststore/ca_truststore ' \
                    '--save-config --dry-run=client -o yaml |  kubectl apply -f -'.format(namespace)
        run_command(ca_secret)

    def get_k8s_install_args(self, cmd_args):
        args = '--set image.tag={} --set persistence.imageTag={}'.format(cmd_args.tag, cmd_args.tag)
        return args

    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != "k8s":
            return
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            for image in ['rda-platform-opensearch', 'rda-platform-busybox']:
                logger.info(f'Pulling {image} on host {host}')
                docker_pull_command = f'docker pull {docker_repo}/{image}:{cmd_args.tag}'
                run_potential_ssh_command(host, docker_pull_command, config_parser)

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        if self.get_deployment_type(config_parser) != "aws":
            opensearch_pv_file = 'opensearch-pv-cluster.yaml' if len(self.get_hosts()) > 1 \
                else 'opensearch-pv.yaml'
            opensearch_pvs = 'kubectl apply -f {} -n {}'.format(
                os.path.join(get_templates_dir_root(), 'k8s-local', opensearch_pv_file), namespace)
            run_command(opensearch_pvs)

        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'opensearch-values.yaml')
        install_command = 'helm install  --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(), deployment_path)
        run_command(install_command)

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'opensearch-values.yaml')
        upgrade_command = 'helm upgrade --install --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(upgrade_command)
        
    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        cmd = 'kubectl get pods -l {} -n {} -o json'.format(self.get_k8s_component_label(), namespace)
        ret, stdout, stderr = execute_command(cmd)
        component = json.loads(stdout)
        for items in component["items"]:
            metadata = items["metadata"]["ownerReferences"]
            for pod in metadata:
                name = pod['name']
                command = f'kubectl scale statefulset.apps/{name} -n {namespace} --replicas=0'
                run_command(command)   
                return

    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        cmd = 'kubectl get all -l {} -n {} -o json'.format(self.get_k8s_component_label(), namespace)
        ret, stdout, stderr = execute_command(cmd)
        component = json.loads(stdout)
        replicas = len(self.get_hosts())
        for comp in component["items"]:
            if comp["kind"] == 'StatefulSet':
                comp_name = comp['metadata']['name']
                command = f'kubectl scale statefulset.apps/{comp_name} -n {namespace} --replicas={replicas}'
                run_command(command)
                return

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        # this is added here as the backup dir can be changed runtime via values.yaml
        # owning the backup dir
        for host in self.get_hosts():
            backup_dir = os.path.join('/opt', 'opensearch-backup')
            command = 'sudo mkdir -p ' + backup_dir + ' && sudo chown -R 1000 ' + backup_dir \
                      + ' && sudo chgrp -R 1000 ' + backup_dir
            run_potential_ssh_command(host, command, config_parser)
        super().install(cmd_args, config_parser)

    def _create_opensearch_yaml_content(self, config_parser) -> str:
        hosts = self.get_hosts()
        opensearch_yaml = os.path.join(get_templates_dir_root(), 'opensearch.yaml')
        with open(opensearch_yaml, 'r') as f:
            template_content = f.read()

        replacements = dict()
        if len(hosts) > 1:
            replacements['OPENSEARCH_DISCOVERY_TYPE'] = 'zen'
        else:
            replacements['OPENSEARCH_DISCOVERY_TYPE'] = 'single-node'

        cert_manager: cert.CertManager = COMPONENT_REGISTRY.require(
            cert.CertManager.COMPONENT_NAME)
        replacements['TRUSTSTORE_PASSWORD'] = cert_manager.get_keystore_pass()
        replacements['KEYSTORE_PASSWORD'] = cert_manager.get_keystore_pass()
        replacements['GEODR_FOLLOWER_VARIABLES'] = ''

        if self.is_geodr_deployment(config_parser):
            replacements['GEODR_FOLLOWER_VARIABLES'] = '''
plugins.replication.follower.index.recovery.chunk_size: 512mb
plugins.replication.follower.index.recovery.max_concurrent_file_chunks: 4
plugins.replication.follower.index.ops_batch_size: 50000
plugins.replication.follower.concurrent_readers_per_shard: 4
plugins.replication.follower.metadata_sync_interval: 30s'''

        original_content = string.Template(template_content).substitute(replacements)
        return original_content

    def get_deployment_env(self, host: str):
        env = dict()
        env['OPENSEARCH_NODE_NAME'] = host
        env['OPENSEARCH_DATA_MOUNT'] = self._default_data_dirs[0]
        env['OPENSEARCH_BACKUP_MOUNT'] = os.path.join('/opt', 'opensearch-backup')
        env['OPENSEARCH_ROOT_USER'] = str_base64_decode(self.configs[self._option_user]).replace("$", "$$")
        env['OPENSEARCH_ROOT_PASSWORD'] = str_base64_decode(
            self.configs[self._option_password]).replace("$", "$$")
        env['OPENSEARCH_YAML_CONFIG'] = os.path.join(self.get_conf_dir(), 'opensearch.yaml')
        opensearch_jks_path = os.path.join('/opt/rdaf/cert/rdaf/rdaf.jks')
        env['SSL_KEYSTORE_JKS_FILENAME'] = 'rdaf.jks'
        env['OPENSEARCH_KEYSTORE_SOURCE_PATH'] = opensearch_jks_path
        env['OPENSEARCH_INTERNAL_USER_YAML'] = os.path.join(self.get_conf_dir(), 'internal_users.yml')
        self.apply_input_values(env)
        return env

    def update_cluster_env(self, component_yaml: dict, host: str):
        if len(self.get_hosts()) == 1:
            return
        component_yaml['environment'].append('discovery.seed_hosts=' + ','.join(self.get_hosts()))
        component_yaml['environment'].append('cluster.initial_cluster_manager_nodes='
                                             + ','.join(self.get_hosts()))

    def healthcheck(self, component_name, host, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        headers = {'Content-type': 'application/json'}
        try:
            with requestsutil.new_session():
                requests.packages.urllib3.disable_warnings(
                    requests.packages.urllib3.exceptions.InsecureRequestWarning)
                logger.debug("Collecting cluster stats from Opensearch")
                register_url = "https://{os_host}:9200/_cluster/stats".format(os_host=host)
                response = requests.get(register_url, verify=False, headers=headers,
                                        auth=(self.get_user(), self.get_password()))
                response.raise_for_status()
        except Exception as e:
            return [component_name, "Service Status", termcolor.colored("Failed", color='red'), str(e)]
        return [component_name, "Service Status", "OK", "N/A"]

    def backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        snapshot_id = datetime.datetime.now().strftime("%Y_%m_%d_%H_%M_%S")
        headers = {'Content-type': 'application/json'}
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)
            logger.info("Creating repo for opensearch backup")
            register_url = "https://{os_host}:9200/_snapshot/cfx-repo".format(os_host=self.get_hosts()[0])
            data = {"type": "fs", "settings": {"location": "/opt/snapshots/cfx-repo", "compress": True}}
            response = requests.put(register_url, verify=False, data=json.dumps(data), headers=headers,
                                    auth=(self.get_user(), self.get_password()))
            response.raise_for_status()
            logger.info("Triggering opensearch snapshot with id " + snapshot_id)
            # trigger snapshot
            snapshot_url = "https://{os_host}:9200/_snapshot/cfx-repo/{snapshot_id}".format(
                os_host=self.get_hosts()[0], snapshot_id=snapshot_id)
            response = requests.put(snapshot_url, verify=False, headers=headers, data='{"indices": "*,-.opendistro*,-security-auditlog*,-.plugins-ml-config,-.opensearch-observability*"}',
                                    auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

        while True:
            logger.info("Waiting for opensearch snapshot to complete...")
            time.sleep(10)
            response = requests.get(snapshot_url, verify=False, headers=headers,
                                    auth=(self.get_user(), self.get_password()))
            content = response.json()
            for entry in content['snapshots']:
                if entry['snapshot'] == snapshot_id:
                    if entry['state'] == 'SUCCESS':
                        if not backup_state.has_section(self.get_name()):
                            backup_state.add_section(self.get_name())
                        backup_state.set(self.get_name(), 'snapshot_id', snapshot_id)
                        return
                    elif entry['state'] != 'IN_PROGRESS':
                        rdafutils.cli_err_exit("Failed to take opensearch snapshot: " + snapshot_id)
                    break

    def k8s_backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        namespace = self.get_namespace(config_parser)
        snapshot_id = datetime.datetime.now().strftime("%Y_%m_%d_%H_%M_%S")
        headers = {'Content-type': 'application/json'}
        pod = self.get_pods_names(config_parser, 'app_component=rda-opensearch')[0]
        pf_proc = subprocess.Popen(f'exec kubectl port-forward {pod} -n {namespace} 9200:9200', shell=True)
        time.sleep(10)
        os_host = '127.0.0.1'
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)
            logger.info("Creating repo for opensearch backup")
            register_url = "https://{os_host}:9200/_snapshot/cfx-repo".format(os_host=os_host)
            data = {"type": "fs", "settings": {"location": "/opt/snapshots/cfx-repo", "compress": True}}
            response = requests.put(register_url, verify=False, data=json.dumps(data), headers=headers,
                                    auth=(self.get_user(), self.get_password()))
            response.raise_for_status()
            logger.info("Triggering opensearch snapshot with id " + snapshot_id)
            # trigger snapshot
            snapshot_url = "https://{os_host}:9200/_snapshot/cfx-repo/{snapshot_id}".format(
                os_host=os_host, snapshot_id=snapshot_id)
            response = requests.put(snapshot_url, verify=False, headers=headers,
                                    data='{"indices": "*,-.opendistro*,-security-auditlog*,-.plugins-ml-config,-.opensearch-observability*"}',
                                    auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

        while True:
            logger.info("Waiting for opensearch snapshot to complete...")
            time.sleep(10)
            response = requests.get(snapshot_url, verify=False, headers=headers,
                                    auth=(self.get_user(), self.get_password()))
            content = response.json()
            for entry in content['snapshots']:
                if entry['snapshot'] == snapshot_id:
                    if entry['state'] == 'SUCCESS':
                        if not backup_state.has_section(self.get_name()):
                            backup_state.add_section(self.get_name())
                        backup_state.set(self.get_name(), 'snapshot_id', snapshot_id)
                        pf_proc.kill()
                        return
                    elif entry['state'] != 'IN_PROGRESS':
                        pf_proc.kill()
                        rdafutils.cli_err_exit("Failed to take opensearch snapshot: " + snapshot_id)
                    break
    
    def required_container_state_before_restore(self):
        return 'running'

    def restore_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                     backup_content_root_dir: os.path,
                     backup_cfg_parser: configparser.ConfigParser):
        if not backup_cfg_parser.has_section(self.get_name()):
            return

        snapshot_id = backup_cfg_parser.get(self.get_name(), 'snapshot_id')
        if not snapshot_id:
            logger.debug("No snapshot id found for opensearch restore")
            return

        headers = {'Content-type': 'application/json'}
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)
            logger.info("Restoring opensearch snapshot with id " + snapshot_id)
            # trigger restore
            indices_close_url = "https://{os_host}:9200/*,.opensearch*,-.opendistro*,-security-auditlog*,-.plugins-ml-config,-.opensearch-observability/_close".format(os_host=self.get_hosts()[0])
            response = requests.post(indices_close_url, verify=False, headers=headers,
                                     auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

            snapshot_url = "https://{os_host}:9200/_snapshot/cfx-repo/{snapshot_id}/_restore".format(
                os_host=self.get_hosts()[0], snapshot_id=snapshot_id)
            data = {"indices": "*,-.plugins-ml-config,-.opensearch-observability", "ignore_unavailable": True, "include_global_state": False}
            response = requests.post(snapshot_url, verify=False, headers=headers,
                                     data=json.dumps(data),
                                     auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

            # open resources
            indices_open_url = "https://{os_host}:9200/*,.opensearch*,-.opendistro*,-security-auditlog*,-.plugins-ml-config,-.opensearch-observability/_open".format(os_host=self.get_hosts()[0])
            response = requests.post(indices_open_url, verify=False, headers=headers,
                                     auth=(self.get_user(), self.get_password()))
            response.raise_for_status()
            print("")

    def setup_user(self, config_parser):
        host = self.get_hosts()[0]
        config_dir = os.path.join(self.get_conf_dir())
        internal_users_yml = os.path.join(config_dir, 'internal_users.yml')
        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            compose_file = os.path.join(tmp, 'docker-compose.yaml')
            image_name = self._get_docker_repo()['DOCKER_REPO'] + '/rda-platform-opensearch:1.0.4'
            content = {"services": {'opensearch': {
                "image": image_name, 'network_mode': 'host'}}}

            with open(compose_file, 'w+') as f:
                yaml.safe_dump(content, f, default_flow_style=False, explicit_start=True,
                               allow_unicode=True, encoding='utf-8', sort_keys=False)
            if not Component.is_local_host(host):
                do_potential_scp(host, tmp, tmp)
            pull_cmd = '/usr/local/bin/docker-compose -f {file} pull '.format(file=compose_file)
            run_potential_ssh_command(host, pull_cmd, config_parser)
            with Component.new_docker_client_(host) as docker_client:
                container_collection = ContainerCollection(client=docker_client.client)
                command = ['plugins/opensearch-security/tools/hash.sh',
                           '-p', self.get_password()]
                env = ['JAVA_HOME=/usr/share/opensearch/jdk']
                hash_generated = container_collection.run(image=image_name,
                                                          command=command,
                                                          environment=env,
                                                          remove=True)
                pass_list = hash_generated.split(b'\n')
                hash_pass = pass_list[-2].decode('utf-8')

            for host in self.get_hosts():
                meta = {'type': 'internalusers', 'config_version': 2}
                user = {'hash': hash_pass, 'reserved': True,
                        'backend_roles': ['admin'],
                        'description': 'Opensearch admin user'}
                data = {'_meta': meta, self.get_user(): user}
                yaml_data = yaml.safe_dump(data, default_flow_style=False, explicit_start=True,
                                           allow_unicode=True, encoding='utf-8', sort_keys=False)
                rdaf.component.create_file(host, yaml_data, internal_users_yml)
        if not Component.is_local_host(host):
            remove_tmp_command = 'rm -rf ' + tmp
            run_potential_ssh_command(host, remove_tmp_command, config_parser)

    def setup_tenant(self, config_parser, tenant_id: str, port='9200'):
        namespace = self.get_namespace(config_parser)
        logger.info("Creating Opensearch tenant configurations...")
        pf_proc = None
        if self.get_deployment_type(config_parser) == 'aws':
            pod = self.get_pods_names(config_parser, 'app_component=rda-opensearch')[0]
            pf_proc = subprocess.Popen('exec kubectl port-forward {} -n {} 9200:9200'.format(pod,namespace),
                                       shell=True)
            time.sleep(10)
            os_host = '127.0.0.1'
        else:
            os_host = self.get_hosts()[0]

        password = rdafutils.gen_password_with_uuid(tenant_id)
        user = tenant_id + "rdauser"
        tenant_data = {"description": "Tenant Name: " + tenant_id}
        run_command('curl -s -k -X PUT -u "{os_user}:{os_password}" '
                    '"https://{os_host}:{port}/_plugins/_security/api/tenants/tenant-{UUID}" '
                    '-H \'Content-Type: application/json\' --data \'{data}\' --insecure'
                    .format(os_user=self.get_user(), os_password=self.get_escaped_password(),
                            os_host=os_host, port=port, UUID=tenant_id, data=json.dumps(tenant_data)), shell=True)
        print("")
        roles_data = {"index_permissions": [{"index_patterns": [tenant_id + "*", f"admin-{tenant_id}*"],
                                             "allowed_actions": ["unlimited"]},
                                            {"index_patterns": ["*"],
                                            "allowed_actions": ["indices:data/write/delete/byquery",
                                                                "indices:data/read/search"]}],
                      "tenant_permissions": [{"tenant_patterns": ["tenant-" + tenant_id],
                                              "allowed_actions": ["unlimited"]}],
                      "cluster_permissions": ["cluster:monitor/health", "cluster:monitor/main", "cluster:monitor/state",
                                              "cluster:monitor/stats", "indices:data/read/scroll/clear", "cluster:admin/opendistro/transform/*",
                                              "cluster:admin/opendistro/ism/policy/*"]}
        run_command('curl -s -k -X PUT -u "{os_user}:{os_password}" '
                    '"https://{os_host}:{port}/_plugins/_security/api/roles/role-{UUID}" '
                    '-H \'Content-Type: application/json\' --data \'{data}\' --insecure'
                    .format(os_user=self.get_user(), os_password=self.get_escaped_password(),
                            os_host=os_host, port=port, UUID=tenant_id, data=json.dumps(roles_data)), shell=True)
        print("")
        internal_users_data = {"password": password}
        run_command('curl -s -k -X PUT -u "{os_user}:{os_password}" '
                    '"https://{os_host}:{port}/_plugins/_security/api/internalusers/{ESUSER}" '
                    '-H \'Content-Type: application/json\' --data \'{data}\' --insecure'
                    .format(os_user=self.get_user(), os_password=self.get_escaped_password(),
                            os_host=os_host, port=port, ESUSER=user, data=json.dumps(internal_users_data)), shell=True)
        print("")
        role_mappings_data = {"users": [user]}
        run_command('curl -s -k -X PUT -u "{os_user}:{os_password}" '
                    '"https://{os_host}:{port}/_plugins/_security/api/rolesmapping/role-{UUID}" '
                    '-H \'Content-Type: application/json\' --data \'{data}\' --insecure'
                    .format(os_user=self.get_user(), os_password=self.get_escaped_password(),
                            os_host=os_host, port=port, UUID=tenant_id, data=json.dumps(role_mappings_data)), shell=True)
        print("")

        self._apply_security_audit_policy(os_host, port)
        if pf_proc:
            pf_proc.kill()
        return user, password
    
    def setup_log_streaming_user(self, config_parser, index, port='9200'):
        namespace = self.get_namespace(config_parser)
        pf_proc = None
        if self.get_deployment_type(config_parser) in ['aws', 'ctr']:
            pod = self.get_pods_names(config_parser, 'app_component=rda-opensearch')[0]
            pf_proc = subprocess.Popen(f'exec kubectl port-forward {pod} -n {namespace} 9200:9200', shell=True)
            time.sleep(10)
            os_host = '127.0.0.1'
        else:
            os_host = self.get_hosts()[0]

        password = rdafutils.gen_password()
        user = "rdaf-log-monitoring"
        user_data = {"password": password, "backend_roles": ["logstash"], "attributes": {}}
        run_command('curl -s -k -X PUT -u "{os_user}:{os_password}" '
                    '"https://{os_host}:{port}/_plugins/_security/api/internalusers/{user}" '
                    '-H \'Content-Type: application/json\' --data \'{data}\' --insecure'
                    .format(os_user=self.get_user(), os_password=self.get_escaped_password(),
                            os_host=os_host, port=port, user=user, data=json.dumps(user_data)))
        print("")
        roles_data = {"index_permissions": [{"index_patterns": [index], "allowed_actions": ["unlimited"]}],
                      "tenant_permissions": [{"tenant_patterns": [index], "allowed_actions": ["unlimited"]}],
                      "cluster_permissions": ["cluster:monitor/main", "indices:data/read/scroll/clear",
                                              "cluster:admin/opendistro/transform/*"]}
        run_command('curl -s -k -X PUT -u "{os_user}:{os_password}" '
                    '"https://{os_host}:{port}/_plugins/_security/api/roles/role-log-monitoring" '
                    '-H \'Content-Type: application/json\' --data \'{data}\' --insecure'
                    .format(os_user=self.get_user(), os_password=self.get_escaped_password(),
                            os_host=os_host, port=port, data=json.dumps(roles_data)))
        print("")
        internal_users_data = {"password": password}
        run_command('curl -s -k -X PUT -u "{os_user}:{os_password}" '
                    '"https://{os_host}:{port}/_plugins/_security/api/internalusers/{ESUSER}" '
                    '-H \'Content-Type: application/json\' --data \'{data}\' --insecure'
                    .format(os_user=self.get_user(), os_password=self.get_escaped_password(),
                            os_host=os_host, port=port, ESUSER=user, data=json.dumps(internal_users_data)))
        print("")
        role_mappings_data = {"users": [user]}
        run_command('curl -s -k -X PUT -u "{os_user}:{os_password}" '
                    '"https://{os_host}:{port}/_plugins/_security/api/rolesmapping/role-log-monitoring" '
                    '-H \'Content-Type: application/json\' --data \'{data}\' --insecure'
                    .format(os_user=self.get_user(), os_password=self.get_escaped_password(),
                            os_host=os_host, port=port, data=json.dumps(role_mappings_data)), shell=True)
        print("")
        if pf_proc:
            pf_proc.kill()
        return user, password

    @staticmethod
    def apply_input_values(env):
        values_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if not os.path.exists(values_path):
            return
        with open(values_path, 'r') as f:
            values = yaml.safe_load(f)

        if 'opensearch' in values['services']:
            opensearch = values['services']['opensearch']
            if 'volumes' in opensearch and opensearch['volumes']['OPENSEARCH_BACKUP_MOUNT']:
                env['OPENSEARCH_BACKUP_MOUNT'] = opensearch['volumes']['OPENSEARCH_BACKUP_MOUNT']

    def _generate_hash_k8s(self,config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        opensearch_hash_deployment = os.path.join(get_templates_dir_root(), 'opensearch-hash.yaml')
        with open(opensearch_hash_deployment, 'r') as f:
            template_content = f.read()
        replacements = self._get_docker_repo()
        replacements['NAMESPACE'] = namespace

        content = string.Template(template_content).substitute(replacements)

        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            deployment_file = os.path.join(tmp, 'opensearch-hash.yaml')
            with open(deployment_file, 'w+') as f:
                f.write(content)
                f.flush()

            run_command('kubectl apply -f ' + deployment_file)
            logger.info("Waiting for opensearch hash pod to be up and running...")
            time.sleep(5)
            pod_status_command = 'kubectl wait --for=condition=Ready pod --timeout=600s -n {} ' \
                                 '-l app_component=rda-opensearch-hash'.format(namespace)
            ret, stdout, stderr = execute_command(pod_status_command)
            if ret != 0:
                cli_err_exit("Failed to get status of opensearch hash generation pod, due to: {}.".format(str(stderr)))

            logger.info("Pod is running, generating hash...")
            hash_pod = self.get_pods_names(config_parser, 'app_component=rda-opensearch-hash')[0]
            hash_command = "kubectl -n {} exec {} -- plugins/opensearch-security/tools/hash.sh " \
                           "-p {}".format(namespace, hash_pod, self.get_password())
            ret, stdout, stderr = execute_command(hash_command)
            if ret != 0:
                print(str(stdout), str(stderr))
                cli_err_exit("Failed to generate opensearch hash.")
            pass_list = str(stdout).split('\n')
            hash_pass = pass_list[-2]
            run_command('kubectl delete -f ' + deployment_file)
            return hash_pass

    def k8s_restore_conf(self, config_parser: configparser.ConfigParser,
                         backup_content_root_dir: os.path):
        self.create_cert_configs(config_parser)

    def check_pod_state_for_restore(self, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = 'kubectl get pods -n ' + namespace + ' --selector=app_component=rda-opensearch ' \
                  '-o jsonpath="{.items[*].status.phase}"'
        exit_code, stdout, stderr = execute_command(command)
        if 'Running' not in str(stdout):
            cli_err_exit('Opensearch pod should be Running for restore')

    def k8s_restore_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                         backup_content_root_dir: os.path, backup_cfg_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        if not backup_cfg_parser.has_section(self.get_name()):
            return

        snapshot_id = backup_cfg_parser.get(self.get_name(), 'snapshot_id')
        if not snapshot_id:
            logger.debug("No snapshot id found for opensearch restore")
            return

        pod = self.get_pods_names(config_parser, 'app_component=rda-opensearch')[0]
        pf_proc = subprocess.Popen(f'exec kubectl port-forward {pod} -n {namespace} 9200:9200', shell=True)
        time.sleep(10)
        os_host = '127.0.0.1'

        headers = {'Content-type': 'application/json'}
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)
            logger.info("Restoring opensearch snapshot with id " + snapshot_id)
            # trigger restore
            indices_close_url = "https://{os_host}:9200/*,.opensearch*,-.opendistro*,-security-auditlog*,-.plugins-ml-config,-.opensearch-observability/_close".format(os_host=os_host)
            response = requests.post(indices_close_url, verify=False, headers=headers,
                        auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

            snapshot_url = "https://{os_host}:9200/_snapshot/cfx-repo/{snapshot_id}/_restore".format(
                os_host=os_host, snapshot_id=snapshot_id)
            data = {"indices": "*,-.plugins-ml-config,-.opensearch-observability", "ignore_unavailable": True, "include_global_state": False}
            response = requests.post(snapshot_url, verify=False, headers=headers,
                                     data=json.dumps(data),
                                     auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

            # open resources
            indices_open_url = "https://{os_host}:9200/*,.opensearch*,-.opendistro*,-security-auditlog*,-.plugins-ml-config,-.opensearch-observability/_open".format(os_host=os_host)
            response = requests.post(indices_open_url, verify=False, headers=headers,
                                     auth=(self.get_user(), self.get_password()))
            response.raise_for_status()
            print("")

        pf_proc.kill()

    def create_data_plane_policy(self,config_parser, tenant_id, k8s=False):
        namespace = self.get_namespace(config_parser)
        logger.info("Creating data plane user for tenant.")

        pf_proc = None
        if k8s:
            pod = self.get_pods_names(config_parser, 'app_component=rda-opensearch')[0]
            pf_proc = subprocess.Popen(f'exec kubectl port-forward {pod} -n {namespace} 9200:9200', shell=True)
            os_host = '127.0.0.1'
            time.sleep(10)
        else:
            os_host = self.get_hosts()[0]

        user = tenant_id + "adminuser"
        password = rdafutils.gen_password_with_uuid(tenant_id)
        roles_data = {"index_permissions": [{"index_patterns": ["admin-" + tenant_id + "*", tenant_id + "*"],
                                             "allowed_actions": ["unlimited"]},
                                            {"index_patterns": ["*"],
                                             "allowed_actions": ["indices:data/write/delete/byquery",
                                                                 "indices:data/read/scroll/clear",
                                                                 "indices:data/read/search",
                                                                 "indices:admin/rollover"]}],
                      "cluster_permissions": ["cluster:monitor/main",
                                              "cluster:monitor/state",
                                              "cluster:monitor/stats",
                                              "cluster:monitor/health",
                                              "cluster:admin/snapshot/*",
                                              "cluster:admin/opendistro/transform/*",
                                              "cluster:admin/settings/update",
                                              "cluster:admin/ingest/pipeline/delete",
                                              "cluster:admin/ingest/pipeline/get",
                                              "cluster:admin/ingest/pipeline/put",
                                              "indices:admin/index_template/get",
                                              "indices:admin/index_template/put",
                                              "indices:admin/index_template/delete",
                                              "cluster:admin/opendistro/ism/managedindex/explain",
                                              "cluster:admin/opendistro/ism/managedindex/change",
                                              "cluster:admin/opendistro/ism/policy/get",
                                              "cluster:admin/opendistro/ism/policy/write",
                                              "cluster:admin/opendistro/ism/policy/delete"],
                      "tenant_permissions": [{"tenant_patterns": ["tenant-" + tenant_id],
                                              "allowed_actions": ["unlimited"]}]}
        run_command('curl -s -k -X PUT -u "{os_user}:{os_password}" '
                    '"https://{os_host}:{port}/_plugins/_security/api/roles/role-{UUID}-dataplane-policy" '
                    '-H \'Content-Type: application/json\' --data \'{data}\' --insecure'
                    .format(os_user=self.get_user(), os_password=self.get_escaped_password(),
                            os_host=os_host, port='9200', UUID=tenant_id, data=json.dumps(roles_data)), shell=True)
        print("")
        internal_users_data = {"password": password}
        run_command('curl -s -k -X PUT -u "{os_user}:{os_password}" '
                    '"https://{os_host}:{port}/_plugins/_security/api/internalusers/{ESUSER}" '
                    '-H \'Content-Type: application/json\' --data \'{data}\' --insecure'
                    .format(os_user=self.get_user(), os_password=self.get_escaped_password(),
                            os_host=os_host, port='9200', ESUSER=user, data=json.dumps(internal_users_data)), shell=True)
        print("")
        role_mappings_data = {"users": [user]}
        run_command('curl -s -k -X PUT -u "{os_user}:{os_password}" '
                    '"https://{os_host}:{port}/_plugins/_security/api/rolesmapping/role-{UUID}-dataplane-policy" '
                    '-H \'Content-Type: application/json\' --data \'{data}\' --insecure'
                    .format(os_user=self.get_user(), os_password=self.get_escaped_password(),
                            os_host=os_host, port='9200', UUID=tenant_id, data=json.dumps(role_mappings_data)),
                    shell=True)
        print("")
        if pf_proc:
            pf_proc.kill()
        return user, password

    def wait_for_opensearch_to_be_up(self):
        logger.info("Waiting for Opensearch to be up and running...")
        retry = 0
        headers = {'Content-type': 'application/json'}
        while retry < 15:
            try:
                with requestsutil.new_session():
                    requests.packages.urllib3.disable_warnings(
                        requests.packages.urllib3.exceptions.InsecureRequestWarning)
                    logger.debug("waiting for opensearch to be up and running... " + str(retry))
                    register_url = "https://{os_host}:9200/_cluster/stats".format(os_host=self.get_hosts()[0])
                    response = requests.get(register_url, verify=False, headers=headers,
                                            auth=(self.get_user(), self.get_password()))
                    response.raise_for_status()
                    break
            except Exception as e:
                retry += 1
                time.sleep(15)

        if retry >= 15:
            raise Exception("Opensearch is not up and running in specified time.")

    def _apply_security_audit_policy(self, host, port):
        logger.info("Creating policy to purge security audit logs.")
        os_user = self.get_user()
        os_password = self.get_escaped_password()

        ret, stdout, stderr = execute_command(
            f'curl -s -o /dev/null -w "%{{http_code}}" -X GET -u "{os_user}:{os_password}" '
            f'"https://{host}:{port}/_plugins/_ism/policies/purge_security_audit_logs" --insecure')
        if stdout.strip() == '200':
            logger.info("security audit purge policy already exists.")
            return

        policy = {"policy": {"description": "Purge security audit log indices older than 15 days",
                             "default_state": "idle",
                             "ism_template": {"index_patterns": ["security-auditlog-*"],
                                              "priority": 100},
                             "states": [{"name": "idle",
                                         "actions": [],
                                         "transitions": [{"state_name": "delete",
                                                          "conditions": {"min_index_age": "15d"}}]},
                                        {"name": "delete",
                                         "actions": [{"delete": {}}],
                                         "transitions": []}
                                        ]
                             }
                  }
        data = json.dumps(policy)
        run_command(
            f'curl -s -k -X PUT -u "{os_user}:{os_password}" '
            f'"https://{host}:{port}/_plugins/_ism/policies/purge_security_audit_logs" '
            f'-H \'Content-Type: application/json\' --data \'{data}\' --insecure', shell=True)
        print("")

        # this will update the policy to any existing indices
        data = json.dumps({"policy_id": "purge_security_audit_logs"})
        run_command(
            f'curl -s -k -X POST -u "{os_user}:{os_password}" '
            f'"https://{host}:{port}/_plugins/_ism/add/security-auditlog-*" '
            f'-H \'Content-Type: application/json\' --data \'{data}\' --insecure', shell=True)
        print("")

    def get_aws_opensearch_endpoint(self, config_parser):
        namespace = self.get_namespace(config_parser)

        ret, stdout, stderr = execute_command(f"kubectl get svc/rda-opensearch-lb -n {namespace} -o json")
        if ret != 0:
            cli_err_exit("Failed to get status of opensearch loadbalancer service, due to: {}.".format(str(stderr)))

        result = json.loads(str(stdout))

        if 'hostname' in result['status']['loadBalancer']['ingress'][0]:
            host = result['status']['loadBalancer']['ingress'][0]['hostname']
        else:
            host = result['status']['loadBalancer']['ingress'][0]['ip']

        if result['spec']['ports']:
            port = str(result['spec']['ports'][0]['port'])

        return host, port

    def geodr_start_replication(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        primary_host = self.get_hosts()[0]
        secondary_host  = peer_configs.get('opensearch', 'host').split(',')[0]
        urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
        headers = {'Content-type': 'application/json'}
        health_url = f'https://{primary_host}:9200/_cluster/health'
        response = requests.get(health_url, verify=False, headers=headers,
                                auth=(self.get_user(), self.get_password()), timeout=30)
        response.raise_for_status()
        health_data = response.json()
        status = health_data.get('status', 'red')
        if status == 'red':
            cli_err_exit("Cannot start replication")
        # Load excluded indexes
        excluded_indexes = set()
        try:
            with open('/opt/rdaf/exclude_index.txt', 'r') as exclude_file:
                excluded_indexes = {line.strip() for line in exclude_file if line.strip()}
        except FileNotFoundError:
            excluded_indexes = []
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)
            logger.info("Adding peers on opensearch primary")
            seeds_url = f'https://{primary_host}:9200/_cluster/settings'
            seeds_data = {"persistent":{"cluster":{"remote":{"opensearch-cluster":{"seeds":[f"{secondary_host}:9300"]}}}}}
            response = requests.put(seeds_url, verify=False, data=json.dumps(seeds_data), headers=headers,
                                    auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

            logger.info("Adding peers on opensearch secondary")
            seeds_url = f'https://{secondary_host}:9200/_cluster/settings'
            seeds_data = {"persistent": {
                "cluster": {"remote": {"opensearch-cluster": {"seeds": [f"{primary_host}:9300"]}}}}}
            response = requests.put(seeds_url, verify=False, data=json.dumps(seeds_data), headers=headers,
                                    auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

            logger.info(f"Starting replication on secondary {secondary_host}.")
            if hasattr(cmd_args, 'index') and cmd_args.index is not None:
                for index_name in cmd_args.index:
                    if index_name in excluded_indexes:
                       logger.info(f"Skipping replication for excluded index: {index_name}")
                       continue
                    replication_url = f'https://{secondary_host}:9200/_plugins/_replication/{index_name}/_start'
                    seeds_data = {"leader_alias":"opensearch-cluster","leader_index": f"{index_name}",
                              "use_roles":{"leader_cluster_role":"all_access","follower_cluster_role":"all_access"}}
                    response = requests.put(replication_url, verify=False, data=json.dumps(seeds_data), headers=headers,
                                        auth=(self.get_user(), self.get_password()))
                    response.raise_for_status()
                return
            logger.info("Applying autofollow replication rule excluding specified indexes.")
            tenant_id = config_parser.get('common', 'tenant_id')
            pattern = f"*{tenant_id}*"
            if excluded_indexes:
                pattern = f"^(?!{'|'.join(excluded_indexes)}).*"
                logger.info(f"Skipping replication for: {', '.join(excluded_indexes)}")
            replication_url = f'https://{secondary_host}:9200/_plugins/_replication/_autofollow'
            seeds_data = {"leader_alias":"opensearch-cluster","name":"my-replication-rule","pattern":pattern,
                          "use_roles":{"leader_cluster_role":"all_access","follower_cluster_role":"all_access"}}
            response = requests.post(replication_url, verify=False, data=json.dumps(seeds_data), headers=headers,
                                    auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

    def geodr_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                 peer_configs: configparser.ConfigParser):
        secondary_host = peer_configs.get('opensearch', 'host').split(',')[0]
        logger.info(f"Checking replication status for opensearch")
        if hasattr(cmd_args, 'index') and cmd_args.index is not None:
            for index_name in cmd_args.index:
                run_command('curl -s -k -X GET -u "{os_user}:{os_password}" "https://{os_host}:9200/_plugins/_replication/{index}/_status"'
                        .format(os_user=self.get_user(),os_password=self.get_escaped_password(),os_host=secondary_host,index=index_name))
            return
        run_command('curl -s -k -X GET -u "{os_user}:{os_password}" "https://{os_host}:9200/_plugins/_replication/autofollow_stats"'
                    .format(os_user=self.get_user(),os_password=self.get_escaped_password(),os_host=secondary_host))
        run_command('curl -s -k -X GET -u "{os_user}:{os_password}" "https://{os_host}:9200/_plugins/_replication/follower_stats"'
                    .format(os_user=self.get_user(),os_password=self.get_escaped_password(),os_host=secondary_host))

    def switch_primary(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        logger.info("Switching opensearch to be a primary instance.")
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)

            indices_response = requests.get(f'https://{self.get_hosts()[0]}:9200/_cat/indices',
                auth=(self.get_user(), self.get_password()), verify=False)
            indices_response.raise_for_status()
            index_names = [line.split()[2] for line in indices_response.text.strip().split('\n') if len(line.split()) > 2 ]
            logger.info(f"There are a total of {len(index_names)} indices")

            # handle exclude list
            try:
                with open('/opt/rdaf/exclude_index.txt', 'r') as exclude_file:
                    exclude_list = [line.strip() for line in exclude_file if line.strip()]
            except FileNotFoundError:
                exclude_list = []

            for index in index_names:
                if index in exclude_list:
                    continue
                try:
                    # Stop replication for the index
                    stop_response = requests.post(f'https://{self.get_hosts()[0]}:9200/_plugins/_replication/{index}/_stop',
                                                 auth=(self.get_user(), self.get_password()), verify=False, json={})
                    stop_response.raise_for_status()
                    logger.info(f"Successfully processed index: {index}")
                except requests.RequestException as index_error:
                    logger.error(f"Error processing index {index}: {str(index_error)}")

        logger.info("Completed switching opensearch as primary")
        time.sleep(10)
        logger.info("Opensearch replication follower status:")
        run_command('curl -s -k -X GET -u "{os_user}:{os_password}" "https://{os_host}:9200/_plugins/_replication/follower_stats"'
            .format(os_user=self.get_user(), os_password=self.get_escaped_password(), os_host=self.get_hosts()[0]))


    def geodr_stop_replication(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        preferred_primary = config_parser.getboolean('rdaf-cli', 'primary', fallback=True)
        if preferred_primary:
            cli_err_exit("Cannot stop/delete indices on primary instance.")
        logger.info("Stopping opensearch replication")
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)

            headers = {'Content-Type': 'application/json'}

            indices_response = requests.get(f'https://{self.get_hosts()[0]}:9200/_cat/indices',
                                            auth=(self.get_user(), self.get_password()), verify=False)
            indices_response.raise_for_status()
            index_names = [line.split()[2] for line in indices_response.text.strip().split('\n') if
                           len(line.split()) > 2]
            logger.info(f"There are a total of {len(index_names)} indices")
            if cmd_args.delete_rule:
                # delete ISM policies
                self.delete_ism_policies()
                # delete templates
                self.delete_templates()
                # delete aliases
                self.delete_aliases()
                try:
                    delete_url = f'https://{self.get_hosts()[0]}:9200/_plugins/_replication/_autofollow'
                    payload = {
                        "leader_alias": "opensearch-cluster",
                        "name": "my-replication-rule"}
                    response = requests.delete(
                        delete_url, headers={'Content-Type': 'application/json'},
                        auth=(self.get_user(), self.get_password()), verify=False,
                        json=payload)
                    response.raise_for_status()
                    logger.info(f"Successfully deleted auto-follow rule: my-replication-rule")
                except requests.exceptions.HTTPError as e:
                    logger.error(f"Failed to delete auto-follow rule: my-replication-rule, Status: {e.response.status_code}, Response: {e.response.text}")
            for index in index_names:
                status_response = requests.get(f'https://{self.get_hosts()[0]}:9200/_plugins/_replication/{index}/_status',
                                              auth=(self.get_user(), self.get_password()), verify=False)
                status_response.raise_for_status()
                status_data = status_response.json()

                if status_data.get('status') != 'REPLICATION NOT IN PROGRESS':
                    replication_stop_url = f'https://{self.get_hosts()[0]}:9200/_plugins/_replication/{index}/_stop'
                    headers = {'Content-Type': 'application/json'}
                    try:
                        stop_response = requests.post(replication_stop_url, headers=headers,
                                                       auth=(self.get_user(), self.get_password()), verify=False, json={})
                        stop_response.raise_for_status()
                        logger.info(f"Successfully stopped replication for index '{index}'.")
                    except requests.exceptions.HTTPError as e:
                        logger.error(f"Failed to stop replication for index '{index}': {e.response.text}")
                else:
                    logger.info(f"Replication is already stopped for index '{index}'.")

                if cmd_args.delete_index:
                    try:
                        delete_response = requests.delete(f'https://{self.get_hosts()[0]}:9200/{index}',
                                                          auth=(self.get_user(), self.get_password()), verify=False)
                        delete_response.raise_for_status()
                        logger.info(f"Successfully deleted index: {index}")
                    except requests.exceptions.HTTPError as e:
                        logger.error(f"Failed to delete index: {index}, Status Code: {e.response.status_code}, Response: {e.response.text}")

    def delete_ism_policies(self):
        logger.info("Deleting ISM policies")
        try:
            with requestsutil.new_session():
                requests.packages.urllib3.disable_warnings(
                    requests.packages.urllib3.exceptions.InsecureRequestWarning)
                response = requests.get(f'https://{self.get_hosts()[0]}:9200/_plugins/_ism/policies',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                response.raise_for_status()
                policies = response.json().get('policies', [])
                for policy in policies:
                    policy_id = policy['_id']
                    logger.debug(f"Deleting ISM policy: {policy_id}")
                    response = requests.delete(f'https://{self.get_hosts()[0]}:9200/_plugins/_ism/policies/{policy_id}',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                    response.raise_for_status()
                    logger.debug(f"Successfully deleted ISM policy: {policy_id}")
        except Exception as e:
            logger.error(f"Failed to delete ISM policies: {e}")
    
    def delete_templates(self):
        logger.info("Deleting Index templates")
        try:
            with requestsutil.new_session():
                requests.packages.urllib3.disable_warnings(
                    requests.packages.urllib3.exceptions.InsecureRequestWarning)
                response = requests.get(f'https://{self.get_hosts()[0]}:9200/_index_template',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                response.raise_for_status()
                templates = response.json().get('index_templates', [])
                for template in templates:
                    template_name = template['name']
                    logger.debug(f"Deleting Index template: {template_name}")
                    response = requests.delete(f'https://{self.get_hosts()[0]}:9200/_index_template/{template_name}',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                    response.raise_for_status()
                    logger.debug(f"Successfully deleted Index template: {template_name}")
        except Exception as e:
            logger.error(f"Failed to delete Index templates: {e}")

    def delete_aliases(self):
        logger.info("Deleting aliases")
        try:
            with requestsutil.new_session():
                requests.packages.urllib3.disable_warnings(
                    requests.packages.urllib3.exceptions.InsecureRequestWarning)
                response = requests.get(f'https://{self.get_hosts()[0]}:9200/_cat/aliases?format=json',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                response.raise_for_status()
                aliases = response.json()
                for alias in aliases:
                    index_name = alias['index']
                    alias_name = alias['alias']
                    logger.debug(f"Deleting alias: {alias_name} on index: {index_name}")
                    response = requests.delete(f'https://{self.get_hosts()[0]}:9200/{index_name}/_alias/{alias_name}',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                    response.raise_for_status()
                    logger.debug(f"Successfully deleted alias: {alias_name} on index: {index_name}")
        except Exception as e:
            logger.error(f"Failed to delete aliases: {e}")

    def geodr_apply_metadata(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        logger.info("Applying metadata for opensearch")
        metadata_file = '/tmp/metadata.json'
        if not self._download_metadata_from_minio(config_parser, metadata_file):
            logger.info(f"Metadata file does not exist on MinIO persistent-streams/ism/ism_config.json")
            return
            
        with open(metadata_file, 'r') as f:
            metadata = json.load(f)

        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)
            for template_item in metadata.get('templates', {}).get('index_templates', []):
                template_name = template_item['name']
                template_data = template_item['index_template']
                logger.debug(f"Applying template: {template_name}")
                try:
                    response = requests.put(f'https://{self.get_hosts()[0]}:9200/_index_template/{template_name}',
                                            auth=(self.get_user(), self.get_password()), verify=False, json=template_data)
                    response.raise_for_status()
                    logger.info(f"Successfully applied template: {template_name}")
                except requests.exceptions.HTTPError as e:
                    if e.response.status_code == 409:
                        logger.info(f"Template {template_name} already exists, skipping")
                    else:
                        logger.error(f"Failed to apply template {template_name}: {e}")
                        logger.error(f"Response content: {e.response.text}")
                        raise

            for policy_item in metadata.get('policies', {}).get('policies', []):
                policy_name = policy_item['policy']['policy_id']
                policy_data = {'policy': policy_item['policy']}
                logger.debug(f"Applying policy: {policy_name}")
                try:
                    response = requests.put(f'https://{self.get_hosts()[0]}:9200/_plugins/_ism/policies/{policy_name}',
                                            auth=(self.get_user(), self.get_password()), verify=False, json=policy_data)
                    response.raise_for_status()
                    logger.info(f"Successfully applied policy: {policy_name}")
                except requests.exceptions.HTTPError as e:
                    if e.response.status_code == 409:
                        logger.info(f"Policy {policy_name} already exists, skipping")
                    else:
                        logger.error(f"Failed to apply policy {policy_name}: {e}")
                        logger.error(f"Response content: {e.response.text}")
                        raise

            for index, alias_data in metadata.get('aliases', {}).items():
                logger.info(f"Applying alias for index: {index}")
                
                try:
                    index_check = requests.head(f'https://{self.get_hosts()[0]}:9200/{index}',
                                              auth=(self.get_user(), self.get_password()), verify=False)
                    if index_check.status_code == 404:
                        logger.warning(f"Index {index} does not exist, skipping alias creation")
                        continue
                except Exception as e:
                    logger.warning(f"Could not check if index {index} exists: {e}")
                
                for alias, properties in alias_data.items():
                    alias_url = f'https://{self.get_hosts()[0]}:9200/{index}/_alias/{alias}'
                    alias_payload = {}
                    
                    if properties.get("is_write_index") is not None:
                        alias_payload["is_write_index"] = properties.get("is_write_index")
                    if "filter" in properties:
                        alias_payload["filter"] = properties["filter"]
                    if "routing" in properties:
                        alias_payload["routing"] = properties["routing"]
                    
                    logger.debug(f"Alias payload for {alias}: {alias_payload}")
                    try:
                        response = requests.put(alias_url, auth=(self.get_user(), self.get_password()), 
                                              verify=False, json=alias_payload, headers={'Content-Type': 'application/json'})
                        response.raise_for_status()
                        logger.info(f"Successfully applied alias: {alias} for index: {index}")
                    except requests.exceptions.HTTPError as e:
                        if e.response.status_code == 409:
                            logger.info(f"Alias {alias} for index {index} already exists, skipping")
                        else:
                            logger.error(f"Failed to apply alias {alias} for index {index}: {e}")
                            logger.error(f"Response content: {e.response.text}")
                            logger.error(f"Payload was: {alias_payload}")
                            raise

            # Attach ISM policies to indices
            self._attach_ism_policies_to_indices(metadata)

        logger.info("Completed applying metadata for opensearch")

    def _download_metadata_from_minio(self, config_parser: configparser.ConfigParser, output_file: str) -> bool:
        try:
            tenant_id = config_parser.get('common', 'tenant_id')
            from rdaf.component import minio as minio_comp
            minio_component = COMPONENT_REGISTRY.require(minio_comp.COMPONENT_NAME)
            val = f'http://{minio_component.get_user()}:{minio_component.get_password()}@{minio_component.get_hosts()[0]}:9000'
            env = dict(os.environ, MC_HOST_myminio=val)
            download_cmd = f'mc cp --insecure myminio/tenants.{tenant_id}/persistent-streams/ism/ism_config.json {output_file}'
            result = subprocess.run(download_cmd, shell=True, env=env, 
                                  capture_output=True, text=True)
            
            if result.returncode == 0:
                logger.info("Successfully downloaded metadata from MinIO")
                return True
            else:
                logger.warning(f"Failed to download metadata from MinIO: {result.stderr}")
                return False         
        except Exception as e:
            logger.warning(f"Error downloading metadata from MinIO: {e}")
            return False

    def _attach_ism_policies_to_indices(self, metadata):
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)
            try:
                logger.info("Attaching ISM policies to indices")
                host = self.get_hosts()[0]
                base_url = f'https://{host}:9200'
                auth = (self.get_user(), self.get_password())
                headers = {'Content-Type': 'application/json'}
                indices_response = requests.get(f"{base_url}/_cat/indices?h=index", 
                                            auth=auth, headers=headers, verify=False)
                indices_response.raise_for_status()
                indices = indices_response.text.strip().split('\n') if indices_response.text.strip() else []
                
                policy_ids = [policy_item['policy']['policy_id'] 
                             for policy_item in metadata.get('policies', {}).get('policies', [])]
                
                for policy_id in policy_ids:
                    matching_indices = [idx for idx in indices if policy_id in idx]
                    
                    for index in matching_indices:
                        # Check if index is a CCR (Cross Cluster Replication) index
                        try:
                            index_response = requests.get(f"{base_url}/{index}", 
                                                        auth=auth, headers=headers, verify=False)
                            index_response.raise_for_status()
                            index_data = index_response.json()
                            
                            settings = index_data[index]['settings']['index']
                            if settings.get('replication.type', '').lower() == 'ccr':
                                logger.info(f"Skipping CCR index: {index}")
                                continue
                        except Exception as e:
                            logger.warning(f"Could not check settings for index {index}: {e}")
                            continue
                        
                        # Try to attach policy using both possible API paths
                        attached = False
                        for api_path in ['_plugins/_ism/add', '_opendistro/_ism/add']:
                            attach_url = f"{base_url}/{api_path}/{index}"
                            payload = {'policy_id': policy_id}
                            
                            try:
                                attach_response = requests.post(attach_url, auth=auth, 
                                                            headers=headers, json=payload, verify=False)
                                if attach_response.ok:
                                    logger.info(f"Successfully attached policy {policy_id} to index {index}")
                                    attached = True
                                    break
                                else:
                                    logger.warning(f"Failed to attach {policy_id} to {index} via {api_path}: {attach_response.text}")
                            except Exception as e:
                                logger.warning(f"Error attaching {policy_id} to {index} via {api_path}: {e}")
                        
                        if not attached:
                            logger.error(f"Could not attach policy {policy_id} to index {index}")
                            
            except Exception as e:
                logger.error(f"Error in ISM policy attachment: {e}")