import argparse
import configparser
import json
import os
import logging
import socket
import string
import tempfile
import time
import glob
from typing import Callable, Any, List, Union, Tuple

import requests
import subprocess
import urllib3

import rdaf.component
import rdaf.component.opensearch as opensearch_comp
import yaml
from docker.models.containers import ContainerCollection
from rdaf.component import cert, check_potential_remote_file_exists, remove_dir_contents
from rdaf import rdafutils, get_templates_dir_root
from rdaf.component import Component, OtherCategoryOrder, _apply_data_dir_defaults, \
    _host_dir_storer, _host_dir_loader, run_potential_ssh_command, run_command, execute_command
from rdaf.component import _comma_delimited_to_list, do_potential_scp, _list_to_comma_delimited
import rdaf.component.platform as platform_comp
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.rdafutils import str_base64_encode, str_base64_decode, cli_err_exit
from rdaf.util import requestsutil

logger = logging.getLogger(__name__)
COMPONENT_NAME = 'opensearch_external'

class OpensearchExternal(Component):
    _option_user = 'user'
    _option_password = 'password'
    _option_data_dir = 'datadir'
    _option_zone = 'zone'
    _option_host = 'host'
    _default_data_dirs = ['/opensearch']
    _option_data_host = 'data_host'
    _option_cluster_manager_host = 'cluster_manager_host'
    _option_client_host = 'client_host'

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'os_external',  category='other',
                         category_order=OtherCategoryOrder.OPENSEARCH_EXTERNAL.value)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name in [self._option_host, self._option_cluster_manager_host, self._option_data_host, self._option_client_host]:
            return _comma_delimited_to_list
        if config_name == self._option_data_dir or config_name == self._option_zone:
            return _host_dir_loader
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name in [self._option_host, self._option_cluster_manager_host, self._option_data_host, self._option_client_host]:
            return _list_to_comma_delimited
        if config_name == self._option_data_dir or config_name == self._option_zone:
            return _host_dir_storer
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_data_dir] = None
        default_configs[self._option_user] = None
        default_configs[self._option_password] = None
        default_configs[self._option_host] = None
        default_configs[self._option_cluster_manager_host] = None
        default_configs[self._option_zone] = None
        default_configs[self._option_data_host] = None
        default_configs[self._option_client_host] = None
        return default_configs

    def gather_setup_inputs(self, cmd_args, config_parser):
        configs = self._init_default_configs()
        pass_desc = 'What is the SSH password for the SSH user used to communicate' \
                    ' between hosts'
        pass_no_prompt_err_msg = 'No SSH password specified. Use --ssh-password to specify one'
        # don't store in the configs
        self.ssh_password = rdaf.component.Component \
            ._parse_or_prompt_value(cmd_args.ssh_password,
                                    None,
                                    pass_no_prompt_err_msg,
                                    pass_desc,
                                    'SSH password',
                                    cmd_args.no_prompt,
                                    password=True,
                                    apply_password_validation=False)

        cluster_manager_host_desc = 'What is the host(s) for cluster manager?'
        cluster_manager_hosts = self._parse_or_prompt_hosts(cmd_args.cluster_manager_host,'',
            'No cluster manager host specified.',
            cluster_manager_host_desc,
            'opensearch cluster manager host(s)',
            cmd_args.no_prompt)
        configs[self._option_cluster_manager_host] = cluster_manager_hosts

        cluster_client_host_desc = 'What is the host(s) for cluster clients?'
        cluster_client_hosts = self._parse_or_prompt_hosts(cmd_args.client_host, '',
                                                            'No cluster client host specified.',
                                                            cluster_client_host_desc,
                                                            'opensearch cluster client host(s)',
                                                            cmd_args.no_prompt)
        configs[self._option_client_host] = cluster_client_hosts

        # now gather app service deployment mode
        desc = 'Do you want to configure cluster zoning?'
        if cmd_args.no_prompt:
            os_zoning = cmd_args.os_zoning
        else:
            os_zoning = rdafutils.query_yes_no(desc, default='no')
        data_zones = []
        if not os_zoning:
            data_host_desc = 'What is the host(s) for data nodes?'
            data_hosts = self._parse_or_prompt_hosts(cmd_args.data_host,'',
                'No data host specified.', data_host_desc,
                'opensearch cluster data host(s)', cmd_args.no_prompt)
            configs[self._option_data_host] = data_hosts
            data_zones = [(host, None) for host in data_hosts]
        else:
            # Check if zones configuration is provided from JSON
            if hasattr(cmd_args, 'zones_config') and cmd_args.zones_config:
                zones_config = cmd_args.zones_config
                all_data_hosts = []
                for zone_name, hosts in zones_config.items():
                    all_data_hosts.extend(hosts)
                    # Extract zone index from zone name (e.g., "zone-0" -> "0")
                    zone_suffix = zone_name.split('-')[-1] if '-' in zone_name else zone_name
                    zone_path = f'/zone-{zone_suffix}'
                    data_zones += [(host, [zone_path]) for host in hosts]
                configs[self._option_data_host] = all_data_hosts
            else:
                zones_desc = 'Please specify the number of zones to be configured'
                zones_input = self._parse_or_prompt_value('', '2',
                                                   "",
                                                   zones_desc,
                                                   'Number of Zones',
                                                   cmd_args.no_prompt,
                                                   False)
                all_data_hosts = []
                for index in range(int(zones_input)):
                    data_host_desc = f'What is the host(s) for data nodes in zone-{index}?'
                    data_hosts = self._parse_or_prompt_hosts(cmd_args.data_host,'',
                        'No data host specified.', data_host_desc,
                        f'opensearch cluster data host(s) for zone-{index}', cmd_args.no_prompt)
                    all_data_hosts += data_hosts
                    data_zones += [(host, [f'/zone-{index}']) for host in data_hosts]
                configs[self._option_data_host] = all_data_hosts
        configs[self._option_zone] = \
            _apply_data_dir_defaults(data_zones, ['/default'])

        user_desc = 'What is the user name you want to give ' \
                               'for opensearch cluster admin user that ' \
                               'will be created and used by the RDAF platform?'
        user_no_prompt_err_msg = 'No opensearch user specified. Use ' \
                                 '--opensearch_external-user to specify one'
        user = self._parse_or_prompt_value(cmd_args.os_external_admin_user, 'rdafadmin',
                                            user_no_prompt_err_msg,
                                            user_desc,
                                           'opensearch user',
                                           cmd_args.no_prompt)
        configs[self._option_user] = str_base64_encode(user)

        pass_desc = 'What is the password you want to use for opensearch admin user?'
        pass_no_prompt_err_msg = 'No password specified for opensearch_external user. Use --opensearch_external-password' \
                                 ' to specify one'
        default_autogen_password = rdafutils.gen_password()
        passwd = self._parse_or_prompt_value(cmd_args.os_external_admin_password,
                                                                 default_autogen_password,
                                                                 pass_no_prompt_err_msg,
                                                                 pass_desc, 'opensearch password',
                                                                 cmd_args.no_prompt,
                                                                 password=True)
        configs[self._option_password] = str_base64_encode(passwd)

        # to maintain order
        all_hosts = []
        for entry in (configs[self._option_cluster_manager_host], configs[self._option_client_host], configs[self._option_data_host]):
            for item in entry:
                if item not in all_hosts:
                    all_hosts.append(item)
        configs[self._option_host] = all_hosts
        host_dirs = [(host, None) for host in configs[self._option_host]]
        configs[self._option_data_dir] = \
            _apply_data_dir_defaults(host_dirs, self._default_data_dirs)

        es = COMPONENT_REGISTRY.require(opensearch_comp.COMPONENT_NAME)
        es_hosts = es.get_hosts()
        for host in configs[self._option_host]:
           if host in es_hosts:
               cli_err_exit(f'OpenSearch is already deployed on host {host}. Please specify a different host.')

        self._mark_configured(configs, config_parser)
        self.write_configs(config_parser)

    def get_hosts(self) -> list:
        if not self.configs[self._option_host]:
            return []
        return self.configs[self._option_host]

    def get_ports(self) -> tuple:
        ports = ['9200', '9300'] if len(self.get_hosts()) == 1 else ['9200', '9300']
        hosts = self.get_hosts()
        return hosts, ports

    def _get_host_data_dirs(self) -> List[Tuple[str, List[str]]]:
        return self.configs[self._option_data_dir]

    def get_user(self) -> str:
        return str_base64_decode(self.configs[self._option_user])

    def get_password(self) -> str:
        return str_base64_decode(self.configs[self._option_password])

    def get_escaped_password(self) -> str:
        return str_base64_decode(self.configs[self._option_password]).replace("$", "\\$").replace("&", "/&")

    def get_data_hosts(self) -> list:
        return self.configs[self._option_data_host]

    def _get_data_hosts_zones(self) -> List[Tuple[str, List[str]]]:
        return self.configs[self._option_zone]

    def get_cluster_manager_hosts(self) -> list:
        return self.configs[self._option_cluster_manager_host]

    def get_cluster_client_hosts(self) -> list:
        return self.configs[self._option_client_host]

    def get_k8s_cluster_endpoint(self, config_parser) -> list:
        namespace = self.get_namespace(config_parser)
        data = self.get_data_hosts()
        master = self.get_cluster_manager_hosts()
        client = self.get_cluster_client_hosts()
    
        # Determine service type and hosts count
        if set(master) == set(client) == set(data):
            service = 'os-ext-master'
            hosts = master
        elif set(master) == set(client) != set(data):
            service = 'os-ext-master'
            hosts = master
        else:
            service = 'os-ext-client'
            hosts = client
    
        # Generate the list of headless service endpoints
        es_hosts = [
            f'{service}-{i}.{service}-headless.{namespace}.svc.cluster.local'
            for i in range(len(hosts))]
        
        return es_hosts

    def do_setup(self, cmd_args, config_parser):
        network_json = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'config.json')
        if not os.path.exists(network_json):
            cli_err_exit('Please install RDAF platform services before installing external opensearch')
        self.gather_setup_inputs(cmd_args, config_parser)
        logger.info('Doing setup for ' + self.component_name)
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME,
                             COMPONENT_NAME]
        )
        docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
        for host in self.get_hosts():
            if host not in all_known_hosts:
                ssh_manager = COMPONENT_REGISTRY.require(
                    rdaf.component.ssh.SSHKeyManager.COMPONENT_NAME)
                # setup ssh keys for this new host
                ssh_manager.setup_keys_for_host(host, self.ssh_password)
            # doing a docker login
            docker_registry.docker_login(host, config_parser)
        # copy docker-compose
        self.docker_compose_check(config_parser)

        self.setup_install_root_dir_hierarchy(self.get_hosts(), config_parser)
        for host in self.get_hosts():
            content = self._create_opensearch_yaml(host, config_parser)
            opensearch_yaml = os.path.join(self.get_conf_dir(), 'opensearch.yaml')
            created_location = rdaf.component.create_file(host, content.encode(encoding='UTF-8'),
                                                          opensearch_yaml)
            logger.info('Created opensearch external configuration at ' + created_location + ' on ' + host)

        # owning the data dir
        for host, data_dirs in self.configs[self._option_data_dir]:
            data_dir = data_dirs[0]
            backup_dir = os.path.join("/opt", "opensearch-external-backup")
            command = (f'sudo mkdir -p {data_dir} {backup_dir} && sudo chown -R 1000 {data_dir} {backup_dir} '
                       f'&& sudo chgrp -R 1000 {data_dir} {backup_dir}')
            run_potential_ssh_command(host, command, config_parser)

        for host in self.get_hosts():
            command = 'sudo mkdir -p ' + self.get_logs_dir() + ' && sudo chmod -R 777 ' + self.get_logs_dir()
            run_potential_ssh_command(host, command, config_parser)

        self.setup_user(config_parser)
        template_path = os.path.join(get_templates_dir_root(), 'opensearch-external-values.yaml')
        destination_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'opensearch-external-values.yaml')
        do_potential_scp(socket.gethostname(), template_path, destination_path)
        logger.info('Setup completed successfully')

    def get_deployment_file_name(self) -> os.path:
        return 'opensearch-external.yaml'

    def get_deployment_file_path(self, host) -> os.path:
        return os.path.join('/opt', 'rdaf', 'deployment-scripts', host, 'opensearch-external.yaml')

    def create_compose_file(self, cmd_args: argparse.Namespace, host: str) -> os.path:
        compose_file = self.get_deployment_file_path(host)
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'opensearch-external.yaml')
        replacements = self.get_deployment_replacements(cmd_args, host)

        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        values = yaml.safe_load(original_content)
        component_value = values['services']['opensearch_external']
        if 'environment' in component_value.keys():
            self.update_cluster_env(component_value, host)
        # override with values.yaml inputs if provided any
        self.apply_user_inputs(host, component_value)

        run_command('mkdir -p ' + os.path.dirname(compose_file))
        with open(compose_file, 'w') as f:
             yaml.safe_dump(values, f, default_flow_style=False, explicit_start=True,
                            allow_unicode=True, encoding='utf-8', sort_keys=False)

        if not self.is_local_host(host):
            do_potential_scp(host, compose_file, compose_file)

    def _apply_security_audit_policy(self, cmd_args, config_parser):
        logger.info("Creating policy to purge security audit logs.")
        os_user = self.get_user()
        os_password = self.get_escaped_password()
        host = self.get_hosts()[0]
        if self.get_deployment_type(config_parser) in ['k8s', 'ctr']:
           port = self.get_service_node_port('os-ext-master', config_parser)
        else:
           port = '9200'
        days = cmd_args.days

        ret, stdout, stderr = execute_command(
            f'curl -s -o /dev/null -w "%{{http_code}}" -X GET -u "{os_user}:{os_password}" '
            f'"https://{host}:{port}/_plugins/_ism/policies/purge_security_audit_logs" --insecure')
        if stdout.strip() == '200':
            logger.info("security audit purge policy already exists.")
            return

        policy = {"policy": {"description": f"Purge security audit log indices older than {days} days",
                             "default_state": "idle",
                             "ism_template": {"index_patterns": ["security-auditlog-*"],
                                              "priority": 100},
                             "states": [{"name": "idle",
                                         "actions": [],
                                         "transitions": [{"state_name": "delete",
                                                          "conditions": {"min_index_age": f"{days}d"}}]},
                                        {"name": "delete",
                                         "actions": [{"delete": {}}],
                                         "transitions": []}
                                        ]
                             }
                  }
        data = json.dumps(policy)
        run_command(
            f'curl -s -k -X PUT -u "{os_user}:{os_password}" '
            f'"https://{host}:{port}/_plugins/_ism/policies/purge_security_audit_logs" '
            f'-H \'Content-Type: application/json\' --data \'{data}\' --insecure', shell=True)
        print("")

        # this will update the policy to any existing indices
        data = json.dumps({"policy_id": "purge_security_audit_logs"})
        run_command(
            f'curl -s -k -X POST -u "{os_user}:{os_password}" '
            f'"https://{host}:{port}/_plugins/_ism/add/security-auditlog-*" '
            f'-H \'Content-Type: application/json\' --data \'{data}\' --insecure', shell=True)
        print("")

    def setup_user(self, config_parser):
        host = self.get_hosts()[0]
        internal_users_yml = os.path.join(self.get_conf_dir(), 'internal_users.yml')
        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            compose_file = os.path.join(tmp, 'docker-compose.yaml')
            image_name = self._get_docker_repo()['DOCKER_REPO'] + '/rda-platform-opensearch:1.0.4'
            content = {"services": {'opensearch': {
                "image": image_name, 'network_mode': 'host'}}}

            with open(compose_file, 'w+') as f:
                yaml.safe_dump(content, f, default_flow_style=False, explicit_start=True,
                               allow_unicode=True, encoding='utf-8', sort_keys=False)
            if not self.is_local_host(host):
                do_potential_scp(host, tmp, tmp)
            pull_cmd = '/usr/local/bin/docker-compose -f {file} pull '.format(file=compose_file)
            run_potential_ssh_command(host, pull_cmd, config_parser)
            with self.new_docker_client_(host) as docker_client:
                container_collection = ContainerCollection(client=docker_client.client)
                command = ['plugins/opensearch-security/tools/hash.sh',
                           '-p', self.get_password()]
                env = ['JAVA_HOME=/usr/share/opensearch/jdk']
                hash_generated = container_collection.run(image=image_name,
                                                          command=command,
                                                          environment=env,
                                                          remove=True)
                pass_list = hash_generated.split(b'\n')
                hash_pass = pass_list[-2].decode('utf-8')

            for host in self.get_hosts():
                meta = {'type': 'internalusers', 'config_version': 2}
                user = {'hash': hash_pass, 'reserved': True,
                        'backend_roles': ['admin'],
                        'description': 'Opensearch admin user'}
                data = {'_meta': meta, self.get_user(): user}
                yaml_data = yaml.safe_dump(data, default_flow_style=False, explicit_start=True,
                                           allow_unicode=True, encoding='utf-8', sort_keys=False)
                rdaf.component.create_file(host, yaml_data, internal_users_yml)
        if not self.is_local_host(host):
            remove_tmp_command = 'rm -rf ' + tmp
            run_potential_ssh_command(host, remove_tmp_command, config_parser)

    def add_opensearch_external_host(self, host: str, config_parser: configparser.ConfigParser):
        es = COMPONENT_REGISTRY.require(opensearch_comp.COMPONENT_NAME)
        es_hosts = es.get_hosts()
        if host in es_hosts:
            cli_err_exit(f'OpenSearch is already deployed on host {host}. Please choose a different host.')
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME]
        )
        if host not in all_known_hosts:
            self.setup_install_root_dir_hierarchy(host, config_parser)

        self.configs[self._option_host].append(host)
        # update the config parser with changes that have happened to our configs
        self.store_config(config_parser)
        self.open_ports(config_parser)
        # doing a docker login
        docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
        docker_registry.docker_login(host, config_parser)
        content = self._create_opensearch_yaml(host, config_parser)
        opensearch_external_yaml = os.path.join(self.get_conf_dir(), 'opensearch.yaml')
        created_location = rdaf.component.create_file(host, content.encode(encoding='UTF-8'),
                                                      opensearch_external_yaml)
        logger.info('Created opensearch_external configuration at ' + created_location + ' on ' + host)


    def setup_install_root_dir_hierarchy(self, hosts: Union[str, List[str]],
                                         config_parser: configparser.ConfigParser):
        known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME,
                             rdaf.component.opensearch_external.COMPONENT_NAME])
        cert_manager = COMPONENT_REGISTRY.require(rdaf.component.cert.CertManager.COMPONENT_NAME)
        install_root = self.get_rdaf_install_root(config_parser)
        uid = os.getuid()
        gid = os.getgid()
        dirs = [install_root, os.path.join(install_root, 'config'),
                os.path.join(install_root, 'data'),
                os.path.join(install_root, 'logs'),
                os.path.join(install_root, 'deployment-scripts')]
        if isinstance(hosts, str):
            hosts = [hosts]
        for host in hosts:
            if host in known_hosts:
                continue
            for path in dirs:
                logger.info(f'Creating directory {path} and setting ownership to user {uid} and group to group {gid} on host {host}')
                command = ('sudo mkdir -p ' + path + ' && sudo chown -R ' + str(uid) + ' ' + path
                           + ' && sudo chgrp -R ' + str(gid) + ' ' + path)
                run_potential_ssh_command(host, command, config_parser)

            # copy certs
            cert_manager.copy_certs([host])

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.open_ports(config_parser)
        for host in self.get_hosts():
            self.create_compose_file(cmd_args, host)
            deployment_file = self.get_deployment_file_path(host)
            command = '/usr/local/bin/docker-compose --project-name os_external -f {file} up -d ' \
                      .format(file=deployment_file)
            run_potential_ssh_command(host, command, config_parser)
        platform = COMPONENT_REGISTRY.require(platform_comp.COMPONENT_NAME)
        config_path = os.path.join('/opt/rdaf/config/network_config/config.json')
        # network config.json
        network_config = json.loads(open(config_path).read())
        if 'os_external' not in network_config:
            backup_command = f'cp {config_path} {config_path}_$(date +"%Y%m%d%H%M%S").bak'
            run_command(backup_command)
            logger.info("Updating config.json with os_external endpoint.")
            network_config['os_external'] = {'hosts': self.get_cluster_client_hosts(), 'user': self.get_user(),
                                             'password': self.get_password(), 'port':'9200',
                                             'scheme': 'https', 'ssl_verify': False}
            self.encrypt_rda_config(network_config)
            output = json.dumps(network_config, indent=4)
            with open(config_path, "w") as f:
                f.write(output)
                f.flush()

            platform._copy_configs_known_hosts(config_parser)

        policy_path = os.path.join('/opt/rdaf/config/network_config/policy.json')
        policy_json = json.loads(open(policy_path).read())
        if 'os_external_default' not in policy_json['credentials']["es"]:
            backup_command = f'cp {policy_path} {policy_path}_$(date +"%Y%m%d%H%M%S").bak'
            run_command(backup_command)
            tenant_id = network_config.get('tenant_id')
            logger.info("Updating policy.json with os_external endpoint.")
            policy_json["pstream-mappings"].insert(2, {'pattern': f'os-external-admin-{tenant_id}.*', 'es_name': 'os_external_default'})
            policy_json["pstream-mappings"].insert(3, {'pattern': 'os-external-rda.*',
                                                       'es_name': 'os_external_default'})
            policy_json["pstream-mappings"].insert(4, {'pattern': 'os-external-.*',
                                                       'es_name': 'os_external_default'})
            policy_json['credentials']["es"]['os_external_default'] = {'hosts': self.get_cluster_client_hosts(),
                                                                       '$user': platform.encrypt(self.get_user(), tenant_id),
                                                                       '$password': platform.encrypt(self.get_password(), tenant_id),
                                                                       'port':'9200', 'scheme': 'https', 'ssl_verify': False}
            output = json.dumps(policy_json, indent=4)
            with open(policy_path, "w") as f:
                f.write(output)
                f.flush()

            for host in platform.get_hosts():
                if not self.is_local_host(host):
                    do_potential_scp(host, policy_path, policy_path)

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            self.create_compose_file(cmd_args, host)
            deployment_file = self.get_deployment_file_path(host)
            command = '/usr/local/bin/docker-compose --project-name os_external -f {file} up -d ' \
                      .format(file=deployment_file)
            run_potential_ssh_command(host, command, config_parser)

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            deployment_file = self.get_deployment_file_path(host)
            if not os.path.exists(deployment_file):
                return
            command = '/usr/local/bin/docker-compose --project-name os_external -f ' + deployment_file + ' up -d '
            logger.info("Creating opensearch-external on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in reversed(self.get_hosts()):
            deployment_file = self.get_deployment_file_path(host)
            if not os.path.exists(deployment_file):
                return
            logger.info("Deleting opensearch-external  on host {}".format(host))
            command = '/usr/local/bin/docker-compose --project-name os_external -f ' \
                      + deployment_file + ' rm -fs opensearch_external'
            run_potential_ssh_command(host, command, config_parser)

    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in reversed(self.get_hosts()):
            deployment_file = self.get_deployment_file_path(host)
            if not os.path.exists(deployment_file):
                return
            logger.info(f"Stopping opensearch_external on host {host}")
            command = '/usr/local/bin/docker-compose --project-name os_external -f ' + deployment_file + \
                      ' stop ' + self.component_name
            run_potential_ssh_command(host, command, config_parser)

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            deployment_file = self.get_deployment_file_path(host)
            if not os.path.exists(deployment_file):
                return
            logger.info("Starting opensearch_external on host {}".format( host))
            command = '/usr/local/bin/docker-compose --project-name os_external -f ' + deployment_file \
                        + ' start ' + self.component_name
            run_potential_ssh_command(host, command, config_parser)

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        if not self.get_hosts():
            return []
        return super().status(cmd_args, config_parser)

    def _create_opensearch_yaml(self, host, config_parser) -> str:
        hosts = self.get_hosts()
        opensearch_external_yaml = os.path.join(get_templates_dir_root(), 'opensearch-external.yaml')
        with open(opensearch_external_yaml, 'r') as f:
            template_content = f.read()

        replacements = dict()
        replacements['DISCOVERY_TYPE'] = 'zen' if len(hosts) > 1 else 'single-node'
        replacements['NODE_NAME'] = host
        replacements['ZONE_DETAILS'] = self._get_zone_details(host)
        node_type = ['remote_cluster_client']
        if host in self.get_data_hosts():
            node_type.append('data')
        if host in self.get_cluster_manager_hosts():
            node_type.append('cluster_manager')
        if host in self.get_cluster_client_hosts():
            node_type.append('client')
        replacements['NODE_ROLE'] = node_type
        cert_manager: cert.CertManager = COMPONENT_REGISTRY.require(
            cert.CertManager.COMPONENT_NAME)
        replacements['TRUSTSTORE_PASSWORD'] = cert_manager.get_keystore_pass()
        replacements['KEYSTORE_PASSWORD'] = cert_manager.get_keystore_pass()
        replacements['GEODR_FOLLOWER_VARIABLES'] = ''

        if self.is_geodr_deployment(config_parser):
            replacements['GEODR_FOLLOWER_VARIABLES'] = '''
plugins.replication.follower.index.recovery.chunk_size: 512mb
plugins.replication.follower.index.recovery.max_concurrent_file_chunks: 4
plugins.replication.follower.index.ops_batch_size: 50000
plugins.replication.follower.concurrent_readers_per_shard: 4
plugins.replication.follower.metadata_sync_interval: 30s'''
            
        original_content = string.Template(template_content).substitute(replacements)
        return original_content

    @staticmethod
    def apply_input_values(env):
        values_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'opensearch-external-values.yaml')
        if not os.path.exists(values_path):
            return
        with open(values_path, 'r') as f:
            values = yaml.safe_load(f)

        if 'client' in values['services']:
            client = values['services']['client']
            if 'volumes' in client and client['volumes']['BACKUP_MOUNT']:
                env['BACKUP_MOUNT'] = client['volumes']['BACKUP_MOUNT']

    def get_deployment_replacements(self, cmd_args: argparse.Namespace, host: str) -> dict:
        replacements = self._get_docker_repo()
        replacements['TAG'] = cmd_args.tag
        replacements.update(self.get_deployment_env(host))
        return replacements

    def get_deployment_env(self, host: str):
        env = dict()
        env['NODE_NAME'] = host
        env['DATA_MOUNT'] = self._default_data_dirs[0]
        env['BACKUP_MOUNT'] = os.path.join('/opt', 'opensearch-external-backup')
        env['ROOT_USER'] = str_base64_decode(self.configs[self._option_user]).replace("$", "$$")
        env['ROOT_PASSWORD'] = str_base64_decode(self.configs[self._option_password]).replace("$", "$$")
        env['YAML_CONFIG'] = os.path.join(self.get_conf_dir(), 'opensearch.yaml')
        opensearch_jks_path = os.path.join('/opt/rdaf/cert/rdaf/rdaf.jks')
        env['SSL_KEYSTORE_JKS_FILENAME'] = 'rdaf.jks'
        env['KEYSTORE_SOURCE_PATH'] = opensearch_jks_path
        env['INTERNAL_USER_YAML'] = os.path.join(self.get_conf_dir(), 'internal_users.yml')
        self.apply_input_values(env)
        return env

    def update_cluster_env(self, component_yaml: dict, host: str):
        if len(self.get_hosts()) == 1:
            return
        component_yaml['environment'].append('discovery.seed_hosts=' + ','.join(self.get_cluster_manager_hosts()))
        component_yaml['environment'].append('cluster.initial_cluster_manager_nodes='
                                             + ','.join(self.get_cluster_manager_hosts()))

    def _get_zone_details(self, host):
        if host not in self.get_data_hosts():
            return ''
        zones = self._get_data_hosts_zones()
        zones_str = ','.join({zone.strip('/') for host, zone_entries in zones for zone in zone_entries})
        for host_entry, zone_entries in zones:
            if 'default' in zone_entries:
                return ''
            if host == host_entry:
                return f'''cluster.routing.allocation.awareness.attributes: zone
node.attr.zone: {zone_entries[0].strip('/')}
cluster.routing.allocation.awareness.force.zone.values: {zones_str}
                '''

    def apply_user_inputs(self, host, component_yaml: dict):
        values_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'opensearch-external-values.yaml')
        if not os.path.exists(values_path):
            return
        with open(values_path, 'r') as f:
            values = yaml.safe_load(f)
        component_name = 'master'
        if host in self.get_cluster_manager_hosts():
            component_name = 'master'
        elif host in self.get_data_hosts():
            component_name = 'data'
        elif host in self.get_cluster_client_hosts():
            component_name = 'client'

        # for now handling mem_limit, memswap_limit, privileged
        if component_name in values['services']:
            user_inputs = values['services'][component_name]
            if 'mem_limit' in user_inputs:
                component_yaml['mem_limit'] = user_inputs['mem_limit']
            if 'memswap_limit' in user_inputs:
                component_yaml['memswap_limit'] = user_inputs['memswap_limit']
            if 'privileged' in user_inputs:
                component_yaml['privileged'] = user_inputs['privileged']
            if 'cap_add' in user_inputs:
                if not isinstance(user_inputs['cap_add'], list):
                     cli_err_exit(f"cap_add must be of type list for service {component_name}")
                component_yaml['cap_add'] = user_inputs['cap_add']

            # handling environment
            if 'environment' in user_inputs:
                user_env = user_inputs['environment']
                env = component_yaml['environment'] if 'environment' in component_yaml \
                    else {}

                if type(env) is dict:
                    for key, value in user_env.items():
                        env[key] = value
                else:
                    # convert list to dict
                    env_dict = {}
                    for item in env:
                        k, v = item.split("=", 1)
                        env_dict[k] = v
                    # update/overwrite entries
                    for key, value in user_env.items():
                        env_dict[key] = value
                    env = []
                    # convert back to list
                    for key, value in env_dict.items():
                        env.append(key + '=' + str(value))

                if len(env) > 0:
                    component_yaml['environment'] = env

    def reset(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser, is_component=False):
        if not self.get_hosts():
            return
        self.down(cmd_args, config_parser)
        self._delete_data(config_parser)
        # clean up is only for the external opensearch component
        if is_component:
            file_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'opensearch-external-values.yaml')
            # Check if the file exists before removing
            if os.path.exists(file_path):
                os.remove(file_path)
            for host in self.get_hosts():
                remove_dir_contents(host,  self.get_conf_dir(), config_parser)
                remove_dir_contents(host, self.get_logs_dir(), config_parser)

            platform = COMPONENT_REGISTRY.require(platform_comp.COMPONENT_NAME)
            config_path = os.path.join('/opt/rdaf/config/network_config/config.json')
            # network config.json
            network_config = json.loads(open(config_path).read())
            if 'os_external' in network_config:
                backup_command = f'cp {config_path} {config_path}_$(date +"%Y%m%d%H%M%S").bak'
                run_command(backup_command)
                logger.info("Updating config.json by removing os_external endpoint.")
                del network_config['os_external']
                output = json.dumps(network_config, indent=4)
                with open(config_path, "w") as f:
                    f.write(output)
                    f.flush()

                platform._copy_configs_known_hosts(config_parser)

            policy_path = os.path.join('/opt/rdaf/config/network_config/policy.json')
            policy_json = json.loads(open(policy_path).read())
            if 'os_external_default' in policy_json['credentials']["es"]:
                backup_command = f'cp {policy_path} {policy_path}_$(date +"%Y%m%d%H%M%S").bak'
                run_command(backup_command)
                # Filter out mappings that start with 'os-external-'
                policy_json['pstream-mappings'] = [
                    mapping for mapping in policy_json['pstream-mappings']
                    if not mapping['pattern'].startswith('os-external-')
                ]

                # Remove the 'os_external_default' entry from credentials
                if 'os_external_default' in policy_json['credentials']['es']:
                    del policy_json['credentials']['es']['os_external_default']

                output = json.dumps(policy_json, indent=4)
                with open(policy_path, "w") as f:
                    f.write(output)
                    f.flush()

                for host in platform.get_hosts():
                    if not self.is_local_host(host):
                        do_potential_scp(host, policy_path, policy_path)

            if config_parser.has_section('os_external'):
                config_parser.remove_section('os_external')
                self.write_configs(config_parser)

    def gather_k8s_setup_inputs(self, cmd_args, config_parser):
        configs = self._init_default_configs()
        pass_desc = 'What is the SSH password for the SSH user used to communicate' \
                    ' between hosts'
        pass_no_prompt_err_msg = 'No SSH password specified. Use --ssh-password to specify one'
        # don't store in the configs
        self.ssh_password = rdaf.component.Component \
            ._parse_or_prompt_value(cmd_args.ssh_password,
                                    None,
                                    pass_no_prompt_err_msg,
                                    pass_desc,
                                    'SSH password',
                                    cmd_args.no_prompt,
                                    password=True,
                                    apply_password_validation=False)

        cluster_manager_host_desc = 'What is the host(s) for cluster manager?'
        cluster_manager_hosts = self._parse_or_prompt_hosts(cmd_args.cluster_manager_host, '',
                                                            'No cluster manager host specified.',
                                                            cluster_manager_host_desc,
                                                            'opensearch cluster manager host(s)',
                                                            cmd_args.no_prompt)
        configs[self._option_cluster_manager_host] = cluster_manager_hosts

        cluster_client_host_desc = 'What is the host(s) for cluster clients?'
        cluster_client_hosts = self._parse_or_prompt_hosts(cmd_args.client_host, '',
                                                           'No cluster client host specified.',
                                                           cluster_client_host_desc,
                                                           'opensearch cluster client host(s)',
                                                           cmd_args.no_prompt)
        configs[self._option_client_host] = cluster_client_hosts

        # now gather app service deployment mode
        desc = 'Do you want to configure cluster zoning?'
        if cmd_args.no_prompt:
            os_zoning = cmd_args.os_zoning
        else:
            os_zoning = rdafutils.query_yes_no(desc, default='no')
        data_zones = []
        if not os_zoning:
            data_host_desc = 'What is the host(s) for data nodes?'
            data_hosts = self._parse_or_prompt_hosts(cmd_args.data_host, '',
                                                     'No data host specified.', data_host_desc,
                                                     'opensearch cluster data host(s)', cmd_args.no_prompt)
            configs[self._option_data_host] = data_hosts
            data_zones = [(host, None) for host in data_hosts]
        else:
            # Check if zones configuration is provided from JSON
            if hasattr(cmd_args, 'zones_config') and cmd_args.zones_config:
                zones_config = cmd_args.zones_config
                all_data_hosts = []
                for zone_name, hosts in zones_config.items():
                    all_data_hosts.extend(hosts)
                    # Extract zone index from zone name (e.g., "zone-0" -> "0")
                    zone_suffix = zone_name.split('-')[-1] if '-' in zone_name else zone_name
                    zone_path = f'/zone-{zone_suffix}'
                    data_zones += [(host, [zone_path]) for host in hosts]
                configs[self._option_data_host] = all_data_hosts
            else:
                zones_desc = 'Please specify the number of zones to be configured'
                zones_input = self._parse_or_prompt_value('', '2',
                                                          "",
                                                          zones_desc,
                                                          'Number of Zones',
                                                          cmd_args.no_prompt,
                                                          False)
                all_data_hosts = []
                for index in range(int(zones_input)):
                    data_host_desc = f'What is the host(s) for data nodes in zone-{index}?'
                    data_hosts = self._parse_or_prompt_hosts(cmd_args.data_host, '',
                                                             'No data host specified.', data_host_desc,
                                                             f'opensearch cluster data host(s) for zone-{index}',
                                                             cmd_args.no_prompt)
                    all_data_hosts += data_hosts
                    data_zones += [(host, [f'/zone-{index}']) for host in data_hosts]
                configs[self._option_data_host] = all_data_hosts
        configs[self._option_zone] = \
            _apply_data_dir_defaults(data_zones, ['/default'])

        user_desc = 'What is the user name you want to give ' \
                    'for opensearch cluster admin user that ' \
                    'will be created and used by the RDAF platform?'
        user_no_prompt_err_msg = 'No opensearch user specified. Use ' \
                                 '--opensearch_external-user to specify one'
        user = self._parse_or_prompt_value(cmd_args.os_external_admin_user, 'rdafadmin',
                                           user_no_prompt_err_msg,
                                           user_desc,
                                           'opensearch user',
                                           cmd_args.no_prompt)
        configs[self._option_user] = str_base64_encode(user)

        pass_desc = 'What is the password you want to use for opensearch admin user?'
        pass_no_prompt_err_msg = 'No password specified for opensearch_external user. Use --opensearch_external-password' \
                                 ' to specify one'
        default_autogen_password = rdafutils.gen_password()
        passwd = self._parse_or_prompt_value(cmd_args.os_external_admin_password,
                                             default_autogen_password,
                                             pass_no_prompt_err_msg,
                                             pass_desc, 'opensearch password',
                                             cmd_args.no_prompt,
                                             password=True)
        configs[self._option_password] = str_base64_encode(passwd)

        # to maintain order
        all_hosts = []
        for entry in (
        configs[self._option_cluster_manager_host], configs[self._option_client_host], configs[self._option_data_host]):
            for item in entry:
                if item not in all_hosts:
                    all_hosts.append(item)
        configs[self._option_host] = all_hosts
        host_dirs = [(host, None) for host in configs[self._option_host]]
        configs[self._option_data_dir] = \
            _apply_data_dir_defaults(host_dirs, self._default_data_dirs)

        es = COMPONENT_REGISTRY.require(opensearch_comp.COMPONENT_NAME)
        es_hosts = es.get_hosts()
        for host in configs[self._option_host]:
            if host in es_hosts:
                cli_err_exit(f'OpenSearch is already deployed on host {host}. Please specify a different host.')

        self._mark_configured(configs, config_parser)
        self.write_configs(config_parser)

    def do_k8s_setup(self, cmd_args, config_parser):
        network_json = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'config.json')
        if not os.path.exists(network_json):
            cli_err_exit('Please install RDAF platform services before installing external opensearch')

        self.gather_k8s_setup_inputs(cmd_args, config_parser)
        logger.info('Doing setup for ' + self.component_name)
        if self.get_deployment_type(config_parser) != "aws":
            all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
                skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME,
                                 COMPONENT_NAME]
            )
            docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
            for host in self.get_hosts():
                if host not in all_known_hosts:
                    ssh_manager = COMPONENT_REGISTRY.require(
                        rdaf.component.ssh.SSHKeyManager.COMPONENT_NAME)
                    # setup ssh keys for this new host
                    ssh_manager.setup_keys_for_host(host, self.ssh_password)
                if self.get_deployment_type(config_parser) == 'k8s':
                    # doing a docker login
                    docker_registry.docker_login(host, config_parser)

        logger.info("Generating hash for external opensearch user.")
        if self.get_deployment_type(config_parser) != "aws":
            command = 'sudo mkdir -p /opt/os-ext-backup'
            for host in self.get_hosts():
                run_potential_ssh_command(host, command, config_parser)

        hash_pass = self._generate_hash_k8s(config_parser)
        # Creating secret for internal users
        logger.info("Creating external users secret.")
        template_path = os.path.join(get_templates_dir_root(), 'k8s-local', 'os-ext-users.yaml')
        dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-users.yaml')
        replacements = {
            'OPENSEARCH_USER': self.get_user(),
            'OPENSEARCH_HASH': hash_pass,
            'NAMESPACE': self.get_namespace(config_parser)
        }
        with open(template_path, 'r') as f:
            template_content = f.read()
        final_content = string.Template(template_content).safe_substitute(replacements)
        with open(dest_path, 'w') as f:
            f.write(final_content)
        run_command('kubectl apply -f ' + dest_path)

        # generate opensearch.yaml
        self.label_os_ext_nodes()
        self.generate_k8s_values_yaml(config_parser)
        self._enable_cluster_and_node_roles()
        self.generate_pv_and_pvc_yaml(config_parser)

    def _enable_cluster_and_node_roles(self):
        zones = self._get_data_hosts_zones()
        for host_entry, zone_entries in zones:
            if 'default' in zone_entries:
                return

        yaml = '''
apiVersion: rbac.authorization.k8s.io/v1
kind: ClusterRole
metadata:
  name: rda-fabric-node-read-access
rules:
- apiGroups: [""]
  resources: ["nodes", "pods"]
  verbs: ["get", "list", "watch", "patch"]
---
apiVersion: rbac.authorization.k8s.io/v1
kind: ClusterRoleBinding
metadata:
  name: rda-fabric-node-access-binding
subjects:
- kind: ServiceAccount
  name: rda-fabric
  namespace: rda-fabric
roleRef:
  kind: ClusterRole
  name: rda-fabric-node-read-access
  apiGroup: rbac.authorization.k8s.io
        '''
        run_command(f'kubectl get clusterrolebinding rda-fabric-node-access-binding || kubectl apply -f - <<EOF\n{yaml}\nEOF')


    def generate_k8s_values_yaml(self, config_parser):
        cert_manager: cert.CertManager = COMPONENT_REGISTRY.require(cert.CertManager.COMPONENT_NAME)
        yaml_configs = []
        data = self.get_data_hosts()
        master = self.get_cluster_manager_hosts()
        client = self.get_cluster_client_hosts()
        namespace = self.get_namespace(config_parser)

        def generate_opensearch_yaml(roles, nodes, name):
            template_dir = 'k8s-local'
            if self.get_deployment_type(config_parser) == "aws":
                template_dir = 'k8s-aws'
            template_path = os.path.join(get_templates_dir_root(), template_dir, 'opensearch-external-values.yaml')
            with open(template_path, 'r') as f:
                template_content = f.read()

            replacements = self._get_docker_repo()
            replacements['IS_SINGLE_NODE'] = "true" if len(self.get_hosts()) == 1 else "false"
            replacements['REPLICAS'] = len(nodes)
            replacements['ROLES'] = str(roles)
            replacements['NAME'] = name
            replacements['NODE_GROUP'] = name
            replacements['LABEL'] = f'rdaf-{name}'
            replacements['CERTS_PASSWORD'] = cert_manager.get_keystore_pass()
            if len(roles) == 1 and roles[0] == 'data':
                replacements['ZONE_DETAILS'] = self._get_k8s_zone_details()
                replacements['ZONING_INIT_CONTAINER'] = self._get_k8s_zone_init_container(namespace)
            else:
                replacements['ZONE_DETAILS'] = ''
                replacements['ZONING_INIT_CONTAINER'] = ''
            final_content = string.Template(template_content).safe_substitute(replacements)
            return final_content


        if set(master) == set(client) == set(data):
            yaml_config = generate_opensearch_yaml(['master', 'data', 'client'], master, 'os-ext-master')
            yaml_configs.append(('os-ext-all-values.yaml', yaml_config))
        elif set(master) == set(client) != set(data):
            yaml_config = generate_opensearch_yaml(['master', 'client'], master, 'os-ext-master')
            yaml_configs.append(('os-ext-master-client-values.yaml', yaml_config))
            yaml_config = generate_opensearch_yaml(['data'], data, 'os-ext-data')
            yaml_configs.append(('os-ext-data-values.yaml', yaml_config))
        else:
            yaml_config = generate_opensearch_yaml(['master'], master, 'os-ext-master')
            yaml_configs.append(('os-ext-master-values.yaml', yaml_config))
            yaml_config = generate_opensearch_yaml(['client'], client, 'os-ext-client')
            yaml_configs.append(('os-ext-client-values.yaml', yaml_config))
            yaml_config = generate_opensearch_yaml(['data'], data, 'os-ext-data')
            yaml_configs.append(('os-ext-data-values.yaml', yaml_config))

        for filename, config in yaml_configs:
            dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', filename)
            with open(dest_path, 'w') as f:
                f.write(config)

    def label_os_ext_nodes(self):
        data = self.get_data_hosts()
        master = self.get_cluster_manager_hosts()
        client = self.get_cluster_client_hosts()
        labels = {}
       
        # all nodes are same
        if set(master) == set(client) == set(data):
            for i, host in enumerate(self.get_hosts()):
                labels[host] = f"os-ext-master=master-{i} rdaf-os-ext-master=allow rdaf-os-ext-backup=allow"
        elif set(master) == set(client) != set(data):
            # master and client are the same, data is different
            for i, host in enumerate(master):
                labels[host] = f"os-ext-master=master-{i} rdaf-os-ext-master=allow rdaf-os-ext-backup=allow"
            for i, host in enumerate(data):
                labels[host] = f"os-ext-data=data-{i} rdaf-os-ext-data=allow rdaf-os-ext-backup=allow"
                for host_entry, zone_entries in  self._get_data_hosts_zones():
                    if 'default' in zone_entries:
                        continue
                    if host == host_entry:
                        labels[host] += f" os-ext-data-zone={zone_entries[0].strip('/')}"
        else:
            for i, host in enumerate(master):
                labels[host] = f"os-ext-master=master-{i} rdaf-os-ext-master=allow rdaf-os-ext-backup=allow"
            for i, host in enumerate(client):
                labels[host] = f"os-ext-client=client-{i} rdaf-os-ext-client=allow rdaf-os-ext-backup=allow"
            for i, host in enumerate(data):
                labels[host] = f"os-ext-data=data-{i} rdaf-os-ext-data=allow rdaf-os-ext-backup=allow"
                for host_entry, zone_entries in  self._get_data_hosts_zones():
                    if 'default' in zone_entries:
                        continue
                    if host == host_entry:
                        labels[host] += f" os-ext-data-zone={zone_entries[0].strip('/')}"

        nodes = self.get_k8s_nodes()
        for host, label in labels.items():
            if host not in nodes:
                rdafutils.cli_err_exit(f"Unknown data host {host}. Please use one of the Kubernetes worker nodes.")
            node_name = nodes[host]
            logger.info(f'Applying node label "{label}" for data host: {host}')
            run_command(f'kubectl label nodes {node_name} {label} --overwrite')

    def _get_k8s_zone_details(self):
        zones = self._get_data_hosts_zones()
        zones_str = ','.join({zone.strip('/') for host, zone_entries in zones for zone in zone_entries})
        for host_entry, zone_entries in zones:
            if 'default' in zone_entries:
                return ''

            return f'''cluster.routing.allocation.awareness.attributes: zone
      node.attr.zone: ${{ZONE}}
      cluster.routing.allocation.awareness.force.zone.values: {zones_str}'''

    def _get_k8s_zone_init_container(self, namespace):
        return f'''rbac: 
  create: true
  serviceAccountName: {namespace}
  automountServiceAccountToken: true
extraInitContainers:
  - name: get-node-zone
    image: {self._get_docker_repo()['DOCKER_REPO']}/rda-platform-kubectl:1.0.4
    command:
    - sh
    - -c 
    - | 
       NODE_NAME=$(kubectl get pod $HOSTNAME -o jsonpath='{{.spec.nodeName}}') && \
LABEL_VALUE=$(kubectl get node $NODE_NAME -o jsonpath='{{.metadata.labels.os-ext-data-zone}}') && \
kubectl annotate pod $HOSTNAME ZONE="$LABEL_VALUE" --overwrite && sleep 15
'''

    def _generate_hash_k8s(self,config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        opensearch_hash_deployment = os.path.join(get_templates_dir_root(), 'opensearch-hash.yaml')
        with open(opensearch_hash_deployment, 'r') as f:
            template_content = f.read()
        replacements = self._get_docker_repo()
        replacements['NAMESPACE'] = namespace

        content = string.Template(template_content).substitute(replacements)

        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            deployment_file = os.path.join(tmp, 'opensearch-hash.yaml')
            with open(deployment_file, 'w+') as f:
                f.write(content)
                f.flush()

            run_command('kubectl apply -f ' + deployment_file)
            logger.info("Waiting for opensearch hash pod to be up and running...")
            time.sleep(5)
            pod_status_command = 'kubectl wait --for=condition=Ready pod --timeout=600s -n {} ' \
                                 '-l app_component=rda-opensearch-hash'.format(namespace)
            ret, stdout, stderr = execute_command(pod_status_command)
            if ret != 0:
                cli_err_exit("Failed to get status of opensearch hash generation pod, due to: {}.".format(str(stderr)))

            logger.info("Pod is running, generating hash...")
            hash_pod = self.get_pods_names(config_parser, 'app_component=rda-opensearch-hash')[0]
            hash_command = "kubectl -n {} exec {} -- plugins/opensearch-security/tools/hash.sh " \
                           "-p {}".format(namespace, hash_pod, self.get_password())
            ret, stdout, stderr = execute_command(hash_command)
            if ret != 0:
                print(str(stdout), str(stderr))
                cli_err_exit("Failed to generate opensearch hash.")
            pass_list = str(stdout).split('\n')
            hash_pass = pass_list[-2]
            run_command('kubectl delete -f ' + deployment_file)
            return hash_pass

    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != "k8s":
            return
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            for image in ['rda-platform-opensearch', 'rda-platform-busybox']:
                logger.info(f'Pulling {image} on host {host}')
                docker_pull_command = f'docker pull {docker_repo}/{image}:{cmd_args.tag}'
                run_potential_ssh_command(host, docker_pull_command, config_parser)

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        data = self.get_data_hosts()
        master = self.get_cluster_manager_hosts()
        client = self.get_cluster_client_hosts()

        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), 'rda-opensearch')
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', 'rda-opensearch-external')
        self.copy_helm_chart(chart_template_path, deployment_path)

        config_path = os.path.join('/opt/rdaf/config/network_config/config.json')
        # network config.json
        network_config = json.loads(open(config_path).read())
        if 'os_external' not in network_config:
            backup_command = f'cp {config_path} {config_path}_$(date +"%Y%m%d%H%M%S").bak'
            run_command(backup_command)
            logger.info("Updating config.json with os_external endpoint.")
            network_config['os_external'] = {'hosts': self.get_k8s_cluster_endpoint(config_parser), 'user': self.get_user(),
                                             'password': self.get_password(), 'port':'9200',
                                             'scheme': 'https', 'ssl_verify': False}
            self.encrypt_rda_config(network_config)
            output = json.dumps(network_config, indent=4)
            with open(config_path, "w") as f:
                f.write(output)
                f.flush()
            run_command('kubectl create configmap rda-network-config -n {} '
                    '--from-file={} --save-config --dry-run=client -o yaml |  kubectl apply -f -'
                    .format(namespace, config_path))

        policy_path = os.path.join('/opt/rdaf/config/network_config/policy.json')
        policy_json = json.loads(open(policy_path).read())
        if 'os_external_default' not in policy_json['credentials']["es"]:
            backup_command = f'cp {policy_path} {policy_path}_$(date +"%Y%m%d%H%M%S").bak'
            run_command(backup_command)
            tenant_id = network_config.get('tenant_id')
            logger.info("Updating policy.json with os_external endpoint.")
            policy_json["pstream-mappings"].insert(2, {'pattern': f'os-external-admin-{tenant_id}.*', 'es_name': 'os_external_default'})
            policy_json["pstream-mappings"].insert(3, {'pattern': 'os-external-rda.*',
                                                       'es_name': 'os_external_default'})
            policy_json["pstream-mappings"].insert(4, {'pattern': 'os-external-.*',
                                                       'es_name': 'os_external_default'})

            policy_json['credentials']["es"]['os_external_default'] = {'hosts': self.get_k8s_cluster_endpoint(config_parser),
                                                                       'user': self.get_user(),
                                                                       'password': self.get_password(), 'port':'9200',
                                                                       'scheme': 'https', 'ssl_verify': False}
            output = json.dumps(policy_json, indent=4)
            with open(policy_path, "w") as f:
                f.write(output)
                f.flush()
            run_command('kubectl create configmap rda-dataplane-config -n {} --from-file=policy.json={} '
                    '--save-config --dry-run=client -o yaml |  kubectl apply -f -'.format(namespace, policy_path))

        if set(master) == set(client) == set(data):
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-all-values.yaml')
            install_command = f'helm install --create-namespace -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-master {deployment_path}'
            logger.info(f"Installing  os-ext-master.")
            run_command(install_command)
        elif set(master) == set(client) != set(data):
            # master, client
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-master-client-values.yaml')
            install_command = f'helm install --create-namespace -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-master {deployment_path}'
            logger.info(f"Installing  os-ext-master.")
            run_command(install_command)
            # data
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-data-values.yaml')
            install_command = f'helm install --create-namespace -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-data {deployment_path}'
            logger.info(f"Installing  os-ext-data.")
            run_command(install_command)
        else:
            # master
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-master-values.yaml')
            install_command = f'helm install --create-namespace -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-master {deployment_path}'
            logger.info(f"Installing  os-ext-master..")
            run_command(install_command)
            # client
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-client-values.yaml')
            install_command = f'helm install --create-namespace -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-client {deployment_path}'
            logger.info(f"Installing  os-ext-client..")
            run_command(install_command)
            # data
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-data-values.yaml')
            install_command = f'helm install --create-namespace -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-data {deployment_path}'
            logger.info(f"Installing  os-ext-data..")
            run_command(install_command)


    def generate_pv_and_pvc_yaml(self, config_parser: configparser.ConfigParser):
        data = self.get_data_hosts()
        master = self.get_cluster_manager_hosts()
        client = self.get_cluster_client_hosts()
        host_labels = {}

        if set(master) == set(client) == set(data):
            for i, host in enumerate(self.get_hosts()):
                host_labels[host] = f"os-ext-master-{i}"
        elif set(master) == set(client) != set(data):
            for i, host in enumerate(set(master + client)):
                host_labels[host] = f"os-ext-master-{i}"
            for i, host in enumerate(data):
                host_labels[host] = f"os-ext-data-{i}"
        else:
            for i, host in enumerate(master):
                host_labels[host] = f"os-ext-master-{i}"
            for i, host in enumerate(client):
                host_labels[host] = f"os-ext-client-{i}"
            for i, host in enumerate(data):
                host_labels[host] = f"os-ext-data-{i}"

        resources = []
        for host, label in host_labels.items():
            pv = {
                "apiVersion": "v1",
                "kind": "PersistentVolume",
                "metadata": {"name": f"{label[:-2]}-{label}",
                             "labels": {
                                 "volume-type": f"{label[:-2]}-{label}",
                                 f"rdaf-{label[:-2]}": "allow"
                             }},
                "spec": {
                    "capacity": {"storage": "50Gi"},
                    "volumeMode": "Filesystem",
                    "accessModes": ["ReadWriteOnce"],
                    "persistentVolumeReclaimPolicy": "Retain",
                    "storageClassName": "local-storage",
                    "local": {
                        "path": "/opensearch"
                    },
                    "nodeAffinity": {
                        "required":{
                            "nodeSelectorTerms": [{
                                "matchExpressions":[{
                                    "key": f"{label[:-2]}",
                                    "operator": "In",
                                    "values": [f"{label[7:]}"]
                                }]
                            }]
                        }
                    }
                }
            }

            pvc = {
                "apiVersion": "v1",
                "kind": "PersistentVolumeClaim",
                "metadata": {"name": f"{label[:-2]}-{label}",
                             "labels": {
                                 "volume-type": f"{label[:-2]}-{label}",
                                 f"rdaf-{label[:-2]}": "allow"
                             }},
                "spec": {
                    "storageClassName": "local-storage",
                    "accessModes": ["ReadWriteOnce"],
                    "resources": {"requests": {"storage": "50Gi"}},
                    "selector": {
                        "matchLabels":{
                            "volume-type": f"{label[:-2]}-{label}"
                        }
                    }
                }
            }

            resources.extend([pv, pvc])

        backup_pv = {
            "apiVersion": "v1",
            "kind": "PersistentVolume",
            "metadata": {"name": "os-ext-backup-pv",
                         "labels": {
                             "volume-type": "os-ext-backup",
                             "rdaf-os-ext-backup": "allow"
                         }},
            "spec": {
                "capacity": {"storage": "50Gi"},
                "volumeMode": "Filesystem",
                "accessModes": ["ReadWriteMany"],
                "persistentVolumeReclaimPolicy": "Retain",
                "storageClassName": "manual",
                "local": {
                    "path": "/opt/os-ext-backup"
                },
                "nodeAffinity": {
                    "required": {
                        "nodeSelectorTerms": [{
                            "matchExpressions": [{
                                "key": "rdaf-os-ext-backup",
                                "operator": "In",
                                "values": ["allow"]
                            }]
                        }]
                    }
                }
            }
        }

        backup_pvc = {
            "apiVersion": "v1",
            "kind": "PersistentVolumeClaim",
            "metadata": {"name": "os-ext-backup-pvc",
                         "labels": {
                             "volume-type": "os-ext-backup",
                             "rdaf-os-ext-backup": "allow"
                         }},
            "spec": {
                "storageClassName": "manual",
                "accessModes": ["ReadWriteMany"],
                "resources": {"requests": {"storage": "50Gi"}},
                "selector": {
                    "matchLabels": {
                        "volume-type": "os-ext-backup",
                        "rdaf-os-ext-backup": "allow"
                    }
                }
            }
        }
        resources.extend([backup_pv, backup_pvc])
        yaml_output = yaml.dump_all(resources, default_flow_style=False)
        pv_yaml = "/opt/rdaf/deployment-scripts/os-ext-pv.yaml"
        with open(pv_yaml, 'w') as f:
            f.write(yaml_output)

        namespace = self.get_namespace(config_parser)
        run_command(f'kubectl apply -f {pv_yaml} -n {namespace}')

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        data = self.get_data_hosts()
        master = self.get_cluster_manager_hosts()
        client = self.get_cluster_client_hosts()

        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), 'rda-opensearch')
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', 'rda-opensearch-external')
        self.copy_helm_chart(chart_template_path, deployment_path)

        if set(master) == set(client) == set(data):
           values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-all-values.yaml')
           upgrade_command = f'helm upgrade -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-master {deployment_path}'
           logger.info(f"Upgrading os-ext-master.")
           run_command(upgrade_command)

        elif set(master) == set(client) != set(data):
            # master, client
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-master-client-values.yaml')
            upgrade_command = f'helm upgrade -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-master {deployment_path}'
            logger.info(f"Upgrading os-ext-master.")
            run_command(upgrade_command)
    
            # data
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-data-values.yaml')
            upgrade_command = f'helm upgrade -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-data {deployment_path}'
            logger.info(f"Upgrading os-ext-data.")
            run_command(upgrade_command)
    
        else:
            # master
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-master-values.yaml')
            upgrade_command = f'helm upgrade -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-master {deployment_path}'
            logger.info(f"Upgrading os-ext-master.")
            run_command(upgrade_command)
    
            # client
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-client-values.yaml')
            upgrade_command = f'helm upgrade -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-client {deployment_path}'
            logger.info(f"Upgrading os-ext-client.")
            run_command(upgrade_command)
    
            # data
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'os-ext-data-values.yaml')
            upgrade_command = f'helm upgrade -n {namespace} -f {values_yaml} {self.get_k8s_install_args(cmd_args)} os-ext-data {deployment_path}'
            logger.info(f"Upgrading os-ext-data.")
            run_command(upgrade_command)

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        namespace = self.get_namespace(config_parser)
        statuses = []
        status_command = 'kubectl get pods -n {} -l app_component=opensearch-external -o json'\
            .format(namespace)
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of external opensearch, due to: {}.".format(str(stderr)))
        result = json.loads(str(stdout))
        items = result['items']
        hosts = self.get_hosts()
        for host in hosts:
            if not items:
                    statuses.append({
                        'component_name': self.get_k8s_component_name(),
                        'host': host,
                        'containers': [{
                            'Id': 'N/A',
                            'Image': 'N/A',
                            'State': 'N/A',
                            'Status': 'Not Provisioned'
                        }]
                    })
        else:
            for item in items:
                pod = dict()
                statuses.append(pod)
                pod['component_name'] = self.get_k8s_component_name()
                pod['host'] = item['status'].get('hostIP', 'Unknown')
                pod['containers'] = []
                if 'containerStatuses' in item['status']:
                    for entry in item['status']['containerStatuses']:
                        if 'containerID' in entry.keys():
                            container = dict()
                            container['Id'] = entry['containerID']
                            container['Image'] = entry['image']
                            for key in entry['state'].keys():
                                container['State'] = key
                                if key == 'running':
                                    container['Status'] = self.get_container_age(entry['state'][key]['startedAt'])
                                else:
                                    container['Status'] = key
                                break
                            pod['containers'].append(container)
            return statuses

    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = f'kubectl get statefulset -n {namespace} -l app_component=opensearch-external -o json'
        ret, stdout, stderr = execute_command(command)
        components = json.loads(stdout)

        values_dir = '/opt/rdaf/deployment-scripts'
        values_files_map = {
            'os-ext-client': [os.path.join(values_dir, 'os-ext-client-values.yaml')],
            'os-ext-data': [os.path.join(values_dir, 'os-ext-data-values.yaml')],
            'os-ext-master': [
                os.path.join(values_dir, 'os-ext-master-values.yaml'),
                os.path.join(values_dir, 'os-ext-master-client-values.yaml'),
                os.path.join(values_dir, 'os-ext-all-values.yaml')
            ]
        }

        for component in components["items"]:
            sts_name = component["metadata"]["name"]
            potential_files = values_files_map.get(sts_name, [])
            # Check each file in the list and stop at the first valid one
            for values_file in potential_files:
                if os.path.exists(values_file):
                    with open(values_file) as f:
                        data = yaml.safe_load(f)
                    break
            else:
                logger.error(f"Values file not found for {sts_name}")
                continue

            replicas = data.get('replicas')
            if replicas is None:
                logger.error(f"Replicas field not found in {values_file} for {sts_name}")
                continue

            logger.info(f"Scaling statefulset: {sts_name} to {replicas} replicas")
            command = f'kubectl scale statefulset.apps/{sts_name} -n {namespace} --replicas={replicas}'
            run_command(command)

    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        cmd = f'kubectl get pods -n {namespace} -l app_component=opensearch-external -o json'
        ret, stdout, stderr = execute_command(cmd)
        component = json.loads(stdout)
        for items in component["items"]:
            metadata = items["metadata"]["ownerReferences"]
            for pod in metadata:
                name = pod['name']
                command = f'kubectl scale statefulset.apps/{name} -n {namespace} --replicas=0'
                run_command(command)

    @staticmethod
    def uninstall_os_ext_releases(namespace):
        list_command = f'helm list -n {namespace} -q'
        ret, stdout, stderr = execute_command(list_command)
        if ret != 0:
            cli_err_exit(f"Failed to retrieve Helm releases, command returned error: {stderr}")

        # Check if any releases contain 'os-ext' in their name
        os_ext_releases = [release for release in stdout.splitlines() if 'os-ext' in release]
        if not os_ext_releases:
            logger.info(f"No os-ext releases found in the {namespace} namespace.")
            return
        logger.info(f"Found os-ext releases: {os_ext_releases}")
        for release in os_ext_releases:
            # Uninstall each os-ext release
            uninstall_command = f'helm uninstall -n {namespace} {release}'
            ret, stdout, stderr = execute_command(uninstall_command)
            if ret != 0:
                logger.error(f"Failed to uninstall release {release}: {stderr}")
            else:
                logger.info(f"Successfully uninstalled release: {release}")


    @staticmethod
    def delete_os_ext_pv_pvc(namespace):
        list_pvc_command = f'kubectl get pvc -n {namespace} -o custom-columns=:metadata.name'
        ret, stdout, stderr = execute_command(list_pvc_command)
        if ret != 0:
            cli_err_exit(f"Failed to retrieve PVCs, command returned error: {stderr}")

        # Check if any PVCs contain 'os-ext' in their name
        os_ext_pvcs = [pvc for pvc in stdout.splitlines() if 'os-ext' in pvc]
        if not os_ext_pvcs:
            logger.info(f"No os-ext PVCs found in the {namespace} namespace.")
            return
        logger.info(f"Found os-ext PVCs: {os_ext_pvcs}")
        for pvc in os_ext_pvcs:
            # Delete each os-ext PVC
            delete_pvc_command = f'kubectl delete pvc -n {namespace} {pvc}'
            ret, stdout, stderr = execute_command(delete_pvc_command)
            if ret != 0:
                logger.error(f"Failed to delete PVC {pvc}: {stderr}")
            else:
                logger.info(f"Successfully deleted PVC: {pvc}")


        list_pv_command = 'kubectl get pv -o custom-columns=:metadata.name'
        ret, stdout, stderr = execute_command(list_pv_command)
        if ret != 0:
           cli_err_exit(f"Failed to retrieve PVs, command returned error: {stderr}")
        # Check if any PVs contain 'os-ext' in their name
        os_ext_pvs = [pv for pv in stdout.splitlines() if 'os-ext' in pv]
        if not os_ext_pvs:
            logger.info("No os-ext PVs found in the cluster.")
            return
        logger.info(f"Found os-ext PVs: {os_ext_pvs}")
        for pv in os_ext_pvs:
            # Delete each os-ext PV
            delete_pv_command = f'kubectl delete pv {pv}'
            ret, stdout, stderr = execute_command(delete_pv_command)
            if ret != 0:
                logger.error(f"Failed to delete PV {pv}: {stderr}")
            else:
                logger.info(f"Successfully deleted PV: {pv}")

    @staticmethod
    def remove_os_ext_files(directory):
        file_pattern = os.path.join(directory, '*os-ext*')
        # Find all files matching the pattern
        files_to_remove = glob.glob(file_pattern)
        if files_to_remove:
            for file_path in files_to_remove:
                try:
                    os.remove(file_path)
                    logger.info(f"Removed file: {file_path}")
                except Exception as e:
                    logger.error(f"Error removing file {file_path}: {e}")
        else:
            logger.info(f"No files containing 'os-ext' found in {directory}")

    def k8s_reset(self,  cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser, is_component=True):
        namespace = self.get_namespace(config_parser)
        self.uninstall_os_ext_releases(namespace)
        delete_pods_command = (
             f'kubectl get pods -n {namespace} | grep "os-ext" | '
             f'awk \'{{print $1}}\' | xargs -I {{}} kubectl delete pod {{}} -n {namespace} --force --grace-period=0')
        logger.info(f"Deleting os-ext pods in namespace: {namespace}")
        ret, stdout, stderr = execute_command(delete_pods_command)
        if ret != 0:
            logger.error(f"Failed to delete os-ext pods: {stderr}")
        else:
            logger.info(f"Successfully deleted os-ext pods: {stdout}")
        self._delete_data(config_parser)
        self.delete_os_ext_pv_pvc(namespace)
        delete_labels = 'kubectl label nodes --all os-ext-master- rdaf-os-ext-master- ' \
                        'os-ext-client- rdaf-os-ext-client- rdaf-os-ext-backup- os-ext-data- ' \
                        'rdaf-os-ext-data- rdaf-os-ext-backup- os-ext-data-zone- '
        execute_command(delete_labels)
        self.remove_os_ext_files('/opt/rdaf/deployment-scripts')
        if is_component:
            config_path = os.path.join('/opt/rdaf/config/network_config/config.json')
            # network config.json
            network_config = json.loads(open(config_path).read())
            if 'os_external' in network_config:
                backup_command = f'cp {config_path} {config_path}_$(date +"%Y%m%d%H%M%S").bak'
                run_command(backup_command)
                logger.info("Updating config.json by removing os_external endpoint.")
                del network_config['os_external']
                output = json.dumps(network_config, indent=4)
                with open(config_path, "w") as f:
                    f.write(output)
                    f.flush()
                run_command('kubectl create configmap rda-network-config -n {} '
                            '--from-file={} --save-config --dry-run=client -o yaml |  kubectl apply -f -'
                            .format(namespace, config_path))

            policy_path = os.path.join('/opt/rdaf/config/network_config/policy.json')
            policy_json = json.loads(open(policy_path).read())
            if 'os_external_default' in policy_json['credentials']["es"]:
                backup_command = f'cp {policy_path} {policy_path}_$(date +"%Y%m%d%H%M%S").bak'
                run_command(backup_command)
                # Filter out mappings that start with 'os-external-'
                policy_json['pstream-mappings'] = [
                    mapping for mapping in policy_json['pstream-mappings']
                    if not mapping['pattern'].startswith('os-external-')
                ]
                # Remove the 'os_external_default' entry from credentials
                if 'os_external_default' in policy_json['credentials']['es']:
                    del policy_json['credentials']['es']['os_external_default']
                output = json.dumps(policy_json, indent=4)
                with open(policy_path, "w") as f:
                    f.write(output)
                    f.flush()
                run_command('kubectl create configmap rda-dataplane-config -n {} --from-file=policy.json={} '
                            '--save-config --dry-run=client -o yaml |  kubectl apply -f -'.format(namespace,
                                                                                                  policy_path))
            if config_parser.has_section('os_external'):
                config_parser.remove_section('os_external')
                self.write_configs(config_parser)

        secret_cmd = f'kubectl delete secret os-ext-internal-users-secret -n {namespace} '
        run_command(secret_cmd)

    def backup_conf(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_dir_root: os.path):
        pass

    def backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        pass

    def restore_conf(self, config_parser: configparser.ConfigParser, backup_content_root_dir: os.path):
        pass

    def restore_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                     backup_content_root_dir: os.path, backup_cfg_parser: configparser.ConfigParser):
        pass

    def docker_compose_check(self, config_parser: configparser.ConfigParser):
        docker_compose_file = os.path.join(rdaf.get_scripts_dir_root(), 'docker-compose')
        docker_compose_file_path = os.path.join('/usr', 'local', 'bin', 'docker-compose')
        remote_tmp_path = '/tmp/docker-compose'
        chmod_cmd = 'sudo chmod +x /usr/local/bin/docker-compose'
        copy_command = 'sudo cp ' + docker_compose_file + ' ' + docker_compose_file_path
        ssh_copy_command = 'sudo cp ' + remote_tmp_path + ' ' + docker_compose_file_path

        for host in self.get_hosts():
            if Component.is_local_host(host):
                run_command(copy_command)
                run_command(chmod_cmd)
            else:
                do_potential_scp(host, docker_compose_file, remote_tmp_path, sudo=True)
                run_potential_ssh_command(host, ssh_copy_command, config_parser)
                run_potential_ssh_command(host, chmod_cmd, config_parser)

    def geodr_start_replication(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        primary_host = self.get_hosts()[0]
        secondary_host  = peer_configs.get('os_external', 'host').split(',')[0]
        urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
        headers = {'Content-type': 'application/json'}
        health_url = f'https://{primary_host}:9200/_cluster/health'
        response = requests.get(health_url, verify=False, headers=headers,
                                auth=(self.get_user(), self.get_password()), timeout=30)
        response.raise_for_status()
        health_data = response.json()
        status = health_data.get('status', 'red')
        if status == 'red':
            cli_err_exit("Cannot start replication")
        # Load excluded indexes
        excluded_indexes = set()
        try:
            with open('/opt/rdaf/exclude_index.txt', 'r') as exclude_file:
                excluded_indexes = {line.strip() for line in exclude_file if line.strip()}
        except FileNotFoundError:
            excluded_indexes = []
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)
            logger.info("Adding peers on opensearch external primary")
            seeds_url = f'https://{primary_host}:9200/_cluster/settings'
            seeds_data = {"persistent":{"cluster":{"remote":{"opensearch-cluster":{"seeds":[f"{secondary_host}:9300"]}}}}}
            response = requests.put(seeds_url, verify=False, data=json.dumps(seeds_data), headers=headers,
                                    auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

            logger.info("Adding peers on opensearch external secondary")
            seeds_url = f'https://{secondary_host}:9200/_cluster/settings'
            seeds_data = {"persistent": {
                "cluster": {"remote": {"opensearch-cluster": {"seeds": [f"{primary_host}:9300"]}}}}}
            response = requests.put(seeds_url, verify=False, data=json.dumps(seeds_data), headers=headers,
                                    auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

            logger.info(f"Starting replication on secondary {secondary_host}.")
            if hasattr(cmd_args, 'index') and cmd_args.index is not None:
                for index_name in cmd_args.index:
                    if index_name in excluded_indexes:
                       logger.info(f"Skipping replication for excluded index: {index_name}")
                       continue
                    replication_url = f'https://{secondary_host}:9200/_plugins/_replication/{index_name}/_start'
                    seeds_data = {"leader_alias":"opensearch-cluster","leader_index": f"{index_name}",
                              "use_roles":{"leader_cluster_role":"all_access","follower_cluster_role":"all_access"}}
                    response = requests.put(replication_url, verify=False, data=json.dumps(seeds_data), headers=headers,
                                        auth=(self.get_user(), self.get_password()))
                    response.raise_for_status()
                return
            logger.info("Applying autofollow replication rule excluding specified indexes.")
            tenant_id = config_parser.get('common', 'tenant_id')
            pattern = f"*{tenant_id}*"
            if excluded_indexes:
                pattern = f"^(?!{'|'.join(excluded_indexes)}).*"
                logger.info(f"Skipping replication for: {', '.join(excluded_indexes)}")
            replication_url = f'https://{secondary_host}:9200/_plugins/_replication/_autofollow'
            seeds_data = {"leader_alias":"opensearch-cluster","name":"my-replication-rule","pattern": pattern,
                          "use_roles":{"leader_cluster_role":"all_access","follower_cluster_role":"all_access"}}
            response = requests.post(replication_url, verify=False, data=json.dumps(seeds_data), headers=headers,
                                    auth=(self.get_user(), self.get_password()))
            response.raise_for_status()

    def geodr_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                 peer_configs: configparser.ConfigParser):
        secondary_host = peer_configs.get('os_external', 'host').split(',')[0]
        logger.info(f"Checking replication status for opensearch external")
        if hasattr(cmd_args, 'index') and cmd_args.index is not None:
            for index_name in cmd_args.index:
                run_command('curl -s -k -X GET -u "{os_user}:{os_password}" "https://{os_host}:9200/_plugins/_replication/{index}/_status"'
                        .format(os_user=self.get_user(),os_password=self.get_escaped_password(),os_host=secondary_host,index=index_name))
            return
        run_command('curl -s -k -X GET -u "{os_user}:{os_password}" "https://{os_host}:9200/_plugins/_replication/autofollow_stats"'
                    .format(os_user=self.get_user(),os_password=self.get_escaped_password(),os_host=secondary_host))
        run_command('curl -s -k -X GET -u "{os_user}:{os_password}" "https://{os_host}:9200/_plugins/_replication/follower_stats"'
                    .format(os_user=self.get_user(),os_password=self.get_escaped_password(),os_host=secondary_host))

    def switch_primary(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        if not self.get_hosts():
            return
        logger.info("Switching opensearch external to be a primary instance.")
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)

            indices_response = requests.get(f'https://{self.get_hosts()[0]}:9200/_cat/indices',
                auth=(self.get_user(), self.get_password()), verify=False)
            indices_response.raise_for_status()
            index_names = [line.split()[2] for line in indices_response.text.strip().split('\n') if len(line.split()) > 2 ]
            logger.info(f"There are a total of {len(index_names)} indices")

            # handle exclude list
            try:
                with open('/opt/rdaf/exclude_index.txt', 'r') as exclude_file:
                    exclude_list = [line.strip() for line in exclude_file if line.strip()]
            except FileNotFoundError:
                exclude_list = []

            for index in index_names:
                if index in exclude_list:
                    continue
                try:
                    # Stop replication for the index
                    stop_response = requests.post(f'https://{self.get_hosts()[0]}:9200/_plugins/_replication/{index}/_stop',
                                                 auth=(self.get_user(), self.get_password()), verify=False, json={})
                    stop_response.raise_for_status()
                    logger.info(f"Successfully processed index: {index}")
                except requests.RequestException as index_error:
                    logger.error(f"Error processing index {index}: {str(index_error)}")

        logger.info("Completed switching opensearch external as primary")
        run_command('curl -s -k -X GET -u "{os_user}:{os_password}" "https://{os_host}:9200/_plugins/_replication/follower_stats"'
            .format(os_user=self.get_user(), os_password=self.get_escaped_password(), os_host=self.get_hosts()[0]))
    
    def geodr_stop_replication(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        preferred_primary = config_parser.getboolean('rdaf-cli', 'primary', fallback=True)
        if preferred_primary:
            cli_err_exit("Cannot stop/delete indices on primary instance.")
        logger.info("Stopping external opensearch replication")
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)

            headers = {'Content-Type': 'application/json'}

            indices_response = requests.get(f'https://{self.get_hosts()[0]}:9200/_cat/indices',
                                            auth=(self.get_user(), self.get_password()), verify=False)
            indices_response.raise_for_status()
            index_names = [line.split()[2] for line in indices_response.text.strip().split('\n') if
                           len(line.split()) > 2]
            logger.info(f"There are a total of {len(index_names)} indices")
            if cmd_args.delete_rule:
                # delete ISM policies
                self.delete_ism_policies()
                # delete templates
                self.delete_templates()
                # delete aliases
                self.delete_aliases()
                try:
                    delete_url = f'https://{self.get_hosts()[0]}:9200/_plugins/_replication/_autofollow'
                    payload = {
                        "leader_alias": "opensearch-cluster",
                        "name": "my-replication-rule"}
                    response = requests.delete(
                        delete_url, headers={'Content-Type': 'application/json'},
                        auth=(self.get_user(), self.get_password()), verify=False,
                        json=payload)
                    response.raise_for_status()
                    logger.info(f"Successfully deleted auto-follow rule: my-replication-rule")
                except requests.exceptions.HTTPError as e:
                    logger.error(f"Failed to delete auto-follow rule: my-replication-rule, Status: {e.response.status_code}, Response: {e.response.text}")
            for index in index_names:
                status_response = requests.get(f'https://{self.get_hosts()[0]}:9200/_plugins/_replication/{index}/_status',
                                              auth=(self.get_user(), self.get_password()), verify=False)
                status_response.raise_for_status()
                status_data = status_response.json()

                if status_data.get('status') != 'REPLICATION NOT IN PROGRESS':
                    replication_stop_url = f'https://{self.get_hosts()[0]}:9200/_plugins/_replication/{index}/_stop'
                    headers = {'Content-Type': 'application/json'}
                    try:
                        stop_response = requests.post(replication_stop_url, headers=headers,
                                                       auth=(self.get_user(), self.get_password()), verify=False, json={})
                        stop_response.raise_for_status()
                        logger.info(f"Successfully stopped replication for index '{index}'.")
                    except requests.exceptions.HTTPError as e:
                        logger.error(f"Failed to stop replication for index '{index}': {e.response.text}")
                else:
                    logger.info(f"Replication is already stopped for index '{index}'.")

                if cmd_args.delete_index:
                    try:
                        delete_response = requests.delete(f'https://{self.get_hosts()[0]}:9200/{index}',
                                                          auth=(self.get_user(), self.get_password()), verify=False)
                        delete_response.raise_for_status()
                        logger.info(f"Successfully deleted index: {index}")
                    except requests.exceptions.HTTPError as e:
                        logger.error(f"Failed to delete index: {index}, Status Code: {e.response.status_code}, Response: {e.response.text}")

    def delete_ism_policies(self):
        logger.info("Deleting ISM policies")
        try:
            with requestsutil.new_session():
                requests.packages.urllib3.disable_warnings(
                    requests.packages.urllib3.exceptions.InsecureRequestWarning)
                response = requests.get(f'https://{self.get_hosts()[0]}:9200/_plugins/_ism/policies',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                response.raise_for_status()
                policies = response.json().get('policies', [])
                for policy in policies:
                    policy_id = policy['_id']
                    logger.debug(f"Deleting ISM policy: {policy_id}")
                    response = requests.delete(f'https://{self.get_hosts()[0]}:9200/_plugins/_ism/policies/{policy_id}',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                    response.raise_for_status()
                    logger.debug(f"Successfully deleted ISM policy: {policy_id}")
        except Exception as e:
            logger.error(f"Failed to delete ISM policies: {e}")
    
    def delete_templates(self):
        logger.info("Deleting Index templates")
        try:
            with requestsutil.new_session():
                requests.packages.urllib3.disable_warnings(
                    requests.packages.urllib3.exceptions.InsecureRequestWarning)
                response = requests.get(f'https://{self.get_hosts()[0]}:9200/_index_template',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                response.raise_for_status()
                templates = response.json().get('index_templates', [])
                for template in templates:
                    template_name = template['name']
                    logger.debug(f"Deleting Index template: {template_name}")
                    response = requests.delete(f'https://{self.get_hosts()[0]}:9200/_index_template/{template_name}',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                    response.raise_for_status()
                    logger.debug(f"Successfully deleted Index template: {template_name}")
        except Exception as e:
            logger.error(f"Failed to delete Index templates: {e}")
    
    def delete_aliases(self):
        logger.info("Deleting aliases")
        try:
            with requestsutil.new_session():
                requests.packages.urllib3.disable_warnings(
                    requests.packages.urllib3.exceptions.InsecureRequestWarning)
                response = requests.get(f'https://{self.get_hosts()[0]}:9200/_cat/aliases?format=json',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                response.raise_for_status()
                aliases = response.json()
                for alias in aliases:
                    index_name = alias['index']
                    alias_name = alias['alias']
                    logger.debug(f"Deleting alias: {alias_name} on index: {index_name}")
                    response = requests.delete(f'https://{self.get_hosts()[0]}:9200/{index_name}/_alias/{alias_name}',
                                        auth=(self.get_user(), self.get_password()), verify=False)
                    response.raise_for_status()
                    logger.debug(f"Successfully deleted alias: {alias_name} on index: {index_name}")
        except Exception as e:
            logger.error(f"Failed to delete aliases: {e}")

    def geodr_apply_metadata(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        logger.info("Applying metadata for opensearch external")
        metadata_file = '/tmp/metadata_external.json'
        if not self._download_metadata_from_minio(config_parser, metadata_file):
            logger.info(f"Metadata file does not exist on MinIO persistent-streams/ism/ism_config_external.json")
            return
        with open(metadata_file, 'r') as f:
            metadata = json.load(f)

        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)
            for template_item in metadata.get('templates', {}).get('index_templates', []):
                template_name = template_item['name']
                template_data = template_item['index_template']
                logger.debug(f"Applying template: {template_name}")
                try:
                    response = requests.put(f'https://{self.get_hosts()[0]}:9200/_index_template/{template_name}',
                                            auth=(self.get_user(), self.get_password()), verify=False, json=template_data)
                    response.raise_for_status()
                    logger.info(f"Successfully applied template: {template_name}")
                except requests.exceptions.HTTPError as e:
                    if e.response.status_code == 409:
                        logger.info(f"Template {template_name} already exists, skipping")
                    else:
                        logger.error(f"Failed to apply template {template_name}: {e}")
                        logger.error(f"Response content: {e.response.text}")
                        raise

            for policy_item in metadata.get('policies', {}).get('policies', []):
                policy_name = policy_item['policy']['policy_id']
                policy_data = {'policy': policy_item['policy']}
                logger.debug(f"Applying policy: {policy_name}")
                try:
                    response = requests.put(f'https://{self.get_hosts()[0]}:9200/_plugins/_ism/policies/{policy_name}',
                                            auth=(self.get_user(), self.get_password()), verify=False, json=policy_data)
                    response.raise_for_status()
                    logger.info(f"Successfully applied policy: {policy_name}")
                except requests.exceptions.HTTPError as e:
                    if e.response.status_code == 409:
                        logger.info(f"Policy {policy_name} already exists, skipping")
                    else:
                        logger.error(f"Failed to apply policy {policy_name}: {e}")
                        logger.error(f"Response content: {e.response.text}")
                        raise

            for index, alias_data in metadata.get('aliases', {}).items():
                logger.info(f"Applying alias for index: {index}")
                
                try:
                    index_check = requests.head(f'https://{self.get_hosts()[0]}:9200/{index}',
                                              auth=(self.get_user(), self.get_password()), verify=False)
                    if index_check.status_code == 404:
                        logger.warning(f"Index {index} does not exist, skipping alias creation")
                        continue
                except Exception as e:
                    logger.warning(f"Could not check if index {index} exists: {e}")
                
                for alias, properties in alias_data.items():
                    alias_url = f'https://{self.get_hosts()[0]}:9200/{index}/_alias/{alias}'
                    alias_payload = {}
                    
                    if properties.get("is_write_index") is not None:
                        alias_payload["is_write_index"] = properties.get("is_write_index")
                    if "filter" in properties:
                        alias_payload["filter"] = properties["filter"]
                    if "routing" in properties:
                        alias_payload["routing"] = properties["routing"]
                    
                    logger.debug(f"Alias payload for {alias}: {alias_payload}")
                    try:
                        response = requests.put(alias_url, auth=(self.get_user(), self.get_password()), 
                                              verify=False, json=alias_payload, headers={'Content-Type': 'application/json'})
                        response.raise_for_status()
                        logger.info(f"Successfully applied alias: {alias} for index: {index}")
                    except requests.exceptions.HTTPError as e:
                        if e.response.status_code == 409:
                            logger.info(f"Alias {alias} for index {index} already exists, skipping")
                        else:
                            logger.error(f"Failed to apply alias {alias} for index {index}: {e}")
                            logger.error(f"Response content: {e.response.text}")
                            logger.error(f"Payload was: {alias_payload}")
                            raise
            
            # Attach ISM policies to indices
            self._attach_ism_policies_to_indices(metadata)

        logger.info("Completed applying metadata for opensearch external")
    
    def _download_metadata_from_minio(self, config_parser: configparser.ConfigParser, output_file: str) -> bool:
        try:
            tenant_id = config_parser.get('common', 'tenant_id')
            from rdaf.component import minio as minio_comp
            minio_component = COMPONENT_REGISTRY.require(minio_comp.COMPONENT_NAME)
            val = f'http://{minio_component.get_user()}:{minio_component.get_password()}@{minio_component.get_hosts()[0]}:9000'
            env = dict(os.environ, MC_HOST_myminio=val)
            download_cmd = f'mc cp --insecure myminio/tenants.{tenant_id}/persistent-streams/ism/ism_config_external.json {output_file}'
            result = subprocess.run(download_cmd, shell=True, env=env, 
                                  capture_output=True, text=True)
            
            if result.returncode == 0:
                logger.info("Successfully downloaded metadata from MinIO")
                return True
            else:
                logger.warning(f"Failed to download metadata from MinIO: {result.stderr}")
                return False         
        except Exception as e:
            logger.warning(f"Error downloading metadata from MinIO: {e}")
            return False

    def _attach_ism_policies_to_indices(self, metadata):
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)
            try:
                logger.info("Attaching ISM policies to indices")
                host = self.get_hosts()[0]
                base_url = f'https://{host}:9200'
                auth = (self.get_user(), self.get_password())
                headers = {'Content-Type': 'application/json'}
                # Get all indices
                indices_response = requests.get(f"{base_url}/_cat/indices?h=index", 
                                            auth=auth, headers=headers, verify=False)
                indices_response.raise_for_status()
                indices = indices_response.text.strip().split('\n') if indices_response.text.strip() else []
                
                policy_ids = [policy_item['policy']['policy_id'] 
                             for policy_item in metadata.get('policies', {}).get('policies', [])]
                
                for policy_id in policy_ids:
                    matching_indices = [idx for idx in indices if policy_id in idx]
                    
                    for index in matching_indices:
                        # Check if index is a CCR (Cross Cluster Replication) index
                        try:
                            index_response = requests.get(f"{base_url}/{index}", 
                                                        auth=auth, headers=headers, verify=False)
                            index_response.raise_for_status()
                            index_data = index_response.json()
                            
                            settings = index_data[index]['settings']['index']
                            if settings.get('replication.type', '').lower() == 'ccr':
                                logger.info(f"Skipping CCR index: {index}")
                                continue
                        except Exception as e:
                            logger.warning(f"Could not check settings for index {index}: {e}")
                            continue
                        
                        # Try to attach policy using both possible API paths
                        attached = False
                        for api_path in ['_plugins/_ism/add', '_opendistro/_ism/add']:
                            attach_url = f"{base_url}/{api_path}/{index}"
                            payload = {'policy_id': policy_id}
                            
                            try:
                                attach_response = requests.post(attach_url, auth=auth, 
                                                            headers=headers, json=payload, verify=False)
                                if attach_response.ok:
                                    logger.info(f"Successfully attached policy {policy_id} to index {index}")
                                    attached = True
                                    break
                                else:
                                    logger.warning(f"Failed to attach {policy_id} to {index} via {api_path}: {attach_response.text}")
                            except Exception as e:
                                logger.warning(f"Error attaching {policy_id} to {index} via {api_path}: {e}")
                        
                        if not attached:
                            logger.error(f"Could not attach policy {policy_id} to index {index}")
                            
            except Exception as e:
                logger.error(f"Error in ISM policy attachment: {e}")