import argparse
import configparser
import json
import logging
import os
import string
from typing import Callable, Any, List

import yaml
import re
import rdaf
import rdaf.component
import rdaf.component.haproxy as haproxy
import rdaf.component.pseudo_platform as pseudo_platform
from rdaf import get_templates_dir_root
from rdaf.component import Component, OtherCategoryOrder, \
    _comma_delimited_to_list, _list_to_comma_delimited, run_potential_ssh_command, do_potential_scp, create_file
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.rdafutils import str_base64_encode, str_base64_decode, cli_err_exit, query_yes_no, prompt_and_validate

COMPONENT_NAME = 'cfx_self_monitor'
logger = logging.getLogger(__name__)

class SelfMonitoring(Component):
    _option_deployment_name = 'name'
    _option_host = 'host'
    _option_portal_uname = 'portal_uname'
    _option_portal_password = 'portal_password'
    _option_slack_token = 'slack_token'
    _option_slack_channel_name = 'slack_channel_name'
    _option_slack_channel_id = 'slack_channel_id'
    _option_slack_members_name = 'slack_member_name'
    _option_slack_members_id = 'slack_member_id'
    _option_webex_token = 'webex_token'
    _option_webex_room_id = 'webex_room_id'
    _option_smtp_server = 'smtp_server'
    _option_smtp_port = 'smtp_port'
    _option_sender_email = 'sender_email'
    _option_sender_password = 'sender_password'
    _option_recipient_emails =  'recipient_emails'

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'self_monitoring', category='other',
                         category_order=OtherCategoryOrder.SELF_MONITORING.value)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name in [self._option_host, self._option_slack_members_name, self._option_slack_members_id, self._option_recipient_emails]:
            return _comma_delimited_to_list
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name in [self._option_host, self._option_slack_members_name, self._option_slack_members_id, self._option_recipient_emails]:
            return _list_to_comma_delimited
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_deployment_name] = None
        default_configs[self._option_host] = None
        default_configs[self._option_portal_uname] = None
        default_configs[self._option_portal_password] =None
        default_configs[self._option_slack_token] = None
        default_configs[self._option_slack_channel_name] = None
        default_configs[self._option_slack_channel_id] = None
        default_configs[self._option_slack_members_name] = None
        default_configs[self._option_slack_members_id] = None
        default_configs[self._option_webex_token] = None
        default_configs[self._option_webex_room_id] = None
        default_configs[self._option_smtp_server] = None
        default_configs[self._option_smtp_port] = None
        default_configs[self._option_sender_email] = None
        default_configs[self._option_sender_password] = None
        default_configs[self._option_recipient_emails] = None
        return default_configs
    
    def validate_smtp_server(self, smtp_server) -> bool:
        # Regular expression for validating an SMTP server format
        smtp_regex = re.compile(
            r'^(?:(?:[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?\.)+(?:[a-zA-Z]{2,}|[0-9]{1,3})|'  # Domain format
            r'(?:(?:\d{1,3}\.){3}\d{1,3}))$'  # IP address format
        )

        # Match the SMTP server string against the regex
        return bool(smtp_regex.match(smtp_server))
    
    def validate_email(self, email) -> bool:
        # Simple regex for email validation
        email_regex = r"^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$"
        return bool(re.match(email_regex, email))

    def gather_setup_inputs(self, cmd_args, config_parser):
        self_monitoring_configs = self._init_default_configs()
        default_host_name = self.get_default_host()
        err_msg = 'No Self Monitoring host specified',
        desc = 'What is the host on which you want the self_monitoring to be installed?'
        self_monitoring_hosts = self._parse_or_prompt_hosts(cmd_args.self_monitoring_host,
                                                   default_host_name,
                                                   err_msg, desc,
                                                   'Self Monitoring host(s)',
                                                   cmd_args.no_prompt)

        self_monitoring_configs[self._option_host] = self_monitoring_hosts[:1]

        deployment_name = self._parse_or_prompt_value(
            cmd_args.deployment_name,
            '',
            'No deployment name specified',
            'What is the deployment name for the self monitoring tool',
            'Self Monitoring deployment name',
            cmd_args.no_prompt,
            lower_case_input=False
        )
        self_monitoring_configs[self._option_deployment_name] = deployment_name

        portal_uname = self._parse_or_prompt_value(
            cmd_args.portal_uname,
            'admin@cfx.com',
            'No portal username specified',
            'What is the portal username for the self monitoring tool',
            'Self Monitoring portal username',
            cmd_args.no_prompt
        )
        if not self.validate_email(portal_uname):
            cli_err_exit(f'Invalid portal username format: {portal_uname}. Please enter a valid portal username.')
        else:       
            self_monitoring_configs[self._option_portal_uname] = portal_uname

        portal_password = self._parse_or_prompt_value(
            cmd_args.portal_password,
            '',
            'No portal password specified',
            'What is the portal password for the self monitoring tool',
            'Self Monitoring portal password',
            cmd_args.no_prompt,
            password=True
        )
        self_monitoring_configs[self._option_portal_password] = str_base64_encode(portal_password)

        if cmd_args.no_prompt:
            post_to_slack = cmd_args.post_to_slack
            if post_to_slack:
                if not(cmd_args.slack_token and cmd_args.slack_channel_name and cmd_args.slack_channel_id and cmd_args.slack_members_ids and cmd_args.slack_member_names):
                    cli_err_exit("If post_to_slack is specified with --no-prompt, you must provide all slack parameters: --slack-token, --slack-channel-name, --slack-channel-id, --slack-members-id, --slack-members-names")
                self_monitoring_configs[self._option_slack_token] = str_base64_encode(cmd_args.slack_token)
                self_monitoring_configs[self._option_slack_channel_name] = cmd_args.slack_channel_name
                self_monitoring_configs[self._option_slack_channel_id] = cmd_args.slack_channel_id
                self_monitoring_configs[self._option_slack_members_name] = cmd_args.slack_member_names
                self_monitoring_configs[self._option_slack_members_id] = _comma_delimited_to_list(cmd_args.slack_members_ids)
        else:
            post_to_slack = query_yes_no("Do you want self_monitoring tool to send messages to Slack?", default='no')
            if post_to_slack:
                slack_token = self._parse_or_prompt_value(
                    cmd_args.slack_token,
                    '',
                    'No slack token specified',
                    'Please specify slack token',
                    'Slack token',
                    cmd_args.no_prompt,
                lower_case_input=False
                )
                self_monitoring_configs[self._option_slack_token] = str_base64_encode(slack_token)
                slack_channel_name = self._parse_or_prompt_value(
                    cmd_args.slack_channel_name,
                    '',
                    'No slack channel name specified',
                    'Please specify the name slack channel ',
                    'Slack channel name',
                    cmd_args.no_prompt,
                    lower_case_input=False
                )
                self_monitoring_configs[self._option_slack_channel_name] = slack_channel_name
                slack_channel_id = self._parse_or_prompt_value(
                    cmd_args.slack_channel_id,
                    '',
                    'No slack channel id specified',
                    'Please specify the id of the slack channel',
                    'Slack channel id',
                    cmd_args.no_prompt,
                    lower_case_input=False
                )
                self_monitoring_configs[self._option_slack_channel_id] = slack_channel_id
                slack_members_names = self._parse_or_prompt_value(
                    cmd_args.slack_member_names,
                    '',
                    'No slack member name specified',
                    'Please specify name of the slack members',
                    'Slack member names',
                    cmd_args.no_prompt,
                    lower_case_input=False
                )
                self_monitoring_configs[self._option_slack_members_name] = _comma_delimited_to_list(slack_members_names)
                slack_members_id = self._parse_or_prompt_value(
                    cmd_args.slack_members_ids,
                    '',
                    'No slack member id specified',
                    'Please specify id of the slack members',
                    'Slack member ids',
                    cmd_args.no_prompt,
                    lower_case_input=False
                )
                self_monitoring_configs[self._option_slack_members_id] = _comma_delimited_to_list(slack_members_id)
        
        if cmd_args.no_prompt:
            post_to_webex = cmd_args.post_to_webex
            if post_to_webex:
                if not (cmd_args.webex_token and cmd_args.webex_room_id and cmd_args):
                    cli_err_exit("If --post-to-webex is specified with --no-prompt, you must provide all webex parameters: --webex-token, --webex-room-id")
                self_monitoring_configs[self._option_webex_token] = str_base64_encode(cmd_args.webex_token)
                self_monitoring_configs[self._option_webex_room_id] = str_base64_encode(cmd_args.webex_room_id)
        else:
            post_to_webex = query_yes_no("Do you want self_monitoring tool to send messages to Webex?", default='no')
            if post_to_webex:
                webex_token = self._parse_or_prompt_value(
                    cmd_args.webex_token,
                    '',
                    'No webex token specified',
                    'Please specify webex token',
                    'Webex token',
                    cmd_args.no_prompt,
                    lower_case_input=False
                )
                self_monitoring_configs[self._option_webex_token] = str_base64_encode(webex_token)
                webex_room_id = self._parse_or_prompt_value(
                    cmd_args.webex_room_id,
                    '',
                    'No webex room_id specified',
                    'Please specify webex room_id',
                    'Webex room_id?',
                    cmd_args.no_prompt,
                    lower_case_input=False
                )
                self_monitoring_configs[self._option_webex_room_id] = str_base64_encode(webex_room_id)
        
        if cmd_args.no_prompt:
            post_to_email = cmd_args.post_to_email
            if post_to_email:
                if not (cmd_args.smtp_server and cmd_args.smtp_port and cmd_args.sender_email and cmd_args.recipient_emails):
                    cli_err_exit("If --post-to-email is specified with --no-prompt, you must provide all email parameters: --smtp-server, --smtp-port, --sender-email, --recipient-email")
        
                self_monitoring_configs[self._option_smtp_server] = cmd_args.smtp_server
                self_monitoring_configs[self._option_smtp_port] = int(cmd_args.smtp_port)
                self_monitoring_configs[self._option_sender_email] = cmd_args.sender_email
                if cmd_args.sender_password:
                    sanitized_password = cmd_args.sender_password.replace('$', '$$')
                    self_monitoring_configs[self._option_sender_password] = str_base64_encode(sanitized_password)
                else:
                    self_monitoring_configs[self._option_sender_password] = ''
                # Split and validate recipient emails
                recipient_list = [email.strip() for email in cmd_args.recipient_emails.split(',') if email.strip()]
                for email in recipient_list:
                    if not self.validate_email(email):
                        cli_err_exit(f"Invalid email format: {email}")
                self_monitoring_configs[self._option_recipient_emails] = recipient_list
        
        else:
            post_to_email = query_yes_no("Do you want self_monitoring tool to send messages to Email?", default='no')
            if post_to_email:
                smtp_server = self._parse_or_prompt_value(
                    cmd_args.smtp_server,
                    '',
                    'No SMTP Server specified',
                    'Please specify SMTP Server',
                    'SMTP Server',
                    cmd_args.no_prompt
                )
                if not self.validate_smtp_server(smtp_server):
                    cli_err_exit(f'{smtp_server} is not a valid SMTP server address')
                self_monitoring_configs[self._option_smtp_server] = smtp_server
        
                smtp_port = self._parse_or_prompt_value(
                    cmd_args.smtp_port,
                    '',
                    'No SMTP port specified',
                    'Please specify SMTP port',
                    'SMTP Port',
                    cmd_args.no_prompt
                )
                smtp_port = int(smtp_port)
                if smtp_port < 1 or smtp_port > 65535:
                    cli_err_exit(f'{smtp_port} is not a valid port number. Please enter a port between 1 and 65535.')
                self_monitoring_configs[self._option_smtp_port] = smtp_port
        
                sender_email = self._parse_or_prompt_value(
                    cmd_args.sender_email,
                    '',
                    'No Sender Email specified',
                    'Please specify Sender Email',
                    'Sender Email',
                    cmd_args.no_prompt
                )
                if not self.validate_email(sender_email):
                    cli_err_exit(f'Invalid email format: {sender_email}. Please enter a valid email address.')
                self_monitoring_configs[self._option_sender_email] = sender_email
        
                sender_password = prompt_and_validate('Sender Password?',
                                                  '',
                                                  help_desc_banner='Please specify sender password',
                                                  lower_case_input=False)
                sanitized_password = sender_password.replace('$', '$$')
                self_monitoring_configs[self._option_sender_password] = str_base64_encode(sanitized_password)
        
                recipient_emails = self._parse_or_prompt_value(
                    cmd_args.recipient_emails,
                    '',
                    'No Recipient Email specified',
                    'Please specify Recipient Email',
                    'Recipient Email',
                    cmd_args.no_prompt
                )
                recipient_list = [email.strip() for email in recipient_emails.split(',') if email.strip()]
                for email in recipient_list:
                    if not self.validate_email(email):
                        cli_err_exit(f'Invalid email format: {email}. Please enter valid email addresses.')
                self_monitoring_configs[self._option_recipient_emails] = recipient_list

        self._mark_configured(self_monitoring_configs, config_parser)
        self.write_configs(config_parser)

    def get_hosts(self) -> list:
        if not self.configs[self._option_host]:
            return []
        return self.configs[self._option_host]

    def get_deployment_name(self) -> str:
        return self.configs[self._option_deployment_name]

    def get_portal_uname(self) -> str:
        return self.configs[self._option_portal_uname]

    def get_portal_password(self) -> str:
        return str_base64_decode(self.configs[self._option_portal_password])
    
    def get_slack_token(self) -> str:
        return str_base64_decode(self.configs[self._option_slack_token])

    def get_slack_channel_name(self) -> str:
        return self.configs[self._option_slack_channel_name]

    def get_slack_channel_id(self) -> str:
        return self.configs[self._option_slack_channel_id]

    def get_slack_member_names(self) -> list:
        return self.configs[self._option_slack_members_name]

    def get_slack_member_ids(self) -> list:
        ids = self.configs[self._option_slack_members_id]
        return [f"<@{id.strip()}>" for id in ids]
    
    def get_webex_token(self) -> str:
        return str_base64_decode(self.configs[self._option_webex_token])

    def get_webex_room_id(self) -> str:
        return str_base64_decode(self.configs[self._option_webex_room_id])
    
    def get_smtp_server(self) -> str:
        return self.configs[self._option_smtp_server]
    
    def get_smtp_port(self) -> int:
        return self.configs[self._option_smtp_port]
    
    def get_sender_email(self) -> str:
        return self.configs[self._option_sender_email]
    
    def get_sender_password(self) -> str:
        return str_base64_decode(self.configs[self._option_sender_password])
    
    def get_recipient_emails(self) -> list:
        return self.configs[self._option_recipient_emails]

    def get_deployment_file_path(self) -> os.path:
        return os.path.join('/opt', 'rdaf', 'deployment-scripts', 'self-monitoring.yaml')

    def create_compose_file_self_monitoring(self, cmd_args: argparse.Namespace):
        compose_file = self.get_deployment_file_path()
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), os.path.basename(compose_file))
        replacements = self.get_deployment_replacements(cmd_args)
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        original_values = yaml.safe_load(original_content)
        with open(compose_file, 'w') as f:
            yaml.safe_dump(original_values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        for host in self.get_hosts():
            if not Component.is_local_host(host):
                do_potential_scp(host, compose_file, compose_file)

    def do_setup(self, cmd_args, config_parser):
        network_config = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'config.json')
        if not os.path.exists(network_config):
            cli_err_exit('Network config file does not exist, Please install rdaf platform.')
        self.gather_setup_inputs(cmd_args, config_parser)
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        for host in self.get_hosts():
            if host not in all_known_hosts:
                self.setup_install_root_dir_hierarchy(host, config_parser)

            # doing a docker login
            docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
            docker_registry.docker_login(host, config_parser)
        if self.get_deployment_type(config_parser) == 'k8s':
            self.k8s_generate_settings_configuration(cmd_args, config_parser)
        else:
            self._generate_settings_configuration(cmd_args, config_parser)
        config_file = os.path.join('/opt', 'rdaf', 'rdaf.cfg')
        for host in self.get_hosts():
            if Component.is_local_host(host):
                continue
            do_potential_scp(host, config_file, config_file)
        
    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
           self.create_compose_file_self_monitoring(cmd_args)
           deployment_file = self.get_deployment_file_path()
           command = '/usr/local/bin/docker-compose -f {file} pull && ' \
                     '/usr/local/bin/docker-compose -f {file} --project-name self_monitoring up -d '.format(file=deployment_file)
           logger.info(f"Installing self_monitoring on host {host}")
           run_potential_ssh_command(host, command, config_parser)
    
    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
           self.create_compose_file_self_monitoring(cmd_args)
           deployment_file = self.get_deployment_file_path()
           command = '/usr/local/bin/docker-compose -f {file} pull && ' \
                     '/usr/local/bin/docker-compose -f {file} --project-name self_monitoring up -d '.format(file=deployment_file)
           logger.info(f"Upgrading self_monitoring on host {host}")
           run_potential_ssh_command(host, command, config_parser)

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        command = f'/usr/local/bin/docker-compose -f {deployment_file} --project-name self_monitoring up -d '
        for host in self.get_hosts():
            logger.info(f"Creating self_monitoring on host {host}")
            run_potential_ssh_command(host, command, config_parser)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        for host in reversed(self.get_hosts()):
            logger.info(f"Deleting self_monitoring on host {host}")
            command = f'/usr/local/bin/docker-compose -f {deployment_file} --project-name self_monitoring rm -fs cfx_self_monitor'
            run_potential_ssh_command(host, command, config_parser)

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        command = f'/usr/local/bin/docker-compose -f {deployment_file} --project-name self_monitoring start '
        for host in self.get_hosts():
            logger.info(f"Starting self_monitoring on host {host}")
            run_potential_ssh_command(host, command, config_parser)

    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        deployment_file = self.get_deployment_file_path()
        if not os.path.exists(deployment_file):
            return
        for host in reversed(self.get_hosts()):
            logger.info(f"stopping self_monitoring on host {host}")
            command = f'/usr/local/bin/docker-compose -f {deployment_file} --project-name self_monitoring stop cfx_self_monitor'
            run_potential_ssh_command(host, command, config_parser)

    def k8s_generate_settings_configuration(self, cmd_args, config_parser):
        logger.info("Generating settings for self-monitor...")
        template_root = get_templates_dir_root()
        portal_template = os.path.join(template_root, 'settings.json')
        with open(portal_template, 'r') as f:
            template_content = f.read()
        
        values_yaml = "/opt/rdaf/deployment-scripts/values.yaml"
        with open(values_yaml, 'r') as file:
            data = yaml.safe_load(file)

        replacement_values = dict()
        replacement_values['DEPLOYMENT_NAME'] =self.get_deployment_name()

        haproxy_advertised_host = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME) \
                .get_internal_access_host()
        replacement_values['PORTAL_URL'] = "https://" + haproxy_advertised_host
        replacement_values['PORTAL_UNAME'] = self.get_portal_uname()
        slack_token = self.get_slack_token()
        if slack_token is not None:
            slack_block = f"""
    "slack": {{
        "token": "{self.get_slack_token()}",
        "channel": "{self.get_slack_channel_name()}",
        "channel_id": "{self.get_slack_channel_id()}",
        "team": {{
            "critical_name": {json.dumps(self.get_slack_member_names())},
            "critical": {json.dumps(self.get_slack_member_ids())},
            "high_names": {json.dumps(self.get_slack_member_names())},
            "high": {json.dumps(self.get_slack_member_ids())},
            "medium_names": {json.dumps(self.get_slack_member_names())},
            "medium": {json.dumps(self.get_slack_member_ids())},
            "low_names": {json.dumps(self.get_slack_member_names())},
            "low": {json.dumps(self.get_slack_member_ids())}
        }}
    }},"""
            slack_placeholder = '"slack": null,'
            template_content = template_content.replace(slack_placeholder, slack_block)

        webex_token = self.get_webex_token()
        if webex_token is not None:
            webex_block = f"""
    "webex": {{
        "token": "{self.get_webex_token()}",
        "roomId": "{self.get_webex_room_id()}"
    }},"""
            webex_placeholder = '"webex": null,'
            template_content = template_content.replace(webex_placeholder, webex_block)

        sender_mail = self.get_sender_email()
        if sender_mail is not None:
            email_block =  f""""email": {{
    "smtp_server": "{self.get_smtp_server()}",
    "smtp_port": {self.get_smtp_port()},
    "sender_email": "{self.get_sender_email()}",
    "sender_password": "{self.get_sender_password()}",
    "recipient_emails": {json.dumps(self.get_recipient_emails())},
    "template": "default.html",
    "send_only_failed": false,
    "subject": "Critical Alert Summary"
    }},"""
            email_placeholder = '"email": null,'
            template_content = template_content.replace(email_placeholder, email_block)

        def get_replicas(service_name):
            return data.get(service_name, {}).get('replicas', 0)
        
        replacement_values['ALERT_INGESTER'] = get_replicas('alert_ingester')
        replacement_values['ALERT_PROCESSOR'] = get_replicas('alert_processor')
        replacement_values['ALERT_PROCESSOR_COMPANION'] = get_replicas('alert_processor_companion')
        replacement_values['IDENTITY'] = get_replicas('rda_identity')
        replacement_values['CFX_APP_CONTROLLER'] = get_replicas('app_controller')
        replacement_values['APP_ACCESS_MANAGER'] = get_replicas('rda_access_manager')
        replacement_values['APP_COLLABORATION'] = get_replicas('collaboration')
        replacement_values['APP_FILE_BROWSER'] = get_replicas('file_browser')
        replacement_values['APP_IRM_SERVICE'] = get_replicas('irm_service')
        replacement_values['APP_NOTIFICATION_SERVICE'] = get_replicas('notification_service')
        replacement_values['APP_RESOURCE_MANAGER'] = get_replicas('rda_resource_manager')
        replacement_values['CHAT_HELPER'] = get_replicas('rda_chat_helper')
        replacement_values['CONFIGURATION_SERVICE'] = get_replicas('configuration_service')
        replacement_values['EVENT_CONSUMER'] = get_replicas('event_consumer')
        replacement_values['FSM'] = get_replicas('rda_fsm')
        replacement_values['INGESTION_TRACKER'] = get_replicas('ingestion_tracker')
        replacement_values['ML_CONFIG'] = get_replicas('ml_config')
        replacement_values['REPORTS_REGISTRY'] = get_replicas('reports_registry')
        replacement_values['SMTP_SERVER'] = get_replicas('smtp_server')
        replacement_values['USER_PREFERENCES'] = get_replicas('rda_user_preferences')
        replacement_values['WEBHOOK_SERVER'] = get_replicas('webhook_server')
        replacement_values['API_SERVER'] = get_replicas('rda_api_server')
        replacement_values['ASM'] = get_replicas('rda_asm')
        replacement_values['COLLECTOR'] = get_replicas('rda_collector')
        replacement_values['REGISTRY'] = get_replicas('rda_registry')
        replacement_values['SCHEDULER'] = get_replicas('rda_scheduler')
        replacement_values['WORKER_HOSTS'] = get_replicas('rda_worker')

        substituted_content = string.Template(template_content).substitute(replacement_values)

        settings = os.path.join('/opt', 'rdaf', 'config', 'cfx_self_monitor', 'resources', 'settings.json')
        uid = os.getuid()
        gid = os.getgid()
        settings_dir_command = (
                    'sudo mkdir -p ' + os.path.join('/opt', 'rdaf', 'config', 'cfx_self_monitor') + ' && sudo chown -R '
                    + str(uid) + ' ' + os.path.join('/opt', 'rdaf', 'config', 'cfx_self_monitor')
                    + ' && sudo chgrp -R ' + str(gid) + ' ' + os.path.join('/opt', 'rdaf', 'config', 'cfx_self_monitor'))
        network_json = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'external-config.json')
        self_monitor_config_json = os.path.join('/opt', 'rdaf', 'config', 'cfx_self_monitor', 'config.json')
        for host in self.get_hosts():
            run_potential_ssh_command(host, settings_dir_command, config_parser)
            created_location = create_file(host, substituted_content.encode(encoding='UTF-8'), settings)
            logger.info(f"Created {created_location} on {host} ")
            created_location = create_file(host, self.get_portal_password().encode(encoding='UTF-8'),
                                           os.path.join('/tmp/portal_pwd'))
            logger.info(f"Created {created_location} on {host} ")
            do_potential_scp(host, network_json, self_monitor_config_json)
        logger.info('Setup completed successfully')

    def _generate_settings_configuration(self, cmd_args, config_parser):
        logger.info("Generating settings for self-monitor...")
        template_root = get_templates_dir_root()
        portal_template = os.path.join(template_root, 'settings.json')
        with open(portal_template, 'r') as f:
            template_content = f.read()

        values_yaml = "/opt/rdaf/deployment-scripts/values.yaml"
        with open(values_yaml, 'r') as file:
            data = yaml.safe_load(file)
        platform_component = COMPONENT_REGISTRY \
            .require(rdaf.component.pseudo_platform.PseudoComponent.COMPONENT_NAME)
        platform_hosts = platform_component.get_platform_services_hosts()

        replacement_values = dict()
        replacement_values['DEPLOYMENT_NAME'] =self.get_deployment_name()

        haproxy_advertised_host = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME) \
                .get_internal_access_host()
        replacement_values['PORTAL_URL'] = "https://" + haproxy_advertised_host
        replacement_values['PORTAL_UNAME'] = self.get_portal_uname()
        slack_token = self.get_slack_token()
        if slack_token is not None:
            slack_block = f"""
    "slack": {{
        "token": "{self.get_slack_token()}",
        "channel": "{self.get_slack_channel_name()}",
        "channel_id": "{self.get_slack_channel_id()}",
        "team": {{
            "critical_name": {json.dumps(self.get_slack_member_names())},
            "critical": {json.dumps(self.get_slack_member_ids())},
            "high_names": {json.dumps(self.get_slack_member_names())},
            "high": {json.dumps(self.get_slack_member_ids())},
            "medium_names": {json.dumps(self.get_slack_member_names())},
            "medium": {json.dumps(self.get_slack_member_ids())},
            "low_names": {json.dumps(self.get_slack_member_names())},
            "low": {json.dumps(self.get_slack_member_ids())}
        }}
    }},"""
            slack_placeholder = '"slack": null,'
            template_content = template_content.replace(slack_placeholder, slack_block)

        webex_token = self.get_webex_token()
        if webex_token is not None:
            webex_block = f"""
    "webex": {{
        "token": "{self.get_webex_token()}",
        "roomId": "{self.get_webex_room_id()}"
    }},"""
            webex_placeholder = '"webex": null,'
            template_content = template_content.replace(webex_placeholder, webex_block)

        sender_mail = self.get_sender_email()
        if sender_mail is not None:
            email_block =  f""""email": {{
    "smtp_server": "{self.get_smtp_server()}",
    "smtp_port": {self.get_smtp_port()},
    "sender_email": "{self.get_sender_email()}",
    "sender_password": "{self.get_sender_password()}",
    "recipient_emails": {json.dumps(self.get_recipient_emails())},
    "template": "default.html",
    "send_only_failed": false,
    "subject": "Critical Alert Summary"
    }},"""
            email_placeholder = '"email": null,'
            template_content = template_content.replace(email_placeholder, email_block)

        replacement_values['ALERT_INGESTER'] = len(data['services'].get('cfx-rda-alert-ingester', {}).get('hosts', []))
        replacement_values['ALERT_PROCESSOR'] = len(data['services'].get('cfx-rda-alert-processor', {}).get('hosts', []))
        replacement_values['ALERT_CORRELATOR'] = len(data['services'].get('cfx-rda-alert-correlator', {}).get('hosts', []))
        replacement_values['ALERT_PROCESSOR_COMPANION'] = len(data['services'].get('cfx-rda-alert-processor-companion', {}).get('hosts', []))
        replacement_values['IDENTITY'] = len(platform_hosts)
        replacement_values['CFX_APP_CONTROLLER'] = len(data['services'].get('cfx-rda-app-controller', {}).get('hosts', []))
        replacement_values['APP_ACCESS_MANAGER'] = len(platform_hosts)
        replacement_values['APP_COLLABORATION'] = len(data['services'].get('cfx-rda-collaboration', {}).get('hosts', []))
        replacement_values['APP_FILE_BROWSER'] = len(data['services'].get('cfx-rda-file-browser', {}).get('hosts', []))
        replacement_values['APP_IRM_SERVICE'] = len(data['services'].get('cfx-rda-irm-service', {}).get('hosts', []))
        replacement_values['APP_NOTIFICATION_SERVICE'] = len(data['services'].get('cfx-rda-notification-service', {}).get('hosts', []))
        replacement_values['APP_RESOURCE_MANAGER'] = len(platform_hosts)
        replacement_values['CHAT_HELPER'] = len(platform_hosts)
        replacement_values['CONFIGURATION_SERVICE'] = len(data['services'].get('cfx-rda-configuration-service', {}).get('hosts', []))
        replacement_values['EVENT_CONSUMER'] = len(data['services'].get('cfx-rda-event-consumer', {}).get('hosts', []))
        replacement_values['FSM'] = len(platform_hosts)
        replacement_values['INGESTION_TRACKER'] = len(data['services'].get('cfx-rda-ingestion-tracker', {}).get('hosts', []))
        replacement_values['ML_CONFIG'] = len(data['services'].get('cfx-rda-ml-config', {}).get('hosts', []))
        replacement_values['REPORTS_REGISTRY'] = len(data['services'].get('cfx-rda-reports-registry', {}).get('hosts', []))
        replacement_values['SMTP_SERVER'] = len(data['services'].get('cfx-rda-smtp-server', {}).get('hosts', []))
        replacement_values['USER_PREFERENCES'] = len(platform_hosts)
        replacement_values['WEBHOOK_SERVER'] = len(data['services'].get('cfx-rda-webhook-server', {}).get('hosts', []))
        replacement_values['API_SERVER'] = len(platform_hosts)
        replacement_values['ASM'] = len(platform_hosts)
        replacement_values['COLLECTOR'] = len(platform_hosts)
        replacement_values['REGISTRY'] = len(platform_hosts)
        replacement_values['SCHEDULER'] = len(platform_hosts)
        replacement_values['WORKER_HOSTS'] = len(self.get_worker_hosts(config_parser))

        substituted_content = string.Template(template_content).substitute(replacement_values)
        settings = os.path.join('/opt', 'rdaf', 'config', 'cfx_self_monitor', 'resources', 'settings.json')
        uid = os.getuid()
        gid = os.getgid()
        settings_dir_command = ('sudo mkdir -p ' + os.path.join('/opt', 'rdaf', 'config', 'cfx_self_monitor') + ' && sudo chown -R '
                                + str(uid) + ' ' + os.path.join('/opt', 'rdaf', 'config', 'cfx_self_monitor')
                                + ' && sudo chgrp -R ' + str(gid) + ' ' + os.path.join('/opt', 'rdaf', 'config', 'cfx_self_monitor'))
        network_json = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'config.json')
        self_monitor_config_json = os.path.join('/opt', 'rdaf', 'config', 'cfx_self_monitor', 'config.json')
        for host in self.get_hosts():
            run_potential_ssh_command(host, settings_dir_command, config_parser)
            created_location = create_file(host, substituted_content.encode(encoding='UTF-8'), settings)
            logger.info(f"Created {created_location} on {host} ")
            created_location = create_file(host, self.get_portal_password().encode(encoding='UTF-8'), os.path.join('/tmp/portal_pwd'))
            logger.info(f"Created {created_location} on {host} ")
            do_potential_scp(host, network_json, self_monitor_config_json)
        logger.info('Setup completed successfully')

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        if self.get_deployment_type(config_parser) != 'k8s':
            return []
        return self.status(cmd_args, config_parser)

    def k8s_reset(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.reset(cmd_args, config_parser)