import configparser
import logging
import os.path

from rdaf import rdafutils
from rdaf.component import Component, OtherCategoryOrder, dockerregistry
from rdaf.contextual import COMPONENT_REGISTRY
import rdaf

logger = logging.getLogger(__name__)


class SSHKeyManager(Component):
    COMPONENT_NAME = 'ssh-key-manager'

    _option_ssh_user = 'ssh_user'
    _option_ssh_key_path = 'ssh_key_path'

    def __init__(self):
        super().__init__(SSHKeyManager.COMPONENT_NAME, 'ssh-key-manager', category='other',
                         category_order=OtherCategoryOrder.SSH_KEY_MANAGER.value)
        self.ssh_password = None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_ssh_key_path] = None
        return default_configs

    def gather_setup_inputs(self, cmd_args, config_parser):
        ssh_configs = self._init_default_configs()
        user_desc = 'What is the SSH user you want to use to communicate ' \
                    'between the hosts'
        user_no_prompt_err_msg = 'No SSH user specified. Use --ssh-user to specify one'
        ssh_configs[self._option_ssh_user] = rdaf.component.Component._parse_or_prompt_value(
            cmd_args.ssh_user,
            None,
            user_no_prompt_err_msg,
            user_desc,
            'SSH user',
            cmd_args.no_prompt)

        if hasattr(cmd_args, 'ssh_key') and cmd_args.ssh_key:
            ssh_key_desc = 'What is the Path for ssh key to communicate with hosts'
            ssh_key_no_prompt_err_msg = 'No SSH key path specified. Use --ssh-key-path to specify one'
            ssh_key_path = rdaf.component.Component \
                ._parse_or_prompt_value(cmd_args.ssh_key_path,
                                        None,
                                        ssh_key_no_prompt_err_msg,
                                        ssh_key_desc,
                                        'SSH key path',
                                        cmd_args.no_prompt,
                                        password=False,
                                        apply_password_validation=False)
            if not os.path.exists(os.path.join(ssh_key_path)) or not os.path.isfile(os.path.join(ssh_key_path)):
                rdafutils.cli_err_exit("ssh key path specified doesn't exist : " + ssh_key_path)
            ssh_configs[self._option_ssh_key_path] = ssh_key_path
        else:
            pass_desc = 'What is the SSH password for the SSH user used to communicate' \
                        ' between hosts'
            pass_no_prompt_err_msg = 'No SSH password specified. Use --ssh-password to specify one'
            # don't store in the configs
            self.ssh_password = rdaf.component.Component \
                ._parse_or_prompt_value(cmd_args.ssh_password,
                                        None,
                                        pass_no_prompt_err_msg,
                                        pass_desc,
                                        'SSH password',
                                        cmd_args.no_prompt,
                                        password=True,
                                        apply_password_validation=False)
        self._mark_configured(ssh_configs, config_parser)

    def do_setup(self, cmd_args, config_parser: configparser.ConfigParser):
        # skip the ssh key generation if using private key to do ssh
        if self.configs[self._option_ssh_key_path]:
            return
        known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])

        # create ssh key (key creation will be skipped if they already exist)
        rdafutils.create_ssh_key()

        if len(known_hosts) == 1 and self.is_local_host(list(known_hosts)[0]):
            logger.debug("ignoring ssh key setup as single host is involved.")
            return
        
        known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        # setup ssh for these hosts so that they "know" each other and
        # don't require password for logins
        logger.info('Setting up SSH between hosts ' + str(known_hosts))
        # first add all these hosts to the current system's ~/.ssh/known_hosts
        # file so that subsequent ssh commands don't prompt for fingerprint acceptance
        # against this hosts
        rdafutils.ssh_add_as_known_host(known_hosts)
        # now copy ssh key of the current system to all these hosts
        for host in known_hosts:
            rdafutils.copy_ssh_key(host, self.configs[self._option_ssh_user], self.ssh_password)

    def do_k8s_setup(self, cmd_args, config_parser):
        self.do_setup(cmd_args, config_parser)

    def get_ssh_user(self) -> str:
        return self.configs[self._option_ssh_user]

    def get_ssh_key_path(self) -> str:
        return self.configs[self._option_ssh_key_path]

    def setup_keys_for_host(self, host: str, ssh_password: str):
        rdafutils.ssh_add_as_known_host([host])
        rdafutils.copy_ssh_key(host, self.configs[self._option_ssh_user], ssh_password)
