#!/usr/bin/python
import logging
import re
import uuid
from typing import Generator

from docker import APIClient, DockerClient
from docker.tls import TLSConfig

logger = logging.getLogger(__name__)


def create_container_name(name_prefix: str) -> str:
    name_part = uuid.uuid4().hex[0:8]
    return name_prefix + '_' + name_part


class cliDockerSession(object):
    def __init__(self, **kwargs):
        import urllib3
        urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
        base_url = kwargs.get('dockersock', None)
        host = kwargs.get('host', None)
        if base_url is None:
            if host is None:
                base_url = 'unix://var/run/docker.sock'
            else:
                # parse the host and create a tcp based endpoint
                parts = host.split(':')
                if len(parts) == 2:
                    base_url = 'tcp://' + str(parts[0]) + ':' + str(parts[1])
                elif len(parts) == 1:
                    base_url = 'tcp://' + str(parts[0]) + ':2376'
                else:
                    # default to whatever value the host holds
                    base_url = host
        tls_config = None
        if re.search('tcp', base_url) and kwargs.get('tls', True):
            client_certs = kwargs.get('client_certs', None)
            if isinstance(client_certs, dict):
                cert, key = client_certs['certificate'], client_certs['key']
                ca_file = client_certs['ca']
            else:
                cert, key = (getattr(client_certs, 'certificate', None),
                             getattr(client_certs, 'key', None))
                ca_file = getattr(client_certs, 'ca', None)
            tls_config = TLSConfig(
                client_cert=(cert, key) if client_certs else None,
                ca_cert=ca_file,
                verify=False
            )
        logger.debug("Creating docker APIClient with base_url: " + base_url)
        self.client = APIClient(base_url=base_url,
                                version=kwargs.get('dockerapiversion', 'auto'),
                                timeout=kwargs.get('timeout', 5),
                                tls=tls_config)
        self.version = self.client.version()
        self.warning = False

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.client is not None:
            self.client.close()

    def getClient(self) -> APIClient:
        return self.client

    def getDockerVersion(self):
        return self.version.get('Version')

    def getContainers(self, **kwargs):
        return self.client.containers(all=kwargs.get('all', True),
                                      filters=kwargs.get('filters', None))

    def getImages(self, **kwargs):
        return self.client.images(all=kwargs.get('all', True))

    def removeImage(self, **kwargs):
        if kwargs.get('image', None):
            self.client.remove_image(image=kwargs.get('image'))

    def pruneImages(self, **kwargs):
        images = self.client.images(filters=kwargs.get('filters', {'dangling': True}))
        for image in images:
            if image.get('Name', None):
                self.removeImage(image=image.get('Name'))

    def docker_exec(self, container_id: str, exec_cmd: str, env: dict = None) -> (bytes, bytes):
        exec_id = self.client.exec_create(container=container_id,
                                          cmd=exec_cmd,
                                          stdout=True, stderr=True,
                                          environment=env)
        output, error = self.client.exec_start(exec_id=exec_id.get('Id'), demux=True)
        return output, error

    def docker_exec_stream(self, container_id: str, exec_cmd: str, env: dict = None) -> Generator:
        exec_id = self.client.exec_create(container=container_id,
                                          cmd=exec_cmd,
                                          stdout=True, stderr=True,
                                          environment=env)

        return self.client.exec_start(exec_id=exec_id.get('Id'), stream=True)


class clientDockerSession(object):
    def __init__(self, **kwargs):
        import urllib3
        urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
        base_url = kwargs.get('dockersock', None)
        host = kwargs.get('host', None)
        if base_url is None:
            if host is None:
                base_url = 'unix://var/run/docker.sock'
            else:
                # parse the host and create a tcp based endpoint
                parts = host.split(':')
                if len(parts) == 2:
                    base_url = 'tcp://' + str(parts[0]) + ':' + str(parts[1])
                elif len(parts) == 1:
                    base_url = 'tcp://' + str(parts[0]) + ':2376'
                else:
                    # default to whatever value the host holds
                    base_url = host
        tls_config = None
        if re.search('tcp', base_url) and kwargs.get('tls', True):
            client_certs = kwargs.get('client_certs', None)
            if isinstance(client_certs, dict):
                cert, key = client_certs['certificate'], client_certs['key']
                ca_file = client_certs['ca']
            else:
                cert, key = (getattr(client_certs, 'certificate', None),
                             getattr(client_certs, 'key', None))
                ca_file = getattr(client_certs, 'ca', None)
            tls_config = TLSConfig(
                client_cert=(cert, key) if client_certs else None,
                ca_cert=ca_file,
                verify=False
            )
        logger.debug("Creating docker Client with base_url: " + base_url)
        self.client = DockerClient(base_url=base_url,
                                   version=kwargs.get('dockerapiversion', 'auto'),
                                   timeout=120,
                                   tls=tls_config)
        self.version = self.client.version()
        self.warning = False

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.client is not None:
            self.client.close()

def truncated_container_id(container_id):
    r"""
    Returns the 12 character truncated representation of the docker container id
    :param  container_id: The docker container id
    :type   container_id: str
    """
    if container_id is None:
        return None
    # Docker commands (most of them) truncate the id to 12 characters
    docker_truncated_id_char_length = 12
    if len(container_id) <= docker_truncated_id_char_length:
        return container_id
    return container_id[:12]
