import argparse
import cmd
import configparser
import logging
import logging.config
import os
import sys
from enum import Enum

import termcolor
import rdaf.cmd.infra as cmd_infra
import rdaf.cmd.backup
import rdaf.cmd.restore
import rdaf.cmd.platform as cmd_platform
import rdaf.cmd.worker as cmd_worker
import rdaf.cmd.event_gateway as cmd_event_gateway
import rdaf.cmd.log_monitoring as cmd_log_monitoring
import rdaf.cmd.bulk_stats as cmd_bulk_stats
import rdaf.cmd.file_object as cmd_file_object
import rdaf.cmd.opensearch_external as cmd_opensearch_external
import rdaf.cmd.dockerregistry as cmd_docker
import rdaf.cmd.geodr as cmd_rdac_geodr
import rdaf.cmd.application as cmd_app
import rdaf.cmd.validate as cmd_validate
import rdaf.cmd.maintenance as cmd_maintenance
import rdaf.cmd.edge_installer as edge_installer
import rdaf.cmd.status as cmd_status
import rdaf.cmd.rdac as cmd_rdac_cli
import rdaf.cmd.telegraf as cmd_telegraf
import rdaf.rdafutils as rdafutils
import rdaf.util.requestsutil
import rdaf.cmd.self_monitoring as cmd_self_monitoring
import rdaf.cmd.logs as cmd_logs
from rdaf import CliException, InvalidCmdUsageException
from rdaf.cmd import setup_cmd, reset, setdockerreg, prune_images, is_rdaf_setup_done


# Windows doesn't have the `pwd` module.
# So we try and use the USERNAME env variable
# to get the current user
def _current_user():
    import platform
    if platform.system() == 'Windows':
        return os.environ.get('USERNAME')
    else:
        import pwd
        return pwd.getpwuid(os.getuid()).pw_name


class CmdName(Enum):
    STATUS = 'status'
    SETUP = 'setup'
    PRUNE_IMAGES = 'prune-images'
    INFRA = 'infra'
    RESET = 'reset'
    PLATFORM = 'platform'
    WORKER = 'worker'
    EVENT_GATEWAY='event_gateway'
    LOG_MONITORING = 'log_monitoring'
    BULK_STATS = 'bulk_stats'
    FILE_OBJECT = 'file_object'
    OPENSEARCH_EXTERNAL = 'opensearch_external'
    APPLICATION = 'app'
    DOCKER_REGISTRY = 'registry'
    SET_DOCKER_REG = 'setregistry'
    VALIDATE = 'validate'
    INSTALL_RDA_EDGE = 'install-rda-edge'
    BACKUP = 'backup'
    RESTORE = 'restore'
    RDACCLI = 'rdac-cli'
    GEODR = 'geodr'
    MAINTENANCE = 'maintenance'
    TELEGRAF = 'telegraf'
    SELF_MONITORING = 'self_monitoring'
    LOGS = 'logs'


insecure_parser = argparse.ArgumentParser(add_help=False)
insecure_parser.add_argument('--insecure',
                             dest='insecure',
                             action='store_true',
                             default=False,
                             help='Ignore SSL certificate issues when communicating'
                                  ' with various hosts')

debug_parser = argparse.ArgumentParser()
debug_parser.add_argument('--debug', dest='debug', action='store_true', default=False,
                          help='Enable debug logs for the CLI operations')


class Cli(cmd.Cmd):
    def __init__(self):
        super().__init__()
        # a dictionary of commands keyed to the CmdName. Value is a tuple of
        # (command parser, command handler)
        self.cmds = dict()
        self._create_commands()

    def _create_commands(self):

        parent_parsers = [insecure_parser, debug_parser]

        ssh_user_parser = argparse.ArgumentParser(add_help=False)
        ssh_user_parser.add_argument('--ssh-user',
                                     dest="ssh_user",
                                     action='store',
                                     default=_current_user(),
                                     help='User name to use to do an initial SSH login'
                                          ' while setting up password-less SSH between hosts')
        ssh_user_parser.add_argument('--ssh-password',
                                     dest="ssh_password",
                                     action='store',
                                     default=None,
                                     help='Password to use to do an initial SSH login'
                                          ' while setting up password-less SSH between hosts')

        ssh_key_parser = argparse.ArgumentParser(add_help=False)

        ssh_key_parser.add_argument('--passphrase',
                                    action='store',
                                    default="",
                                    dest="passphrase",
                                    help='SSH Key Pass Phrase')
        # setup command
        setup_cmd_parser = _new_arg_parser(command=CmdName.SETUP,
                                           description='Installs and sets up RDAF',
                                           parent_parsers=[insecure_parser, debug_parser,
                                                           ssh_key_parser, ssh_user_parser])
        setup_cmd_handler = setup_cmd.SetupCmdHandler()
        setup_cmd_handler.configure_parser(setup_cmd_parser)
        self.cmds[CmdName.SETUP] = (setup_cmd_parser, setup_cmd_handler)

        # prune_images command
        prune_images_cmd_parser = _new_arg_parser(command=CmdName.PRUNE_IMAGES,
                                                  description='Prune unused images from all hosts',
                                                  parent_parsers=[debug_parser])
        prune_images_cmd_handler = prune_images.PruneImageCmdHandler()
        prune_images_cmd_handler.configure_parser(prune_images_cmd_parser)
        self.cmds[CmdName.PRUNE_IMAGES] = (prune_images_cmd_parser, prune_images_cmd_handler)

        # infra command
        infra_cmd_parser = _new_arg_parser(command=CmdName.INFRA,
                                           description='Manage infra services',
                                           parent_parsers=parent_parsers)
        infra_cmd_handler = cmd_infra.InfraCmdHandler()
        infra_cmd_handler.configure_parser(infra_cmd_parser)
        self.cmds[CmdName.INFRA] = (infra_cmd_parser, infra_cmd_handler)

        # reset command
        reset_cmd_parser = _new_arg_parser(command=CmdName.RESET,
                                           description='Reset the RDAF installation',
                                           parent_parsers=parent_parsers)
        reset_cmd_handler = reset.ResetCmdHandler()
        reset_cmd_handler.configure_parser(reset_cmd_parser)
        self.cmds[CmdName.RESET] = (reset_cmd_parser, reset_cmd_handler)

        # status command
        status_cmd_parser = _new_arg_parser(command=CmdName.STATUS,
                                            description='Manages status of infra and platform',
                                            parent_parsers=parent_parsers)
        status_cmd_handler = cmd_status.StatusCmdHandler()
        status_cmd_handler.configure_parser(status_cmd_parser)
        self.cmds[CmdName.STATUS] = (status_cmd_parser, status_cmd_handler)

        # platform command
        platform_cmd_parser = _new_arg_parser(command=CmdName.PLATFORM,
                                              parent_parsers=[debug_parser],
                                              description='Manage the RDAF Platform')
        platform_handler = cmd_platform.PlatformCmdHandler()
        platform_handler.configure_parser(platform_cmd_parser)
        self.cmds[CmdName.PLATFORM] = (platform_cmd_parser, platform_handler)

        # worker command
        worker_cmd_parser = _new_arg_parser(command=CmdName.WORKER,
                                            parent_parsers=[debug_parser],
                                            description='Manage the RDAF Worker')
        worker_handler = cmd_worker.WorkerCmdHandler()
        worker_handler.configure_parser(worker_cmd_parser)
        self.cmds[CmdName.WORKER] = (worker_cmd_parser, worker_handler)

        # event_gateway command
        event_gateway_cmd_parser = _new_arg_parser(command=CmdName.EVENT_GATEWAY,
                                            parent_parsers=[debug_parser],
                                            description='Manage the RDAF Event Gateway')
        event_gateway_handler = cmd_event_gateway.EventGatewayCmdHandler()
        event_gateway_handler.configure_parser(event_gateway_cmd_parser)
        self.cmds[CmdName.EVENT_GATEWAY] = (event_gateway_cmd_parser, event_gateway_handler)

        # log_monitoring command
        log_monitoring_cmd_parser = _new_arg_parser(command=CmdName.LOG_MONITORING,
                                                    parent_parsers=[debug_parser],
                                                    description='Manage the RDAF log monitoring')
        log_monitoring_handler = cmd_log_monitoring.LogMonitoringCmdHandler()
        log_monitoring_handler.configure_parser(log_monitoring_cmd_parser)
        self.cmds[CmdName.LOG_MONITORING] = (log_monitoring_cmd_parser, log_monitoring_handler)

        # bulk_stats command
        bulk_stats_cmd_parser = _new_arg_parser(command=CmdName.BULK_STATS,
                                                    parent_parsers=[debug_parser],
                                                    description='Manage the RDAF bulkstats')
        bulk_stats_handler = cmd_bulk_stats.BulkStatsCmdHandler()
        bulk_stats_handler.configure_parser(bulk_stats_cmd_parser)
        self.cmds[CmdName.BULK_STATS] = (bulk_stats_cmd_parser, bulk_stats_handler)

        # file_object command
        file_object_cmd_parser = _new_arg_parser(command=CmdName.FILE_OBJECT,
                                                    parent_parsers=[debug_parser],
                                                    description='Manage the RDAF file object')
        file_object_handler = cmd_file_object.FileObjectCmdHandler()
        file_object_handler.configure_parser(file_object_cmd_parser)
        self.cmds[CmdName.FILE_OBJECT] = (file_object_cmd_parser, file_object_handler)

        # opensearch_external command
        opensearch_external_cmd_parser = _new_arg_parser(command=CmdName.OPENSEARCH_EXTERNAL,
                                                    parent_parsers=[debug_parser],
                                                    description='Manage the Opensearch External')
        opensearch_external_handler = cmd_opensearch_external.opensearch_externalCmdHandler()
        opensearch_external_handler.configure_parser(opensearch_external_cmd_parser)
        self.cmds[CmdName.OPENSEARCH_EXTERNAL] = (opensearch_external_cmd_parser, opensearch_external_handler)

        # app command
        app_cmd_parser = _new_arg_parser(command=CmdName.APPLICATION,
                                         parent_parsers=[debug_parser],
                                         description='Manage the RDAF Apps')
        app_handler = cmd_app.ApplicationCmdHandler()
        app_handler.configure_parser(app_cmd_parser)
        self.cmds[CmdName.APPLICATION] = (app_cmd_parser, app_handler)

        # docker reg command
        docker_reg_cmd_parser = _new_arg_parser(command=CmdName.DOCKER_REGISTRY,
                                                description='Manage the Docker registry',
                                                parent_parsers=parent_parsers)
        docker_reg_cmd_handler = cmd_docker.DockerCmdHandler()
        docker_reg_cmd_handler.configure_parser(docker_reg_cmd_parser)
        self.cmds[CmdName.DOCKER_REGISTRY] = (docker_reg_cmd_parser, docker_reg_cmd_handler)

        # set docker registry command
        set_docker_reg_cmd_parser = _new_arg_parser(command=CmdName.SET_DOCKER_REG,
                                                    parent_parsers=[debug_parser],
                                                    description='Configure the Docker registry '
                                                                'for the platform')
        set_docker_reg_cmd_handler = rdaf.cmd.setdockerreg.SetDockerRegCmdHandler()
        set_docker_reg_cmd_handler.configure_parser(set_docker_reg_cmd_parser)
        self.cmds[CmdName.SET_DOCKER_REG] = (set_docker_reg_cmd_parser, set_docker_reg_cmd_handler)

        # validate command
        validate_cmd_parser = _new_arg_parser(command=CmdName.VALIDATE,
                                              description='validate rdaf configurations',
                                              parent_parsers=parent_parsers)
        validate_handler = cmd_validate.ValidateCmdHandler()
        validate_handler.configure_parser(validate_cmd_parser)
        self.cmds[CmdName.VALIDATE] = (validate_cmd_parser, validate_handler)

         # logs command
        logs_cmd_parser = _new_arg_parser(command=CmdName.LOGS,
                                          parent_parsers=parent_parsers,
                                          description='tar logs file of all service')
        logs_cmd_handler = cmd_logs.LogsCmdHandler()
        logs_cmd_handler.configure_parser(logs_cmd_parser)
        self.cmds[CmdName.LOGS] = (logs_cmd_parser, logs_cmd_handler)


        # install_rda_edge command
        install_edge_cmd_parser = _new_arg_parser(command=CmdName.INSTALL_RDA_EDGE,
                                                  description='Install RDA Edge deployment',
                                                  parent_parsers=[debug_parser])
        edge_handler = edge_installer.EdgeInstallCmdHandler()
        edge_handler.configure_parser(install_edge_cmd_parser)
        self.cmds[CmdName.INSTALL_RDA_EDGE] = (install_edge_cmd_parser, edge_handler)

        # backup command
        backup_cmd_parser = _new_arg_parser(command=CmdName.BACKUP, parent_parsers=parent_parsers,
                                            description='Backup the RDAF platform')
        backup_cmd_handler = rdaf.cmd.backup.BackupCmdHandler()
        backup_cmd_handler.configure_parser(backup_cmd_parser)
        self.cmds[CmdName.BACKUP] = (backup_cmd_parser, backup_cmd_handler)

        # restore command
        restore_cmd_parser = _new_arg_parser(command=CmdName.RESTORE,
                                             parent_parsers=parent_parsers,
                                             description='Restore the RDAF platform from a '
                                                         'previously backed up state')
        restore_cmd_handler = rdaf.cmd.restore.RestoreCmdHandler()
        restore_cmd_handler.configure_parser(restore_cmd_parser)
        self.cmds[CmdName.RESTORE] = (restore_cmd_parser, restore_cmd_handler)

        # rdac-cli command
        rdac_cli_cmd_parser = _new_arg_parser(command=CmdName.RDACCLI,
                                              parent_parsers=[debug_parser],
                                              description='Install RDAC CLI')
        rdac_cli_handler = cmd_rdac_cli.RDACCmdHandler()
        rdac_cli_handler.configure_parser(rdac_cli_cmd_parser)
        self.cmds[CmdName.RDACCLI] = (rdac_cli_cmd_parser, rdac_cli_handler)

        # geodr command
        geodr_cmd_parser = _new_arg_parser(command=CmdName.GEODR,
                                              parent_parsers=[debug_parser],
                                              description='Geodr Replication')
        geodr_handler = cmd_rdac_geodr.GeoDRCmdHandler()
        geodr_handler.configure_parser(geodr_cmd_parser)
        self.cmds[CmdName.GEODR] = (geodr_cmd_parser, geodr_handler)

        # telegraf command
        telegraf_cmd_parser = _new_arg_parser(command=CmdName.TELEGRAF, parent_parsers=[debug_parser],
                                         description='Telegraf monitoring solution')
        telegraf_cmd_handler = cmd_telegraf.TelegrafCmdHandler()
        telegraf_cmd_handler.configure_parser(telegraf_cmd_parser)
        self.cmds[CmdName.TELEGRAF] = (telegraf_cmd_parser, telegraf_cmd_handler)

        # self_monitoring command
        self_monitoring_cmd_parser = _new_arg_parser(command=CmdName.SELF_MONITORING,
                                                    parent_parsers=[debug_parser],
                                                    description='Manage the RDAF self monitoring')
        self_monitoring_handler = cmd_self_monitoring.SelfMonitoringCmdHandler()
        self_monitoring_handler.configure_parser(self_monitoring_cmd_parser)
        self.cmds[CmdName.SELF_MONITORING] = (self_monitoring_cmd_parser, self_monitoring_handler)


    def emptyline(self):
        return

    def help_status(self):
        self._help_command(CmdName.STATUS)

    def do_status(self, arg):
        self._handle_command(CmdName.STATUS, arg)

    def help_prune_images(self):
        self._help_command(CmdName.PRUNE_IMAGES)

    def do_prune_images(self, arg):
        self._handle_command(CmdName.PRUNE_IMAGES, arg)

    def do_setup(self, arg):
        self._handle_command(CmdName.SETUP, arg)

    def help_setup(self):
        self._help_command(CmdName.SETUP)

    def do_infra(self, arg):
        self._handle_command(CmdName.INFRA, arg)

    def help_infra(self):
        self._help_command(CmdName.INFRA)

    def do_platform(self, arg):
        self._handle_command(CmdName.PLATFORM, arg)

    def help_platform(self):
        self._help_command(CmdName.PLATFORM)

    def do_worker(self, arg):
        self._handle_command(CmdName.WORKER, arg)

    def help_worker(self):
        self._help_command(CmdName.WORKER)

    def do_event_gateway(self, arg):
        self._handle_command(CmdName.EVENT_GATEWAY, arg)

    def help_event_gateway(self):
        self._help_command(CmdName.EVENT_GATEWAY)

    def do_log_monitoring(self, arg):
        self._handle_command(CmdName.LOG_MONITORING, arg)

    def help_log_monitoring(self):
        self._help_command(CmdName.LOG_MONITORING)

    def do_bulk_stats(self, arg):
        self._handle_command(CmdName.BULK_STATS, arg)

    def help_bulk_stats(self):
        self._help_command(CmdName.BULK_STATS)

    def do_file_object(self, arg):
        self._handle_command(CmdName.FILE_OBJECT, arg)

    def help_file_object(self):
        self._help_command(CmdName.FILE_OBJECT)

    def do_opensearch_external(self, arg):
        self._handle_command(CmdName.OPENSEARCH_EXTERNAL, arg)

    def help_opensearch_external(self):
        self._help_command(CmdName.OPENSEARCH_EXTERNAL)

    def do_app(self, arg):
        self._handle_command(CmdName.APPLICATION, arg)

    def help_app(self):
        self._help_command(CmdName.APPLICATION)

    def do_reset(self, arg):
        self._handle_command(CmdName.RESET, arg)

    def help_reset(self):
        self._help_command(CmdName.RESET)

    def do_registry(self, arg):
        self._handle_command(CmdName.DOCKER_REGISTRY, arg)

    def help_registry(self):
        self._help_command(CmdName.DOCKER_REGISTRY)

    def help_setregistry(self):
        self._help_command(CmdName.SET_DOCKER_REG)

    def do_setregistry(self, arg):
        self._handle_command(CmdName.SET_DOCKER_REG, arg)

    def do_validate(self, arg):
        self._handle_command(CmdName.VALIDATE, arg)

    def help_validate(self):
        self._help_command(CmdName.VALIDATE)

    def do_logs(self, arg):
        self._handle_command(CmdName.LOGS, arg)
    
    def help_logs(self):
        self._help_command(CmdName.LOGS)

    def do_install_rda_edge(self, arg):
        self._handle_command(CmdName.INSTALL_RDA_EDGE, arg)

    def help_install_rda_edge(self):
        self._help_command(CmdName.INSTALL_RDA_EDGE)

    def help_backup(self):
        self._help_command(CmdName.BACKUP)

    def do_backup(self, arg):
        self._handle_command(CmdName.BACKUP, arg)

    def do_rdac_cli(self, arg):
        self._handle_command(CmdName.RDACCLI, arg)

    def help_rdac_cli(self):
        self._help_command(CmdName.RDACCLI)

    def do_geodr(self, arg):
        self._handle_command(CmdName.GEODR, arg)

    def help_geodr(self):
        self._help_command(CmdName.GEODR)

    def do_telegraf(self, arg):
        self._handle_command(CmdName.TELEGRAF, arg)

    def help_telegraf(self):
        self._help_command(CmdName.TELEGRAF)

    def help_restore(self):
        self._help_command(CmdName.RESTORE)

    def do_restore(self, arg):
        self._handle_command(CmdName.RESTORE, arg)

    def do_self_monitoring(self, arg):
        self._handle_command(CmdName.SELF_MONITORING, arg)

    def help_self_monitoring(self):
        self._help_command(CmdName.SELF_MONITORING)
        

    def _handle_command(self, command_name: CmdName, arg: str):
        if command_name not in self.cmds:
            raise CliException('Unknown command ' + command_name.value)
        parser, cmd_handler = self.cmds[command_name]
        args = arg.split()
        parsed_args = parser.parse_args(args)
        #consecutive setup prevention logic
        if command_name.value == "setup" and is_rdaf_setup_done():
            message = "RDAF is already set up. To reconfigure, please run 'reset' command first."
            raise CliException(message)
        # handle debug logging argument
        prev_level = logging.getLogger('rdaf').level
        if 'debug' in parsed_args and parsed_args.debug:
            logging.getLogger('rdaf').setLevel('DEBUG')
        if command_name.value not in ['registry', 'telegraf']:
            self._validate_cmd_invoked()
        try:
            config_parser = configparser.ConfigParser(allow_no_value=True)
            cmd_handler.before_handle(parsed_args, config_parser)
            # create a global (requests) session for use in the CLI command
            # handling
            cmd_handler.handle(parsed_args, config_parser)
        except InvalidCmdUsageException:
            #   print usage
            self._help_command(command_name)
        finally:
            # close the global session
            try:
                rdaf.util.requestsutil.close_global_session()
            except Exception:
                # ignore
                pass
            if prev_level is not logging.NOTSET:
                # reset to previous level
                logging.getLogger('rdaf').setLevel(prev_level)

    def _help_command(self, command_name):
        if command_name not in self.cmds:
            raise CliException('Unknown command ' + command_name.value)
        parser, cmd_handler = self.cmds[command_name]
        print(parser.format_help())

    @staticmethod
    def _validate_cmd_invoked():
        config_file_path = os.path.expanduser(os.path.join('/opt', 'rdaf', 'rdaf.cfg'))
        if not os.path.isfile(config_file_path):
            return
        config_parser = configparser.ConfigParser(allow_no_value=True)
        with open(config_file_path, 'r') as f:
            config_parser.read_file(f)

        if (config_parser.has_section('rdaf-cli') and
                config_parser.get('rdaf-cli', 'namespace', fallback=None)):
            message = "This is a K8s deployment, please use 'rdafk8s'."
            sys.stdout.write(message + '\n')
            sys.exit(1)


class CliK8S(cmd.Cmd):
    def __init__(self):
        super().__init__()
        # a dictionary of commands keyed to the CmdName. Value is a tuple of
        # (command parser, command handler)
        self.cmds = dict()
        self._create_commands()

    def _create_commands(self):
        parent_parsers = [debug_parser]

        ssh_user_parser = argparse.ArgumentParser(add_help=False)
        ssh_user_parser.add_argument('--ssh-user',
                                     dest="ssh_user",
                                     action='store',
                                     default=_current_user(),
                                     help='User name to use to do an initial SSH login'
                                          ' while setting up password-less SSH between hosts')
        ssh_user_parser.add_argument('--ssh-password',
                                     dest="ssh_password",
                                     action='store',
                                     default=None,
                                     help='Password to use to do an initial SSH login'
                                          ' while setting up password-less SSH between hosts')

        ssh_key_parser = argparse.ArgumentParser(add_help=False)

        ssh_key_parser.add_argument('--passphrase',
                                    action='store',
                                    default="",
                                    dest="passphrase",
                                    help='SSH Key Pass Phrase')
        # setup command
        setup_cmd_parser = _new_arg_parser(command=CmdName.SETUP,
                                           description='Installs and sets up RDAF',
                                           parent_parsers=[debug_parser, ssh_key_parser,
                                                           ssh_user_parser])
        setup_cmd_handler = setup_cmd.SetupK8SCmdHandler()
        setup_cmd_handler.configure_parser(setup_cmd_parser)
        self.cmds[CmdName.SETUP] = (setup_cmd_parser, setup_cmd_handler)

        # prune_images command
        prune_images_cmd_parser = _new_arg_parser(command=CmdName.PRUNE_IMAGES,
                                                  description='Prune unused images from all hosts',
                                                  parent_parsers=[debug_parser])
        prune_images_cmd_handler = prune_images.K8SPruneImageCmdHandler()
        prune_images_cmd_handler.configure_parser(prune_images_cmd_parser)
        self.cmds[CmdName.PRUNE_IMAGES] = (prune_images_cmd_parser, prune_images_cmd_handler)

        # infra command
        infra_cmd_parser = _new_arg_parser(command=CmdName.INFRA,
                                           description='Manage infra services',
                                           parent_parsers=parent_parsers)
        infra_cmd_handler = cmd_infra.K8SInfraCmdHandler()
        infra_cmd_handler.configure_parser(infra_cmd_parser)
        self.cmds[CmdName.INFRA] = (infra_cmd_parser, infra_cmd_handler)

        # reset command
        reset_cmd_parser = _new_arg_parser(command=CmdName.RESET,
                                           description='Reset the RDAF installation',
                                           parent_parsers=parent_parsers)
        reset_cmd_handler = reset.ResetCmdHandler(k8s=True)
        reset_cmd_handler.configure_parser(reset_cmd_parser)
        self.cmds[CmdName.RESET] = (reset_cmd_parser, reset_cmd_handler)

        # status command
        status_cmd_parser = _new_arg_parser(command=CmdName.STATUS,
                                            description='Manages status of infra and platform',
                                            parent_parsers=parent_parsers)
        status_cmd_handler = cmd_status.StatusCmdHandler(k8s=True)
        status_cmd_handler.configure_parser(status_cmd_parser)
        self.cmds[CmdName.STATUS] = (status_cmd_parser, status_cmd_handler)

        # platform command
        platform_cmd_parser = _new_arg_parser(command=CmdName.PLATFORM,
                                              parent_parsers=[debug_parser],
                                              description='Manage the RDAF Platform')
        platform_handler = cmd_platform.K8SPlatformCmdHandler()
        platform_handler.configure_parser(platform_cmd_parser)
        self.cmds[CmdName.PLATFORM] = (platform_cmd_parser, platform_handler)

        # log_monitoring command
        log_monitoring_cmd_parser = _new_arg_parser(command=CmdName.LOG_MONITORING,
                                                    parent_parsers=[debug_parser],
                                                    description='Manage the RDAF log monitoring')
        log_monitoring_handler = cmd_log_monitoring.K8SLogMonitoringCmdHandler()
        log_monitoring_handler.configure_parser(log_monitoring_cmd_parser)
        self.cmds[CmdName.LOG_MONITORING] = (log_monitoring_cmd_parser, log_monitoring_handler)

         # bulk_stats command
        bulk_stats_cmd_parser = _new_arg_parser(command=CmdName.BULK_STATS,
                                                    parent_parsers=[debug_parser],
                                                    description='Manage the RDAF bulk_stats')
        bulk_stats_handler = cmd_bulk_stats.K8SBulkStatsCmdHandler()
        bulk_stats_handler.configure_parser(bulk_stats_cmd_parser)
        self.cmds[CmdName.BULK_STATS] = (bulk_stats_cmd_parser, bulk_stats_handler)

        # opensearch_external command
        opensearch_external_cmd_parser = _new_arg_parser(command=CmdName.OPENSEARCH_EXTERNAL,
                                                    parent_parsers=[debug_parser],
                                                    description='Manage the Opensearch External')
        opensearch_external_handler = cmd_opensearch_external.K8Sopensearch_externalCmdHandler()
        opensearch_external_handler.configure_parser(opensearch_external_cmd_parser)
        self.cmds[CmdName.OPENSEARCH_EXTERNAL] = (opensearch_external_cmd_parser, opensearch_external_handler)

        # worker command
        worker_cmd_parser = _new_arg_parser(command=CmdName.WORKER,
                                            parent_parsers=[debug_parser],
                                            description='Manage the RDAF Worker')
        worker_handler = cmd_worker.K8SWorkerCmdHandler()
        worker_handler.configure_parser(worker_cmd_parser)
        self.cmds[CmdName.WORKER] = (worker_cmd_parser, worker_handler)

        # event_gateway command
        event_gateway_cmd_parser = _new_arg_parser(command=CmdName.EVENT_GATEWAY,
                                            parent_parsers=[debug_parser],
                                            description='Manage the RDAF Event Gateway')
        event_gateway_handler = cmd_event_gateway.K8SEventGatewayCmdHandler()
        event_gateway_handler.configure_parser(event_gateway_cmd_parser)
        self.cmds[CmdName.EVENT_GATEWAY] = (event_gateway_cmd_parser, event_gateway_handler)

        # app command
        app_cmd_parser = _new_arg_parser(command=CmdName.APPLICATION,
                                         parent_parsers=[debug_parser],
                                         description='Manage the RDAF Apps')
        app_handler = cmd_app.K8SApplicationCmdHandler()
        app_handler.configure_parser(app_cmd_parser)
        self.cmds[CmdName.APPLICATION] = (app_cmd_parser, app_handler)

        # set docker registry command
        set_docker_reg_cmd_parser = _new_arg_parser(command=CmdName.SET_DOCKER_REG,
                                                    parent_parsers=[debug_parser],
                                                    description='Configure the Docker registry '
                                                                'for the platform')
        set_docker_reg_cmd_handler = rdaf.cmd.setdockerreg.SetDockerRegCmdHandler()
        set_docker_reg_cmd_handler.configure_parser(set_docker_reg_cmd_parser)
        self.cmds[CmdName.SET_DOCKER_REG] = (set_docker_reg_cmd_parser, set_docker_reg_cmd_handler)

        # rdac-cli command
        rdac_cli_cmd_parser = _new_arg_parser(command=CmdName.RDACCLI,
                                              parent_parsers=[debug_parser],
                                              description='Install RDAC CLI')
        rdac_cli_handler = cmd_rdac_cli.RDACCmdHandler()
        rdac_cli_handler.configure_parser(rdac_cli_cmd_parser)
        self.cmds[CmdName.RDACCLI] = (rdac_cli_cmd_parser, rdac_cli_handler)

        # maintenance command
        maintenance_cmd_parser = _new_arg_parser(command=CmdName.MAINTENANCE,
                                              description='Start Maintenance Mode',
                                              parent_parsers=parent_parsers)
        maintenance_handler = cmd_maintenance.MaintenanceCmdHandler()
        maintenance_handler.configure_parser(maintenance_cmd_parser)
        self.cmds[CmdName.MAINTENANCE] = (maintenance_cmd_parser, maintenance_handler)

        # validate command
        validate_cmd_parser = _new_arg_parser(command=CmdName.VALIDATE,
                                              description='validate rdaf configurations',
                                              parent_parsers=parent_parsers)
        validate_handler = cmd_validate.K8sValidateCmdHandler()
        validate_handler.configure_parser(validate_cmd_parser)
        self.cmds[CmdName.VALIDATE] = (validate_cmd_parser, validate_handler)

        # backup command
        backup_cmd_parser = _new_arg_parser(command=CmdName.BACKUP, parent_parsers=parent_parsers,
                                            description='Backup the RDAF platform')
        backup_cmd_handler = rdaf.cmd.backup.K8SBackupCmdHandler()
        backup_cmd_handler.configure_parser(backup_cmd_parser)
        self.cmds[CmdName.BACKUP] = (backup_cmd_parser, backup_cmd_handler)

        # restore command
        restore_cmd_parser = _new_arg_parser(command=CmdName.RESTORE,
                                             parent_parsers=parent_parsers,
                                             description='Restore the RDAF platform from a '
                                                         'previously backed up state')
        restore_cmd_handler = rdaf.cmd.restore.K8SRestoreCmdHandler()
        restore_cmd_handler.configure_parser(restore_cmd_parser)
        self.cmds[CmdName.RESTORE] = (restore_cmd_parser, restore_cmd_handler)

        # self_monitoring command
        self_monitoring_cmd_parser = _new_arg_parser(command=CmdName.SELF_MONITORING,
                                                    parent_parsers=[debug_parser],
                                                    description='Manage the RDAF self monitoring')
        self_monitoring_handler = cmd_self_monitoring.K8sSelfMonitoringCmdHandler()
        self_monitoring_handler.configure_parser(self_monitoring_cmd_parser)
        self.cmds[CmdName.SELF_MONITORING] = (self_monitoring_cmd_parser, self_monitoring_handler)

    def emptyline(self):
        return

    def help_status(self):
        self._help_command(CmdName.STATUS)

    def do_status(self, arg):
        self._handle_command(CmdName.STATUS, arg)

    def help_prune_images(self):
        self._help_command(CmdName.PRUNE_IMAGES)

    def do_prune_images(self, arg):
        self._handle_command(CmdName.PRUNE_IMAGES, arg)

    def do_setup(self, arg):
        self._handle_command(CmdName.SETUP, arg)

    def help_setup(self):
        self._help_command(CmdName.SETUP)

    def do_infra(self, arg):
        self._handle_command(CmdName.INFRA, arg)

    def help_infra(self):
        self._help_command(CmdName.INFRA)

    def do_platform(self, arg):
        self._handle_command(CmdName.PLATFORM, arg)

    def help_platform(self):
        self._help_command(CmdName.PLATFORM)

    def do_worker(self, arg):
        self._handle_command(CmdName.WORKER, arg)

    def help_worker(self):
        self._help_command(CmdName.WORKER)

    def do_event_gateway(self, arg):
        self._handle_command(CmdName.EVENT_GATEWAY, arg)

    def help_event_gateway(self):
        self._help_command(CmdName.EVENT_GATEWAY)

    def do_log_monitoring(self, arg):
        self._handle_command(CmdName.LOG_MONITORING, arg)

    def help_log_monitoring(self):
        self._help_command(CmdName.LOG_MONITORING)

    def do_bulk_stats(self, arg):
        self._handle_command(CmdName.BULK_STATS, arg)

    def help_bulk_stats(self):
        self._help_command(CmdName.BULK_STATS)

    def do_opensearch_external(self, arg):
        self._handle_command(CmdName.OPENSEARCH_EXTERNAL, arg)

    def help_opensearch_external(self):
        self._help_command(CmdName.OPENSEARCH_EXTERNAL)

    def do_app(self, arg):
        self._handle_command(CmdName.APPLICATION, arg)

    def help_app(self):
        self._help_command(CmdName.APPLICATION)

    def do_reset(self, arg):
        self._handle_command(CmdName.RESET, arg)

    def help_reset(self):
        self._help_command(CmdName.RESET)

    # def do_registry(self, arg):
    #     self._handle_command(CmdName.DOCKER_REGISTRY, arg)
    #
    # def help_registry(self):
    #     self._help_command(CmdName.DOCKER_REGISTRY)

    def help_setregistry(self):
        self._help_command(CmdName.SET_DOCKER_REG)

    def do_setregistry(self, arg):
        self._handle_command(CmdName.SET_DOCKER_REG, arg)

    def do_rdac_cli(self, arg):
        self._handle_command(CmdName.RDACCLI, arg)

    def help_rdac_cli(self):
        self._help_command(CmdName.RDACCLI)

    def do_maintenance(self, arg):
        self._handle_command(CmdName.MAINTENANCE, arg)

    def help_maintenance(self):
        self._help_command(CmdName.MAINTENANCE)

    def do_validate(self, arg):
        self._handle_command(CmdName.VALIDATE, arg)

    def help_validate(self):
        self._help_command(CmdName.VALIDATE)

    def help_backup(self):
        self._help_command(CmdName.BACKUP)

    def do_backup(self, arg):
        self._handle_command(CmdName.BACKUP, arg)

    def help_restore(self):
        self._help_command(CmdName.RESTORE)

    def do_restore(self, arg):
        self._handle_command(CmdName.RESTORE, arg)

    def do_self_monitoring(self, arg):
        self._handle_command(CmdName.SELF_MONITORING, arg)

    def help_self_monitoring(self):
        self._help_command(CmdName.SELF_MONITORING)

    @staticmethod
    def _validate_cmd_invoked():
        config_file_path = os.path.expanduser(os.path.join('/opt', 'rdaf', 'rdaf.cfg'))
        if not os.path.isfile(config_file_path):
            return
        config_parser = configparser.ConfigParser(allow_no_value=True)
        with open(config_file_path, 'r') as f:
            config_parser.read_file(f)

        if config_parser.has_section('rdaf-cli'):
            # Check the 'deployment' key
            deployment_type = config_parser.get('rdaf-cli', 'deployment', fallback=None)
            if deployment_type == "non-k8s":
                sys.stdout.write("This is a non-kubernetes deployment. Please use 'rdaf'.\n")
                sys.exit(1)

    def _handle_command(self, command_name: CmdName, arg: str):
        self._validate_cmd_invoked()
        if command_name not in self.cmds:
            raise CliException('Unknown command ' + command_name.value)
        parser, cmd_handler = self.cmds[command_name]
        args = arg.split()
        parsed_args = parser.parse_args(args)
        #consecutive setup prevention logic
        if command_name.value == "setup" and is_rdaf_setup_done():
            message = "RDAF is already set up. To reconfigure, please run 'reset' command first."
            raise CliException(message)
        # handle debug logging argument
        prev_level = logging.getLogger('rdaf').level
        if 'debug' in parsed_args and parsed_args.debug:
            logging.getLogger('rdaf').setLevel('DEBUG')
        try:
            config_parser = configparser.ConfigParser(allow_no_value=True)
            cmd_handler.before_handle(parsed_args, config_parser)
            # create a global (requests) session for use in the CLI command
            # handling
            cmd_handler.handle(parsed_args, config_parser)
        except InvalidCmdUsageException:
            # print usage
            self._help_command(command_name)
        finally:
            # close the global session
            try:
                rdaf.util.requestsutil.close_global_session()
            except Exception:
                # ignore
                pass
            if prev_level is not logging.NOTSET:
                # reset to previous level
                logging.getLogger('rdaf').setLevel(prev_level)

    def _help_command(self, command_name):
        if command_name not in self.cmds:
            raise CliException('Unknown command ' + command_name.value)
        parser, cmd_handler = self.cmds[command_name]
        print(parser.format_help())


def _new_arg_parser(command: CmdName = None, description: str = None, parent_parsers: list = []) \
        -> argparse.ArgumentParser:
    return argparse.ArgumentParser(conflict_handler='resolve', parents=parent_parsers,
                                   prog=command.value,
                                   description=description)


def _init_logging():
    log_config_file = os.path.join(os.path.dirname(os.path.abspath(__file__)), "logrdaf.ini")
    if not os.path.exists(log_config_file):
        return
    # create the cli log dir
    os.makedirs(os.path.expanduser('~/.rdafcli/'), exist_ok=True)
    logging.config.fileConfig(log_config_file)


def main(k8s=False):
    # setup logging of CLI
    _init_logging()
    # get the command line
    command_line = sys.argv[1:]
    # TODO: temporary hack - see if we can do this in a better way
    if len(command_line) == 1:
        if command_line[0] == '--version' or command_line[0] == '-v':
            version = 'RDAF CLI version: ' + rdafutils.get_cli_version()
            sys.stdout.write(version + '\n')
            sys.exit(0)
        elif command_line[0] == '--help' or command_line[0] == '-h':
            command_line = ['help']
    if k8s:
        cli = CliK8S()
    else:
        cli = Cli()
    # parse the one shot command
    try:
        cli.onecmd(' '.join(command_line))
    except CliException as e:
        if e.message is None:
            raise
        print(termcolor.colored(e.message, color='red'))
        sys.exit(1)


def k8s_main():
    main(k8s=True)


if __name__ == '__main__':
    main()
