#!/bin/bash

# Expected container name prefixes
declare -a expected=(
  "infra-graphdb-1-coordinator"
  "infra-graphdb-1-dbserver"
  "infra-graphdb-1-agent"
  "infra-graphdb-1"   # starter
)

check_all_running() {
    for prefix in "${expected[@]}"; do
        if ! docker ps --format '{{.Names}}' | grep -q "^$prefix"; then
            return 1
        fi
    done
    return 0
}

while true; do
    starter_status=$(docker inspect -f '{{.State.Status}}' infra-graphdb-1 2>/dev/null || echo "missing")

    if [[ "$starter_status" == "exited" || "$starter_status" == "missing" ]]; then
        echo "Starter (infra-graphdb-1) is not running. Doing nothing..."
    elif [[ "$starter_status" == "running" ]]; then
        echo "Starter is running. Checking other containers..."

        for prefix in "${expected[@]}"; do
            if [[ "$prefix" == "infra-graphdb-1" ]]; then
                continue  # skip starter
            fi

            cname=$(docker ps -a --format '{{.Names}}' | grep "^$prefix" | head -n1)

            if [[ -z "$cname" ]]; then
                echo "No container found with prefix $prefix"
                echo "Restarting starter to reprovision missing container..."
                docker restart infra-graphdb-1
                sleep 20
                continue
            fi

            status=$(docker inspect -f '{{.State.Status}}' "$cname" 2>/dev/null || echo "missing")

            if [[ "$status" == "exited" ]]; then
                echo "$cname is exited. Removing..."
                docker rm -f "$cname"

                echo "Waiting 10s for reprovision..."
                sleep 10

                # Check if it came back
                if ! docker ps --format '{{.Names}}' | grep -q "^$prefix"; then
                    echo "$prefix not reprovisioned. Restarting starter..."
                    docker restart infra-graphdb-1
                else
                    echo "$prefix reprovisioned by starter"
                fi
            fi
        done
    fi

    if check_all_running; then
        echo "All containers are running!"
        exit 0
    else
        echo "Waiting... not all containers are up yet."
    fi

    sleep 10
done