import contextvars
import logging
import os
from urllib.parse import urlparse, quote, ParseResult, urlunparse

from requests import Session
from requests.adapters import HTTPAdapter

import rdaf.component

logger = logging.getLogger(__name__)

_global_session = contextvars.ContextVar('rdaf.requestutil.global_session', default=None)

COMPONENT_REGISTRY = rdaf.component.Registry()


def global_session():
    """
    Returns a Session object which is active throughout the lifetime of the CLI process
    """
    current_global = _global_session.get(None)
    if current_global is not None:
        return current_global
    raise RuntimeError('No global session is available')


def create_global_session():
    current_global = _global_session.get(None)
    if current_global is not None:
        raise RuntimeError('A global session already exists')
    s = _GlobalSession()
    import rdaf.component.cert
    cert_manager = COMPONENT_REGISTRY.get(rdaf.component.cert.CertManager.COMPONENT_NAME)
    if cert_manager is not None:
        ca_cert_file = cert_manager.get_ca_cert_file()
        if os.path.isfile(ca_cert_file):
            s.verify = ca_cert_file
    _global_session.set(s)
    return s


def close_global_session():
    current_global: Session = _global_session.get(None)
    if current_global is None:
        # don't raise an error, just return
        return
    current_global.close()


def new_session():
    """
    Creates and returns a new Session
    """
    s = _CustomSession()
    import rdaf.component.cert
    cert_manager = COMPONENT_REGISTRY.get(rdaf.component.cert.CertManager.COMPONENT_NAME)
    if cert_manager is not None:
        ca_cert_file = cert_manager.get_ca_cert_file()
        if os.path.isfile(ca_cert_file):
            s.verify = ca_cert_file
    return s


class _CustomSession(Session):

    def __init__(self) -> None:
        super().__init__()
        # we use our own RDAF specific adapters here, to
        # add some specific proxy handling logic
        self.mount('https://', _RDAFHTTPAdapter())
        self.mount('http://', _RDAFHTTPAdapter())


class _GlobalSession(_CustomSession):
    def __init__(self):
        super().__init__()

    def __exit__(self, *args):
        # we don't close a global session when used in "with" statement
        return


class _RDAFHTTPAdapter(HTTPAdapter):

    def proxy_manager_for(self, proxy, **proxy_kwargs):
        # encode the proxy url if necessary
        possibly_encoded_proxy_url = _safe_encode_proxy_url(proxy)
        return super().proxy_manager_for(possibly_encoded_proxy_url, **proxy_kwargs)


def _safe_encode_proxy_url(proxy_url):
    r"""
    A best-effort based implementation which parses the raw proxy_url string
    and encodes any components of these proxy_url.
    This is necessary for cases where the standard environment variables
    'http_proxy' and 'https_proxy' are sometimes of the form:
    http://user:pass@host:port and the 'user' value in that string happens
    to be of the form 'foo\bar' (i.e. Windows domain user representation).
    The urllib3 implementation (in its 'urllib3.util.url#parse_url) ends up
    parsing such URLs incorrectly and as a result returns incorrect host
    and other component values. We prevent this issue by converting such
    proxy_url (before they get sent to that function) to the form
    http://foo%5Cbar:pass@host:port
    More details at https://github.com/urllib3/urllib3/issues/1963
    """
    if proxy_url is None:
        return None
    try:
        result = urlparse(proxy_url)
        netloc = result.netloc
        if '@' not in netloc:
            return proxy_url
        username = ''
        password = ''
        if result.username is not None:
            username = quote(result.username)
        if result.password is not None:
            password = quote(result.password)
        idx = netloc.index('@')
        no_user_pass_netloc = netloc[idx + 1:]
        escaped_netloc = username + ':' + password + '@' + no_user_pass_netloc
        # recreate the url with this new netloc and the rest of the
        # components as-is
        new_components = ParseResult(result.scheme, escaped_netloc,
                                     result.path, result.params,
                                     result.query, result.fragment)
        return str(urlunparse(new_components))
    except Exception:
        logger.debug('Failed to parse proxy url %s', proxy_url, exc_info=1)
        return proxy_url
