import argparse
import configparser
import logging
import os

import rdaf
from rdaf import InvalidCmdUsageException, rdafutils
from rdaf.cmd import CliCmdHandler, status
from rdaf.component import _comma_delimited_to_list, Component
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.component import dockerregistry, ssh, bulk_stats

logger = logging.getLogger(__name__)


class BulkStatsCmdHandler(CliCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = False

    def configure_parser(self, parser):
        bulk_stats_commands_parser = parser.add_subparsers(dest='bulk_stats_op', help='commands')

        upgrade_parser = bulk_stats_commands_parser.add_parser('upgrade',
                                                                   help='Upgrade bulk_stats components')
        upgrade_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='Docker image tag')
        
        add_bulk_stats_parser = bulk_stats_commands_parser.add_parser('add-bulk-stats-host',
                                                                   help='Add extra bulk-stats vm')
        add_bulk_stats_parser.add_argument('host', help='Host name or IP address of the ' 
                                                         'host which needs to be ' 
                                                         'added as a bulk-stats host')
        add_bulk_stats_parser.add_argument('--ssh-password', dest='ssh_password', required=True,
                                            action='store',
                                            help='Password for the SSH user for initial' 
                                                 ' login to this host')
        add_bulk_stats_parser.add_argument('--tag',
                                            dest='tag',
                                            action='store',
                                            required=True,
                                            help='Docker image tag')

        install_parser = bulk_stats_commands_parser.add_parser('install',
                                                                   help='Install bulk_stats components')
        install_parser.add_argument('--host',
                                    dest='bulk_stats_host',
                                    action='store',
                                    required=True,
                                    help='bulk_stats host')

        install_parser.add_argument('--ssh-password', dest='ssh_password', required=True,
                                            action='store',
                                            help='Password for the SSH user for initial'
                                                 ' login to this host')
        install_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='bulk_stats image tag')
        install_parser.add_argument('--no-prompt',
                                    dest='no_prompt',
                                    action='store_true',
                                    default=False,
                                    help='Don\'t prompt for inputs')

        status_parser = bulk_stats_commands_parser.add_parser('status', help='Status of the RDAF bulk_stats')

        up_parser = bulk_stats_commands_parser.add_parser('up',
                                                              help='Create the RDAF bulk_stats Containers')

        down_parser = bulk_stats_commands_parser.add_parser('down',
                                                                help='Delete the RDAF bulk_stats Containers')
        
        down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                                                help='Don\'t prompt for inputs')

        start_parser = bulk_stats_commands_parser.add_parser('start',
                                                                 help='Start the RDAF bulk_stats Containers')

        stop_parser = bulk_stats_commands_parser.add_parser('stop',
                                                                help='Stop the RDAF bulk_stats Containers')
        
        stop_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                                                help='Don\'t prompt for inputs')

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.bulk_stats_op
        component = COMPONENT_REGISTRY.get('rda_bulk_stats')
        if cmd == 'install':
            network_config = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'config.json')
            if not os.path.exists(network_config):
                rdafutils.cli_err_exit('config.json is missing.. please install rdaf platform')
            if component.configs['host']:
                rdafutils.cli_err_exit('bulk stats service is already installed.')
            bulk_stats_hosts = cmd_args.bulk_stats_host
            all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
                skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME]
            )
            for host in _comma_delimited_to_list(bulk_stats_hosts):
                if host not in all_known_hosts:
                    ssh_manager = COMPONENT_REGISTRY.require(
                        rdaf.component.ssh.SSHKeyManager.COMPONENT_NAME)
                    # setup ssh keys for this new host
                    ssh_manager.setup_keys_for_host(host, cmd_args.ssh_password)
                # doing a docker login
                docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
                docker_registry.docker_login(host, config_parser)
            if self.k8s:
                component.k8s_pull_images(cmd_args, config_parser)
                component.k8s_install(cmd_args, config_parser)
            else:
                component.pull_images(cmd_args, config_parser)
                component.configs['host'] = _comma_delimited_to_list(bulk_stats_hosts)
                component._mark_configured(component.configs, config_parser)
                component.write_configs(config_parser)
                component.install(cmd_args, config_parser)

        elif cmd == 'add-bulk-stats-host':
            self.add_bulk_stats_host(cmd_args, config_parser)
            return
        
        elif cmd == 'upgrade':
            if self.k8s:
                component.k8s_pull_images(cmd_args, config_parser)
                component.k8s_upgrade(cmd_args, config_parser)
            else:
                component.pull_images(cmd_args, config_parser)
                component.upgrade(cmd_args, config_parser)
        elif cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF bulk_stats container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the bulk_stats container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('bulk_stats down operation has been cancelled')
                    return
            if self.k8s:
                component.k8s_down(cmd_args, config_parser)
            else:
                component.down(cmd_args, config_parser)
        elif cmd == 'up':
            if self.k8s:
               component.k8s_up(cmd_args, config_parser)
            else: 
                component.up(cmd_args, config_parser)
        elif cmd == 'status':
            return status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, components=[component])
        elif cmd == 'stop':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Stopping of RDAF bulk_stats container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to stop the bulk_stats container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('bulk_stats stop operation has been cancelled')
                    return
            component.stop(cmd_args, config_parser)
        elif cmd == 'start':
            component.start(cmd_args, config_parser)
        else:
            raise InvalidCmdUsageException()
        
    def add_bulk_stats_host(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        hosts = [host.strip() for host in cmd_args.host.split(',')]
        bulk_stats_comp = COMPONENT_REGISTRY.require(rdaf.component.bulk_stats.COMPONENT_NAME)
        existing_bulk_stats_hosts = bulk_stats_comp.get_hosts()
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        
        for host in hosts:
            rdafutils.validate_host_name(host)
            if host in existing_bulk_stats_hosts:
                logger.warning(f"{host} is already configured as a bulk-stats host.")
                continue
            if host not in all_known_hosts:
                ssh_manager = COMPONENT_REGISTRY.require(rdaf.component.ssh.SSHKeyManager.COMPONENT_NAME)
                ssh_manager.setup_keys_for_host(host, cmd_args.ssh_password)
            bulk_stats_comp.add_bulk_stats_host(host, cmd_args, config_parser)
        
        # Write out the configuration changes to rdaf.cfg
        Component.write_configs(config_parser)

class K8SBulkStatsCmdHandler(BulkStatsCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = True

    def configure_parser(self, parser):
        bulk_stats_commands_parser = parser.add_subparsers(dest='bulk_stats_op', help='commands')

        upgrade_parser = bulk_stats_commands_parser.add_parser('upgrade',
                                                                   help='Upgrade bulk_stats components')
        upgrade_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='Docker image tag')

        install_parser = bulk_stats_commands_parser.add_parser('install',
                                                                   help='Install bulk_stats components')
        install_parser.add_argument('--host',
                                    dest='bulk_stats_host',
                                    action='store',
                                    required=True,
                                    help='bulk_stats host')
        install_parser.add_argument('--ssh-password',
                                    dest="ssh_password",
                                    action='store',
                                    required=True,
                                    help='Password to use to do an initial SSH login'
                                         ' while setting up password-less SSH between hosts')
        install_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='bulk_stats image tag')

        status_parser = bulk_stats_commands_parser.add_parser('status', help='Status of the RDAF bulk_stats')

        up_parser = bulk_stats_commands_parser.add_parser('up',
                                                              help='Create the RDAF bulk_stats Containers')
        
        down_parser = bulk_stats_commands_parser.add_parser('down',
                                                                help='Delete the RDAF bulk_stats Containers')
        
        down_parser.add_argument('--force', action='store_true',
                                help='Delete the RDAF bulk_stats Containers Forcefully')
        
        down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                                                help='Don\'t prompt for inputs')