import argparse
import configparser
import logging.config
import os

import rdaf.component.proxy
import rdaf.component.dockerregistrymirror
from rdaf import rdafutils, InvalidCmdUsageException
from rdaf.cmd import CliCmdHandler, _read_configs
from rdaf.component import run_command, Component, execute_command
from rdaf.contextual import COMPONENT_REGISTRY

logger = logging.getLogger(__name__)


def get_config_file_path() -> os.path:
    return os.path.join('/opt/rdaf-registry/', 'rdaf-registry.cfg')


def get_install_root(cmd_args: argparse.Namespace) -> str:
    if hasattr(cmd_args, 'install_root') and cmd_args.install_root is not None:
        return cmd_args.install_root
    return '/opt/rdaf-registry/'


class DockerCmdHandler(CliCmdHandler):

    def configure_parser(self, parser):
        install_root_parser = argparse.ArgumentParser(add_help=False)
        install_root_parser.add_argument('--install-root',
                                         dest='install_root',
                                         action='store',
                                         required=False,
                                         help='Path to a directory where the Docker '
                                              'registry will be installed and managed')

        docker_commands_parser = parser.add_subparsers(dest='docker_cmd')
        setup_parser = docker_commands_parser.add_parser('setup',
                                                         help='Setup Docker Registry',
                                                         parents=[install_root_parser])
        setup_parser.add_argument('--docker-server-host',
                                  dest='docker_server_host',
                                  action='append',
                                  default=None,
                                  help='Host name or IP address of the host where the'
                                       ' Docker registry will be installed')
        setup_parser.add_argument('--docker-registry-source-host',
                                  dest='docker_source_host',
                                  action='store',
                                  default=None,
                                  help='The hostname/IP of the source docker registry')
        setup_parser.add_argument('--docker-registry-source-port',
                                  dest='docker_source_port',
                                  action='store',
                                  default='',
                                  help='port of the docker registry')
        setup_parser.add_argument('--docker-registry-source-project',
                                  dest='docker_source_project',
                                  action='store',
                                  default='',
                                  help='project of the docker registry')
        setup_parser.add_argument('--docker-registry-source-user',
                                  dest='docker_source_user',
                                  action='store',
                                  default='',
                                  help='The username to use while connecting to the '
                                       'source docker registry')
        setup_parser.add_argument('--docker-registry-source-password',
                                  dest='docker_source_password',
                                  action='store',
                                  default='',
                                  help='The password to use while connecting to the '
                                       'source docker registry')
        setup_parser.add_argument('--no-prompt',
                                  dest='no_prompt',
                                  action='store_true',
                                  default=False,
                                  help='Don\'t prompt for inputs')

        setup_parser.add_argument('--offline',
                                  dest='offline',
                                  action='store_true',
                                  default=False,
                                  help='Will ignore pulling image and docker login')

        upgrade_parser = docker_commands_parser.add_parser('upgrade',
                                                           help='Upgrade Registry locally')
        upgrade_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='Docker image tag')

        install_parser = docker_commands_parser.add_parser('install',
                                                           help='Install Registry locally')
        install_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='Docker image tag')
        install_parser.add_argument('--no-prompt',
                                    dest='no_prompt',
                                    action='store_true',
                                    default=False,
                                    help='Don\'t prompt for inputs')

        image_parser = argparse.ArgumentParser(add_help=False)
        image_parser.add_argument('--image',
                                  dest="images",
                                  action='append',
                                  default=None,
                                  help='Restrict the scope of the command to a particular image')

        fetch_parser = docker_commands_parser.add_parser('fetch', parents=[image_parser],help='Fetch from configured '
                                                                       'Docker registries')

        group = fetch_parser.add_mutually_exclusive_group(required=True)
        group.add_argument('--tag',
                                  dest='tag',
                                  action='store',
                                  required=False,
                                  help='Comma separated list of tag names')

        group.add_argument('--minio-tag', dest='minio_tag',
                                  action='store',
                                  required=False,
                                  help='Fetch minio tags from quay.io')

        delete_parser = docker_commands_parser.add_parser('delete-images',
                                                          help="Deletes tag(s) and corresponding "
                                                               "docker images")
        delete_parser.add_argument('--tag',
                                   dest='tag',
                                   action='store',
                                   required=True,
                                   help='Comma separated list of tag names')

        import_parser = docker_commands_parser.add_parser('import',
                                                          help="Import images(s) provided as tar file")
        import_parser.add_argument('--file',
                                   dest='file',
                                   action='store',
                                   required=True,
                                   help='File Name that needs to be imported')

        list_tag_parser = docker_commands_parser.add_parser('list-tags',
                                                            help='Lists all the tags for all images in '
                                                                 'the docker registry')

        reset = docker_commands_parser.add_parser('reset', help='Reset the docker registry deployment')

    def before_handle(self, cmd_args: argparse.Namespace,
                      config_parser: configparser.ConfigParser):
        # skip if we are in "rdaf setup" command
        if hasattr(cmd_args, 'docker_cmd') and cmd_args.docker_cmd == 'setup':
            return
        # read the configuration file into the configparser
        cfg_file = get_config_file_path()
        logger.debug('Using configuration file ' + cfg_file)
        if not os.path.isfile(cfg_file):
            rdafutils.cli_err_exit('Missing ' + cfg_file
                                   + '. Please run "rdaf registry setup"')
        _read_configs(config_parser, cfg_file=cfg_file)
        # load components
        components = [rdaf.component.proxy.Proxy(),
                      rdaf.component.dockerregistrymirror.DockerRegistryMirror()]
        # register each of these components
        for component in components:
            rdaf.contextual.COMPONENT_REGISTRY.register(component)
        # let all the components load their configs
        for component in components:
            component.load_config(config_parser)
        return

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        docker_cmd = cmd_args.docker_cmd
        if docker_cmd == 'setup':
            self.setup(cmd_args, config_parser)
            return
        docker_reg_mirror = COMPONENT_REGISTRY.require(rdaf.component.dockerregistrymirror.COMPONENT_NAME)
        if docker_cmd == 'list-tags':
            docker_reg_mirror.list_tags()
        elif docker_cmd == 'import':
            docker_reg_mirror.import_file(cmd_args, config_parser)
        elif docker_cmd == 'install':
            logger.info('Installing ' + docker_reg_mirror.get_name())
            docker_reg_mirror.install(cmd_args, config_parser)
        elif docker_cmd == 'upgrade':
            logger.info('Upgrading ' + docker_reg_mirror.get_name())
            docker_reg_mirror.upgrade(cmd_args, config_parser)
        elif docker_cmd == 'fetch':
            tag_type = 'default'
            if cmd_args.minio_tag:
                tag_type = 'minio'
                tags = rdafutils.delimited_to_list(cmd_args.minio_tag)
            else:
                tags = rdafutils.delimited_to_list(cmd_args.tag)
            # write this updated state to the config file
            cfg_file = get_config_file_path()
            Component.write_configs(config_parser, config_file=cfg_file)
            docker_reg_mirror.mirror(tags, tag_type, cmd_args, config_parser)
        elif docker_cmd == 'delete-images':
            tags = rdafutils.delimited_to_list(cmd_args.tag)
            docker_reg_mirror.delete_images(tags, config_parser)
        elif docker_cmd == 'reset':
            if os.path.exists(os.path.join('/opt', 'rdaf-registry', 'rdaf-registry.cfg')):
                registry_cfg = os.path.join('/opt', 'rdaf-registry', 'rdaf-registry.cfg')
                self.reset_registry(cmd_args, registry_cfg, config_parser)
        else:
            raise InvalidCmdUsageException()

    def setup(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        install_root = get_install_root(cmd_args)
        self._setup_install_root_dir_hierarchy(install_root)
        components = [rdaf.component.proxy.Proxy(),
                      rdaf.component.dockerregistrymirror.DockerRegistryMirror()]
        # register each of these components
        for component in components:
            COMPONENT_REGISTRY.register(component)
        # phase 1 of the setup - gather inputs for each component
        for component in components:
            logger.info('Gathering inputs for ' + component.component_name)
            component.gather_setup_inputs(cmd_args, config_parser)
        # validate these setup inputs
        for component in components:
            component.validate_setup_inputs(cmd_args, config_parser)
        # now that the setup inputs have been gathered and validated, persist them
        cfg_file = get_config_file_path()
        config_parser.add_section('common')
        config_parser.set('common', 'install_root', install_root)
        Component.write_configs(config_parser, config_file=cfg_file)
        # phase 2 of the setup - do the necessary setup for each component
        # Note that this doesn't involve installing any components
        for component in components:
            logger.info('Doing setup for ' + component.component_name)
            component.do_setup(cmd_args, config_parser)
        Component.write_configs(config_parser, config_file=cfg_file)
        os.chmod(cfg_file, 0o600)
        logger.info('Setup completed successfully, configuration written to ' + cfg_file)
        return

    @staticmethod
    def _setup_install_root_dir_hierarchy(install_root: os.path):
        uid = os.getuid()
        gid = os.getgid()
        dirs = [os.path.join('/opt', 'rdaf'), install_root, os.path.join(install_root, 'config'),
                os.path.join(install_root, 'data'), os.path.join(install_root, 'log'),
                os.path.join(install_root, 'import'),
                os.path.join(install_root, 'deployment-scripts')]
        for path in dirs:
            logger.info('Creating directory ' + path + ' and setting ownership to user '
                        + str(uid) + ' and group to group ' + str(gid))
            command = 'sudo mkdir -p ' + path + ' && sudo chown -R ' + str(
                uid) + ' ' + path + ' && sudo chgrp -R ' + str(
                gid) + ' ' + path
            run_command(command)

    @staticmethod
    def reset_registry(cmd_args, registry_cfg, config_parser):
        docker_reg_mirror = rdaf.component.dockerregistrymirror.DockerRegistryMirror()
        _read_configs(config_parser, cfg_file=registry_cfg)

        docker_reg_mirror.load_config(config_parser)
        logger.info('Resetting ' + docker_reg_mirror.get_name())
        docker_reg_mirror.reset(cmd_args, config_parser)
        command = 'docker image prune -a -f'
        execute_command(command)