import argparse
import configparser
import logging
import rdaf.rdafutils as rdafutils
from rdaf import InvalidCmdUsageException
from rdaf.cmd import CliCmdHandler, status
from rdaf.contextual import COMPONENT_REGISTRY


logger = logging.getLogger(__name__)


class EventGatewayCmdHandler(CliCmdHandler):

    def __init__(self, k8s=False):
        super().__init__()
        self.k8s = k8s

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)
        tag_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the event_gateway components')

        gateway_commands_parser = parser.add_subparsers(dest='rda_event_gateway', metavar='{}',
                                                        help='commands')

        gateway_commands_parser .add_parser('status', help='Status of the RDAF Event-Gateway')

        up_parser = gateway_commands_parser.add_parser('up', help='Create the RDAF Event-Gateway Containers')

        down_parser = gateway_commands_parser.add_parser('down', help='Delete the RDAF Event-Gateway Containers')

        down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')

        install_parser = gateway_commands_parser.add_parser('install',
                                                            parents=[tag_parser],
                                                            help='Install the RDAF Event-Gateway containers')
        upgrade_parser = gateway_commands_parser.add_parser('upgrade',
                                                            parents=[tag_parser],
                                                            help='Upgrade the RDAF Event-Gateway containers')
        start_parser = gateway_commands_parser.add_parser('start', help='Start the RDAF Event-Gateway Containers')

        stop_parser = gateway_commands_parser.add_parser('stop', help='Stop the RDAF Event-Gateway Containers')

        stop_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.rda_event_gateway
        component = COMPONENT_REGISTRY.get('rda_event_gateway')
        if cmd == 'install':
            component.pull_images(cmd_args, config_parser)
            component.install(cmd_args, config_parser)
        elif cmd == 'upgrade':
            component.pull_images(cmd_args, config_parser)
            component.upgrade(cmd_args, config_parser)
        elif cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF event_gateway container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the event_gateway container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('event_gateway down operation has been cancelled')
                    return
            component.down(cmd_args, config_parser)
        elif cmd == 'up':
            component.up(cmd_args, config_parser)
        elif cmd == 'status':
            status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, components=[component])
        elif cmd == 'stop':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Stopping of RDAF event_gateway container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to stop the event_gateway container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('event_gateway stop operation has been cancelled')
                    return
            component.stop(cmd_args, config_parser)
        elif cmd == 'start':
            component.start(cmd_args, config_parser)
        else:
            raise InvalidCmdUsageException()

class K8SEventGatewayCmdHandler(CliCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = True

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)
        tag_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the Event-Gateway components')

        gateway_commands_parser = parser.add_subparsers(dest='rda_event_gateway', metavar='{}',
                                                        help='commands')

        gateway_commands_parser.add_parser('status', help='Status of the RDAF Event-Gateway')

        install_parser = gateway_commands_parser.add_parser('install',
                                                            parents=[tag_parser],
                                                            help='Install the RDAF Event-Gateway '
                                                                 'containers')
        upgrade_parser = gateway_commands_parser.add_parser('upgrade',
                                                            parents=[tag_parser],
                                                            help='Upgrade the RDAF Event-Gateway '
                                                                 'containers')
        event_gateway_up_parser = gateway_commands_parser.add_parser('up',
                                                                     help='Create the RDAF Event-Gateway Containers')

        event_gateway_down_parser = gateway_commands_parser.add_parser('down',
                                                                       help='Delete the RDAF '
                                                                            'Event-Gateway containers')
        
        event_gateway_down_parser.add_argument('--force', action='store_true',
                                               help='Delete the RDAF Event-Gateway Containers Forefully')
        
        event_gateway_down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')


    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.rda_event_gateway
        component = COMPONENT_REGISTRY.get('rda_event_gateway')
        if cmd == 'install':
            component.k8s_pull_images(cmd_args, config_parser)
            component.k8s_install(cmd_args, config_parser)
        elif cmd == 'upgrade':
            component.k8s_pull_images(cmd_args, config_parser)
            component.k8s_upgrade(cmd_args, config_parser)
        elif cmd == 'status':
            status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, components=[component])
        elif cmd == 'up':
            component.k8s_up(cmd_args, config_parser)
        elif cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF event_gateway container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the event_gateway container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('event_gateway down operation has been cancelled')
                    return
            component.k8s_down(cmd_args, config_parser)
        else:
            raise InvalidCmdUsageException()



