import argparse
import configparser
import logging
import os

import rdaf
from rdaf import InvalidCmdUsageException, rdafutils
from rdaf.cmd import CliCmdHandler, status
from rdaf.component import Component
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.component import dockerregistry, ssh

logger = logging.getLogger(__name__)


class FileObjectCmdHandler(CliCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = False

    def configure_parser(self, parser):
        file_object_commands_parser = parser.add_subparsers(dest='file_object_op', help='commands')

        upgrade_parser = file_object_commands_parser.add_parser('upgrade',
                                                                   help='Upgrade file_object components')
        upgrade_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='Docker image tag')

        install_parser = file_object_commands_parser.add_parser('install',
                                                                   help='Install file_object components')
        install_parser.add_argument('--host',
                                    dest='file_object_host',
                                    action='append',
                                    required=True,
                                    help='file_object host')

        install_parser.add_argument('--ssh-password', dest='ssh_password', required=True,
                                            action='store',
                                            help='Password for the SSH user for initial'
                                                 ' login to this host')
        install_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='file_object image tag')

        install_parser.add_argument('--no-prompt',
                                    dest='no_prompt',
                                    action='store_true',
                                    default=False,
                                    help='Don\'t prompt for inputs')

        status_parser = file_object_commands_parser.add_parser('status', help='Status of the RDAF file_object')

        up_parser = file_object_commands_parser.add_parser('up',
                                                              help='Create the RDAF file_object Containers')

        down_parser = file_object_commands_parser.add_parser('down',
                                                                help='Delete the RDAF file_object Containers')
        
        down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                                                help='Don\'t prompt for inputs')

        start_parser = file_object_commands_parser.add_parser('start',
                                                                 help='Start the RDAF file_object Containers')

        stop_parser = file_object_commands_parser.add_parser('stop',
                                                                help='Stop the RDAF file_object Containers')
        
        stop_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                                                help='Don\'t prompt for inputs')

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.file_object_op
        component = COMPONENT_REGISTRY.get('rda_file_object')
        if cmd == 'install':
            network_config = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'config.json')
            if not os.path.exists(network_config):
                rdafutils.cli_err_exit('config.json is missing.. please install rdaf platform')
            if component.configs['host']:
                rdafutils.cli_err_exit('file object service is already installed.')
            file_object_hosts = cmd_args.file_object_host
            all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
                skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME]
            )
            for host in file_object_hosts:
                if host not in all_known_hosts:
                    ssh_manager = COMPONENT_REGISTRY.require(
                        rdaf.component.ssh.SSHKeyManager.COMPONENT_NAME)
                    # setup ssh keys for this new host
                    ssh_manager.setup_keys_for_host(host, cmd_args.ssh_password)
                # doing a docker login
                docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
                docker_registry.docker_login(host, config_parser)
                component.pull_images(cmd_args, config_parser)
                component.configs['host'] = file_object_hosts
                component._mark_configured(component.configs, config_parser)
                component.write_configs(config_parser)
                component.install(cmd_args, config_parser)
        elif cmd == 'upgrade':
                component.pull_images(cmd_args, config_parser)
                component.upgrade(cmd_args, config_parser)
        elif cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF file_object container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the file_object container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('file_object down operation has been cancelled')
                    return
            component.down(cmd_args, config_parser)
        elif cmd == 'up': 
                component.up(cmd_args, config_parser)
        elif cmd == 'status':
            return status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, components=[component])
        elif cmd == 'stop':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Stopping of RDAF file_object container(s) is a disruptive operation\n '
                'and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to stop the file_object container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('file_object stop operation has been cancelled')
                    return
            component.stop(cmd_args, config_parser)
        elif cmd == 'start':
            component.start(cmd_args, config_parser)
        else:
            raise InvalidCmdUsageException()