import argparse
import configparser
import logging.config
import os
from docker.errors import DockerException
from paramiko.ssh_exception import NoValidConnectionsError
from requests.exceptions import ConnectionError
from rdaf.component import Component
import rdaf.contextual
import rdaf.component.keepalived
from rdaf import InvalidCmdUsageException, rdafutils
from rdaf.cmd import CliCmdHandler, status, validate
from rdaf.rdafutils import query_yes_no

logger = logging.getLogger(__name__)


class InfraCmdHandler(CliCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = False

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)

        tag_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the infra components')

        service_parser = argparse.ArgumentParser(add_help=False)
        service_parser.add_argument('--service',
                                    dest="services",
                                    action='append',
                                    default=None,
                                    help='Restrict the scope of the command to a specific service')

        infra_commands_parser = parser.add_subparsers(dest='rdaf_infra_cmd')

        infra_status_parser = infra_commands_parser.add_parser('status',
                                                               help='Status of the RDAF Infra')

        infra_install_parser = infra_commands_parser.add_parser('install',
                                                                parents=[tag_parser,
                                                                         service_parser],
                                                                help='Install the '
                                                                     'RDAF Infra containers')
        
        infra_install_parser.add_argument('--external-url',dest='external_url', type=str,
                                       help='provide the external url')
        
        infra_upgrade_parser = infra_commands_parser.add_parser('upgrade',
                                                                parents=[tag_parser,
                                                                         service_parser],
                                                                help='Upgrade the '
                                                                     'RDAF Infra containers')

        infra_up_parser = infra_commands_parser.add_parser('up', parents=[service_parser],
                                                           help='Create the RDAF Infra Containers')

        infra_down_parser = infra_commands_parser.add_parser('down',
                                                             parents=[service_parser],
                                                             help='Delete the RDAF'
                                                                  ' Infra Containers')
        infra_down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')
        infra_healthcheck_parser = infra_commands_parser.add_parser('healthcheck',
                                                                    parents=[service_parser],
                                                                    help='Check the liveness/health of Infra services.')
        
        infra_start_parser = infra_commands_parser.add_parser('start', parents=[service_parser],
                                                           help='Start the RDAF Infra Containers')

        infra_stop_parser = infra_commands_parser.add_parser('stop',
                                                             parents=[service_parser],
                                                             help='Stop the RDAF'
                                                                  ' Infra Containers')
        infra_stop_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = Component.get_namespace(config_parser)
        if cmd_args.rdaf_infra_cmd == 'install':
            specific_services = [] if cmd_args.services is None else cmd_args.services
            infra_components = rdaf.contextual.COMPONENT_REGISTRY.get_by_category('infra')
            # to pull all the needed images
            for component in infra_components:
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    continue
                if len(specific_services) == 0 and component_name == 'nginx':
                    logger.debug('Skip pulling of ' + component_name)
                    continue
                if self.k8s:
                    if component.get_deployment_type(config_parser) == 'k8s':
                       component.k8s_pull_images(cmd_args, config_parser)
                else:
                    component.pull_images(cmd_args, config_parser)

            for component in infra_components:
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    logger.debug('Skipping installation of ' + component_name)
                    continue
                if len(specific_services) == 0 and component_name == 'nginx':
                    logger.debug('Skipping installation of ' + component_name)
                    continue
                logger.info('Installing ' + component_name)
                if self.k8s:
                    ignored_components = [rdaf.component.keepalived.COMPONENT_NAME]
                    if component_name in ignored_components:
                        continue
                    component.k8s_install(cmd_args, config_parser)
                else:
                    component.install(cmd_args, config_parser)
            if self.k8s:
                status_cmd = '\033[1;4m kubectl get pods -n {} -l app_category=rdaf-infra \033[0m'.format(namespace)
                print('\n')
                logger.info('Please check infra pods status using - ' + status_cmd)
            else:
                status.StatusCmdHandler().handle(cmd_args, config_parser, category='infra')
            return
        if cmd_args.rdaf_infra_cmd == 'upgrade':
            specific_services = [] if cmd_args.services is None else cmd_args.services
            infra_components = rdaf.contextual.COMPONENT_REGISTRY.get_by_category('infra')
            # to pull all the needed images
            for component in infra_components:
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    continue
                if len(specific_services) == 0 and component_name == 'nginx':
                    logger.debug('Skip pulling of ' + component_name)
                    continue
                if self.k8s:
                    if component.get_deployment_type(config_parser) == 'k8s':
                       component.k8s_pull_images(cmd_args, config_parser)
                else:
                    component.pull_images(cmd_args, config_parser)

            for component in infra_components:
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    logger.debug('Skipping upgrade of ' + component_name)
                    continue
                if len(specific_services) == 0 and component_name == 'nginx':
                    logger.debug('Skipping upgrade of ' + component_name)
                    continue
                logger.info('Upgrading ' + component_name)
                if self.k8s:
                    ignored_components = [rdaf.component.keepalived.COMPONENT_NAME]
                    if component_name in ignored_components:
                        continue
                    component.k8s_upgrade(cmd_args, config_parser)
                else:
                    component.upgrade(cmd_args, config_parser)
            status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, category='infra')
            return
        if cmd_args.rdaf_infra_cmd == 'down':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Deleting of RDAF infra container(s) is a disruptive operation\n'
                ' and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if query_yes_no("Are you sure you want to delete the infra container(s)?"):
                    if query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('infra down operation has been cancelled')
                    return
            specific_services = [] if cmd_args.services is None else cmd_args.services
            infra_components = rdaf.contextual.COMPONENT_REGISTRY.get_by_category('infra')
            for component in reversed(infra_components):
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    logger.debug('Skipping delete of ' + component_name)
                    continue
                if len(specific_services) == 0 and component_name == 'nginx':
                    logger.debug('Skipping delete of ' + component_name)
                    continue
                logger.info('Deleting ' + component_name)
                if self.k8s:
                    ignored_components = [rdaf.component.keepalived.COMPONENT_NAME]
                    if component_name in ignored_components:
                        continue
                    component.k8s_down(cmd_args, config_parser)
                else:    
                    component.down(cmd_args, config_parser)
            return
        if cmd_args.rdaf_infra_cmd == 'up':
            specific_services = [] if cmd_args.services is None else cmd_args.services
            infra_components = rdaf.contextual.COMPONENT_REGISTRY.get_by_category('infra')
            for component in infra_components:
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    logger.debug('Skipping creation of ' + component_name)
                    continue
                if len(specific_services) == 0 and component_name == 'nginx':
                    logger.debug('Skipping creation of ' + component_name)
                    continue
                logger.info('Creating ' + component_name)
                if self.k8s:
                    ignored_components = [rdaf.component.keepalived.COMPONENT_NAME]
                    if component_name in ignored_components:
                        continue
                    component.k8s_up(cmd_args, config_parser)
                else:    
                    component.up(cmd_args, config_parser)
            status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, category='infra')
            return
        
        if cmd_args.rdaf_infra_cmd == 'stop':
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Stopping of RDAF infra container(s) is a disruptive operation\n'
                ' and may cause applications to stop working\n')
            print(warn_message)
            cancelled = True
            if not cmd_args.no_prompt:
                if query_yes_no("Are you sure you want to Stop the infra container(s)?"):
                    if query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('infra Stop operation has been cancelled')
                    return
            specific_services = [] if cmd_args.services is None else cmd_args.services
            infra_components = rdaf.contextual.COMPONENT_REGISTRY.get_by_category('infra')
            for component in reversed(infra_components):
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    logger.debug('Skipping Stopping of ' + component_name)
                    continue
                if len(specific_services) == 0 and component_name == 'nginx':
                    logger.debug('Skipping Stopping of ' + component_name)
                    continue
                logger.info('Stopping ' + component_name)
                component.stop(cmd_args, config_parser)
            return

        if cmd_args.rdaf_infra_cmd == 'start':
            specific_services = [] if cmd_args.services is None else cmd_args.services
            infra_components = rdaf.contextual.COMPONENT_REGISTRY.get_by_category('infra')
            for component in infra_components:
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    logger.debug('Skipping Starting of ' + component_name)
                    continue
                if len(specific_services) == 0 and component_name == 'nginx':
                    logger.debug('Skipping Starting of ' + component_name)
                    continue
                logger.info('Starting ' + component_name)
                component.start(cmd_args, config_parser)
            status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, category='infra')
            return

        if cmd_args.rdaf_infra_cmd == 'status':
            return status.StatusCmdHandler(self.k8s).handle(cmd_args, config_parser, category='infra')
        if cmd_args.rdaf_infra_cmd == 'healthcheck':
            logger.info("Running Health Check on Infra services")
            specific_services = [] if cmd_args.services is None else cmd_args.services
            infra_components = rdaf.contextual.COMPONENT_REGISTRY.get_by_category('infra')
            statuses = []
            unreachable_hosts = set()
            for component in infra_components:
                for host in component.get_hosts():
                    component_name = component.get_name()
                    if host in unreachable_hosts:
                        statuses.append(
                            [component_name, "Host Connectivity", "Failed", f"Host is unreachable", host, "N/A"])
                        continue
                    _status = []
                    if len(specific_services) > 0 and component_name not in specific_services:
                        logger.debug('Skipping health check of ' + component_name)
                        continue
                    if len(specific_services) == 0 and component_name == 'nginx':
                        logger.debug('Skipping health check of ' + component_name)
                        continue

                    logger.info('Running Health Check on ' + component_name + ' on host ' + host)
                    try:
                        healthcheck = component.healthcheck(component_name, host, cmd_args, config_parser)
                        port_check = validate.ValidateCmdHandler.check_ports_on_host(host, config_parser, component_name)
                    except NoValidConnectionsError:
                        logger.error(f"Host {host} is not accessible. Skipping health checks for this host.")
                        statuses.append(
                            [component_name, "Host Connectivity", "Failed", f"Host is unreachable", host, "N/A"])
                        unreachable_hosts.add(host)
                        continue
                    ports = component.get_ports()
                    if ports:
                        is_cluster = True if len(component.get_hosts()) > 1 else False
                        tcp_check = validate.ValidateCmdHandler.check_tcp(component_name, host, 
                                                                          ports[1], is_cluster=is_cluster)
                        _status.append(tcp_check)
                    
                    _status.append(healthcheck)
                    _status.extend(port_check)
                    try:
                        with component.new_docker_client(host, config_parser) as docker_client:
                            if component.get_name() == 'graphdb':
                                if len(component.get_hosts()) == 1:
                                    service = 'server'
                                else:
                                    service = 'coordinator'
                                containers = component._find_component_container_on_host(docker_client, service)
                                if containers:
                                    selected_container = containers[0]
                                    for st in _status:
                                        if len(st) == 5:
                                            temp_st = st
                                            temp_st.append(selected_container["Id"][:12])
                                            statuses.append(temp_st)
                                        else:
                                            temp_st = st
                                            temp_st.append(host)
                                            temp_st.append(selected_container["Id"][:12])
                                            statuses.append(temp_st)
                                else:
                                    for st in _status:
                                        if len(st) == 5:
                                            temp_st = st
                                            temp_st.append("N/A")
                                            statuses.append(temp_st)
                                        else:
                                            temp_st = st
                                            temp_st.append(host)
                                            temp_st.append("N/A")
                                            statuses.append(temp_st)
                            else:
                                containers = component.find_component_container_on_host(docker_client)
                                for container in containers:
                                    for st in _status:
                                        if len(st) == 5:
                                            temp_st = st
                                            temp_st.append(container["Id"][:12])
                                            statuses.append(temp_st)
                                        else:
                                            temp_st = st
                                            temp_st.append(host)
                                            temp_st.append(container["Id"][:12])
                                            statuses.append(temp_st)
                                if not containers:
                                    for st in _status:
                                        if len(st) == 5:
                                            temp_st = st
                                            temp_st.append("N/A")
                                            statuses.append(temp_st)
                                        else:
                                            temp_st = st
                                            temp_st.append(host)
                                            temp_st.append("N/A")
                                            statuses.append(temp_st)
                    except (DockerException, ConnectionError) as e:
                        if "[Errno 113]" in str(e):
                            logger.error(f"Host {host} is not accessible. Skipping health checks for this host.")
                            statuses.append(
                                [component_name, "Host Connectivity", "Failed", f"Host is unreachable", host, "N/A"])
                            unreachable_hosts.add(host)
                            continue
                        else:
                            raise

            # platform portal frontend healthcheck
            platform = rdaf.contextual.COMPONENT_REGISTRY.get('platform')
            for host in platform.get_involved_hosts('portal-frontend'):
                if not os.path.exists((platform.get_deployment_file_path(host))):
                    continue
                _status = []
                component_name = 'portal'
                logger.info('Running Health Check on ' + component_name + ' on host ' + host)
                try:
                    healthcheck = platform.healthcheck(component_name, host, cmd_args, config_parser)
                    portal_port = ['7780']
                    port_check = validate.ValidateCmdHandler.check_all_ports_on_host(host, portal_port, config_parser, component_name)
                    tcp_check = validate.ValidateCmdHandler.check_tcp(component_name, host,
                                                                        portal_port)
                except NoValidConnectionsError:
                        logger.error(f"Host {host} is not accessible. Skipping health checks for this host.")
                        statuses.append(
                            [component_name, "Host Connectivity", "Failed", f"Host is unreachable", host, "N/A"])
                        unreachable_hosts.add(host)
                        continue
                _status.append(tcp_check)
                _status.append(healthcheck)
                _status.extend(port_check)

                service = 'portal-frontend'
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = platform._find_component_container_on_host(docker_client,
                                                                        service, all_states=True)
                    for container in containers:
                        for st in _status:
                            if len(st) == 5:
                                temp_st = st
                                temp_st.append(container["Id"][:12])
                                statuses.append(temp_st)
                            else:
                                temp_st = st
                                temp_st.append(host)
                                temp_st.append(container["Id"][:12])
                                statuses.append(temp_st)
                    if not containers:
                        for st in _status:
                            if len(st) == 5:
                                temp_st = st
                                temp_st.append("N/A")
                                statuses.append(temp_st)
                            else:
                                temp_st = st
                                temp_st.append(host)
                                temp_st.append("N/A")
                                statuses.append(temp_st)
            col_header = ['Name', 'Check', "Status", "Reason", 'Host', "Container Id"]
            rdafutils.print_tabular(col_header, statuses, add_row_spacing=False)
            return
        else:
            raise InvalidCmdUsageException()


class K8SInfraCmdHandler(InfraCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = True

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)
        tag_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the infra components')

        service_parser = argparse.ArgumentParser(add_help=False)
        service_parser.add_argument('--service',
                                    dest="services",
                                    action='append',
                                    default=None,
                                    help='Restrict the scope of the command to a specific service')

        infra_commands_parser = parser.add_subparsers(dest='rdaf_infra_cmd')

        infra_status_parser = infra_commands_parser.add_parser('status',
                                                               help='Status of the RDAF Infra')

        infra_install_parser = infra_commands_parser.add_parser('install',
                                                                parents=[tag_parser,
                                                                         service_parser],
                                                                help='Install the '
                                                                     'RDAF Infra containers')
        
        infra_install_parser.add_argument('--external-url',dest='external_url', type=str,
                                       help='provide the external url')
        
        infra_upgrade_parser = infra_commands_parser.add_parser('upgrade',
                                                                parents=[tag_parser,
                                                                         service_parser],
                                                                help='Upgrade the '
                                                                     'RDAF Infra containers')

        infra_up_parser = infra_commands_parser.add_parser('up',
                                                           parents=[service_parser],
                                                           help='Create the RDAF Infra Containers')

        infra_down_parser = infra_commands_parser.add_parser('down',
                                                             parents=[service_parser],
                                                             help='Delete the RDAF Infra Containers')
        infra_down_parser.add_argument('--no-prompt', dest='no_prompt', action='store_true', default=False,
                                       help='Don\'t prompt for inputs')

