import configparser
from rdaf.cmd import CliCmdHandler
import rdaf
import argparse
import tarfile
import os
import logging
import rdaf.component as comp
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.component import Component
from rdaf import rdafutils
import datetime
import paramiko
import os
import shutil

logger = logging.getLogger(__name__)

class LogsCmdHandler(CliCmdHandler):

    def configure_parser(self, parser):
        parser.add_argument('--service',
                            dest="services",
                            action='append',
                            default=None,
                            help='Restrict the scope of the command to a specific service')
        
        parser.add_argument('--dest-dir',
                            dest='dest_dir',
                            action='store',
                            default=None,
                            required=True,
                            help='Specify the destination directory for the tar file')
        
    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if cmd_args.services:
            self.tar_logs_specific_service(cmd_args.services,cmd_args.dest_dir,  config_parser)
        else:
            self.tar_logs_all_services(cmd_args.dest_dir, config_parser)
        
    def tar_logs_all_services(self, dest_dir, config_parser):
        log_dir = os.path.join('/opt', 'rdaf', 'logs')
        if not os.path.exists(dest_dir):
            os.makedirs(dest_dir, exist_ok=True)
    
        rdaf_logs_dir = os.path.join(dest_dir, 'rdaf_logs')
        os.makedirs(rdaf_logs_dir, exist_ok=True)
                
        now = datetime.datetime.now()
        current_time = str(now.date()) + '-' + str(now.timestamp())
        tar_name = f"rdaf-{current_time}.tar.gz"
        tar_path = os.path.join(dest_dir, tar_name)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])

        try:
            with tarfile.open(tar_path, 'w:gz') as tar:
                for host in all_known_hosts:
                    try:
                        chmod_command = f"sudo chmod -R 755 {log_dir}"
                        comp.execute_command_ssh(chmod_command, host, config_parser)
                        comp.do_potential_scp_fetch(host, log_dir, rdaf_logs_dir)
                    except Exception as e:
                        logger.warning(f"skipping host {host} due to error: {e} ")
                        continue    
                for root, dirs, files in os.walk(rdaf_logs_dir):
                    for file in files:
                        file_path = os.path.join(root, file)
                        arcname = os.path.relpath(file_path, start=rdaf_logs_dir)
                        tar.add(file_path, arcname=arcname)
                shutil.rmtree(rdaf_logs_dir)

            logger.info(f'All services logs have been successfully tarred to {tar_path}')
        except Exception as e:
            rdafutils.cli_err_exit(f'Failed to tar logs: {e}')
    
    def tar_logs_specific_service(self, services, dest_dir, config_parser):
        log_dir = "/opt/rdaf/logs"
        if not os.path.exists(dest_dir):
            os.makedirs(dest_dir, exist_ok=True)
        
        rdaf_logs_dir = os.path.join(dest_dir, "rdaf_logs_specific")
        os.makedirs(rdaf_logs_dir, exist_ok=True)

        now = datetime.datetime.now()
        current_time = str(now.date()) + '-' + str(now.timestamp())
        if len(services) == 1:
            tar_name = f"{services[0]}-{current_time}.tar.gz"
        else:
            tar_name = f"rdaf-{current_time}.tar.gz"
        tar_path = os.path.join(dest_dir, tar_name)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])

        try:
            with tarfile.open(tar_path, 'w:gz') as tar:
                for host in all_known_hosts:
                    try:
                        chmod_command = f"sudo chmod -R 755 {log_dir}"
                        comp.execute_command_ssh(chmod_command, host, config_parser)

                        for service in services:
                            service_pattern = f"{service}"
                            search_cmd = f"ls {log_dir} | grep {service} || true"
                            result = comp.execute_command_ssh(search_cmd, host, config_parser)

                            if result:
                                for log in result[1].strip().splitlines():
                                    log_path = os.path.join(log_dir, log)
                                    cmd = f"test -d {log_path} && echo 'directory' || echo 'file'"
                                    res = comp.execute_command_ssh(cmd, host, config_parser)
                                    if res[1].strip() == 'directory':
                                        comp.do_potential_scp_fetch(host, log_path, rdaf_logs_dir)
                                    else :
                                        comp.do_potential_scp_fetch(host, log_path, os.path.join(rdaf_logs_dir, log), is_dir=False)
                            else:
                                continue
                    except Exception as e:
                        logger.warning(f"skipping host {host} due to error: {e} ")
                        continue          
                for root, dirs, files in os.walk(rdaf_logs_dir):
                    for file in files:
                        file_path = os.path.join(root, file)
                        arcname = os.path.relpath(file_path, start=rdaf_logs_dir)
                        tar.add(file_path, arcname=arcname)
                shutil.rmtree(rdaf_logs_dir)

            logger.info(f'{', '.join(services)} logs have been successfully tarred to {tar_path}')
        except Exception as e:
            rdafutils.cli_err_exit(f'Failed to tar logs: {e}')


    
                    
      
            
            

    