import argparse
import configparser
import logging
import os
import string
import socket
import tempfile
from typing import Tuple

import rdaf
from rdaf.component import do_potential_scp, run_command, execute_command
from rdaf import InvalidCmdUsageException
from rdaf.cmd import CliCmdHandler

logger = logging.getLogger(__name__)


class RDACCmdHandler(CliCmdHandler):

    @staticmethod
    def _get_docker_repo() -> Tuple[str, str, str]:
        from rdaf.component.dockerregistry import COMPONENT_NAME
        from rdaf.contextual import COMPONENT_REGISTRY
        docker_registry = COMPONENT_REGISTRY.require(COMPONENT_NAME)
        user = docker_registry.configs['username']
        password = docker_registry.configs['password']

        return docker_registry.get_docker_registry_url(), user, password

    def install_rdac(self, cmd_args, upgrade=False):
        template_name = "rdac.py"  # default to Docker template
        
        # Check if ctr (containerd CLI) is available
        ctr_check_command = "ctr containers list"
        ctr_return_code, ctr_stdout, ctr_stderr = execute_command(ctr_check_command)
        if ctr_return_code == 0:
            docker_check_command = "docker ps"
            docker_return_code, docker_stdout, docker_stderr = execute_command(docker_check_command)
            if docker_return_code != 0:
                template_name = "rdac_ctr.py"
       
        rdac_path = os.path.join(rdaf.get_templates_dir_root(), template_name)
        with open(rdac_path, "r") as f:
            rdac_template = f.read()

        registry, user, password = self._get_docker_repo()
        substitutions = {
            "REGISTRY": registry,
            "TAG": cmd_args.tag,
            "USER": user,
            "PASSWORD": password
        }
        original_content = string.Template(rdac_template).substitute(substitutions)
        dest_location = os.path.join(os.path.expanduser('~'), ".local/bin/rdac")
        tf = tempfile.NamedTemporaryFile(mode='w')
        with open(tf.name, 'w') as f:
            f.write(original_content)
        do_potential_scp(socket.gethostname(), tf.name, dest_location)
        tf.close()
        chmod_cmd = 'sudo chmod +x ' + dest_location
        run_command(chmod_cmd)

        if upgrade:
            run_command("rdac update")

    def configure_parser(self, parser):
        tag_parser = argparse.ArgumentParser(add_help=False)
        tag_parser.add_argument('--tag',
                                dest='tag',
                                action='store',
                                required=True,
                                help='Tag to use for the docker images of the platform components')

        platform_commands_parser = parser.add_subparsers(dest='rdac_op', metavar='{}',
                                                         help='commands')

        install_parser = platform_commands_parser.add_parser('install',
                                                             parents=[tag_parser],
                                                             help='Install the RDAC CLI')

        upgrade_parser = platform_commands_parser.add_parser('upgrade',
                                                             parents=[tag_parser],
                                                             help='Upgrade the RDAC CLI')
        
    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        cmd = cmd_args.rdac_op
        if cmd == 'install':
            logger.info("Installing RDAC CLI")
            self.install_rdac(cmd_args)
        elif cmd == 'upgrade':
            logger.info("Upgrading RDAC CLI")
            self.install_rdac(cmd_args, upgrade=True)
        else:
            raise InvalidCmdUsageException()
