import configparser
import json
import logging
import os

import rdaf

import rdaf.component.pseudo_platform as pseudo_platform
import rdaf.component.opensearch_external as opensearch_external
import rdaf.component.graphdb as graphdb
import rdaf.component.dockerregistrymirror
import rdaf.component.dockerregistry
from rdaf.component import Component
from rdaf import rdafutils
from rdaf.cmd import CliCmdHandler
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.rdafutils import query_yes_no
from rdaf.component import execute_command_ssh, execute_command

logger = logging.getLogger(__name__)


class ResetCmdHandler(CliCmdHandler):

    def __init__(self, k8s=False):
        super().__init__()
        self.k8s = k8s

    def configure_parser(self, parser):
        parser.add_argument('--no-prompt',
                            dest="no_prompt",
                            action='store_true',
                            default=False,
                            help='Don\'t prompt for inputs')
        parser.add_argument('--no-purge',
                            dest="no_purge",
                            action='store_true',
                            default=False,
                            help='Don\'t purge older images')
        
    def handle(self, cmd_args, config_parser: configparser.ConfigParser):
        if not os.path.exists(os.path.join('/opt', 'rdaf', 'rdaf.cfg')):
            return
        
        if not cmd_args.no_prompt:
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Resetting the system will remove all RDAF installed components\n'
                ' and delete their data from the systems belonging\n'
                ' to this RDAF installation\n')
            print(warn_message)
            cancelled = True
            if query_yes_no("Are you sure you want to reset the system?"):
                if query_yes_no("Please confirm again?"):
                    cancelled = False
            if cancelled:
                logger.info('RDAF reset has been cancelled')
                return

        if self.k8s:
            self.k8s_reset(cmd_args, config_parser)
            return

        # stop apps
        app_components = COMPONENT_REGISTRY.get_by_category(category='apps')
        logger.info('Initiating reset on application components')
        for component in reversed(app_components):
            logger.info('Resetting ' + component.get_name())
            component.reset(cmd_args, config_parser)

        # stop platform
        platform_components = COMPONENT_REGISTRY.get_by_category(category='platform')
        logger.info('Initiating reset on platform components')
        for component in reversed(platform_components):
            logger.info('Resetting ' + component.get_name())
            component.reset(cmd_args, config_parser)

        # stop infra
        infra_components = COMPONENT_REGISTRY.get_by_category(category='infra')
        logger.info('Initiating reset on infra components')
        for component in reversed(infra_components):
            logger.info('Resetting ' + component.get_name())
            component.reset(cmd_args, config_parser)

        other_components = COMPONENT_REGISTRY.get_by_category(category='other')
        logger.info('Initiating reset on other components')
        for component in reversed(other_components):
            logger.info('Resetting ' + component.get_name())
            component.reset(cmd_args, config_parser)

        pseudo_comp = COMPONENT_REGISTRY.require(pseudo_platform.PseudoComponent.COMPONENT_NAME)
        pseudo_comp.reset(cmd_args, config_parser)

        if not cmd_args.no_purge:
            self.prune_images(config_parser)
            
        logger.info('RDAF reset completed successfully')
        
    def prune_images(self, config_parser: configparser.ConfigParser):
        command = 'docker image prune -a -f'
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        for host in all_known_hosts:
            execute_command_ssh(command, host, config_parser)
            logger.info('Removed all component images on host ' + host)

        execute_command(command)
    
    def prune_ctr_images(self, config_parser: configparser.ConfigParser):
        command = 'sudo ctr -n k8s.io images prune --all'
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        for host in all_known_hosts:
            execute_command_ssh(command, host, config_parser)
            logger.info('Removed all component images on host ' + host)

        execute_command(command)

    def k8s_reset(self, cmd_args, config_parser):
        logger.info('Initiating reset on graphdb components')
        graphdb_comp = COMPONENT_REGISTRY.require(graphdb.COMPONENT_NAME)
        graphdb_comp.k8s_reset(cmd_args, config_parser)
        opensearch_external_comp  = COMPONENT_REGISTRY.require(opensearch_external.COMPONENT_NAME)
        if config_parser.has_section('os_external'):
            opensearch_external_comp.k8s_reset(cmd_args, config_parser)
        logger.info('Initiating namespace deletion')
        namespace = Component.get_namespace(config_parser)
        charts_list_cmd = f'helm uninstall -n {namespace} $(helm ls --all --short -n {namespace}) '
        execute_command(charts_list_cmd)

        # deleting namespace
        logger.info(f'Deleting namespace {namespace}')
        namespace_cmd = f'kubectl delete namespaces {namespace} '
        execute_command(namespace_cmd)

        if not config_parser.has_section("rdaf-cli"):
            deployment_type = "k8s"
        else:
            deployment_type = config_parser.get("rdaf-cli", "deployment")

        if deployment_type in ['k8s', 'ctr']:
            # deleting pv for all components
            delete_pv = 'kubectl delete pv nats-js-0 nats-js-1 nats-jwt-pv-0 nats-jwt-pv-1 ' \
                        'opensearch-master-pv-0 opensearch-master-pv-1 opensearch-master-pv-2 rda-kafka-controller-0 rda-kafka-controller-1 '\
                         'rda-kafka-controller-2 rda-mariadb-0 rda-mariadb-1 rda-mariadb-2 rda-minio-pv-0 ' \
                        'rda-minio-pv-1 rda-minio-pv-2 rda-minio-pv-3 rda-minio-pv opensearch-backup-pv rda-portal-nats-keys-pv'
            execute_command(delete_pv)

            # stop infra
            infra_components = COMPONENT_REGISTRY.get_by_category(category='infra')
            logger.info('Initiating reset on infra components')
            for component in reversed(infra_components):
                logger.info('Resetting ' + component.get_name())
                component.k8s_reset(cmd_args, config_parser)
            
            if deployment_type == 'k8s':
                component = COMPONENT_REGISTRY.get('cfx_self_monitor')
                if component:
                    logger.info('Resetting ' + component.get_name())
                    component.reset(cmd_args, config_parser)

            # delete the install root
            pseudo_comp = COMPONENT_REGISTRY.require(pseudo_platform.PseudoComponent.COMPONENT_NAME)
            pseudo_comp.k8s_reset(cmd_args, config_parser)
        if not cmd_args.no_purge:
            if deployment_type == 'k8s':
               self.prune_images(config_parser)
            elif deployment_type == 'ctr':
                self.prune_ctr_images(config_parser)

        delete_labels = 'kubectl label nodes --all rdaf_application_services- rdaf_infra_minio- rdaf_infra_haproxy- ' \
                        'rdaf_infra_nats- rdaf_infra_graphdb- rdaf_infra_services- rdaf_platform_services- ' \
                        'rdaf_worker_services- rdaf_event_gateway- rdaf_monitoring_services- rdaf_infra_node- ' \
                        'rdaf_infra_minio_node- rdaf_bulk_stats_services- rdaf_infra_graphdb_node- rdaf_infra_nats_node-'
        execute_command(delete_labels)

        logger.info('RDAF reset completed successfully')
