import argparse
import configparser
import logging
import os
import shutil
import socket
import tarfile
import tempfile

import rdaf.cmd
import rdaf.cmd.reset
import rdaf.component.dockerregistry
import rdaf.component.pseudo_platform
import rdaf.component.ssh
from rdaf import rdafutils
from rdaf.cmd import CliCmdHandler
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.rdafutils import query_yes_no
from rdaf.component import remove_dir_contents, Component, run_command, do_potential_scp

logger = logging.getLogger(__name__)


class RestoreCmdHandler(CliCmdHandler):

    def configure_parser(self, parser: argparse.ArgumentParser):
        parser.add_argument('--no-prompt',
                            dest="no_prompt",
                            action='store_true',
                            default=False,
                            help='Don\'t prompt for inputs')

        parser.add_argument('--service',
                            dest="services",
                            action='append',
                            default=None,
                            help='Restore only the specified components')

        backed_up_src = parser.add_mutually_exclusive_group(required=True)
        backed_up_src.add_argument('--from-dir', dest='backup_src_dir',
                                   action='store',
                                   help='The directory which contains the backed up '
                                        'installation state')
        backed_up_src.add_argument('--from-tar',
                                   dest='backup_src_tar',
                                   action='store',
                                   help='The tar.gz file which contains the backed up '
                                        'installation state')

    def before_handle(self, cmd_args: argparse.Namespace,
                      config_parser: configparser.ConfigParser):
        # unlike other cmd handlers, we don't parse the rdaf.cfg and load the
        # components
        return

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not cmd_args.no_prompt:
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Restoring the system will result in data overwrites on the current system\n')
            print(warn_message)
            cancelled = True
            if query_yes_no('Are you sure you want to restore the system from a previous backup?'):
                if query_yes_no('Please confirm again?'):
                    cancelled = False
            if cancelled:
                logger.info('rdaf restore has been cancelled')
                return
        backup_content_dir_root: os.path
        tmp_dir: tempfile.TemporaryDirectory = None
        try:
            if cmd_args.backup_src_tar is not None:
                if not os.path.isfile(cmd_args.backup_src_tar):
                    rdafutils.cli_err_exit(cmd_args.backup_src_tar + ' is not a file')
                    return
                tmp_dir = tempfile.TemporaryDirectory(dir=os.path.dirname(cmd_args.backup_src_tar))
                # extract the tar to the temp directory
                with tarfile.open(os.path.abspath(cmd_args.backup_src_tar)) as tar:
                    tar.extractall(path=tmp_dir.name)
                backup_content_dir_root = tmp_dir.name
            else:
                if not os.path.isdir(cmd_args.backup_src_dir):
                    rdafutils.cli_err_exit(cmd_args.backup_src_dir + ' is not a directory')
                    return
                backup_content_dir_root = os.path.abspath(cmd_args.backup_src_dir)
            _assert_valid_backup_content_dir(backup_content_dir_root)
            # start with copying the rdaf.cfg to the install root
            logger.info('Starting restoration of the system, from a previous backed up content at '
                        + backup_content_dir_root)
            # load components using the backed up rdaf.cfg
            rdaf.cmd._read_configs(config_parser,
                                   os.path.join(backup_content_dir_root, 'rdaf.cfg'))
            self._load_all_components(config_parser)
            rdaf_cfg_file = os.path.abspath(os.path.join('/opt/rdaf', 'rdaf.cfg'))
            shutil.copyfile(os.path.join(backup_content_dir_root, 'rdaf.cfg'),
                            rdaf_cfg_file)
            logger.debug('Copied rdaf.cfg from backed up content to ' + rdaf_cfg_file)
            # restore each component
            self._do_restore(cmd_args, config_parser, backup_content_dir_root)
        finally:
            if tmp_dir is not None:
                try:
                    remove_dir_contents(Component.get_default_host(), tmp_dir.name,
                                        config_parser, use_sudo=True)
                    tmp_dir.cleanup()
                except Exception:
                    # ignore
                    pass

    def _do_restore(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_content_root_dir: os.path):
        components = []
        components.extend(COMPONENT_REGISTRY.get_by_category('other'))
        components.extend(COMPONENT_REGISTRY.get_by_category('infra'))
        components.extend(COMPONENT_REGISTRY.get_by_category('platform'))

        component_names = ['config']
        for component in components:
            component_names.append(component.get_name())
        specific_services = [] if cmd_args.services is None else cmd_args.services
        if len(specific_services) > 0:
            for entry in specific_services:
                if entry not in component_names:
                    rdafutils.cli_err_exit("Unknown component specified: " + entry)

        # read the rdaf-backup.cfg in the backed up content
        backup_cfg_parser = configparser.ConfigParser(allow_no_value=True)
        rdaf.cmd._read_configs(backup_cfg_parser,
                               os.path.join(backup_content_root_dir, 'rdaf-backup.cfg'))

        # create the install root hierarchy for all relevant hosts
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        pseudo_platform = COMPONENT_REGISTRY.require(
            rdaf.component.pseudo_platform.PseudoComponent.COMPONENT_NAME)
        pseudo_platform.setup_install_root_dir_hierarchy(all_known_hosts, config_parser)
        # restoring the rdaf.cfg to all hosts
        config_file = os.path.join('/opt', 'rdaf', 'rdaf.cfg')
        for host in all_known_hosts:
            if Component.is_local_host(host):
                continue
            do_potential_scp(host, config_file, config_file)
        if 'config' in specific_services:
            for component in components:
                logger.info('Restoring config of ' + component.get_name())
                component.restore_conf(config_parser, backup_content_root_dir)
        else:
            # now do the config and data restore of the components
            for component in components:
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    continue
                component.before_restore(config_parser)
            for component in components:
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    logger.debug('Skipping restoration of ' + component.get_name())
                    continue
                logger.info('Restoring config of ' + component.get_name())
                component.restore_conf(config_parser, backup_content_root_dir)
                logger.info('Restoring data of ' + component.get_name())
                component.restore_data(cmd_args, config_parser, backup_content_root_dir, backup_cfg_parser)


class K8SRestoreCmdHandler(CliCmdHandler):

    def configure_parser(self, parser: argparse.ArgumentParser):
        parser.add_argument('--no-prompt',
                            dest="no_prompt",
                            action='store_true',
                            default=False,
                            help='Don\'t prompt for inputs')

        parser.add_argument('--service',
                            dest="services",
                            action='append',
                            default=None,
                            help='Restore only the specified components')

        backed_up_src = parser.add_mutually_exclusive_group(required=True)
        backed_up_src.add_argument('--from-dir', dest='backup_src_dir',
                                   action='store',
                                   help='The directory which contains the backed up '
                                        'installation state')
        backed_up_src.add_argument('--from-tar',
                                   dest='backup_src_tar',
                                   action='store',
                                   help='The tar.gz file which contains the backed up '
                                        'installation state')

    def before_handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        pass

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not cmd_args.no_prompt:
            # ask for confirmation
            warn_message = rdafutils.center_text_on_terminal(
                'Restoring the system will result in data overwrites on the current system\n')
            print(warn_message)
            cancelled = True
            if query_yes_no('Are you sure you want to restore the system from a previous backup?'):
                if query_yes_no('Please confirm again?'):
                    cancelled = False
            if cancelled:
                logger.info('rdaf restore has been cancelled')
                return
        backup_content_dir_root: os.path
        tmp_dir: tempfile.TemporaryDirectory = None
        try:
            if cmd_args.backup_src_tar is not None:
                if not os.path.isfile(cmd_args.backup_src_tar):
                    rdafutils.cli_err_exit(cmd_args.backup_src_tar + ' is not a file')
                tmp_dir = tempfile.TemporaryDirectory(dir=os.path.dirname(cmd_args.backup_src_tar))
                # extract the tar to the temp directory
                with tarfile.open(os.path.abspath(cmd_args.backup_src_tar)) as tar:
                    tar.extractall(path=tmp_dir.name)
                backup_content_dir_root = tmp_dir.name
            else:
                if not os.path.isdir(cmd_args.backup_src_dir):
                    rdafutils.cli_err_exit(cmd_args.backup_src_dir + ' is not a directory')
                backup_content_dir_root = os.path.abspath(cmd_args.backup_src_dir)
            _assert_valid_backup_content_dir(backup_content_dir_root)
            # start with copying the rdaf.cfg to the install root
            logger.info('Starting restoration of the system, from a previous backed up content at '
                        + backup_content_dir_root)
            # load components using the backed up rdaf.cfg
            rdaf.cmd._read_configs(config_parser, os.path.join(backup_content_dir_root, 'rdaf.cfg'))
            self._load_all_components(config_parser)
            rdaf_cfg_file = os.path.abspath(os.path.join('/opt/rdaf', 'rdaf.cfg'))
            shutil.copyfile(os.path.join(backup_content_dir_root, 'rdaf.cfg'),
                            rdaf_cfg_file)
            logger.debug('Copied rdaf.cfg from backed up content to ' + rdaf_cfg_file)
            # restore each component
            self._do_restore(cmd_args, config_parser, backup_content_dir_root)
        finally:
            if tmp_dir is not None:
                try:
                    run_command('sudo rm -rf {}/*'.format(tmp_dir.name))
                    tmp_dir.cleanup()
                except Exception:
                    # ignore
                    pass

    @staticmethod
    def _do_restore(cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_content_root_dir: os.path):
        components = []
        components.extend(COMPONENT_REGISTRY.get_by_category('platform'))
        components.extend(COMPONENT_REGISTRY.get_by_category('other'))
        components.extend(COMPONENT_REGISTRY.get_by_category('infra'))

        component_names = ['config']
        for component in components:
            component_names.append(component.get_name())
        specific_services = [] if cmd_args.services is None else cmd_args.services
        if len(specific_services) > 0:
            for entry in specific_services:
                if entry not in component_names:
                    rdafutils.cli_err_exit("Unknown component specified: " + entry)

        # read the rdaf-backup.cfg in the backed up content
        backup_cfg_parser = configparser.ConfigParser(allow_no_value=True)
        rdaf.cmd._read_configs(backup_cfg_parser,
                               os.path.join(backup_content_root_dir, 'rdaf-backup.cfg'))

        pseudo_platform = COMPONENT_REGISTRY.require(
            rdaf.component.pseudo_platform.PseudoComponent.COMPONENT_NAME)
        pseudo_platform.setup_install_root_dir_hierarchy([socket.gethostname()], config_parser)

        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        # restoring the rdaf.cfg to all hosts
        config_file = os.path.join('/opt', 'rdaf', 'rdaf.cfg')
        for host in all_known_hosts:
            if Component.is_local_host(host):
                continue
            do_potential_scp(host, config_file, config_file)

        if 'config' in specific_services:
            for component in components:
                logger.info('Restoring config of ' + component.get_name())
                component.k8s_restore_conf(config_parser, backup_content_root_dir)
        else:
            # now do the config and data restore of the components
            for component in components:
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    continue
                component.check_pod_state_for_restore(config_parser)
            for component in components:
                component_name = component.get_name()
                if len(specific_services) > 0 and component_name not in specific_services:
                    logger.debug('Skipping restoration of ' + component.get_name())
                    continue
                logger.info('Restoring config of ' + component.get_name())
                component.k8s_restore_conf(config_parser, backup_content_root_dir)
                logger.info('Restoring data of ' + component.get_name())
                component.k8s_restore_data(cmd_args, config_parser, backup_content_root_dir, backup_cfg_parser)


def _assert_valid_backup_content_dir(content_dir: os.path):
    if not os.path.isdir(content_dir):
        rdafutils.cli_err_exit(content_dir + ' is not a directory')
    if not os.path.isfile(os.path.join(content_dir, 'rdaf-backup.cfg')):
        rdafutils.cli_err_exit('Backed up content at ' + content_dir
                               + ' is missing rdaf-backup.cfg')
    if not os.path.isfile(os.path.join(content_dir, 'rdaf.cfg')):
        rdafutils.cli_err_exit('Backed up content at ' + content_dir
                               + ' is missing rdaf.cfg')
    expected_dirs = ['config', 'cert', 'data', 'deployment-scripts']
    for d in expected_dirs:
        if not os.path.isdir(os.path.join(content_dir, d)):
            rdafutils.cli_err_exit('Backed up content at ' + content_dir
                                   + ' is missing "' + d + '" directory')
