import argparse
import configparser
import json
import logging
import os
import psutil
import shutil
import socket
import subprocess
from configparser import ConfigParser

import rdaf
from rdaf.cmd import CliCmdHandler, _read_configs
from rdaf.component import Component, do_potential_scp, run_potential_ssh_command, execute_command
from rdaf.component.cert import CertManager
from rdaf.component.pseudo_platform import PseudoComponent
from rdaf.component.ssh import SSHKeyManager
from rdaf.component.dockerregistry import DockerRegistry
from rdaf.component.proxy import Proxy
from rdaf.component.nats import Nats
from rdaf.component.graphdb import GraphDB
from rdaf.component.haproxy import HAProxy
from rdaf.component.worker import Worker
from rdaf.component.keepalived import Keepalived
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.component.minio import Minio
from rdaf.component.kafka import Kafka
from rdaf.component.opensearch import Opensearch
from rdaf.component.mariadb import MariaDB
from rdaf import rdafutils
from rdaf.component.event_gateway import EventGateway
from rdaf.rdafutils import cli_err_exit, str_base64_decode

logger = logging.getLogger(__name__)



class SetupCmdHandler(CliCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = False

    def configure_parser(self, parser):
        parser.add_argument('--no-prompt',
                            dest='no_prompt',
                            action='store_true',
                            default=False,
                            help='Don\'t prompt for inputs')

        parser.add_argument('--rda-edge',
                            dest='rda_edge',
                            action='store_true',
                            default=False,
                            help='Specify that setup is an RDA edge deployment')

        parser.add_argument('--config-file',
                            dest='config_file',
                            action='store',
                            default=None,
                            help='Specify the path to the file where all the input configs are present')
        
        parser.add_argument('--primary',
                            dest='primary',
                            action='store_true',
                            default=None,
                            help='Flag to indicate primary setup when setting up geo-dr')
        
        parser.add_argument('--secondary',
                            dest='secondary',
                            action='store_true',
                            default=None,
                            help='Flag to indicate secondary ssetup when setting up geo-dr')
        
        parser.add_argument('--primary-config',
                            dest='primary_config',
                            action='store',
                            default=None,
                            help='Specify the path to the file where the primary rdaf.cfg file is present')

        parser.add_argument('--accept-eula',
                            dest='accept_eula',
                            action='store_true',
                            default=False,
                            help='Accept the RDAF EULA')

        parser.add_argument('--advertised-external-host',
                            dest='advertised_ext_host',
                            action='store',
                            default=None,
                            help='Host name or IP address of the host which will be externally '
                                 ' used to access the installation')

        parser.add_argument('--advertised-external-interface',
                            dest='advertised_ext_interface',
                            action='store',
                            default=None,
                            help='The interface on which the platform should be externally '
                                 'accessible')

        parser.add_argument('--bind-internal-interface',
                            dest='bind_internal_inf',
                            action='store_true',
                            default=False,
                            help='Bind to any internal interface.')

        parser.add_argument('--advertised-internal-host',
                            dest='advertised_int_host',
                            action='store',
                            default=None,
                            help='Host name or IP address of the host which will be internally '
                                 ' used to access the installation')

        parser.add_argument('--advertised-internal-interface',
                            dest='advertised_int_interface',
                            action='store',
                            default=None,
                            help='The interface on which the platform should be internally '
                                 'accessible')

        parser.add_argument('--platform-service-host',
                            dest='platform_service_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' RDAF platform services will be installed')

        parser.add_argument('--app-services-ha',
                            dest='app_service_ha',
                            action='store_true',
                            default=False,
                            help='Are the app services be deployed in HA mode')
        
        parser.add_argument('--platform-services-ha',
                            dest='platform_service_ha',
                            action='store_true',
                            default=False,
                            help='Are the platform services be deployed in HA mode')

        parser.add_argument('--service-host',
                            dest='service_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where '
                                 'application services will be installed')

        parser.add_argument('--admin-organization',
                            dest='admin_organization',
                            action='store',
                            default=None,
                            help='Organization name to associate with the default admin user')

        parser.add_argument('--worker-host',
                            dest='worker_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where '
                                 'Worker containers will be installed')

        parser.add_argument('--event-gateway-host',
                            dest='rda_event_gateway_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where '
                                 'Event-Gateway containers will be installed')

        parser.add_argument('--http-proxy',
                            dest='http_proxy',
                            action='store',
                            default=None,
                            help='URL to any HTTP proxy to be used by this platform')
        parser.add_argument('--https-proxy',
                            dest='https_proxy',
                            action='store',
                            default=None,
                            help='URL to any HTTPS proxy to be used by this platform')
        parser.add_argument('--no-proxy',
                            dest='no_proxy',
                            action='store',
                            nargs="*",
                            default=None,
                            help='Any hosts that should be added to the no-proxy list')

        parser.add_argument('--ssh-user',
                            dest="ssh_user",
                            action='store',
                            default=_current_user(),
                            help='User name to use to do an initial SSH login'
                                 ' while setting up password-less SSH between hosts')

        parser.add_argument('--ssh-key',
                            dest='ssh_key',
                            action='store_true',
                            default=False,
                            help='Specify that setup is to use ssh key instead of password')

        parser.add_argument('--ssh-key-path',
                            dest='ssh_key_path',
                            action='store',
                            default=None,
                            help='Specify the ssh private key path to login between hosts')

        parser.add_argument('--ssh-password',
                            dest="ssh_password",
                            action='store',
                            default=None,
                            help='Password to use to do an initial SSH login'
                                 ' while setting up password-less SSH between hosts')

        parser.add_argument('--alt-names',
                            dest='alt_names',
                            action='store',
                            default="",
                            help='Alt name(s) if any to generate self signed SAN certs')

        parser.add_argument('--nats-host',
                            dest='nats_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' Nats needs to be installed')
        parser.add_argument('--minio-server-host',
                            dest='minio_server_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' Minio server for the platform will be'
                                 ' installed')

        parser.add_argument('--minio-user',
                            dest='minio_user',
                            action='store',
                            default=None,
                            help='User name for the root user that will be created'
                                 ' for Minio usage')

        parser.add_argument('--minio-password',
                            dest='minio_password',
                            action='store',
                            default=None,
                            help='Password to assign for the newly created Minio root'
                                 ' user')

        parser.add_argument('--opensearch-server-host',
                            dest='opensearch_server_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host '
                                 'where the opensearch server will be installed')

        parser.add_argument('--opensearch-user',
                            dest='opensearch_user',
                            action='store',
                            default=None,
                            help='User name for the admin user that will be created'
                                 ' for Opensearch platform\'s usage')

        parser.add_argument('--opensearch-password',
                            dest='opensearch_password',
                            action='store',
                            default=None,
                            help='Password to assign for the newly created Opensearch admin'
                                 ' user')

        parser.add_argument('--mariadb-server-host',
                            dest='mariadb_server_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' MariaDB server for the platform will be installed')

        parser.add_argument('--mariadb-user',
                            dest='mariadb_user',
                            action='store',
                            default=None,
                            help='User name for the admin user that will be created'
                                 ' for MariaDB platform\'s usage')

        parser.add_argument('--mariadb-password',
                            dest='mariadb_password',
                            action='store',
                            default=None,
                            help='Password to assign for MariaDB root user')

        parser.add_argument('--kafka-server-host',
                            dest='kafka_server_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' Kafka server to be installed')

        parser.add_argument('--graphdb-host',
                            dest='graphdb_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' GraphDB needs to be installed')

        parser.add_argument('--graphdb-user',
                            dest='graphdb_user',
                            action='store',
                            default=None,
                            help='User name to access GraphDB')

        parser.add_argument('--graphdb-password',
                            dest='graphdb_password',
                            action='store',
                            default=None,
                            help='Password to assign for GraphDb user')

        parser.add_argument('--haproxy-host',
                            dest='haproxy_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' HAProxy for the platform will be'
                                 ' installed')

        parser.add_argument('--docker-registry-ca',
                            dest='docker_registry_ca',
                            action='store',
                            default=None,
                            help='Use the ca cert specified to setup the hosts '
                                 'to interact to docker registry')

    def before_handle(self, cmd_args: argparse.Namespace,
                      config_parser: configparser.ConfigParser):
        return

    def handle(self, cmd_args, config_parser: configparser.ConfigParser):
        # check for the input file and update all the cmd args using that
        config_file = hasattr(cmd_args, 'config_file') and cmd_args.config_file
        if config_file:
            self.populate_from_json(config_file, cmd_args)

        if bool(cmd_args.secondary) != bool(cmd_args.primary_config):
            rdafutils.cli_err_exit("--secondary flag requires --primary-config to be specified and vice versa")
        if not cmd_args.accept_eula:
            if cmd_args.no_prompt:
                # accepting eula is mandatory
                rdafutils.cli_err_exit("EULA hasn't been accepted. Please use interactive "
                                       "mode to read and accept the EULA or use --accept-eula "
                                       "to accept the EULA in --no-prompt mode")
            # display the EULA
            eula = rdaf.read_eula_file()
            print(rdafutils.center_text_on_terminal(eula))
            if not rdafutils.query_yes_no("Do you accept the EULA?"):
                rdafutils.cli_err_exit('EULA not accepted. Exiting')

        # Check if --primary_config file is present and populate credentials
        primary_config_file = hasattr(cmd_args, 'primary_config') and cmd_args.primary_config
        if primary_config_file:
            self.populate_from_primary_config(primary_config_file, cmd_args)
            
        command = 'sudo mkdir -p /opt/rdaf  && sudo chown -R ' + str(
            os.getuid()) + ' /opt/rdaf && sudo chgrp -R ' + str(os.getuid()) + ' /opt/rdaf'
        completed_process = subprocess.run([command], cwd=os.getcwd(), shell=True, text=True)
        if completed_process.returncode != 0:
            rdafutils.cli_err_exit('Failed to create /opt/rdaf')

        rda_edge = hasattr(cmd_args, 'rda_edge') and cmd_args.rda_edge
        if rda_edge:
            components = [CertManager(), PseudoComponent(), DockerRegistry(), Worker(), EventGateway(),
                          Nats(), Minio(), MariaDB(), Opensearch(), GraphDB(), HAProxy()]
        elif hasattr(cmd_args, 'aws') and cmd_args.aws:
            components = [CertManager(), PseudoComponent(), DockerRegistry(), Worker(), Nats(), EventGateway(),
                          Minio(), MariaDB(), Opensearch(), Kafka(), GraphDB()]
        else:
            if self.k8s:
                components = [SSHKeyManager(), CertManager(), PseudoComponent(), DockerRegistry(), Proxy(),
                              Worker(), EventGateway(), Nats(), Minio(), MariaDB(), Opensearch(), Kafka(),
                              GraphDB(), HAProxy()]
            else:
                components = [SSHKeyManager(), CertManager(), PseudoComponent(), DockerRegistry(), Proxy(),
                              Worker(), EventGateway(), Nats(), Minio(), MariaDB(), Opensearch(), Kafka(),
                              GraphDB(), HAProxy(), Keepalived()]

        # register each of these components
        for component in components:
            COMPONENT_REGISTRY.register(component)

        config_parser_handle = ConfigParser(allow_no_value=True)
        skip_docker_reg_gather_setup_inputs = False
        # docker registries can potentially be configured even before rdaf setup
        # in invoked. Here we check if the platform configuration file exists and if it
        # does, we load only that component configs
        config_file = os.path.join('/opt', 'rdaf', 'rdaf.cfg')
        if os.path.isfile(config_file):
            throw_away_config_parser = ConfigParser(allow_no_value=True)
            _read_configs(throw_away_config_parser)
            # copy over docker registry sections into the setup command's
            # config parser
            docker_reg = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
            copy_section(docker_reg.section_name, throw_away_config_parser, config_parser_handle)
            if config_parser_handle.has_section(docker_reg.section_name):
                docker_reg.load_config(config_parser_handle)
                # already configured, do not overwrite or ask for inputs again
                skip_docker_reg_gather_setup_inputs = True
        # phase 1 of the setup - gather inputs for each component
        for component in components:
            if component.get_name() == rdaf.component.dockerregistry.COMPONENT_NAME:
                if rda_edge and not skip_docker_reg_gather_setup_inputs:
                    component.gather_minimal_setup_inputs(cmd_args, config_parser_handle)
                else:
                    component.gather_setup_inputs(cmd_args, config_parser_handle,
                                                  skip_setup=skip_docker_reg_gather_setup_inputs)
                continue
            logger.info('Gathering inputs for ' + component.component_name)
            if rda_edge:
                component.gather_minimal_setup_inputs(cmd_args, config_parser_handle)
            elif self.k8s:
                component.gather_k8s_setup_inputs(cmd_args, config_parser_handle)
            else:
                component.gather_setup_inputs(cmd_args, config_parser_handle)

        # validate these setup inputs
        for component in components:
            component.validate_setup_inputs(cmd_args, config_parser_handle)

        # now that the setup inputs have been gathered and validated, persist them
        if not config_parser_handle.has_section("rdaf-cli"):
            config_parser_handle.add_section("rdaf-cli")

        if rda_edge:
            deployment_type = 'rda-edge'
        elif not self.k8s:
            deployment_type = "non-k8s"
        else:
            maint_pods_command = "kubectl get nodes -o json | jq -r '.items[].status.nodeInfo.containerRuntimeVersion'"
            return_code, stdout, stderr = execute_command(maint_pods_command)
            if "containerd" in stdout:
                deployment_type = "ctr"
            else:
                deployment_type = "k8s"
            if cmd_args.aws:
                deployment_type = "aws"
            namespace = cmd_args.namespace if cmd_args.namespace else 'rda-fabric'
            config_parser_handle.set("rdaf-cli", 'namespace', namespace)
        config_parser_handle.set("rdaf-cli", 'deployment', deployment_type)

        cli_host = get_local_host()
        config_parser_handle.set("rdaf-cli", 'cli-host', cli_host)

        if hasattr(cmd_args, 'primary') and cmd_args.primary:
            config_parser_handle.set("rdaf-cli", 'primary', 'true')
        elif hasattr(cmd_args, 'secondary') and cmd_args.secondary:
            config_parser_handle.set("rdaf-cli", 'primary', 'false')

        Component.write_configs(config_parser_handle)
        destination_path = os.path.join('/opt', 'rdaf', 'rdaf-peer.cfg')
        if primary_config_file and destination_path != primary_config_file:
           logger.info(f"Copying the primary config file to {destination_path}")
           shutil.copyfile(primary_config_file, destination_path)

        for component in components:
            logger.info('Doing setup for ' + component.component_name)
            if self.k8s:
                component.do_k8s_setup(cmd_args, config_parser_handle)
            else:
                component.do_setup(cmd_args, config_parser_handle)

        # explicitly setting the cfg file to have 600 permissions
        os.chmod(config_file, 0o600)
        known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])

        if deployment_type != "aws":
            for host in known_hosts:
                if Component.is_local_host(host):
                    continue
                run_potential_ssh_command(host, command, config_parser_handle)
                do_potential_scp(host, config_file, config_file, sudo=True)

        if hasattr(cmd_args, 'secondary') and cmd_args.secondary:
            peer_hosts = set()
            config = configparser.ConfigParser(allow_no_value=True)
            with open(destination_path, 'r') as f:
                config.read_file(f)

            for section in config.sections():
                if section == 'docker' or 'host' not in config[section]:  # Skip the docker section
                    continue
                host_entries = set(config.get(section, 'host').split(','))
                peer_hosts.update(host_entries)

            ssh_manager = COMPONENT_REGISTRY.require(rdaf.component.ssh.SSHKeyManager.COMPONENT_NAME)
            ssh_password = ssh_manager.ssh_password
            for host in peer_hosts:
                ssh_manager.setup_keys_for_host(host, ssh_password)
        logger.info('Setup completed successfully')

    @staticmethod
    def populate_from_json(config_file, cmd_args):
        try:
            with open(config_file, 'r') as f:
                config = json.load(f)
            for key, value in config.items():
                attr_key = key.replace('-', '_')
                if "host" in attr_key.lower() and isinstance(value, str):
                    value_list = rdafutils.delimited_to_list(value)
                    if value_list:
                        setattr(cmd_args, attr_key, value_list)
                else:
                    setattr(cmd_args, key.replace('-', '_'), value)
            # explicitly set noprompt true
            if not 'no_prompt' in config:
                setattr(cmd_args, 'no_prompt', True)

        except json.JSONDecodeError:
            cli_err_exit(f"The file '{config_file}' is not a valid JSON file.")
        except FileNotFoundError:
            cli_err_exit(f"Error: The file '{config_file}' was not found.")
        except Exception as e:
            cli_err_exit(f"An error occurred while loading the config: {str(e)}")

    @staticmethod
    def populate_from_primary_config(primary_config, cmd_args):
        if not os.path.isfile(primary_config):
            cli_err_exit(f"Error: The file '{primary_config}' was not found.")
        rdaf_configs = configparser.ConfigParser(allow_no_value=True)
        try:
            with open(primary_config, 'r') as f:
                rdaf_configs.read_file(f)
            components = ['minio', 'mariadb', 'opensearch', 'graphdb']
            for component in components:
                if not rdaf_configs.has_section(component):
                    cli_err_exit(f"Error: Missing section '{component}' in the config file.")
                user = rdaf_configs.get(component, 'user')
                passwd = rdaf_configs.get(component, 'password')
                if user:
                    setattr(cmd_args, f'{component}_user', str_base64_decode(user))
                if passwd:
                    setattr(cmd_args, f'{component}_password', str_base64_decode(passwd))
            if not rdaf_configs.has_section('common'):
                cli_err_exit(f"Error: Missing section 'common' in the config file.")
            admin_organization = rdaf_configs.get('common', 'admin_organization')
            if admin_organization:
                cmd_args.admin_organization = admin_organization

        except configparser.Error as e:
            cli_err_exit(f"Error reading primary config file: {str(e)}")


class SetupK8SCmdHandler(SetupCmdHandler):

    def __init__(self):
        super().__init__()
        self.k8s = True

    def configure_parser(self, parser):
        parser.add_argument('--no-prompt',
                            dest='no_prompt',
                            action='store_true',
                            default=False,
                            help='Don\'t prompt for inputs')

        parser.add_argument('--accept-eula',
                            dest='accept_eula',
                            action='store_true',
                            default=False,
                            help='Accept the RDAF EULA')

        parser.add_argument('--config-file',
                            dest='config_file',
                            action='store',
                            default=None,
                            help='Specify the path to the file where all the input configs are present')
        
        parser.add_argument('--primary',
                            dest='primary',
                            action='store_true',
                            default=None,
                            help='Flag to indicate primary setup in rdaf.cfg')
        
        parser.add_argument('--secondary',
                            dest='secondary',
                            action='store_true',
                            default=None,
                            help='Flag to indicate secondary setup in rdaf.cfg')
        
        parser.add_argument('--primary-config',
                            dest='primary_config',
                            action='store',
                            default=None,
                            help='Specify the path to the file where the primary setup rdaf.cfg file is present')

        parser.add_argument('--advertised-external-host',
                            dest='advertised_ext_host',
                            action='store',
                            default=None,
                            help='Host name or IP address of the host which will be externally '
                                 ' used to access the installation')

        parser.add_argument('--advertised-external-interface',
                            dest='advertised_ext_interface',
                            action='store',
                            default=None,
                            help='The interface on which the platform should be externally '
                                 'accessible')

        parser.add_argument('--bind-internal-interface',
                            dest='bind_internal_inf',
                            action='store_true',
                            default=False,
                            help='Bind to any internal interface.')

        parser.add_argument('--advertised-internal-host',
                            dest='advertised_int_host',
                            action='store',
                            default=None,
                            help='Host name or IP address of the host which will be internally '
                                 ' used to access the installation')
        parser.add_argument('--advertised-internal-interface',
                            dest='advertised_int_interface',
                            action='store',
                            default=None,
                            help='The interface on which the platform should be internally '
                                 'accessible')

        parser.add_argument('--platform-service-host',
                            dest='platform_service_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' RDAF platform services will be installed')

        parser.add_argument('--app-services-ha',
                            dest='app_service_ha',
                            action='store_true',
                            default=False,
                            help='Are the app services be deployed in HA mode')

        parser.add_argument('--service-host',
                            dest='service_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where '
                                 'application services will be installed')

        parser.add_argument('--admin-organization',
                            dest='admin_organization',
                            action='store',
                            default=None,
                            help='Organization name to associate with the default admin user')

        parser.add_argument('--worker-host',
                            dest='worker_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where '
                                 'Worker containers will be installed')
        
        parser.add_argument('--event-gateway-host',
                            dest='rda_event_gateway_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where '
                                 'Event-Gateway containers will be installed')

        parser.add_argument('--http-proxy',
                            dest='http_proxy',
                            action='store',
                            default=None,
                            help='URL to any HTTP proxy to be used by this platform')

        parser.add_argument('--https-proxy',
                            dest='https_proxy',
                            action='store',
                            default=None,
                            help='URL to any HTTPS proxy to be used by this platform')

        parser.add_argument('--no-proxy',
                            dest='no_proxy',
                            action='store',
                            nargs="*",
                            default=None,
                            help='Any hosts that should be added to the no-proxy list')

        parser.add_argument('--aws',
                            dest='aws',
                            action='store_true',
                            default=False,
                            help='Specify that setup is for AWS cloud')

        parser.add_argument('--namespace',
                            dest='namespace',
                            default='rda-fabric',
                            help='Specify the namespace (default: rda-fabric)')
        
        parser.add_argument('--ssh-user',
                            dest="ssh_user",
                            action='store',
                            default=_current_user(),
                            help='User name to use to do an initial SSH login'
                                 ' while setting up password-less SSH between hosts')

        parser.add_argument('--ssh-password',
                            dest="ssh_password",
                            action='store',
                            default=None,
                            help='Password to use to do an initial SSH login'
                                 ' while setting up password-less SSH between hosts')

        parser.add_argument('--alt-names',
                            dest='alt_names',
                            action='store',
                            default="",
                            help='Alt name(s) if any to generate self signed SAN certs')

        parser.add_argument('--nats-host',
                            dest='nats_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' Nats needs to be installed')

        parser.add_argument('--minio-server-host',
                            dest='minio_server_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' Minio server for the platform will be'
                                 ' installed')

        parser.add_argument('--minio-user',
                            dest='minio_user',
                            action='store',
                            default=None,
                            help='User name for the root user that will be created'
                                 ' for Minio usage')

        parser.add_argument('--minio-password',
                            dest='minio_password',
                            action='store',
                            default=None,
                            help='Password to assign for the newly created Minio root'
                                 ' user')

        parser.add_argument('--opensearch-server-host',
                            dest='opensearch_server_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host '
                                 'where the opensearch server will be installed')

        parser.add_argument('--opensearch-user',
                            dest='opensearch_user',
                            action='store',
                            default=None,
                            help='User name for the admin user that will be created'
                                 ' for Opensearch platform\'s usage')

        parser.add_argument('--opensearch-password',
                            dest='opensearch_password',
                            action='store',
                            default=None,
                            help='Password to assign for the newly created Opensearch admin'
                                 ' user')

        parser.add_argument('--mariadb-server-host',
                            dest='mariadb_server_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' MariaDB server for the platform will be installed')

        parser.add_argument('--mariadb-user',
                            dest='mariadb_user',
                            action='store',
                            default=None,
                            help='User name for the admin user that will be created'
                                 ' for MariaDB platform\'s usage')

        parser.add_argument('--mariadb-password',
                            dest='mariadb_password',
                            action='store',
                            default=None,
                            help='Password to assign for MariaDB root user')

        parser.add_argument('--kafka-server-host',
                            dest='kafka_server_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' Kafka server to be installed')

        parser.add_argument('--graphdb-host',
                            dest='graphdb_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' GraphDB needs to be installed')

        parser.add_argument('--graphdb-user',
                            dest='graphdb_user',
                            action='store',
                            default=None,
                            help='User name to access GraphDB')

        parser.add_argument('--graphdb-password',
                            dest='graphdb_password',
                            action='store',
                            default=None,
                            help='Password to assign for GraphDB user')

        parser.add_argument('--haproxy-host',
                            dest='haproxy_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' HAProxy for the platform will be'
                                 ' installed')

        parser.add_argument('--docker-registry-ca',
                            dest='docker_registry_ca',
                            action='store',
                            default=None,
                            help='Use the ca cert specified to setup the hosts '
                                 'to interact to docker registry')



def get_local_host():
    def get_ip_for_interface(interface_name):
        for interface, snics in psutil.net_if_addrs().items():
            if interface.startswith(interface_name):
                for snic in snics:
                    if snic.family == socket.AF_INET:
                        return f"{snic.address}"
        return None

    def compare_ip_address(local_ips, known_host):
        known_host_prefix = '.'.join(known_host.split('.')[:3])
        for local_ip in local_ips:
            if '.'.join(local_ip.split('.')[:3]) == known_host_prefix:
                return local_ip
        return None

    ipv4s = []
    for interface, snics in psutil.net_if_addrs().items():
        ipv4s.extend([snic.address for snic in snics if snic.family == socket.AF_INET])

    platform = COMPONENT_REGISTRY.require(PseudoComponent.COMPONENT_NAME)
    known_host_to_compare = platform.get_platform_services_hosts()[0]
    cli_host = compare_ip_address(ipv4s, known_host_to_compare)
    if cli_host:
        return cli_host

    cli_host = get_ip_for_interface('eth0')
    if not cli_host:
        cli_host = get_ip_for_interface('ens160')
    if not cli_host:
        return socket.gethostname()
    return cli_host

def copy_section(section: str, src_config_parser: configparser.ConfigParser,
                 dest_config_parser: configparser.ConfigParser):
    if section is None:
        return
    if not src_config_parser.has_section(section):
        return
    src_options = src_config_parser.options(section)
    if src_options is None or len(src_options) == 0:
        return
    if not dest_config_parser.has_section(section):
        # create the section
        dest_config_parser.add_section(section)
    # copy over the options
    for option_name in src_options:
        dest_config_parser.set(section, option_name,
                               src_config_parser.get(section, option_name, fallback=None))
    return dest_config_parser


# Windows doesn't have the `pwd` module.
# So we try and use the USERNAME env variable
# to get the current user
def _current_user():
    import platform
    if platform.system() == 'Windows':
        return os.environ.get('USERNAME')
    else:
        import pwd
        return pwd.getpwuid(os.getuid()).pw_name
