import argparse
import configparser
import logging
import os
from typing import List

import termcolor

import rdaf.contextual
from rdaf import rdafutils
from rdaf.cmd import CliCmdHandler


logger = logging.getLogger(__name__)


class StatusCmdHandler(CliCmdHandler):

    def __init__(self, k8s=False):
        super().__init__()
        self.k8s = k8s

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
               category: str = None, components=None, target_file=None):
        if category is not None:
            components = rdaf.contextual.COMPONENT_REGISTRY.get_by_category(category)
        elif not components:
            components = rdaf.contextual.COMPONENT_REGISTRY.get_all_components()

        statuses = []
        if os.path.exists(os.path.join('/opt', 'rdaf', 'rdaf.cfg')):
            for component in components:
                logger.debug('Getting status of component ' + component.get_name())
                if self.k8s and component.get_name() == 'keepalived':
                   logger.debug('Skipping keepalived component in k8s')
                   continue
                component_status = component.k8s_status(cmd_args, config_parser) if self.k8s else \
                    component.status(cmd_args, config_parser)
                if component_status is None:
                    continue
                statuses.extend(component_status)
        elif os.path.exists(os.path.join('/opt', 'rdaf', 'rdaf-registry.cfg')):
            config_parser = configparser.ConfigParser(allow_no_value=True)
            registry_cfg = os.path.join('/opt', 'rdaf', 'rdaf-registry.cfg')
            with open(registry_cfg, 'r') as f:
                config_parser.read_file(f)
            docker_reg_mirror = rdaf.component.dockerregistrymirror.DockerRegistryMirror()
            docker_reg_mirror.load_config(config_parser)
            component_status = docker_reg_mirror.status(cmd_args, config_parser)
            statuses.extend(component_status)

        print_status(statuses, target_file=target_file, k8s=self.k8s)
        return


def print_status(component_statuses: List[dict], table_header_text=None, target_file=None, k8s=False):
    if component_statuses is None or len(component_statuses) == 0:
        return
    col_header = ['Name', 'Host', 'Status', 'Container Id', 'Tag']
    rows = []
    for status_entry in component_statuses:
        if 'status' in status_entry:
            status = status_entry['status']
            row = rdafutils.create_row_for_tabular_display(len(col_header))
            row[0] = status_entry['component_name']
            row[1] = status_entry['host']
            status_msg = status['message'] if 'message' in status else 'Unknown'
            if 'error' in status and status['error']:
                row[2] = termcolor.colored(status_msg, color='red')
            else:
                row[2] = status_msg
            row[3] = 'N/A'
            row[4] = 'N/A'
            rows.append(row)
            continue
        containers = status_entry['containers'] if 'containers' in status_entry else []
        if len(containers) == 0:
            row = rdafutils.create_row_for_tabular_display(len(col_header))
            row[0] = status_entry['component_name']
            row[1] = status_entry['host']
            row[2] = termcolor.colored('Not Provisioned', color='red')
            row[3] = 'N/A'
            row[4] = 'N/A'
            rows.append(row)
        else:
            for container in status_entry['containers']:
                row = rdafutils.create_row_for_tabular_display(len(col_header))
                row[0] = status_entry['component_name']
                row[1] = status_entry['host']
                if container['State'] != 'running':
                    status = termcolor.colored(container['Status'], color='red')
                else:
                    status = container['Status']
                row[2] = status
                row[3] = container['Id'].split('//')[-1][:12] if k8s else container['Id'][:12]
                row[4] = container['Image'].rpartition(':')[2]
                rows.append(row)
    if table_header_text:
        print(termcolor.colored(table_header_text, attrs=['bold']), file=target_file)
    rdafutils.print_tabular(col_header, rows, add_row_spacing=True, target_file=target_file)
