import argparse
import configparser
import logging.config
import os

import termcolor
import rdaf.component.ssh
import rdaf.component.proxy
import rdaf.component.dockerregistry
import rdaf.component.telegraf

from rdaf import rdafutils, InvalidCmdUsageException
from rdaf.cmd import CliCmdHandler, _read_configs
from rdaf.component import run_command, Component, run_potential_ssh_command
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.rdafutils import query_yes_no

logger = logging.getLogger(__name__)


def get_config_file_path() -> os.path:
    return os.path.join('/opt/rdaf-telegraf/', 'rdaf-telegraf.cfg')


class TelegrafCmdHandler(CliCmdHandler):

    def configure_parser(self, parser):
        telegraf_commands_parser = parser.add_subparsers(dest='telegraf_cmd')
        set_registry_parser = telegraf_commands_parser.add_parser('setregistry', help='set docker registry '
                                                                          'to pull images from')
        set_registry_parser.add_argument('--host',
                            dest="docker_registry_host",
                            required=True,
                            action='store',
                            help='Hostname/IP of the Docker registry')
        set_registry_parser.add_argument('--port',
                            dest="docker_registry_port",
                            required=True,
                            action='store',
                            default='',
                            help='Port of the Docker registry')
        set_registry_parser.add_argument('--user',
                            dest="docker_registry_user",
                            required=False,
                            action='store',
                            help='Username of the Docker registry')
        set_registry_parser.add_argument('--password',
                            dest="docker_registry_password",
                            required=False,
                            action='store',
                            help='Password of the Docker registry')
        set_registry_parser.add_argument('--cert-path',
                            dest="cert_path",
                            required=False,
                            action='store',
                            help='path of the Docker registry ca cert')

        setup_parser = telegraf_commands_parser.add_parser('setup', help='Setup Telegraf monitoring')
        setup_parser.add_argument('--telegraf-host',
                            dest='telegraf_host',
                            action='append',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' Telegraf server to be installed')

        setup_parser.add_argument('--ssh-user',
                            dest="ssh_user",
                            action='store',
                            default=_current_user(),
                            help='User name to use to do an initial SSH login'
                                 ' while setting up password-less SSH between hosts')
        
        setup_parser.add_argument('--ssh-password',
                            dest="ssh_password",
                            action='store',
                            default=None,
                            help='Password to use to do an initial SSH login'
                                 ' while setting up password-less SSH between hosts')
        
        setup_parser.add_argument('--project-id',
                            dest="project_id",
                            action='store',
                            default=None,
                            help='Project Id for the current tag metrics')
        
        setup_parser.add_argument('--kafka-host',
                            dest="kafka_host",
                            action='store',
                            default=None,
                            help='Host name or IP address of the host where the'
                                 ' Kafka server to be installed')

        setup_parser.add_argument('--kafka-user',
                            dest="kafka_user",
                            action='store',
                            default=None,
                            help='User name for the admin user that will be created'
                                 ' for Kafka usage')

        setup_parser.add_argument('--kafka-password',
                            dest="kafka_password",
                            action='store',
                            default=None,
                            help='Password to assign for Kafka root user')

        setup_parser.add_argument('--cert-dir',
                            dest="cert_dir",
                            action='store',
                            default=None,
                            help='Path for certs to be used')

        setup_parser.add_argument('--no-prompt',
                                  dest='no_prompt',
                                  action='store_true',
                                  default=False,
                                  help='Don\'t prompt for inputs')

        upgrade_parser = telegraf_commands_parser.add_parser('upgrade', help='Upgrade Telegraf components')
        upgrade_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='Docker image tag')

        install_parser = telegraf_commands_parser.add_parser('install',
                                                        help='Install Telegraf components')
        install_parser.add_argument('--tag',
                                    dest='tag',
                                    action='store',
                                    required=True,
                                    help='Docker image tag')

        parser.add_argument('--docker-registry-ca',
                            dest='docker_registry_ca',
                            action='store',
                            default=None,
                            help='Use the ca cert specified to setup the hosts '
                                 'to interact to docker registry')

        telegraf_commands_parser.add_parser('up', help='Create the Telegraf Containers')

        down_parser = telegraf_commands_parser.add_parser('down', help='Deleting the Telegraf Containers')
        down_parser.add_argument('--no-prompt',
                                  dest='no_prompt',
                                  action='store_true',
                                  default=False,
                                  help='Don\'t prompt for inputs')

        telegraf_commands_parser.add_parser('start', help='Start the Telegraf Containers')

        stop_parser = telegraf_commands_parser.add_parser('stop', help='Stop the Telegraf Containers')
        stop_parser.add_argument('--no-prompt',
                                         dest='no_prompt',
                                         action='store_true',
                                         default=False,
                                         help='Don\'t prompt for inputs')

        telegraf_commands_parser.add_parser('status', help='Status of the Telegraf components')

        reset_parser = telegraf_commands_parser.add_parser('reset', help='Reset the Telegraf components')
        reset_parser.add_argument('--no-prompt',
                                  dest='no_prompt',
                                  action='store_true',
                                  default=False,
                                  help='Don\'t prompt for inputs')

    def before_handle(self, cmd_args: argparse.Namespace,
                      config_parser: configparser.ConfigParser):
        # skip if we are in "rdaf setup" command
        if (hasattr(cmd_args, 'telegraf_cmd') and
                (cmd_args.telegraf_cmd == 'setup' or cmd_args.telegraf_cmd == 'setregistry')):
            return
        # read the configuration file into the configparser
        cfg_file = get_config_file_path()
        logger.debug('Using configuration file ' + cfg_file)
        if not os.path.isfile(cfg_file):
            rdafutils.cli_err_exit('Missing ' + cfg_file + '. Please run "rdaf telegraf setup"')
        _read_configs(config_parser, cfg_file=cfg_file)
        # load components
        components = [rdaf.component.proxy.Proxy(), rdaf.component.dockerregistry.DockerRegistry(),
                      rdaf.component.ssh.SSHKeyManager(), rdaf.component.telegraf.Telegraf()]
        # register each of these components
        for component in components:
            rdaf.contextual.COMPONENT_REGISTRY.register(component)
        # let all the components load their configs
        for component in components:
            component.load_config(config_parser)
        return

    def handle(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        telegraf_cmd = cmd_args.telegraf_cmd
        if telegraf_cmd == 'setregistry':
            components = [rdaf.component.dockerregistry.DockerRegistry()]
            for component in components:
                COMPONENT_REGISTRY.register(component)
            docker_reg = rdaf.contextual.COMPONENT_REGISTRY.require(
                rdaf.component.dockerregistry.COMPONENT_NAME)
            config_parser = configparser.ConfigParser(allow_no_value=True)
            _read_configs(config_parser, cfg_file=get_config_file_path())
            docker_reg.set_telegraf_docker_registry(cmd_args, config_parser, get_config_file_path())
        elif telegraf_cmd == 'setup':
            self.setup(cmd_args, config_parser)
        elif telegraf_cmd == 'install':
            component = COMPONENT_REGISTRY.get('telegraf')
            logger.info('Installing ' + component.get_name())
            component.install(cmd_args, config_parser)
            self._handle_status(cmd_args, config_parser)
        elif telegraf_cmd == 'upgrade':
            component = COMPONENT_REGISTRY.get('telegraf')
            logger.info('Upgrading ' + component.get_name())
            component.upgrade(cmd_args, config_parser)
            self._handle_status(cmd_args, config_parser)
        elif telegraf_cmd == 'status':
            self._handle_status(cmd_args, config_parser)
        elif telegraf_cmd == 'reset':
            cfg_path=os.path.join('/opt', 'rdaf-telegraf', 'rdaf-telegraf.cfg')
            if os.path.exists(cfg_path):
                self.reset(cmd_args, cfg_path, config_parser)
        elif telegraf_cmd == 'down':
            # ask for confirmation
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to delete the Telegraf container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('Telegraf down operation has been cancelled')
                    return
            component = COMPONENT_REGISTRY.get('telegraf')
            logger.info('Deleting ' + component.get_name())
            component.down(cmd_args, config_parser)
        elif telegraf_cmd == 'up':
            component = COMPONENT_REGISTRY.get('telegraf')
            component.up(cmd_args, config_parser)
            self._handle_status(cmd_args,config_parser)
        elif telegraf_cmd == 'stop':
            cancelled = True
            if not cmd_args.no_prompt:
                if rdafutils.query_yes_no(
                        "Are you sure you want to stop the Telegraf container(s)?"):
                    if rdafutils.query_yes_no("Please confirm again?"):
                        cancelled = False
                if cancelled:
                    logger.info('Telegraf stop operation has been cancelled')
                    return
            component = COMPONENT_REGISTRY.get('telegraf')
            logger.info('Stopping ' + component.get_name())
            component.stop(cmd_args, config_parser)
        elif telegraf_cmd == 'start':
            component = COMPONENT_REGISTRY.get('telegraf')
            logger.info('Starting ' + component.get_name())
            component.start(cmd_args, config_parser)
            self._handle_status(cmd_args, config_parser)
        else:
            raise InvalidCmdUsageException()

    def setup(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self._setup_install_root_dir_hierarchy()
        components = [rdaf.component.ssh.SSHKeyManager(), rdaf.component.proxy.Proxy(),
                      rdaf.component.dockerregistry.DockerRegistry(),rdaf.component.telegraf.Telegraf()]
        # register each of these components
        for component in components:
            COMPONENT_REGISTRY.register(component)
        # phase 1 of the setup - gather inputs for each component
        for component in components:
            logger.info('Gathering inputs for ' + component.component_name)
            component.gather_setup_inputs(cmd_args, config_parser)
        # validate these setup inputs
        for component in components:
            component.validate_setup_inputs(cmd_args, config_parser)
        # now that the setup inputs have been gathered and validated, persist them
        cfg_file = get_config_file_path()
        _read_configs(config_parser, cfg_file=cfg_file)
        config_parser.add_section('common')
        config_parser.set('common', 'install_root', os.path.join('/opt', 'rdaf-telegraf'))
        Component.write_configs(config_parser, config_file=cfg_file)
        # phase 2 of the setup - do the necessary setup for each component
        # Note that this doesn't involve installing any components
        for component in components:
            logger.info('Doing setup for ' + component.component_name)
            component.load_config(config_parser)
            component.do_setup(cmd_args, config_parser)
        Component.write_configs(config_parser, config_file=cfg_file)
        os.chmod(cfg_file, 0o600)
        logger.info('Setup completed successfully, configuration written to ' + cfg_file)
        return

    @staticmethod
    def _setup_install_root_dir_hierarchy():
        uid = os.getuid()
        gid = os.getgid()
        install_root = os.path.join('/opt', 'rdaf-telegraf')
        dirs = [install_root, os.path.join(install_root, 'config'),
                os.path.join(install_root, 'certs'), os.path.join(install_root, 'deployment-scripts')]
        for path in dirs:
            logger.info('Creating directory ' + path + ' and setting ownership to user '
                        + str(uid) + ' and group to group ' + str(gid))
            command = 'sudo mkdir -p ' + path + ' && sudo chown -R ' + str(
                uid) + ' ' + path + ' && sudo chgrp -R ' + str(
                gid) + ' ' + path
            run_command(command)

    def _handle_status(self, cmd_args, config_parser):
        component_statuses=[]
        component = COMPONENT_REGISTRY.get('telegraf')
        component_statuses.extend(component.status(cmd_args, config_parser))
        if component_statuses is None or len(component_statuses) == 0:
            return
        col_header = ['Name', 'Host', 'Status', 'Container Id', 'Tag']
        rows = []
        for status_entry in component_statuses:
            if 'status' in status_entry:
                status = status_entry['status']
                row = rdafutils.create_row_for_tabular_display(len(col_header))
                row[0] = status_entry['component_name']
                row[1] = status_entry['host']
                status_msg = status['message'] if 'message' in status else 'Unknown'
                if 'error' in status and status['error']:
                    row[2] = termcolor.colored(status_msg, color='red')
                else:
                    row[2] = status_msg
                row[3] = 'N/A'
                row[4] = 'N/A'
                rows.append(row)
                continue
            containers = status_entry['containers'] if 'containers' in status_entry else []
            if len(containers) == 0:
                row = rdafutils.create_row_for_tabular_display(len(col_header))
                row[0] = status_entry['component_name']
                row[1] = status_entry['host']
                row[2] = termcolor.colored('Not Provisioned', color='red')
                row[3] = 'N/A'
                row[4] = 'N/A'
                rows.append(row)
            else:
                for container in status_entry['containers']:
                    row = rdafutils.create_row_for_tabular_display(len(col_header))
                    row[0] = status_entry['component_name']
                    row[1] = status_entry['host']
                    if container['State'] != 'running':
                        status = termcolor.colored(container['Status'], color='red')
                    else:
                        status = container['Status']
                    row[2] = status
                    row[3] = container['Id'][:12]
                    row[4] = container['Image'].rpartition(':')[2]
                    rows.append(row)
        rdafutils.print_tabular(col_header, rows, add_row_spacing=True)

    @staticmethod
    def reset(cmd_args, telegraf_cfg, config_parser):
        # ask for confirmation
        if not cmd_args.no_prompt:
            warn_message = rdafutils.center_text_on_terminal(
                'Do you want to delete all the Telegraf components and data\n')
            print(warn_message)
            cancelled = True
            if query_yes_no("Are you sure you want to reset the Telegraf?"):
                if query_yes_no("Please confirm again?"):
                    cancelled = False
            if cancelled:
                logger.info('Telegraf reset has been cancelled')
                return

        component = COMPONENT_REGISTRY.get('telegraf')
        logger.info('Initiating reset on Telegraf components')
        involved_hosts = set()
        logger.info('Resetting ' + component.get_name())
        component.reset(cmd_args, config_parser)
        involved_hosts.update(component.get_hosts())

        command = 'sudo find /opt/rdaf-telegraf/. -name . -o -prune -exec rm -rf -- {} +'
        for host in involved_hosts:
            run_potential_ssh_command(host, command, config_parser)
            run_potential_ssh_command(host, 'docker image prune -a -f', config_parser)

        if os.path.exists('/opt/rdaf-telegraf/'):
            run_command(command)

# Windows doesn't have the `pwd` module.
# So we try and use the USERNAME env variable
# to get the current user
def _current_user():
    import platform
    if platform.system() == 'Windows':
        return os.environ.get('USERNAME')
    else:
        import pwd
        return pwd.getpwuid(os.getuid()).pw_name