import argparse
import base64
import configparser
import datetime
import hashlib
import json
import random

import psutil
import logging
import os
import shutil
import distro
import socket
import string
import subprocess
import tempfile
import textwrap

from cryptography.fernet import Fernet
from dateutil import parser
from enum import IntEnum
from typing import Callable, Any, List, Union, Tuple

import yaml

from rdaf import rdafutils
import rdaf
from rdaf.dockerutils import cliDockerSession, clientDockerSession
from rdaf.rdafutils import cliHost, cliSSH, cli_err_exit

logger = logging.getLogger(__name__)


class InfraCategoryOrder(IntEnum):
    NATS = 1,
    MINIO = 2,
    MARIADB = 3,
    OPENSEARCH = 4,
    KAFKA = 5,
    GRAPHDB = 6,
    HAPROXY = 7,
    KEEPALIVED = 8,
    NGINX = 9


class PlatformCategoryOrder(IntEnum):
    PLATFORM = 1

class AppCategoryOrder(IntEnum):
    OIA = 1

class OtherCategoryOrder(IntEnum):
    SSH_KEY_MANAGER = 1,
    PSEUDO_COMPONENT = 2,
    WORKER = 3,
    EVENT_GATEWAY=4,
    CERT_MANAGER = 5,
    LOG_MONITORING = 6,
    BULK_STATS = 7,
    SELF_MONITORING = 8,
    OPENSEARCH_EXTERNAL = 9
    FILE_OBJECT = 10


def _host_dir_loader(val: str) -> list:
    converted = []
    for host_dir in _comma_delimited_to_list(val):
        host, data_dirs = _split_as_host_dir(host_dir)
        converted.append((host, data_dirs))
    return converted


def _host_dir_storer(val: list) -> str:
    converted = ''
    first_val = True
    for host, data_dirs in val:
        if not first_val:
            converted += ','
        if data_dirs is None:
            converted += host
        else:
            converted += host + _list_to_plus_delimited(data_dirs)
        first_val = False
    return converted


def _comma_delimited_to_list(val: str) -> list or None:
    if val is None:
        return None
    return rdafutils.delimited_to_list(val)


def _list_to_comma_delimited(val: list) -> str or None:
    if val is None:
        return None
    return ','.join(val)


def _list_to_plus_delimited(val: list) -> str or None:
    if val is None:
        return None
    return '+'.join(val)


def _int_to_str(val: int) -> str or None:
    if val is None:
        return None
    return str(val)


def _str_to_int(val: str) -> int or None:
    if val is None:
        return None
    return int(val)


class Component:
    def __init__(self, name, section_name, category=None, category_order: int = 1000):
        self.component_name = name
        self.configs = None
        self.section_name = section_name
        self.category = category
        self.category_order = category_order

    def get_name(self):
        return self.component_name

    def get_category(self):
        return self.category

    def get_k8s_component_name(self):
        return self.component_name

    def get_k8s_chart_name(self):
        return self.component_name

    def gather_setup_inputs(self, cmd_args, config_parser):
        return

    def gather_minimal_setup_inputs(self, cmd_args, config_parser):
        pass

    def gather_k8s_setup_inputs(self, cmd_args, config_parser):
        self.gather_setup_inputs(cmd_args, config_parser)
        return

    def validate_setup_inputs(self, cmd_args, config_parser):
        return

    def do_setup(self, cmd_args, config_parser):
        return

    def do_k8s_setup(self, cmd_args, config_parser):
        return

    @staticmethod
    def get_deployment_type(config_parser: configparser.ConfigParser):
        if not config_parser.has_section("rdaf-cli"):
            deployment_type = "k8s"
        else:
            deployment_type = config_parser.get("rdaf-cli", "deployment")
        return deployment_type

    @staticmethod
    def get_namespace(config_parser: configparser.ConfigParser):
        if config_parser.has_option("rdaf-cli", "namespace"):
            return config_parser.get("rdaf-cli", "namespace")

        return 'rda-fabric'

    @staticmethod
    def is_geodr_deployment(config_parser: configparser.ConfigParser) -> bool:
        return config_parser.has_option("rdaf-cli", 'primary')

    def get_hosts(self) -> list:
        return []

    def get_ports(self) -> tuple:
        return ()

    def load_config(self, config_parser: configparser.ConfigParser):
        self.configs = dict()
        default_configs = self._init_default_configs()
        if default_configs is not None:
            self.configs.update(default_configs)
        if not config_parser.has_section(self.section_name):
            return
        configured_option_names = config_parser.options(self.section_name)
        if len(configured_option_names) == 0:
            return
        for configured_option_name in configured_option_names:
            option_val = config_parser.get(self.section_name, configured_option_name)
            config_loader = self._get_config_loader(configured_option_name)
            if config_loader is None:
                self.configs[configured_option_name] = option_val
            else:
                # call the config_loader to load/convert the option value
                self.configs[configured_option_name] = config_loader(option_val)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        return None

    def _init_default_configs(self) -> dict:
        return dict()

    def _mark_configured(self, configs, config_parser):
        self.configs = configs
        self.store_config(config_parser)

    def store_config(self, config_parser):
        """
        :type config_parser: configparser.ConfigParser
        """
        if self.configs is None:
            return
        if not config_parser.has_section(self.section_name):
            config_parser.add_section(self.section_name)
        for k, v in self.configs.items():
            config_val = v
            config_storer = self._get_config_storer(k)
            if config_storer is None:
                # TODO: remove this once all components specify proper config loader and storers
                if isinstance(config_val, list):
                    config_parser.set(self.section_name, k, _list_to_comma_delimited(config_val))
                elif isinstance(config_val, int):
                    config_parser.set(self.section_name, k, _int_to_str(config_val))
                else:
                    config_parser.set(self.section_name, k, config_val)
            else:
                # call the config_storer to convert the value for storing
                val_to_store = config_storer(config_val)
                config_parser.set(self.section_name, k, val_to_store)

    def get_deployment_env(self, host: str) -> dict:
        return dict()

    def _get_docker_repo(self) -> dict:
        env = dict()
        from rdaf.component.dockerregistry import COMPONENT_NAME
        from rdaf.contextual import COMPONENT_REGISTRY
        docker_registry = COMPONENT_REGISTRY.require(COMPONENT_NAME)
        env['DOCKER_REPO'] = docker_registry.get_docker_registry_url()
        return env

    def _get_infra_component_replacements(self, cmd_args: argparse.Namespace, host: str) -> dict:
        replacements = self._get_docker_repo()
        replacements['TAG'] = cmd_args.tag
        replacements.update(self.get_deployment_env(host))
        return replacements

    def get_deployment_replacements(self, cmd_args: argparse.Namespace) -> dict:
        replacements = self._get_docker_repo()
        replacements['TAG'] = cmd_args.tag
        return replacements

    def create_compose_file(self, cmd_args: argparse.Namespace, host: str, component_name=None) -> os.path:
        if not component_name:
            component_name = self.component_name
        compose_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', host,
                                    self.get_deployment_file_name())
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(),
                                 self.get_deployment_file_name())
        if self.get_category() == 'infra':
            replacements = self._get_infra_component_replacements(cmd_args, host)
        else:
            replacements = self.get_deployment_replacements(cmd_args)

        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        values = yaml.safe_load(original_content)
        component_value = values['services'][component_name]
        if 'environment' in component_value.keys():
            self.update_cluster_env(component_value, host)
        # override with values.yaml inputs if provided any
        self.apply_user_inputs(component_name, component_value)
        if not os.path.exists(compose_file):
            content = {"services": {component_name: component_value}}
        else:
            with open(compose_file, 'r') as f:
                content = yaml.safe_load(f)
                content['services'][component_name] = component_value

        run_command('mkdir -p ' + os.path.dirname(compose_file))
        with open(compose_file, 'w') as f:
            yaml.safe_dump(content, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        if not Component.is_local_host(host):
            do_potential_scp(host, compose_file, compose_file)
        return compose_file

    def get_deployment_file_name(self) -> os.path:
        if self.get_category() == 'infra':
            return 'infra.yaml'
        elif self.get_category() == 'platform':
            return 'platform.yaml'
        return None
        
    def generate_ipv6_network_block(self):
        daemon_file = os.path.join('/etc', 'docker', 'daemon.json')
        with open(daemon_file, 'r') as f:
            daemon_config = json.load(f)
        if not daemon_config.get("ipv6", False):
            return
        return { 'driver': 'bridge', 'enable_ipv6': True }

    def apply_user_inputs(self, component_name: str, component_yaml: dict):
        values_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if not os.path.exists(values_path):
            return
        with open(values_path, 'r') as f:
            values = yaml.safe_load(f)

        # for now handling mem_limit, memswap_limit, privileged
        if component_name not in values['services']:
            return

        user_inputs = values['services'][component_name]
        if 'mem_limit' in user_inputs:
            component_yaml['mem_limit'] = user_inputs['mem_limit']
        if 'memswap_limit' in user_inputs:
            component_yaml['memswap_limit'] = user_inputs['memswap_limit']
        if 'privileged' in user_inputs:
            component_yaml['privileged'] = user_inputs['privileged']
        if 'cap_add' in user_inputs:
            if not isinstance(user_inputs['cap_add'], list):
                cli_err_exit(f"cap_add must be of type list for service {component_name}")
            component_yaml['cap_add'] = user_inputs['cap_add']

        # handling environment
        user_env = user_inputs.get('environment', {})
        configs = configparser.ConfigParser(allow_no_value=True)
        with open(os.path.join('/opt', 'rdaf', 'rdaf.cfg'), 'r') as f:
            configs.read_file(f)
        if configs.has_option('rdaf-cli', 'primary'):
            if 'IS_GEODR_ENABLED' not in user_env:
                user_env['IS_GEODR_ENABLED'] = 'true'

        if user_env:
            env = component_yaml.get('environment', {})
            if type(env) is dict:
                for key, value in user_env.items():
                    env[key] = value
            else:
                # convert list to dict
                env_dict = {}
                for item in env:
                    k, v = item.split("=", 1)
                    env_dict[k] = v
                # update/overwrite entries
                for key, value in user_env.items():
                    env_dict[key] = value
                env = []
                # convert back to list
                for key, value in env_dict.items():
                    env.append(key + '=' + str(value))

            if len(env) > 0:
                component_yaml['environment'] = env

    def copy_logging_config(self, config_parser: configparser.ConfigParser,
                            host: str, service: str):
        dest_log_config = os.path.join(self.get_rdaf_install_root(), 'config', 'log',
                                       service, 'logging.yaml')
        if check_potential_remote_file_exists(host, dest_log_config):
            return

        logging_yaml = os.path.join(rdaf.get_templates_dir_root(), 'logging.yaml')
        with open(logging_yaml, 'r') as f:
            content = string.Template(f.read()).safe_substitute(pod_type=service)

        command = 'mkdir -p ' + os.path.dirname(dest_log_config)
        run_potential_ssh_command(host, command, config_parser)
        rdaf.component.create_file(host, content.encode(encoding='UTF-8'), dest_log_config)

    def get_k8s_install_args(self, cmd_args):
        return ''

    def update_k8s_values_yaml(self, values={}, component_values={}):
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        with open(values_yaml, 'r') as f:
            data = yaml.safe_load(f)

        for key, value in values.items():
            data[key] = value

        for key, value in component_values.items():
            data[self.get_k8s_chart_name()][key] = value

        with open(values_yaml, 'w') as f:
            yaml.safe_dump(data, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)
            
    def get_images_involved(self):
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), self.get_deployment_file_name())
        with open(template_path, 'r') as f:
            template_content = f.read()
        values = yaml.safe_load(template_content)
        image = values['services'][self.component_name]['image']
        image_name = image.rsplit(':', 1)[0].split('/', 1)[1]
        return [image_name]

    def pull_images(self, cmd_args, config_parser):
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            for image_name in self.get_images_involved():
                docker_pull_command = f'docker pull {docker_repo}/{image_name}:{cmd_args.tag}'
                run_potential_ssh_command(host, docker_pull_command, config_parser)

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.open_ports(config_parser)
        for host in self.get_hosts():
            compose_file_path = self.create_compose_file(cmd_args, host)
            command = '/usr/local/bin/docker-compose --project-name infra -f {file} up -d {service}' \
                    .format(file=compose_file_path, service=self.component_name)
            run_potential_ssh_command(host, command, config_parser)

    def k8s_pull_images(self, cmd_args, config_parser):
        pass

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        install_command = 'helm install  --create-namespace -n {} -f {} {} {} {} '\
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(), deployment_path)
        run_command(install_command)

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            compose_file_path = self.create_compose_file(cmd_args, host)
            command = '/usr/local/bin/docker-compose --project-name infra -f {file} up -d {service}' \
                    .format(file=compose_file_path, service=self.component_name)
            run_potential_ssh_command(host, command, config_parser)

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        upgrade_command = 'helm upgrade --install --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(), deployment_path)
        run_command(upgrade_command)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        # we down in reverse order
        for host in reversed(self.get_hosts()):
            compose_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', host,
                                        self.get_deployment_file_name())
            if not os.path.exists(compose_file):
                continue
            if self.component_name not in self.get_involved_services(compose_file):
                continue

            command = '/usr/local/bin/docker-compose --project-name infra -f ' + compose_file \
                      + ' rm -fsv ' + self.component_name
            run_potential_ssh_command(host, command, config_parser)

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            compose_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', host,
                                        self.get_deployment_file_name())
            if not os.path.exists(compose_file):
                continue
            if self.component_name not in self.get_involved_services(compose_file):
                continue

            command = '/usr/local/bin/docker-compose --project-name infra -f ' + compose_file + ' up -d ' \
                      + self.component_name
            run_potential_ssh_command(host, command, config_parser)

    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in reversed(self.get_hosts()):
            compose_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', host,
                            self.get_deployment_file_name())
            if not os.path.exists(compose_file):
                return
            if self.component_name not in self.get_involved_services(compose_file):
                continue
            logger.info("Stopping service: {} on host {}".format(self.component_name, host))
            command = '/usr/local/bin/docker-compose --project-name infra -f ' + compose_file + \
                      ' stop ' + self.component_name
            run_potential_ssh_command(host, command, config_parser)

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            compose_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', host,
                                        self.get_deployment_file_name())
            if not os.path.exists(compose_file):
                continue
            if self.component_name not in self.get_involved_services(compose_file):
                continue

            logger.info("Starting service: {} on host {}".format(self.component_name, host))
            command = '/usr/local/bin/docker-compose --project-name infra -f ' + compose_file \
                        + ' start ' + self.component_name
            run_potential_ssh_command(host, command, config_parser)

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        statuses = []
        for host in self.get_hosts():
            component_status = dict()
            statuses.append(component_status)
            component_status['component_name'] = self.get_name()
            component_status['host'] = host
            try:
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = self.find_component_container_on_host(
                        docker_client, all_states=True)
                    if len(containers) == 0:
                        logger.debug(
                            'No container found for ' + self.get_name() + ' on host ' + host)
                        component_status['containers'] = []
                    else:
                        component_status['containers'] = containers
            except Exception:
                logger.debug('Failed to get status of ' + self.get_name() + ' on host ' + host,
                             exc_info=1)
                # set the status as error
                component_status['status'] = {'error': True, 'message': 'Unknown'}
        return statuses

    def get_k8s_component_label(self):
        return 'app_component={}'.format(self.get_k8s_component_name())

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        if self.get_deployment_type(config_parser) == 'aws':
            return self.k8s_aws_status(cmd_args, config_parser)
        namespace = self.get_namespace(config_parser)
        statuses = []
        host_pod_map = {}
        status_command = 'kubectl get pods -n {} -l {} -o json'.format(namespace, self.get_k8s_component_label())
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of component {}, due to: {}."
                         .format(self.get_k8s_component_name(), str(stderr)))

        result = json.loads(str(stdout))
        items = result['items']
        for item in items:
            pod = dict()
            pod['component_name'] = item['metadata']['labels']['app_component']
            pod['host'] = item['status'].get('hostIP', 'Unknown')
            pod['containers'] = []
            if pod['component_name'] == 'rda-nats' and any(entry['name'] == 'nats-box' for entry in item.get('status', {}).get('containerStatuses', [])):
               continue
            if 'containerStatuses' in item['status']:
                for entry in item['status']['containerStatuses']:
                    if 'containerID' in entry.keys():
                        container = dict()
                        container['Id'] = entry['containerID']
                        container['Image'] = entry['image']
                        container['name'] = entry['name']
                        for key in entry['state'].keys():
                            container['State'] = key
                            if key == 'running':
                                container['Status'] = self.get_container_age(entry['state'][key]['startedAt'])
                            else:
                                container['Status'] = key
                            break
                        if container['name'] in ['prom-exporter', 'reloader']:
                            continue
                        pod['containers'].append(container)
            if pod['host'] not in host_pod_map.keys():
                host_pod_map[pod['host']] = []
            host_pod_map[pod['host']].append(pod)

        for host in self.get_hosts():
            if host in host_pod_map.keys():
                statuses.extend(host_pod_map.get(host))
            else:
                entry = dict()
                statuses.append(entry)
                entry['component_name'] = self.get_k8s_component_name()
                entry['host'] = host
                entry['containers'] = []

        return statuses
    
    def k8s_aws_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        namespace = self.get_namespace(config_parser)
        statuses = []
        status_command = 'kubectl get pods -n {} -l {} -o json'.format(namespace, self.get_k8s_component_label())
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of component {}, due to: {}."
                         .format(self.get_k8s_component_name(), str(stderr)))
    
        result = json.loads(str(stdout))
        items = result['items']
        if not items:
            component_name = self.get_k8s_component_name()
            if component_name in ['ssh-key-manager', 'cert-manager', 'proxy']:
                return statuses

            statuses.append({
                'component_name': component_name,
                'host': 'Not Provisioned',
                'containers': [{
                    'Id': 'N/A',
                    'Image': 'N/A',
                    'name': component_name,
                    'State': 'Not Provisioned',
                    'Status': 'Not Provisioned'
                }]
            })
            return statuses
        if not items:
            # No pods scheduled — use expected host count
            host_count = self.get_host() or 1
            for _ in range(host_count):
                statuses.append({
                    'component_name': self.get_k8s_component_name(),
                    'host': 'Not Provisioned',
                    'containers': [{
                        'Id': 'N/A',
                        'Image': 'N/A',
                        'name': self.get_k8s_component_name(),
                        'State': 'Not Provisioned',
                        'Status': 'Not Provisioned'
                    }]
                })
            return statuses
        for item in items:
            pod = dict()
            pod['component_name'] = item['metadata']['labels'].get('app_component', 'Unknown')
            pod['host'] = item['status'].get('hostIP', 'Unknown')
            pod['containers'] = []
    
            if pod['component_name'] == 'rda-nats' and any(entry['name'] == 'nats-box' for entry in item.get('status', {}).get('containerStatuses', [])):
                continue
    
            if 'containerStatuses' in item['status']:
                for entry in item['status']['containerStatuses']:
                    if 'containerID' in entry:
                        container = dict()
                        container['Id'] = entry['containerID']
                        container['Image'] = entry['image']
                        container['name'] = entry['name']
                        for key in entry['state']:
                            container['State'] = key
                            if key == 'running':
                                container['Status'] = self.get_container_age(entry['state'][key]['startedAt'])
                            else:
                                container['Status'] = key
                            break
                        if container['name'] in ['prom-exporter', 'reloader']:
                            continue
                        pod['containers'].append(container)
    
            statuses.append(pod)
    
        return statuses

    def create_cert_configs(self, config_parser):
        pass

    def setup_install_root_dir_hierarchy(self, hosts: Union[str, List[str]],
                                         config_parser: configparser.ConfigParser):
        install_root = self.get_rdaf_install_root(config_parser)
        uid = os.getuid()
        gid = os.getgid()
        dirs = [install_root, os.path.join(install_root, 'config'),
                os.path.join(install_root, 'data'),
                os.path.join(install_root, 'logs'),
                os.path.join(install_root, 'deployment-scripts')]
        if isinstance(hosts, str):
            hosts = [hosts]
        for host in hosts:
            for path in dirs:
                logger.info('Creating directory ' + path + ' and setting ownership to user '
                            + str(uid) + ' and group to group ' + str(gid) + ' on host ' + host)
                command = 'sudo mkdir -p ' + path + ' && sudo chown -R ' + str(
                    uid) + ' ' + path + ' && sudo chgrp -R ' + str(
                    gid) + ' ' + path
                run_potential_ssh_command(host, command, config_parser)

    @staticmethod
    def copy_helm_chart(template_path, deployment_path):
        if os.path.exists(deployment_path):
            shutil.rmtree(deployment_path)
        os.makedirs(deployment_path, exist_ok=True)
        native_local_copy_dir(template_path, deployment_path)

    def healthcheck(self, component_name, host, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        # Run healthcheck on all infra components
        return [component_name, "N/A", "N/A", "N/A"]

    def k8s_reset(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self._delete_data(config_parser)

    def reset(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        # find the relevant container(s) for this component
        # for each of those containers, issue a down and delete
        # remove any other data for the component (like named volumes and filesystem directories)
        self.down(cmd_args, config_parser)
        self._delete_data(config_parser)

    def backup_conf(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_dir_root: os.path):
        component_conf_dir = self.get_conf_dir()
        # copy/backup from each component specific host
        for host in self.get_hosts():
            if not check_potential_remote_file_exists(host, component_conf_dir):
                return
            dest_config_dir = os.path.join(backup_dir_root, 'config', host, self.get_name())
            logger.info('Backing up ' + component_conf_dir + ' from host ' + host + ' to '
                        + dest_config_dir)
            os.makedirs(dest_config_dir, exist_ok=True)
            do_potential_scp_fetch(host, component_conf_dir, dest_config_dir)

    def backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        component_data_dir = self.get_data_dir()

        # copy/backup from each component specific host
        for host in self.get_hosts():
            if not check_potential_remote_file_exists(host, component_data_dir):
                return
            dest_data_dir = os.path.join(backup_dir_root, 'data', host, self.get_name())
            logger.info('Backing up ' + component_data_dir + ' from host ' + host + ' to '
                        + dest_data_dir)
            os.makedirs(dest_data_dir, exist_ok=True)
            do_potential_scp_fetch(host, component_data_dir, dest_data_dir)

    def k8s_backup_conf(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_dir_root: os.path):
        pass

    def k8s_backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        pass

    def k8s_restore_conf(self, config_parser: configparser.ConfigParser,
                         backup_content_root_dir: os.path):
        pass

    def k8s_restore_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                         backup_content_root_dir: os.path, backup_cfg_parser: configparser.ConfigParser):
        pass

    def check_pod_state_for_restore(self, config_parser: configparser.ConfigParser):
        return

    def required_container_state_before_restore(self):
        return None

    def before_restore(self, config_parser: configparser.ConfigParser):
        required_container_state = self.required_container_state_before_restore()
        if required_container_state is None:
            # no checks to do
            return
        # make sure the containers are in correct state
        for host in self.get_hosts():
            with Component.new_docker_client(host, config_parser) as docker_client:
                # fetch only running containers
                containers = self.find_component_container_on_host(docker_client)
                # only exited and running states are supported
                if required_container_state == 'exited' and len(containers) != 0:
                    rdafutils.cli_err_exit(
                        'Cannot trigger a restore of '
                        + self.get_name() + ' since the component\'s containers '
                        + ' are not in exited state on host ' + host)
                elif required_container_state == 'running' and len(containers) == 0:
                    rdafutils.cli_err_exit(
                        'Cannot trigger a restore of '
                        + self.get_name() + ' since the component\'s containers '
                        + ' are not in running state on host ' + host)

    def restore_conf(self, config_parser: configparser.ConfigParser,
                     backup_content_root_dir: os.path):
        conf_dir = self.get_conf_dir()
        for host in self.get_hosts():
            config_backup_dir = os.path.join(backup_content_root_dir, 'config', host, self.get_name())
            if not os.path.exists(config_backup_dir):
                # no backed up content exists
                return

            logger.info('Restoring ' + config_backup_dir + ' to ' + conf_dir + ' on host ' + host)
            do_potential_scp(host, config_backup_dir, conf_dir, sudo=True)

    def restore_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                     backup_content_root_dir: os.path,
                     backup_cfg_parser: configparser.ConfigParser):
        component_data_dir = self.get_data_dir()
        # copy/restore to each component specific host
        for host in self.get_hosts():
            backedup_data_dir = os.path.join(backup_content_root_dir, 'data', host,
                                             self.get_name())
            if not os.path.exists(backedup_data_dir):
                # no backed up content exists
                return
            logger.info('Restoring ' + backedup_data_dir + ' to ' + component_data_dir
                        + ' on host ' + host)
            do_potential_scp(host, backedup_data_dir, component_data_dir, sudo=True)


    def geodr_prepare_replication(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        pass


    def geodr_start_replication(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        pass

    def geodr_stop_replication(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        pass

    def geodr_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        pass

    def switch_primary(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                                peer_configs: configparser.ConfigParser):
        pass

    @staticmethod
    def filter_involved_services(services) -> List:
        values_yaml_path = '/opt/rdaf/deployment-scripts/values.yaml'
        if not os.path.exists(values_yaml_path):
            return services
        with open(values_yaml_path, 'r') as f:
            user_inputs = yaml.safe_load(f.read())
        if user_inputs is None or user_inputs.get('services') is None:
            return services
        involved_services = [ service for service in services
            if user_inputs.get('services', {}).get(service, {}).get('deployment', True)]
        return involved_services

    def filter_k8s_involved_services(self, services) -> List:
        values_yaml_path = '/opt/rdaf/deployment-scripts/values.yaml'
        if not os.path.exists(values_yaml_path):
            return services
        with open(values_yaml_path, 'r') as f:
            user_inputs = yaml.safe_load(f.read())

        services_map = self.get_k8s_services_map()
        involved_services = [service for service in services
                             if user_inputs.get(services_map.get(service, ''), {}).get('deployment', True)]
        return involved_services

    @staticmethod
    def get_k8s_services_map():
        return {}

    def get_k8s_involved_services(self):
        helm_dir = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        services = [i for i in os.listdir(helm_dir) if os.path.isdir(os.path.join(helm_dir, i))]
        return self.filter_k8s_involved_services(services)

    @staticmethod
    def get_involved_services(compose_file_path: os.path) -> List:
        if not os.path.exists(compose_file_path):
            return []
        with open(compose_file_path, 'r') as f:
            values = yaml.safe_load(f.read())
        services = [service for service in values['services']]
        return Component.filter_involved_services(services)

    @staticmethod
    def get_pods_names(config_parser: configparser.ConfigParser, label: str) -> List:
        namespace = Component.get_namespace(config_parser)
        pods = list()
        return_code, stdout, stderr = execute_command(
            f'kubectl -n {namespace} get pods -l {label} -o name')
        for line in stdout.splitlines():
            pods.append(line.strip())
        return pods

    @staticmethod
    def get_k8s_nodes():
        nodes = {}
        ret, stdout, stderr = execute_command('kubectl get nodes -o json')
        if ret != 0:
            cli_err_exit("Failed to get nodes of kubernetes cluster, due to: {}.".format(str(stderr)))

        result = json.loads(str(stdout))
        for node in result['items']:
            # we don't use master nodes for deploying
            if 'node-role.kubernetes.io/control-plane' in node['metadata']['labels']:
                continue

            for i in node['status']['addresses']:
                if i['type'] == "InternalIP":
                    nodes[i['address']] = node['metadata']['name']
                    break

        return nodes

    def get_container_age(self, started_at):
        started_time = parser.parse(started_at).replace(tzinfo=None)
        since = datetime.datetime.utcnow() - started_time
        weeks = since.days // 7
        days = since.days
        hours = since.seconds // 3600
        minutes = since.seconds // 60
        seconds = since.seconds
        if weeks > 0:
            p_status = "Up {} Weeks ago".format(weeks)
        elif days > 0:
            p_status = "Up {} Days ago".format(days)
        elif hours > 0:
            p_status = "Up {} Hours ago".format(hours)
        elif minutes > 0:
            p_status = "Up {} Minutes ago".format(minutes)
        elif seconds > 0:
            p_status = "Up {} Seconds ago".format(seconds)
        else:
            p_status = "Up Second ago"

        return p_status

    def get_container_identification_labels(self) -> dict:
        labels = {'com.docker.compose.service': self.component_name}
        return labels

    def find_component_container_on_host(self, docker_client: cliDockerSession,
                                         all_states: bool = False) -> List:
        component_filter_labels = []
        for k, v in self.get_container_identification_labels().items():
            component_filter_labels.append(str(k + '=' + v))
        return docker_client.client.containers(all=all_states,
                                               filters={'label': component_filter_labels})

    @staticmethod
    def new_docker_client(target_host: str,
                          config_parser: configparser.ConfigParser,
                          timeout: int = 5,
                          client_cert_dir: os.path = None) -> cliDockerSession:
        if target_host is None:
            raise ValueError('Host cannot be None')
        # TODO: fix client_certs
        if Component.is_local_host(target_host):
            return cliDockerSession(dockersock='unix://var/run/docker.sock', client_certs=None, timeout=timeout)
        cli_cert_dir = rdaf.get_cli_cert_dir()
        ca_cert = os.path.join(cli_cert_dir, 'ca.pem')
        cli_host_cert = os.path.join(cli_cert_dir, 'cli.pem')
        cli_host_key = os.path.join(cli_cert_dir, 'cli.key')
        client_certs = {'certificate': cli_host_cert,
                        'key': cli_host_key,
                        'ca': ca_cert}
        return cliDockerSession(dockersock='tcp://' + target_host + ':' + str(2376),
                                client_certs=client_certs, timeout=timeout)

    @staticmethod
    def new_docker_client_(target_host: str) -> clientDockerSession:
        if target_host is None:
            raise ValueError('Host cannot be None')
        # TODO: fix client_certs
        if Component.is_local_host(target_host):
            return clientDockerSession(dockersock='unix://var/run/docker.sock', client_certs=None)
        cli_cert_dir = rdaf.get_cli_cert_dir()
        ca_cert = os.path.join(cli_cert_dir, 'ca.pem')
        cli_host_cert = os.path.join(cli_cert_dir, 'cli.pem')
        cli_host_key = os.path.join(cli_cert_dir, 'cli.key')
        client_certs = {'certificate': cli_host_cert,
                        'key': cli_host_key,
                        'ca': ca_cert}
        return clientDockerSession(dockersock='tcp://' + target_host + ':' + str(2376),
                                   client_certs=client_certs)

    @staticmethod
    def write_configs(config_parser: configparser.ConfigParser, config_file: os.path = None):
        config_file_path = config_file if config_file is not None \
            else os.path.join('/opt', 'rdaf', 'rdaf.cfg')
        if not os.path.exists(config_file_path):
            path = os.path.dirname(config_file_path)
            command = 'sudo mkdir -p ' + path + ' && sudo chown -R ' + str(
                os.getuid()) + ' ' + path + ' && sudo chgrp -R ' + str(os.getuid()) + ' ' + path
            run_command(command)
        with open(config_file_path, 'w') as f:
            config_parser.write(f)

    def _wait_till_healthy(self, component_host: str) -> bool:
        return True

    def _delete_data(self, config_parser: configparser.ConfigParser):
        host_data_dirs = self._get_host_data_dirs()
        if host_data_dirs is None or len(host_data_dirs) == 0:
            return
        for host, data_dirs in host_data_dirs:
            for data_dir in data_dirs:
                logger.info('Deleting ' + self.get_name() + ' data from path '
                            + data_dir + ' on host ' + host)
                try:
                    remove_dir_contents(host, data_dir, config_parser, use_sudo=True)
                except Exception:
                    logger.debug(
                        'Deletion of data at ' + data_dir + ' on host ' + host + ' failed',
                        exc_info=1)
                    logger.warning('Ignoring exception that occurred during data deletion of '
                                   + self.get_name() + ' at path ' + data_dir + ' on host ' + host)

    def _get_host_data_dirs(self) -> List[Tuple[str, List[str]]]:
        return []

    @staticmethod
    def get_default_host():
        from rdaf.contextual import COMPONENT_REGISTRY
        import rdaf.component.pseudo_platform
        pseudo_platform = COMPONENT_REGISTRY.get(
            rdaf.component.pseudo_platform.PseudoComponent.COMPONENT_NAME)
        if pseudo_platform is None:
            return socket.gethostname()
        platform_services_hosts = pseudo_platform.get_platform_services_hosts()
        # return the first host in the order list of platform hosts
        return platform_services_hosts[0]

    @staticmethod
    def get_service_hosts(config_parser: configparser.ConfigParser):
        if not config_parser.has_option('common', 'service_host'):
            # default to platform host
            return [Component.get_default_host()]
        hosts = config_parser.get('common', 'service_host')
        return hosts.split(',')
    
    @staticmethod
    def get_platform_hosts(config_parser: configparser.ConfigParser):
        if not config_parser.has_option('common', 'platform_service_host'):
            # default to platform host
            return [Component.get_default_host()]
        hosts = config_parser.get('common', 'platform_service_host')
        return hosts.split(',')

    @staticmethod
    def get_worker_hosts(config_parser: configparser.ConfigParser):
        if not config_parser.has_option('rda_worker', 'host'):
            # default to platform host
            return [Component.get_default_host()]
        hosts = config_parser.get('rda_worker', 'host')
        return hosts.split(',')

    @staticmethod
    def get_rdaf_install_root(config_parser=None) -> os.path:
        # TODO: fix this import management
        from rdaf.contextual import COMPONENT_REGISTRY
        from rdaf.component.pseudo_platform import PseudoComponent
        pseudo_component: PseudoComponent = COMPONENT_REGISTRY.require(
            PseudoComponent.COMPONENT_NAME)
        return pseudo_component.get_install_root()

    def get_conf_dir(self) -> os.path:
        return os.path.join(self.get_rdaf_install_root(), 'config', self.get_name())

    def get_data_dir(self) -> os.path:
        return os.path.join(self.get_rdaf_install_root(), 'data', self.get_name())

    def get_logs_dir(self) -> os.path:
        return os.path.join(self.get_rdaf_install_root(), 'logs', self.get_name())


    @staticmethod
    def _parse_or_prompt_hosts(cmd_arg_val: Union[list, str], default_host: str,
                               no_prompt_err_msg: str, prompt_help_text: str,
                               prompt_msg: str, no_prompt: bool,
                               allow_multiple_hosts: bool = True) -> list:
        specified_hosts = cmd_arg_val
        if isinstance(specified_hosts, str):
            # create a list with one element
            specified_hosts = [specified_hosts]
        if specified_hosts:
            # parse the explicitly passed hosts
            for host in specified_hosts:
                resolved_host = cliHost(host)
                if not resolved_host.ipv4_addr:
                    rdafutils.cli_err_exit(host + ' is not a recognized IP address')
            return specified_hosts

        # no hosts have been explicitly specified and prompting is disabled.
        # we won't know of any hosts - throw an error
        if no_prompt:
            # see if a default is specified
            if default_host is not None:
                return [default_host]
            # no host specified and we are told not to prompt
            rdafutils.cli_err_exit(no_prompt_err_msg)
        # prompt for the host
        return rdafutils.prompt_host_name(prompt_help_text, prompt_msg, default_host,
                                          multiple_hosts=allow_multiple_hosts)

    @staticmethod
    def _parse_or_prompt_host_dirs(cmd_arg_val: Union[list, str], default_val: str,
                                   no_prompt_err_msg: str, prompt_help_text: str,
                                   prompt_msg: str, no_prompt: bool) \
            -> List[Tuple[str, List[str]]]:
        specified_host_dirs = cmd_arg_val
        if isinstance(specified_host_dirs, str):
            # create a list with one element
            specified_host_dirs = [specified_host_dirs]
        host_dir_values = []
        if specified_host_dirs:
            # parse the explicitly passed value of the form [host]/[dir]
            for host_dir in specified_host_dirs:
                host, dir_on_host = _split_as_host_dir(host_dir)
                if host is None:
                    rdafutils.cli_err_exit('Invalid value ' + host_dir)
                rdafutils.validate_host_name(host)
                host_dir_values.append((host, dir_on_host))
            return host_dir_values

        # no "host/dir" have been explicitly specified and prompting is disabled.
        # we won't know of any "host/dir" combinations - throw an error
        if no_prompt:
            # see if a default is specified
            if default_val is not None:
                host, dir_on_host = _split_as_host_dir(default_val)
                if host is None:
                    rdafutils.cli_err_exit('Invalid value ' + default_val)
                rdafutils.validate_host_name(host)
                return [(host, dir_on_host)]
            # no host specified and we are told not to prompt
            rdafutils.cli_err_exit(no_prompt_err_msg)
        # prompt for the host
        return Component._prompt_host_dir(prompt_help_text,
                                          prompt_msg,
                                          default_val)

    @staticmethod
    def _parse_or_prompt_value(cmd_arg_val, default_value,
                               no_prompt_err_msg, prompt_help_text,
                               prompt_msg, no_prompt, password=False,
                               apply_password_validation=True,
                               lower_case_input=True, password_min_length=6):
        val = cmd_arg_val
        if not val or len(val.strip()) == 0:
            if no_prompt:
                # see if a default is specified
                if default_value is not None:
                    return default_value
                # no value specified and we are told not to prompt
                rdafutils.cli_err_exit(no_prompt_err_msg)
            else:
                # prompt for the value
                val = rdafutils.prompt_and_validate(
                    prompt_msg, default_value, help_desc_banner=prompt_help_text,
                    password=password, apply_password_validation=apply_password_validation,
                    lower_case_input=lower_case_input, password_min_length=password_min_length)
        elif password and apply_password_validation:
            # validate the explicitly specified password
            valid, reason = rdafutils.validate_password(val, password_min_length=password_min_length)
            if not valid:
                rdafutils.cli_err_exit('Password is invalid: ' + reason)
        return val

    @staticmethod
    def _prompt_host_dir(help_desc_banner, prompt_string, default_value):
        if help_desc_banner:
            print(textwrap.dedent(help_desc_banner))
        while True:
            val = rdafutils.prompt_and_validate(
                prompt_string if prompt_string is not None else
                'Host and dir of the form '
                'host/dir (ex: 10.95.1.2/data1)',
                default_value if default_value is not None else '')
            if val == '':
                continue

            host_dir_values = []
            host_dirs = rdafutils.delimited_to_list(val)
            for host_dir in host_dirs:
                host, dir_on_host = _split_as_host_dir(host_dir)
                if host is None:
                    rdafutils.cli_err_exit('Invalid value ' + host_dir)
                rdafutils.validate_host_name(host)
                host_dir_values.append((host, dir_on_host))
            return host_dir_values

    @staticmethod
    def is_local_host(host: str):
        hostname = socket.gethostname()
        if host == hostname:
            return True
        ipv4s = []
        for interface, snics in psutil.net_if_addrs().items():
            ipv4s.extend([snic.address for snic in snics if snic.family == socket.AF_INET])
        return host in ipv4s

    def open_ports(self, config_parser: configparser.ConfigParser):
        port_host_tup = self.get_ports()
        if not port_host_tup:
            return
        if distro.id() == 'ubuntu':
            ports_list_cmd = 'sudo ufw status'
            port_open_cmd = 'sudo ufw allow {}/tcp'
            delimiter = '\n'
        else:
            ports_list_cmd = 'sudo firewall-cmd --list-ports'
            port_open_cmd = 'sudo firewall-cmd --add-port={}/tcp --permanent; ' \
                            'sudo firewall-cmd --reload'
            delimiter = ' '

        for host in port_host_tup[0]:
            output = execute_command_ssh(ports_list_cmd, host, config_parser)
            entries = {x.split(' ', 1)[0].strip() for x in output[1].split(delimiter)}
            for port in port_host_tup[1]:
                port_tcp = port + '/tcp'
                if port_tcp in entries:
                    logger.debug("port " + str(port) + " on host "
                                 + host + ' for ' + self.get_name() + ' is open')
                else:
                    logger.info("opening port " + str(port) + " on host "
                                + host + ' for ' + self.get_name())
                    run_potential_ssh_command(host, port_open_cmd.format(str(port)), config_parser)

    def update_cluster_env(self, component_value: dict, host: str):
        return

    def get_service_node_port(self, service_name, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        ret, stdout, stderr = execute_command(f'kubectl get svc {service_name} -n {namespace} -o json')
        if ret != 0:
            cli_err_exit("Failed to get services of kubernetes cluster, due to: {}."
                         .format(self.get_k8s_component_name(), str(stderr)))
        result = json.loads(str(stdout))
        port = result['spec']['ports'][0]
        return str(port['nodePort'])

    @staticmethod
    def __gen_key(tenant_id, salt_text=None):
        allowed_salt_chars = "{}{}".format(string.ascii_letters, string.digits)
        if not salt_text:
            salt_text = ''.join(random.choice(allowed_salt_chars) for i in range(4))
        elif len(salt_text) != 4:
            raise Exception("Invalid salt")

        key1 = salt_text + tenant_id
        key2 = hashlib.md5(key1.encode()).hexdigest()
        key2 = key2[:32]
        return salt_text, base64.b64encode(key2.encode()).decode()

    def encrypt(self, text, tenant_id):
        salt_text, key_text = self.__gen_key(tenant_id)
        fkey = Fernet(key_text)
        edata = fkey.encrypt(text.encode()).decode()

        data = {
            "salt": salt_text,
            "data": edata
        }
        datastr = base64.b64encode(json.dumps(data).encode()).decode()
        return datastr

    def decrypt(self, tenant_id, encrypted_text):
        try:
            datastr = base64.b64decode(encrypted_text.encode()).decode()
            data = json.loads(datastr)
            if not isinstance(data, dict) or "salt" not in data or "data" not in data:
                raise Exception()
        except:
            raise Exception("Invalid encrypted data format: Code-001")
        salt_text = data.get("salt")
        edata = data.get("data")
        salt_text, key_text = self.__gen_key(tenant_id, salt_text=salt_text)
        fkey = Fernet(key_text)
        return fkey.decrypt(edata.encode()).decode()

    def encrypt_rda_config(self, config_obj):
        tenant_id = config_obj.get("tenant_id")
        if not tenant_id:
            raise Exception("Invalid tenant_id : {} in configuration".format(tenant_id))

        def process(obj, params):
            for param in params:
                config = obj.get(param, None)
                if config:
                    config = self.encrypt(str(config), tenant_id)
                    obj.pop(param, None)
                    obj["$" + param] = config
            return obj

        nats = config_obj.get("nats")
        if nats:
            process(nats, ["token"])
        minio = config_obj.get("minio")
        if minio:
            process(minio, ["access_key", "secret_key"])
        mariadb = config_obj.get("mariadb")
        if mariadb:
            process(mariadb, ["username", "password"])
        service_config = config_obj.get("service_config")
        if service_config:
            dimensions_services = service_config.get("dimensions_services")
            if dimensions_services:
                process(dimensions_services, ["username", "password"])
            platform_mysql = service_config.get("platform_mysql")
            if platform_mysql:
                process(platform_mysql, ["username", "password"])
        mariadb = config_obj.get("mariadb")
        if mariadb:
            process(mariadb, ["username", "password"])
        kafka = config_obj.get("kafka")
        if kafka:
            process(kafka, ["sasl.username", "sasl.password"])
        kafka = config_obj.get("kafka-external")
        if kafka:
            process(kafka, ["sasl.username", "sasl.password"])
        es = config_obj.get("es")
        if es:
            process(es, ["user", "password"])
        graphdb = config_obj.get("graphdb")
        if graphdb:
            process(graphdb, ["username", "password"])
        os_external = config_obj.get("os_external")
        if os_external:
            process(os_external, ["user", "password"])
        return config_obj



class Registry:
    def __init__(self):
        self.components = dict()
        self.categorized_components = dict()

    def register(self, component: Component):
        component_name = component.get_name()
        if component_name is None:
            raise ValueError('Component name cannot be None')
        self.components[component_name] = component
        category = component.get_category()
        if category is not None:
            if category not in self.categorized_components:
                self.categorized_components[category] = []
            self.categorized_components[category].append(component)
            self.categorized_components[category].sort(key=lambda c: c.category_order)

    def get(self, component_name: str) -> Component:
        return self.components[component_name] if component_name in self.components else None

    def get_by_category(self, category: str) -> List[Component]:
        if category not in self.categorized_components:
            return []
        return self.categorized_components[category]

    def require(self, component_name: str) -> Component:
        component = self.get(component_name)
        if component is None:
            raise KeyError(component_name + ' component isn\'t registered')
        return component

    def get_all_components(self) -> iter:
        return self.components.values()

    def get_all_known_component_hosts(self, skip_components: List[str] = None) -> set:
        known_hosts = set()
        for component in self.get_all_components():
            component_name = component.get_name()
            if skip_components is not None and component_name in skip_components:
                continue
            if component.get_hosts():
                known_hosts.update(component.get_hosts())
        return known_hosts



def to_host_ports(val: List[str]) -> List[Tuple[str, int]]:
    if len(val) == 0:
        return [None, None]
    host_ports = []
    for host_port in val:
        # item will be of the form host:port
        parts = host_port.split(':')
        host_ports.append((parts[0], int(parts[1])))
    return host_ports


def _split_as_host_dir(val) -> Tuple:
    # for a value of the form 10.95.12.12/minio01, this is expected
    # to create host = 10.95.12.12 and dir_on_host = /minio01
    if val is None:
        return None, None
    host = None
    try:
        slash_index = val.index('/')
    except ValueError:
        host = val
        slash_index = -1
        # the entire string is expected to be a host value
        # in the absence of the slash character
    dir_on_host = None
    if slash_index != -1:
        host = val[:slash_index]
        # convert (any + character delimited) value to list
        dir_on_host = rdafutils.delimited_to_list(val[slash_index:], delimiter='+')
    if host is not None and len(host) == 0:
        host = None
    if dir_on_host is not None and len(dir_on_host) == 0:
        dir_on_host = None
    return host, dir_on_host


def _apply_data_dir_defaults(host_dirs: List[Tuple[str, List[str]]], data_dir_defaults: List) \
        -> List[Tuple[str, List[str]]]:
    converted = []
    for host, data_dirs in host_dirs:
        if data_dirs is None or len(data_dirs) == 0:
            data_dirs = data_dir_defaults
        converted.append((host, data_dirs))
    return converted


def find_all_files(name, path):
    result = []
    for root, dirs, files in os.walk(path):
        if name in files:
            result.append(os.path.join(root, name))
    return result


def run_command(command: str, env: dict = None, cwd=os.getcwd(), shell=True,
                ignore_error_stream_message: str = None):
    capture_output = ignore_error_stream_message is not None
    completed_process = subprocess.run([command], cwd=cwd, shell=shell,
                                       text=True, env=env, capture_output=capture_output)
    if completed_process.stdout:
        logger.info(completed_process.stdout)
    if completed_process.returncode != 0:
        if ignore_error_stream_message is not None \
                and completed_process.stderr is not None \
                and ignore_error_stream_message in completed_process.stderr:
            logger.debug('Ignoring failed exit code ' + str(completed_process.returncode)
                         + ' caused by: '
                         + ignore_error_stream_message)
            return
        if completed_process.stderr is not None:
            logger.error(completed_process.stderr)
        rdafutils.cli_err_exit('Command execution failed')


def run_potential_ssh_command(target_host: str, command: str,
                              config_parser: configparser.ConfigParser, env: dict = None,
                              cwd=os.getcwd(), shell=True,
                              ignore_error_stream_message: str = None):
    if Component.is_local_host(target_host):
        run_command(command, env=env, cwd=cwd, shell=shell,
                    ignore_error_stream_message=ignore_error_stream_message)
        return
    else:
        import rdaf.component.ssh as comp_ssh
        from rdaf.contextual import COMPONENT_REGISTRY
        ssh_key_manager = COMPONENT_REGISTRY.require(comp_ssh.SSHKeyManager.COMPONENT_NAME)
        ssh_user = ssh_key_manager.get_ssh_user()
        ssh_key_path = ssh_key_manager.get_ssh_key_path()
        # execute over ssh
        ssh_session = cliSSH(user=ssh_user, keyfile=ssh_key_path, host=target_host)
        try:
            exit_code, stdout, stderr = ssh_session.command_over_ssh_session_v2(command, environment=env)
            if stdout:
                logger.info(stdout)
            if exit_code != 0:
                if ignore_error_stream_message is not None \
                        and ((stderr is not None
                              and ignore_error_stream_message in stderr)
                             or (stdout is not None
                                 and ignore_error_stream_message in stdout)):
                    logger.debug('Ignoring failed exit code ' + str(exit_code)
                                 + ' because of presence of message in <STDERR/STDOUT>: '
                                 + ignore_error_stream_message)
                    return
                if stderr is not None:
                    logger.error(stderr)
                rdafutils.cli_err_exit('Command execution failed on ' + target_host)
        finally:
            ssh_session.close()


def do_potential_scp(target_host: str, src: os.path, dst: os.path, sudo=False) -> str:
    if Component.is_local_host(target_host):
        # execute a local copy
        if os.path.isfile(src):
            return shutil.copy(src, dst)
        native_local_copy_dir(src, dst, sudo=sudo)
        return dst
    import rdaf.component.ssh as comp_ssh
    from rdaf.contextual import COMPONENT_REGISTRY
    ssh_key_manager = COMPONENT_REGISTRY.require(comp_ssh.SSHKeyManager.COMPONENT_NAME)
    ssh_user = ssh_key_manager.get_ssh_user()
    ssh_key_path = ssh_key_manager.get_ssh_key_path()
    # execute over ssh
    ssh_session = cliSSH(user=ssh_user, keyfile=ssh_key_path, host=target_host)
    try:
        if os.path.isfile(src):
            ssh_session.transfer_file(src, dst, sudo=sudo)
        else:
            ssh_session.transfer_dir(src, dst, sudo=sudo)
        return dst
    finally:
        ssh_session.close()


def do_potential_scp_fetch(target_host: str, src_file: os.path, dest_file: os.path,
                           is_dir=True) -> str or None :
    if Component.is_local_host(target_host):
        if not os.path.exists(src_file):
            return None
        # execute a local copy
        if is_dir:
            return _copytree(src_file, dest_file)
        return shutil.copy(src_file, dest_file)
    else:
        import rdaf.component.ssh as comp_ssh
        from rdaf.contextual import COMPONENT_REGISTRY
        ssh_key_manager = COMPONENT_REGISTRY.require(comp_ssh.SSHKeyManager.COMPONENT_NAME)
        ssh_user = ssh_key_manager.get_ssh_user()
        ssh_key_path = ssh_key_manager.get_ssh_key_path()
        # execute over ssh
        ssh_session = cliSSH(user=ssh_user, keyfile=ssh_key_path, host=target_host)
        try:
            if is_dir:
                ssh_session.fetch_dir(src_file, dest_file)
            else:
                ssh_session.fetch(src_file, dest_file, mkdirs=is_dir)
            return dest_file
        finally:
            ssh_session.close()


def remove_dir_contents(target_host: str, dir_path: str,
                        config_parser: configparser.ConfigParser,
                        use_sudo=False):
    if not dir_path.endswith('/'):
        dir_path = dir_path + '/'
    dir_path = dir_path.strip()
    if len(dir_path) == 0:
        return False
    if dir_path == '/' or os.path.realpath(dir_path) == '/':
        # do not delete root filesystem
        return False
    command = 'find ' + dir_path + '. -name . -o -prune -exec rm -rf -- {} +'
    if use_sudo:
        command = 'sudo ' + command
    run_potential_ssh_command(target_host, command, config_parser)


def create_file(target_host: str, file_content: bytes, dest_location: str) -> str:
    if Component.is_local_host(target_host):
        # execute a local file write operation
        os.makedirs(os.path.dirname(dest_location), exist_ok=True)
        with open(dest_location, 'w+b') as f:
            f.write(file_content)
        return dest_location
    # create a temp file and then scp it to target host's dest location
    with tempfile.NamedTemporaryFile(mode='w+b') as f:
        f.write(file_content)
        f.flush()
        # scp the file
        return do_potential_scp(target_host, f.name, dest_location)


def check_potential_remote_file_exists(target_host: str, file_path: os.path):
    if Component.is_local_host(target_host):
        if os.path.exists(file_path):
            return True
    else:
        import rdaf.component.ssh as comp_ssh
        from rdaf.contextual import COMPONENT_REGISTRY
        ssh_key_manager = COMPONENT_REGISTRY.require(comp_ssh.SSHKeyManager.COMPONENT_NAME)
        ssh_user = ssh_key_manager.get_ssh_user()
        ssh_key_path = ssh_key_manager.get_ssh_key_path()
        # execute over ssh
        ssh_session = cliSSH(user=ssh_user, keyfile=ssh_key_path, host=target_host)
        return ssh_session.is_remote_file_exists(file_path)

    return False


def copy_content_to_root_owned_file(target_host: str, content: str, dest_file: str,
                                    config_parser: configparser.ConfigParser,
                                    append=True):
    if Component.is_local_host(target_host):
        # write to local temp file then "cat" its content and sudo tee it to the target file
        with tempfile.NamedTemporaryFile(mode='w') as tf:
            tf.write(content)
            tf.flush()
            command = 'cat ' + tf.name + ' | sudo tee ' \
                      + ('-a ' if append else '') + dest_file
            completed_process = subprocess.run([command], text=True, shell=True)
            if completed_process.returncode != 0:
                rdafutils.cli_err_exit('Failed to copy file content to file '
                                       + dest_file + ' on host ' + target_host)
        return
    # first create a temp file locally with the content.
    # then copy it over to the remote host using scp/sftp.
    # then do a local copy as sudo, locally on the remote host
    # to the destination path on that host
    with tempfile.NamedTemporaryFile(mode='w') as tf:
        tf.write(content)
        tf.flush()
        # scp/sftp
        import rdaf.component.ssh as comp_ssh
        from rdaf.contextual import COMPONENT_REGISTRY
        ssh_key_manager = COMPONENT_REGISTRY.require(comp_ssh.SSHKeyManager.COMPONENT_NAME)
        ssh_user = ssh_key_manager.get_ssh_user()
        ssh_key_path = ssh_key_manager.get_ssh_key_path()
        # execute over ssh
        ssh_session = cliSSH(user=ssh_user, keyfile=ssh_key_path, host=target_host)
        try:
            # copy to the same location as the local temp file
            ssh_session.transfer_file(tf.name, tf.name)
            # now locally on the remote host, copy over the content to the dest file
            command = 'cat ' + tf.name + ' | sudo tee ' \
                      + ('-a ' if append else '') + dest_file + ' ; rm ' + tf.name
            exit_code = ssh_session.command_over_ssh_session(command)
            if exit_code != 0:
                rdafutils.cli_err_exit('Failed to copy file content to file '
                                       + dest_file + ' on host ' + target_host)
        finally:
            ssh_session.close()


# This has been literally copied from shutil.copytree(...). The only difference
# between this and the shutil implementation is that this method doesn't fail
# if any of the directories in the destination directory already exists. shutil
# in Python 3.8 introduces a new method optional param which allows callers to
# configure this behaviour. Until we move to that version, we use this copied
# implementation
# TODO: remove this and its usage when we move to Python 3.8
def _copytree(src, dst, symlinks=False, ignore=None, copy_function=shutil.copy2,
              ignore_dangling_symlinks=False):
    """Recursively copy a directory tree.

    The destination directory must not already exist.
    If exception(s) occur, an Error is raised with a list of reasons.

    If the optional symlinks flag is true, symbolic links in the
    source tree result in symbolic links in the destination tree; if
    it is false, the contents of the files pointed to by symbolic
    links are copied. If the file pointed by the symlink doesn't
    exist, an exception will be added in the list of errors raised in
    an Error exception at the end of the copy process.

    You can set the optional ignore_dangling_symlinks flag to true if you
    want to silence this exception. Notice that this has no effect on
    platforms that don't support os.symlink.

    The optional ignore argument is a callable. If given, it
    is called with the `src` parameter, which is the directory
    being visited by copytree(), and `names` which is the list of
    `src` contents, as returned by os.listdir():

        callable(src, names) -> ignored_names

    Since copytree() is called recursively, the callable will be
    called once for each directory that is copied. It returns a
    list of names relative to the `src` directory that should
    not be copied.

    The optional copy_function argument is a callable that will be used
    to copy each file. It will be called with the source path and the
    destination path as arguments. By default, copy2() is used, but any
    function that supports the same signature (like copy()) can be used.

    """
    names = os.listdir(src)
    if ignore is not None:
        ignored_names = ignore(src, names)
    else:
        ignored_names = set()

    # Don't fail if directories exist
    os.makedirs(dst, exist_ok=True)
    errors = []
    for name in names:
        if name in ignored_names:
            continue
        srcname = os.path.join(src, name)
        dstname = os.path.join(dst, name)
        try:
            if os.path.islink(srcname):
                linkto = os.readlink(srcname)
                if symlinks:
                    # We can't just leave it to `copy_function` because legacy
                    # code with a custom `copy_function` may rely on copytree
                    # doing the right thing.
                    os.symlink(linkto, dstname)
                    shutil.copystat(srcname, dstname, follow_symlinks=not symlinks)
                else:
                    # ignore dangling symlink if the flag is on
                    if not os.path.exists(linkto) and ignore_dangling_symlinks:
                        continue
                    # otherwise let the copy occurs. copy2 will raise an error
                    if os.path.isdir(srcname):
                        _copytree(srcname, dstname, symlinks, ignore,
                                  copy_function)
                    else:
                        copy_function(srcname, dstname)
            elif os.path.isdir(srcname):
                _copytree(srcname, dstname, symlinks, ignore, copy_function)
            else:
                # Will raise a SpecialFileError for unsupported file types
                copy_function(srcname, dstname)
        # catch the Error from the recursive copytree so that we can
        # continue with other files
        except shutil.Error as err:
            errors.extend(err.args[0])
        except OSError as why:
            errors.append((srcname, dstname, str(why)))
    try:
        shutil.copystat(src, dst)
    except OSError as why:
        # Copying file access times may fail on Windows
        if getattr(why, 'winerror', None) is None:
            errors.append((src, dst, str(why)))
    if errors:
        raise shutil.Error(errors)
    return dst


def native_local_copy_dir(src_dir: os.path, dest_dir: os.path, excludes: List[str] = None,
                          sudo=False, cwd=None, dest_chown_grp=None) \
        -> Tuple[int, str, str]:
    # tar -cf - --exclude <ex1> --exclude <ex2> -C <src-dir> . | tar -xC <dest-dir>
    exclusions = ''
    if excludes is not None:
        for exclude in excludes:
            exclusions += ' --exclude ' + exclude
    command = 'tar -cf - ' + exclusions + ' -C ' + src_dir + ' . | ' \
              + (' sudo' if sudo else '') + ' tar -xC ' + dest_dir
    if sudo:
        command = 'sudo ' + command
    if dest_chown_grp:
        command += ' &&'
        if sudo:
            command += ' sudo'
        command += ' chown -R ' + dest_chown_grp + ' ' + dest_dir
    completed_process = subprocess.run([command], cwd=cwd, shell=True, text=True)
    return completed_process.returncode, completed_process.stdout, completed_process.stderr


def execute_script(target_host: str, local_script_path: os.path, *script_args) \
        -> Tuple[int, str, str]:
    args = [*script_args]
    if Component.is_local_host(target_host):
        command = 'bash ' + local_script_path + ' ' + ' '.join(args)
    else:
        command = 'cat ' + local_script_path + ' | ssh ' + target_host + ' bash -s - ' \
                  + ' '.join(args)
    completed_process = subprocess.run([command], shell=True, text=True)
    return completed_process.returncode, completed_process.stdout, completed_process.stderr


def execute_command(cmd: os.path, *script_args) -> Tuple[int, str, str]:
    args = [*script_args]
    command = cmd + ' ' + ' '.join(args)
    completed_process = subprocess.run([command], shell=True, text=True, capture_output=True)
    return completed_process.returncode, completed_process.stdout, completed_process.stderr


def run_command_exitcode(command: str, target_host: str, config_parser: configparser.ConfigParser,
                         env: dict = None, cwd=os.getcwd(), shell=True,
                         ignore_error_stream_message: str = None):
    if Component.is_local_host(target_host):
        completed_process = subprocess.run([command], shell=shell, cwd=cwd, text=True,
                                           capture_output=True, env=env)
        return completed_process.returncode, completed_process.stdout, completed_process.stderr
    else:
        import rdaf.component.ssh as comp_ssh
        from rdaf.contextual import COMPONENT_REGISTRY
        ssh_key_manager = COMPONENT_REGISTRY.require(comp_ssh.SSHKeyManager.COMPONENT_NAME)
        ssh_user = ssh_key_manager.get_ssh_user()
        ssh_key_path = ssh_key_manager.get_ssh_key_path()
        # execute over ssh
        ssh_session = cliSSH(user=ssh_user, keyfile=ssh_key_path, host=target_host)
        try:
            exit_code, stdout, stderr = ssh_session.command_over_ssh_session_v2(command, environment=env)
            return exit_code, stdout, stderr
        finally:
            ssh_session.close()


def execute_command_ssh(command: str, target_host: str, config_parser: configparser.ConfigParser, env: dict = None,
                        cwd=os.getcwd(), shell=True, ignore_error_stream_message: str = None):
    if Component.is_local_host(target_host):
        completed_process = subprocess.run([command], shell=True, text=True, capture_output=True)
        if completed_process.returncode != 0:
            if ignore_error_stream_message is not None \
                    and ((completed_process.stderr is not None
                          and ignore_error_stream_message in completed_process.stderr)
                         or (completed_process.stdout is not None
                             and ignore_error_stream_message in completed_process.stdout)):
                logger.debug('Ignoring failed exit code ' + str(completed_process.returncode)
                             + ' because of presence of message in <STDERR/STDOUT>: '
                             + ignore_error_stream_message)
                return
            if completed_process.stderr is not None:
                logger.error(completed_process.stderr)
            rdafutils.cli_err_exit('Command execution failed on ' + target_host)
        return completed_process.returncode, completed_process.stdout, completed_process.stderr
    else:
        import rdaf.component.ssh as comp_ssh
        from rdaf.contextual import COMPONENT_REGISTRY
        ssh_key_manager = COMPONENT_REGISTRY.require(comp_ssh.SSHKeyManager.COMPONENT_NAME)
        ssh_user = ssh_key_manager.get_ssh_user()
        ssh_key_path = ssh_key_manager.get_ssh_key_path()
        # execute over ssh
        ssh_session = cliSSH(user=ssh_user, keyfile=ssh_key_path, host=target_host)
        try:
            exit_code, stdout, stderr = ssh_session.command_over_ssh_session_v2(command, environment=env)
            if exit_code != 0:
                if ignore_error_stream_message is not None \
                        and ((stderr is not None
                              and ignore_error_stream_message in stderr)
                             or (stdout is not None
                                 and ignore_error_stream_message in stdout)):
                    logger.debug('Ignoring failed exit code ' + str(exit_code)
                                 + ' because of presence of message in <STDERR/STDOUT>: '
                                 + ignore_error_stream_message)
                    return
                if stderr is not None:
                    logger.error(stderr)
                rdafutils.cli_err_exit('Command execution failed on ' + target_host)
            return exit_code, stdout, stderr
        finally:
            ssh_session.close()

