import argparse
import json
import os
import logging

import datetime
import string
from rdaf.contextual import COMPONENT_REGISTRY
from typing import Callable, Any, List, Union
import configparser
import rdaf
import yaml
from rdaf import rdafutils
from rdaf.component import dockerregistry
from rdaf.component import Component, OtherCategoryOrder, _comma_delimited_to_list, _list_to_comma_delimited, \
    do_potential_scp, execute_command, run_command, run_potential_ssh_command

from rdaf.rdafutils import cli_err_exit

logger = logging.getLogger(__name__)
COMPONENT_NAME = 'rda_bulk_stats'


class BulkStats(Component):
    _option_host = 'host'

    def __init__(self):
        super().__init__(COMPONENT_NAME, COMPONENT_NAME,  category='other',
                         category_order=OtherCategoryOrder.BULK_STATS.value)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host:
            return _comma_delimited_to_list
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host:
            return _list_to_comma_delimited
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_host] = None
        return default_configs

    def get_hosts(self) -> List:
        if not self.configs[self._option_host]:
            return []
        return self.configs[self._option_host]

    def get_k8s_component_name(self):
        return 'rda-bulk-stats'

    def get_k8s_chart_name(self):
        return 'rda_bulk_stats'

    def get_deployment_file_path(self) -> os.path:
        return os.path.join("/opt/rdaf/deployment-scripts", 'bulkstats.yaml')


    def create_compose_file(self, cmd_args: argparse.Namespace):
        compose_file = self.get_deployment_file_path()
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), os.path.basename(compose_file))
        replacements = self.get_deployment_replacements(cmd_args)
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        original_values = yaml.safe_load(original_content)
        self.apply_user_inputs('rda-bulk-stats', original_values['services']['rda-bulk-stats'])
        with open(compose_file, 'w') as f:
            yaml.safe_dump(original_values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        for host in self.get_hosts():
            if not Component.is_local_host(host):
                do_potential_scp(host, compose_file, compose_file)

    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != "k8s":
            return
        self.pull_images(cmd_args, config_parser)

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        namespace = self.get_namespace(config_parser)
        statuses = []
        status_command = 'kubectl get pods -n {} -l app_component={} -o json'\
            .format(namespace, self.get_k8s_component_name())
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of event-gateway, due to: {}.".format(str(stderr)))
        result = json.loads(str(stdout))
        items = result['items']
        hosts = self.get_hosts()
        for host in hosts:
            deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
            if not items and os.path.exists(deployment_path):
                    statuses.append({
                        'component_name': self.get_k8s_component_name(),
                        'host': host,
                        'containers': [{
                            'Id': 'N/A',
                            'Image': 'N/A',
                            'State': 'N/A',
                            'Status': 'Not Provisioned'
                        }]
                    })
        else:
            for item in items:
                pod = dict()
                statuses.append(pod)
                pod['component_name'] = self.get_k8s_component_name()
                pod['host'] = item['status'].get('hostIP', 'Unknown')
                pod['containers'] = []
                if 'containerStatuses' in item['status']:
                    for entry in item['status']['containerStatuses']:
                        if 'containerID' in entry.keys():
                            container = dict()
                            container['Id'] = entry['containerID']
                            container['Image'] = entry['image']
                            for key in entry['state'].keys():
                                container['State'] = key
                                if key == 'running':
                                    container['Status'] = self.get_container_age(entry['state'][key]['startedAt'])
                                else:
                                    container['Status'] = key
                                break
                            pod['containers'].append(container)
            return statuses

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        bulk_stats_hosts = _comma_delimited_to_list(cmd_args.bulk_stats_host)
        self.label_nodes(bulk_stats_hosts)
        self.configs[self._option_host] = bulk_stats_hosts
        self._mark_configured(self.configs, config_parser)
        self.write_configs(config_parser)

        self.k8s_pull_images(cmd_args, config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_yaml):
            with open(values_yaml, 'r') as f:
                data = yaml.safe_load(f)
            
            # Modify the replica count for rda_bulk_stats only
            if 'rda_bulk_stats' in data:
                data['rda_bulk_stats']['replicas'] = len(self.get_hosts())
            
            # Write the updated YAML back to the file
            with open(values_yaml, 'w') as f:
                yaml.safe_dump(data, f, default_flow_style=False, explicit_start=True, allow_unicode=True, encoding='utf-8', sort_keys=False)
            
            logger.info(f"Updated {values_yaml} with {len(self.get_hosts())} replicas for rda_bulk_stats.")
        install_command = 'helm install  --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(), deployment_path)
        run_command(install_command)

        logger.info('Successfully installed bulk stats service')

    def get_k8s_install_args(self, cmd_args):
        args = '--set tag={} '.format(cmd_args.tag)
        # adding the timestamp for daily tags which will work for upgrade scenarios to pull the
        # latest image, with a change to the chart templates.
        if cmd_args.tag == 'daily':
            timestamp = datetime.datetime.now().strftime("%Y-%m-%d-%H:%M:%S")
            args += ' --set {service}.extraEnvs[0].name=timestamp,{service}.extraEnvs[0].value={value}' \
                .format(service='rda_bulk_stats', value=str(timestamp))
        return args

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        install_command = 'helm upgrade --install  --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(install_command)

    def pull_images(self, cmd_args, config_parser):
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        if self.configs.get(self._option_host):
            hosts = self.get_hosts()
        else:
            hosts = _comma_delimited_to_list(cmd_args.bulk_stats_host)
        for host in hosts:
            logger.info(f'Pulling bulk_stats image on host {host}')
            docker_pull_command = f'docker pull {docker_repo}/ubuntu-rda-bulkstats:{cmd_args.tag}'
            run_potential_ssh_command(host, docker_pull_command, config_parser)

    def setup_install_root_dir_hierarchy(self, hosts: Union[str, List[str]],
                                         config_parser: configparser.ConfigParser):
        known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME, COMPONENT_NAME])
        install_root = self.get_rdaf_install_root(config_parser)
        uid = os.getuid()
        gid = os.getgid()
        dirs = [install_root, os.path.join(install_root, 'config'),
                os.path.join(install_root, 'data'),
                os.path.join(install_root, 'logs'),
                os.path.join(install_root, 'deployment-scripts')]
        if isinstance(hosts, str):
            hosts = [hosts]
        for host in hosts:
            if host in known_hosts:
                continue
            for path in dirs:
                logger.info(f'Creating directory {path} and setting ownership to user {uid} and group to group {gid} on host {host}')
                command = ('sudo mkdir -p ' + path + ' && sudo chown -R ' + str(uid) + ' ' + path
                           + ' && sudo chgrp -R ' + str(gid) + ' ' + path)
                run_potential_ssh_command(host, command, config_parser)

    @staticmethod
    def docker_compose_check(hosts, config_parser: configparser.ConfigParser):
        docker_compose_file = os.path.join(rdaf.get_scripts_dir_root(), 'docker-compose')
        docker_compose_file_path = os.path.join('/usr', 'local', 'bin', 'docker-compose')
        remote_tmp_path = '/tmp/docker-compose'
        chmod_cmd = 'sudo chmod +x /usr/local/bin/docker-compose'
        copy_command = 'sudo cp ' + docker_compose_file + ' ' + docker_compose_file_path
        ssh_copy_command = 'sudo cp ' + remote_tmp_path + ' ' + docker_compose_file_path

        for host in hosts:
            if Component.is_local_host(host):
                run_command(copy_command)
                run_command(chmod_cmd)
            else:
                do_potential_scp(host, docker_compose_file, remote_tmp_path, sudo=True)
                run_potential_ssh_command(host, ssh_copy_command, config_parser)
                run_potential_ssh_command(host, chmod_cmd, config_parser)

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        network_config = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'config.json')
        data_plane_policy = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'policy.json')
        self.setup_install_root_dir_hierarchy(self.get_hosts(), config_parser)
        self.docker_compose_check(self.get_hosts(), config_parser)
        for host in self.get_hosts():
            if Component.is_local_host(host):
                continue
            logger.info(f'Copying  {network_config} to {host}' )
            do_potential_scp(host, network_config, network_config)
            logger.info(f'Copying  {data_plane_policy} to {host}' )
            do_potential_scp(host, data_plane_policy, data_plane_policy)
    
        self.create_compose_file(cmd_args)
        command = '/usr/local/bin/docker-compose --project-name bulk_stats -f {file} up -d '\
            .format(file=self.get_deployment_file_path())
        for host in self.get_hosts():
            self.copy_logging_config(config_parser, host, 'rda_bulk_stats')
            logger.info("Installing bulk_stats on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

        logger.info('Successfully installed rdaf bulk_stats service')

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.create_compose_file(cmd_args)
        command = '/usr/local/bin/docker-compose --project-name bulk_stats -f {file} up -d ' \
            .format(file=self.get_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Upgrading bulk_stats on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_deployment_file_path()):
            return

        command = '/usr/local/bin/docker-compose --project-name bulk_stats -f {} up -d ' \
            .format(self.get_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Creating bulk_stats services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        cmd = f'kubectl get pods -n {namespace} -l app_component=rda-bulk-stats -o json'
        ret, stdout, stderr = execute_command(cmd)
        components = json.loads(stdout)
        if not components["items"]:
            logger.warning(f"No Pods found for bulk_stats in namespace {namespace}")
        else:
            for component in components["items"]:
                comp = component["metadata"]["labels"]["app_component"]
                if comp == 'rda-bulk-stats':
                    logger.info(f"Deleting service: {comp} ")
                    command = f'kubectl scale deployments.apps/{comp} -n {namespace} --replicas=0'
                    run_command(command)
                    if cmd_args.force:
                        delete_pods_command = f'kubectl delete pods -n {namespace} -l app_component={comp} --grace-period=0 --force'
                        run_command(delete_pods_command)

    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        cmd = f'kubectl get deployments -n {namespace} -l app_component=rda-bulk-stats -o json'
        ret, stdout, stderr = execute_command(cmd)
        components = json.loads(stdout)

        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_yaml):
            with open(values_yaml) as f:
                data = yaml.safe_load(f)
        service_map = {'rda-bulk-stats': 'rda_bulk_stats'}
        for component in components["items"]:
            comp = component["metadata"]["labels"]["app_component"]
            replicas = data[service_map[comp]]['replicas']
            command = f'kubectl scale deployment.apps/{comp} -n {namespace} --replicas={replicas}'
            run_command(command)
            logger.info(f"Creating service: {comp} ")


    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_deployment_file_path()):
            return

        command = '/usr/local/bin/docker-compose --project-name bulk_stats -f {file} rm -fsv'\
            .format(file=self.get_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Deleting bulk_stats service on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_deployment_file_path()):
            return

        command = '/usr/local/bin/docker-compose --project-name bulk_stats -f {} start ' \
            .format(self.get_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Starting bulk_stats services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)


    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_deployment_file_path()):
            return
        command = '/usr/local/bin/docker-compose --project-name bulk_stats -f {} stop ' \
            .format(self.get_deployment_file_path())
        for host in reversed(self.get_hosts()):
            logger.info("Stopping bulk_stats services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def get_container_identification_labels(self) -> dict:
        labels = {'com.docker.compose.service': 'rda-bulk-stats'}
        return labels

    def label_nodes(self, hosts):
        # label the nodes
        logger.info("Labelling the bulk-stats nodes")
        # <ip, node>
        nodes = {}
        ret, stdout, stderr = execute_command('kubectl get nodes -o json')
        if ret != 0:
            cli_err_exit("Failed to get nodes of kubernetes cluster, due to: {}."
                         .format(self.get_k8s_component_name(), str(stderr)))

        result = json.loads(str(stdout))
        for node in result['items']:
            # we don't use master nodes for deploying
            if 'node-role.kubernetes.io/control-plane' in node['metadata']['labels']:
                continue

            for i in node['status']['addresses']:
                if i['type'] == "InternalIP":
                    nodes[i['address']] = node['metadata']['name']
                    break

        for host in hosts:
            if not nodes[host]:
                rdafutils.cli_err_exit("unknown host {}, Please use one of the "
                                       "kubernetes worker nodes".format(host))

            logger.info('applying node label "rdaf_bulk_stats_services=allow" on host: ' + host)
            run_command('kubectl label nodes {} {} --overwrite'.format(nodes[host], 'rdaf_bulk_stats_services=allow'))
    
    def _copy_configs(self, host):
        network_config = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'config.json')
        data_plane_policy = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'policy.json')
        if not Component.is_local_host(host):
            logger.info('Creating directory ' + os.path.dirname(network_config))
            do_potential_scp(host, network_config, network_config)
            do_potential_scp(host, data_plane_policy, data_plane_policy)

    def add_bulk_stats_host(self, host: str, cmd_args, config_parser: configparser.ConfigParser):
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        if host not in all_known_hosts:
            self.setup_install_root_dir_hierarchy(host, config_parser)
            self.docker_compose_check([host], config_parser)
    
        docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
        docker_registry.docker_login(host, config_parser)
        self._copy_configs(host)
        self.configs[self._option_host].append(host)
        self.create_compose_file(cmd_args)
        command = '/usr/local/bin/docker-compose -f {file} pull && ' \
                  '/usr/local/bin/docker-compose --project-name bulk_stats -f {file} up -d '\
            .format(file=self.get_deployment_file_path())
        self.copy_logging_config(config_parser, host, 'rda_bulk_stats')
        logger.info("Installing bulk_stats on host {}".format(host))
        run_potential_ssh_command(host, command, config_parser)

        logger.info('Successfully installed rdaf bulk_stats service')
        self.store_config(config_parser)