import argparse
import configparser
import logging
import os
import tempfile
from typing import Callable, Any

import rdaf.component
import rdaf.component.dockerregistry
import rdaf.component.pseudo_platform
from rdaf import rdafutils, get_scripts_dir_root
from rdaf.component import Component, OtherCategoryOrder, do_potential_scp, _list_to_comma_delimited,\
    _comma_delimited_to_list
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.rdafutils import str_base64_decode, str_base64_encode, execScript, delimited_to_list

logger = logging.getLogger(__name__)


class CertManager(Component):
    COMPONENT_NAME = 'cert-manager'
    _option_keystore_pass = 'keystore_pass'
    _option_alt_names = 'alt_names'
    _option_nlb_arn = 'nlb_arn'

    def __init__(self):
        super().__init__(CertManager.COMPONENT_NAME, 'cert-manager', category='other',
                         category_order=OtherCategoryOrder.CERT_MANAGER.value)
        self.tmp_cert_location = tempfile.TemporaryDirectory(prefix='rdaf')
        self.use_tmp = False

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_alt_names:
            return _comma_delimited_to_list
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_alt_names:
            return _list_to_comma_delimited
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_alt_names] = ''
        return default_configs

    def gather_minimal_setup_inputs(self, cmd_args, config_parser):
        cert_configs = self._init_default_configs()
        cert_configs[self._option_keystore_pass] = str_base64_encode(rdafutils.gen_password())
        self._mark_configured(cert_configs, config_parser)

    def gather_setup_inputs(self, cmd_args, config_parser):
        cert_configs = self._init_default_configs()
        cert_configs[self._option_keystore_pass] = str_base64_encode(rdafutils.gen_password())
        alt_names_desc = 'Provide any Subject alt name(s) to be used while generating SAN certs'
        alt_names = rdaf.component.Component ._parse_or_prompt_value(cmd_args.alt_names, '', '',
                                                                     alt_names_desc, 'Subject alt name(s) for certs',
                                                                     cmd_args.no_prompt,
                                                                     password=False,
                                                                     apply_password_validation=False)
        cert_configs[self._option_alt_names] = delimited_to_list(alt_names)
        if hasattr(cmd_args, 'aws') and cmd_args.aws:
            arn_desc = 'Provide cert ARN to configure load balancer with'
            arn_name = rdaf.component.Component._parse_or_prompt_value(
                '', '', '', arn_desc, 'Cert ARN for LoadBalancer', no_prompt=False, password=False,
                apply_password_validation=False)
            cert_configs[self._option_nlb_arn] = arn_name
        self._mark_configured(cert_configs, config_parser)

    def do_setup(self, cmd_args, config_parser: configparser.ConfigParser):
        cmd_args.cert_root_dir = self.tmp_cert_location.name
        cmd_args.overwrite = True
        self.generate_certs(cmd_args, config_parser)

    def do_k8s_setup(self, cmd_args, config_parser):
        cmd_args.cert_root_dir = self.tmp_cert_location.name
        cmd_args.overwrite = True
        self.generate_certs(cmd_args, config_parser, k8s=True)

    def generate_certs(self, cmd_args: argparse.Namespace, config_parser, k8s=False):
        dest_dir = cmd_args.cert_root_dir
        if dest_dir is None:
            dest_dir = self.get_cert_root_dir()
        known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        # To support tls for nats in k8s env adding the service endpoint to certs
        alt_names = self.configs[self._option_alt_names]

        if not self.get_deployment_type(config_parser) == 'aws':
            import rdaf.component.haproxy as haproxy
            haproxy_comp = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
            if haproxy_comp.get_advertised_external_host():
                alt_names.append(haproxy_comp.get_advertised_external_host())
            if haproxy_comp.get_advertised_internal_host():
                alt_names.append(haproxy_comp.get_advertised_internal_host())
        if k8s:
            namespace = self.get_namespace(config_parser)
            alt_names = alt_names if alt_names else []
            alt_names.append(f'*.rda-nats.{namespace}.svc.cluster.local')
            alt_names.append(f'rda-nats.{namespace}.svc.cluster.local')
            alt_names.append(f'*.rda-kafka-headless.{namespace}.svc.cluster.local')
            alt_names.append('127.0.0.1')
            if self.get_deployment_type(config_parser) == 'aws':
                # parsing the region from arn
                region = self.get_cert_arn().split(':')[3]
                alt_names.append('*.elb.{}.amazonaws.com '.format(region))
                alt_names.append('*.amazonaws.com')
                alt_names.append('*.cloudfabrix.io')
        # create certs for each of the known hosts
        logger.info('Generating certs at ' + dest_dir)
        create_self_signed_san_cert(known_hosts, dest_dir, self.get_keystore_pass(),
                                    overwrite_cert=cmd_args.overwrite,
                                    alt_names=alt_names)

    def use_temp_cert_dir(self):
        tmp_crt_dir = self.get_temp_cert_dir()
        logger.debug('Copying certs to directory ' + tmp_crt_dir)
        # copy the certs to the temp dir and then set a flag to use this temp cert dir
        rdaf.component._copytree(self.get_cert_root_dir(), tmp_crt_dir)
        self.use_tmp = True

    def get_temp_cert_dir(self) -> os.path:
        return self.tmp_cert_location.name

    def get_cert_arn(self) -> str:
        return self.configs[self._option_nlb_arn]

    def delete_temp_cert_dir(self):
        logger.info('Deleting the temporary certs directory ' + self.tmp_cert_location.name)
        self.tmp_cert_location.cleanup()

    def get_keystore_pass(self):
        return str_base64_decode(self.configs[self._option_keystore_pass])

    def get_cert_root_dir(self) -> os.path:
        if self.use_tmp:
            return self.get_temp_cert_dir()
        return os.path.join(self.get_rdaf_install_root(), 'cert')

    def get_truststore_location(self) -> os.path:
        return os.path.join(self.get_cert_root_dir(), 'truststore')

    def get_ca_cert_file(self) -> os.path:
        return os.path.join(self.get_cert_root_dir(), 'ca', 'ca.pem')

    def copy_certs(self, hosts: list = None):
        if hosts:
            known_hosts = hosts
        else:
            known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
                skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        cert_dir = self.get_cert_root_dir()
        ca_cert_path = os.path.join(cert_dir, 'ca')
        truststore_path = os.path.join(cert_dir, 'truststore')
        rdaf_cert_path = os.path.join(cert_dir, 'rdaf')

        for host in known_hosts:
            logger.info("Copying certs to host: " + host)
            do_potential_scp(host, rdaf_cert_path, rdaf_cert_path)
            do_potential_scp(host, ca_cert_path, ca_cert_path)
            do_potential_scp(host, truststore_path, truststore_path)


def create_self_signed_san_cert(hosts: list, certs_root_dir: os.path,
                                keystore_pass: str, overwrite_cert: bool = False,
                                alt_names: list = None):
    rdaf_cert_dir = os.path.join(certs_root_dir, 'rdaf')
    if not overwrite_cert:
        # see if certs exist for this host
        if os.path.isdir(rdaf_cert_dir) and \
                os.path.isfile(os.path.join(rdaf_cert_dir,  'rdaf.pem')):
            logger.info('Skipping cert generation since certificate already exists.')
            return
    logger.info('Creating self-signed certificate for rdaf')
    cert_configs = dict(os.environ)
    cert_configs['KEYSTORE_PASSWORD'] = keystore_pass
    cert_gen_script = os.path.join(get_scripts_dir_root(), 'certGenerator')
    command = "/bin/bash " + cert_gen_script
    if overwrite_cert:
        cert_configs['CERT_OVERWRITE'] = 'yes'
    cert_configs['CERT_PURPOSE'] = "serverAuth,clientAuth"
    ip_addrs = []
    for host in hosts:
        resolved_host = rdafutils.cliHost(host)
        ip_addrs.append("IP:" + resolved_host.ipv4_addr)
    if alt_names:
        for entry in alt_names:
            if rdafutils.isIPAddress(entry):
                ip_addrs.append("IP:" + entry)
            else:
                ip_addrs.append("DNS:" + entry)
    extensions = ','.join(ip_addrs)
    custom_alt_name = os.getenv('CERT_ALTNAMES', None)
    cert_configs['CERT_ALTNAMES'] = extensions + "," + custom_alt_name \
        if custom_alt_name else extensions
    cert_configs['CERT_CN'] = 'rdaf'
    cert_configs['CERTS_ROOT_DIR'] = certs_root_dir
    os.makedirs(certs_root_dir, exist_ok=True)
    cert_name = 'rdaf'
    cert_configs['CERT_FOR'] = cert_name
    cert_configs['CERT_CA_GENERATE'] = 'yes'
    execScript(command=command, shell=True, env=cert_configs, cwd=certs_root_dir)

