import argparse
import configparser
import json
import logging
import os
import socket
from collections import OrderedDict
from typing import Callable, Any, List

import rdaf.component as comp
import rdaf.component.dockerregistrymirror as docker_registry
from rdaf import get_repository_templates_dir_root
from rdaf import rdafutils
from rdaf.component import Component, _str_to_int, _int_to_str, do_potential_scp, \
    run_potential_ssh_command, run_command
from rdaf.contextual import COMPONENT_REGISTRY

COMPONENT_NAME = 'docker'
logger = logging.getLogger(__name__)


class DockerRegistry(Component):
    _option_name = 'name'
    _option_description = 'description'
    _option_host = 'host'
    _option_port = 'port'
    _option_protocol = 'protocol'
    _option_project = 'project'
    _option_username = 'username'
    _option_password = 'password'

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'docker', 'other')
        self.named_docker_repos = dict()
        self._cert_path = None

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_port:
            return _str_to_int
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_port:
            return _int_to_str
        return None

    def get_ports(self) -> tuple:
        hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[comp.dockerregistry.COMPONENT_NAME, docker_registry.COMPONENT_NAME])
        ports = ['2376']
        return hosts, ports

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_host] = None
        default_configs[self._option_port] = None
        default_configs[self._option_protocol] = None
        default_configs[self._option_project] = None
        default_configs[self._option_username] = None
        default_configs[self._option_password] = None
        return default_configs

    @staticmethod
    def from_config(host: str, username: str, password: str,
                    port: str, protocol: str = 'https', project: str = 'internal'):
        docker_reg = DockerRegistry()
        docker_reg.configs = dict()
        docker_reg.configs.update(docker_reg._init_default_configs())
        docker_reg.configs[DockerRegistry._option_host] = host
        docker_reg.configs[DockerRegistry._option_port] = port
        docker_reg.configs[DockerRegistry._option_protocol] = protocol
        docker_reg.configs[DockerRegistry._option_project] = project
        docker_reg.configs[DockerRegistry._option_username] = username
        docker_reg.configs[DockerRegistry._option_password] = password
        return docker_reg

    def gather_k8s_setup_inputs(self, cmd_args, config_parser):
        if cmd_args.aws:
            return
        self.gather_setup_inputs(cmd_args, config_parser)

    def gather_minimal_setup_inputs(self, cmd_args, config_parser):
        repository_templates_root = get_repository_templates_dir_root()
        docker_repo_template = os.path.join(repository_templates_root, 'docker.repo')
        if not os.path.isfile(docker_repo_template):
            raise IOError(docker_repo_template + ' is either missing or is not a file')
        with open(docker_repo_template, 'r') as f:
            template = json.loads(f.read(), object_pairs_hook=OrderedDict)
        for k, v in template.items():
            self.named_docker_repos[k] = v
        docker_configs = self._init_default_configs()
        docker_registry_template = template['CloudFabrix'][0]['docker-registry']
        for attr in docker_registry_template:
            docker_configs[attr] = docker_registry_template[attr]
        self._mark_configured(docker_configs, config_parser)

    def gather_setup_inputs(self, cmd_args, config_parser, skip_setup=False):
        if not skip_setup:
            repository_templates_root = get_repository_templates_dir_root()
            docker_repo_template = os.path.join(repository_templates_root, 'docker.repo')
            if not os.path.isfile(docker_repo_template):
                raise IOError(docker_repo_template + ' is either missing or is not a file')
            with open(docker_repo_template, 'r') as f:
                template = json.loads(f.read(), object_pairs_hook=OrderedDict)
            for k, v in template.items():
                self.named_docker_repos[k] = v
            docker_configs = self._init_default_configs()
            docker_registry_template = template['CloudFabrix'][0]['docker-registry']
            for attr in docker_registry_template:
                docker_configs[attr] = docker_registry_template[attr]
            self._mark_configured(docker_configs, config_parser)

        cert_dir = "/etc/docker/certs.d/" + self.configs[self._option_host] \
                   + ":" + str(self.configs[self._option_port])
        cert_path = os.path.join(cert_dir, "ca.crt")
        if not os.path.exists(cert_path):
            cert_desc = 'What is the ca cert to use to communicate to on-prem docker registry'
            cert_no_prompt_err_msg = 'No docker ca cert specified. ' \
                                     'Use --docker-registry-ca to specify one'
            default_val = ''  # default to empty to signify that CA cert copying isn't needed
            docker_ca_cert_path = Component._parse_or_prompt_value(
                cmd_args.docker_registry_ca,
                default_val,
                cert_no_prompt_err_msg,
                cert_desc,
                'Docker Registry CA cert path',
                cmd_args.no_prompt,
                lower_case_input=False)
            if len(docker_ca_cert_path.strip()) > 0:
                self._cert_path = docker_ca_cert_path.strip()
            else:
                self._cert_path = None
            if self._cert_path is not None and not os.path.exists(self._cert_path):
                rdafutils.cli_err_exit(self._cert_path
                                       + ' does not exist, please provide a valid path '
                                         'to docker registry ca cert')
        elif cmd_args.docker_registry_ca:
            if os.path.exists(cmd_args.docker_registry_ca):
                self._cert_path = cmd_args.docker_registry_ca
            else:
                rdafutils.cli_err_exit(cmd_args.docker_registry_ca
                                       + ' does not exist, please provide a valid path '
                                         'to docker registry ca cert')

    def do_setup(self, cmd_args, config_parser: configparser.ConfigParser):
        self.open_ports(config_parser)
        if self._cert_path is None:
            self.docker_login_known_hosts(config_parser)
            return

        remote_cert_dir = "/etc/docker/certs.d/" + self.configs[self._option_host] \
                          + ":" + str(self.configs[self._option_port])
        remote_path = os.path.join(remote_cert_dir, "ca.crt")
        uid = os.getuid()
        gid = os.getgid()
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[comp.dockerregistry.COMPONENT_NAME, docker_registry.COMPONENT_NAME])
        all_known_hosts.add(socket.gethostname())
        for host in all_known_hosts:
            logger.debug("Copying docker ca cert for host " + host)
            command = 'sudo mkdir -p ' + remote_cert_dir + ' && sudo chown -R ' + str(
                uid) + ' ' + remote_cert_dir + ' && sudo chgrp -R ' + str(
                gid) + ' ' + remote_cert_dir
            run_potential_ssh_command(host, command, config_parser)
            do_potential_scp(host, self._cert_path, remote_path, sudo=True)

        self.docker_login_known_hosts(config_parser)

    def do_k8s_setup(self, cmd_args, config_parser):
        logger.info("Creating secret for docker registry...")
        if self._cert_path and self.get_deployment_type(config_parser) in ['k8s', 'ctr']:
            remote_cert_dir = "/etc/docker/certs.d/" + self.configs[self._option_host] \
                              + ":" + str(self.configs[self._option_port])
            remote_path = os.path.join(remote_cert_dir, "ca.crt")
            uid = os.getuid()
            gid = os.getgid()
            all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
                skip_components=[comp.dockerregistry.COMPONENT_NAME, docker_registry.COMPONENT_NAME])
            all_known_hosts.add(socket.gethostname())
            for host in all_known_hosts:
                logger.debug("Copying docker ca cert for host " + host)
                command = 'sudo mkdir -p ' + remote_cert_dir + ' && sudo chown -R ' + str(
                    uid) + ' ' + remote_cert_dir + ' && sudo chgrp -R ' + str(
                    gid) + ' ' + remote_cert_dir
                run_potential_ssh_command(host, command, config_parser)
                do_potential_scp(host, self._cert_path, remote_path, sudo=True)

        docker_host, port = self.get_docker_registry_host_port()
        if port:
            docker_host = docker_host + ":" + str(port)
        user_name = self.configs[self._option_username]
        reg_pass = self.configs[self._option_password]
        docker_email = 'registry@cloudfabrix.com'
        registry_secret = 'kubectl create secret docker-registry cfxregistry-cred ' \
                          '--docker-server={} --docker-username={} --docker-password={} ' \
                          '--docker-email={} -n {}'.format(docker_host, user_name, reg_pass,
                                                           docker_email, self.get_namespace(config_parser))
        run_command(registry_secret)

        self.update_k8s_values_yaml(values={'registry': self.get_docker_registry_url()})

    def get_hosts(self) -> list:
        return [self.configs[self._option_host]]

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        # nothing to do
        return

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        # nothing to do
        return

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
               k8s=False) -> List[dict]:
        return []

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        return []

    def get_docker_repos(self, repo_name: str) -> dict:
        return self.named_docker_repos[repo_name]

    def get_docker_registry_host_port(self) -> tuple:
        return self.configs[self._option_host], self.configs[self._option_port]

    def get_docker_registry_project(self) -> str:
        return self.configs[self._option_project]

    def get_docker_registry_url(self):
        registry_host, registry_port = self.get_docker_registry_host_port()
        if registry_port:
            registry_host = registry_host + ':' + str(registry_port)
        if self.get_docker_registry_project():
            registry_host = f'{registry_host}/{self.get_docker_registry_project()}'
        return registry_host

    def docker_login_known_hosts(self, config_parser: configparser.ConfigParser):
        user_name = self.configs[self._option_username]
        password = self.configs[self._option_password]
        host = self.configs[self._option_host]
        port = self.configs[self._option_port]
        if not user_name:
            return
        command = 'docker login -u=' + user_name + ' -p=' + password + ' ' + host
        if port:
            command += ':' + str(port)
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[comp.dockerregistry.COMPONENT_NAME,
                             docker_registry.COMPONENT_NAME])
        for host in all_known_hosts:
            run_potential_ssh_command(host, command, config_parser)

        # docker login on current host too
        run_potential_ssh_command(socket.gethostname(), command, config_parser)

    def docker_login(self, remote_host, config_parser: configparser.ConfigParser):
        user_name = self.configs[self._option_username]
        password = self.configs[self._option_password]
        host = self.configs[self._option_host]
        port = self.configs[self._option_port]
        if not user_name:
            return
        command = 'docker login -u=' + user_name + ' -p=' + password + ' ' + host
        if port:
            command += ':' + str(port)

        run_potential_ssh_command(remote_host, command, config_parser)

    def set_docker_registry(self, cmd_args: argparse.Namespace,
                            config_parser: configparser.ConfigParser):
        from rdaf.cmd import is_rdaf_setup_done
        if cmd_args.cert_path is not None and is_rdaf_setup_done():
            if not os.path.exists(cmd_args.cert_path):
                rdafutils.cli_err_exit("--cert-path: " + cmd_args.cert_path
                                       + "is missing or not a valid path.")
            self._copy_ca_cert(cmd_args, config_parser)
            logger.info("Completed copying docker ca cert on known machines")

        reg_configs = self._init_default_configs()
        reg_configs[self._option_host] = cmd_args.docker_registry_host
        reg_configs[self._option_port] = cmd_args.docker_registry_port
        reg_configs[self._option_username] = cmd_args.docker_registry_user
        reg_configs[self._option_password] = cmd_args.docker_registry_password
        reg_configs[self._option_protocol] = 'https'
        # TODO: This needs to be moved in to Component.store_config()
        # cleanup existing configs of docker registry from the current config_parser
        config_parser.remove_section(self.section_name)
        self._mark_configured(reg_configs, config_parser)
        Component.write_configs(config_parser)
        logger.info('Updated configuration to use Docker registry '
                    + self.configs[self._option_host])

    def set_telegraf_docker_registry(self, cmd_args: argparse.Namespace,
                            config_parser: configparser.ConfigParser,
                                config_file: os.path = None):
        from rdaf.cmd import is_rdaf_telegraf_setup_done
        if cmd_args.cert_path is not None and is_rdaf_telegraf_setup_done():
            if not os.path.exists(cmd_args.cert_path):
                rdafutils.cli_err_exit("--cert-path: " + cmd_args.cert_path
                                       + "is missing or not a valid path.")
            self._copy_ca_cert(cmd_args, config_parser)
            logger.info("Completed copying docker ca cert on known machines")

        reg_configs = self._init_default_configs()
        reg_configs[self._option_host] = cmd_args.docker_registry_host
        reg_configs[self._option_port] = cmd_args.docker_registry_port
        reg_configs[self._option_username] = cmd_args.docker_registry_user
        reg_configs[self._option_password] = cmd_args.docker_registry_password
        reg_configs[self._option_protocol] = 'https'
        # TODO: This needs to be moved in to Component.store_config()
        # cleanup existing configs of docker registry from the current config_parser
        config_parser.remove_section(self.section_name)
        self._mark_configured(reg_configs, config_parser)
        Component.write_configs(config_parser, config_file)
        logger.info('Updated configuration to use Docker registry '
                    + self.configs[self._option_host])

    def registry_config_as_json(self) -> dict:
        return {
            'name': 'Docker registry',
            'description': 'Docker registry',
            'host': self.configs[self._option_host],
            'port': self.configs[self._option_port],
            'protocol': self.configs[self._option_protocol],
            'project': self.configs[self._option_project],
            'username': self.configs[self._option_username],
            'password': self.configs[self._option_password]
        }

    @staticmethod
    def _copy_ca_cert(cmd_args: argparse.Namespace,
                      config_parser: configparser.ConfigParser):
        remote_cert_dir = "/etc/docker/certs.d/" + cmd_args.docker_registry_host \
                          + ":" + cmd_args.docker_registry_port
        remote_path = os.path.join(remote_cert_dir, "ca.crt")
        command = 'sudo mkdir -p ' + remote_cert_dir + ' && sudo chown -R ' + str(
            os.getuid()) + ' ' + remote_cert_dir + ' && sudo chgrp -R ' + str(
            os.getgid()) + ' ' + remote_cert_dir
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[comp.dockerregistry.COMPONENT_NAME, docker_registry.COMPONENT_NAME])
        for host in all_known_hosts:
            logger.debug("Copying docker ca cert for host " + host)
            run_potential_ssh_command(host, command, config_parser)
            do_potential_scp(host, cmd_args.cert_path, remote_path, sudo=True)

        if not os.path.exists(remote_path):
            logger.debug("Copying docker ca cert on localhost...")
            run_potential_ssh_command(socket.gethostname(), command, config_parser)
            do_potential_scp(socket.gethostname(), cmd_args.cert_path, remote_path, sudo=True)