import argparse
import configparser
import json
import logging
import os
import shutil
import socket
import string
import tarfile
from collections import OrderedDict
from configparser import ConfigParser
from typing import Callable, Any, List

import requests
import yaml

import rdaf
import rdaf.contextual
import rdaf.component.cert
import rdaf.component.proxy
import rdaf.util.requestsutil as requestsutil
import rdaf.rdafutils as rdafutils
from rdaf import get_templates_dir_root
from rdaf.rdafutils import cli_err_exit, str_base64_decode, str_base64_encode
from rdaf.component import Component, _comma_delimited_to_list, \
    _list_to_comma_delimited, run_command

COMPONENT_NAME = 'docker-registry'
logger = logging.getLogger(__name__)
_docker_image_name = 'docker-registry'


class DockerRegistryMirror(Component):
    _image_tag_listing_header = ["Service-Name", "Tags"]
    _option_host = 'host'
    _option_source_host = 'source_host'
    _option_source_port = 'source_port'
    _option_source_project = 'source_project'
    _option_source_user = 'source_user'
    _option_source_password = 'source_password'
    _option_offline = 'offline_deployment'

    def __init__(self):
        super().__init__(COMPONENT_NAME, COMPONENT_NAME)
        self._install_root = None

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host:
            return _comma_delimited_to_list
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host:
            return _list_to_comma_delimited
        return None

    def load_config(self, config_parser: configparser.ConfigParser):
        super().load_config(config_parser)
        # additionally keep track of install root
        self._install_root = config_parser.get('common', 'install_root')

    def get_log_dir(self) -> os.path:
        return os.path.join(self._install_root, 'log')

    def get_conf_dir(self):
        return os.path.join(self._install_root, 'config')

    def get_import_dir(self):
        return os.path.join(self._install_root, 'import')

    def get_cert_dir(self):
        return os.path.join(self._install_root, 'cert')

    def get_data_dir(self) -> os.path:
        return os.path.join(self._install_root, 'data')

    def get_deployment_dir(self) -> os.path:
        return os.path.join(self._install_root, 'deployment-scripts')

    def is_offline_deployment(self) -> bool:
        return self.configs[self._option_offline]

    def validate_setup_inputs(self, cmd_args, config_parser):
        if len(self.get_hosts()) > 1:
            cli_err_exit('Only one docker registry host is supported, '
                         'current configured hosts are: ' + str(self.get_hosts()))

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_host] = None
        default_configs[self._option_source_host] = None
        default_configs[self._option_source_project] = None
        default_configs[self._option_source_port] = None
        default_configs[self._option_source_user] = None
        default_configs[self._option_source_password] = None
        default_configs[self._option_offline] = False
        return default_configs

    def gather_setup_inputs(self, cmd_args, config_parser):
        mirror_configs = self._init_default_configs()
        mirror_configs[self._option_offline] = cmd_args.offline
        default_host_name = socket.gethostname()
        no_prompt_err_msg = 'No Docker registry host specified. Use --docker-server-host ' \
                            'to specify one'
        host_desc = 'What is the host on which you want the Docker registry mirror ' \
                    'server to be provisioned?'
        hosts = Component._parse_or_prompt_hosts(cmd_args.docker_server_host, default_host_name,
                                                 no_prompt_err_msg, host_desc,
                                                 'Docker registry server host', cmd_args.no_prompt)
        mirror_configs[self._option_host] = hosts
        mirror_configs[self._option_source_host] = Component._parse_or_prompt_value(
            cmd_args.docker_source_host,
            None,
            'No source host specified for Docker registry mirror, please use'
            ' --docker-registry-source-host to specify one',
            'What is the hostname/IP of the docker registry whose contents need to be mirrored?',
            'Docker registry source host',
            cmd_args.no_prompt, lower_case_input=False)
        mirror_configs[self._option_source_port] = Component._parse_or_prompt_value(
            cmd_args.docker_source_port,
            '',
            'No port specified for Docker registry source, please use'
            ' --docker-registry-source-port to specify one',
            'What is the port for the docker registry whose contents need to be mirrored?',
            'Docker registry source port',
            cmd_args.no_prompt, lower_case_input=False)
        mirror_configs[self._option_source_project] = Component._parse_or_prompt_value(
            cmd_args.docker_source_project,
            None,
            'No source project specified for Docker registry mirror, please use'
            ' --docker-registry-source-project to specify one',
            'What is the project of the docker registry whose contents need to be mirrored?',
            'Docker registry source project',
            cmd_args.no_prompt, lower_case_input=False)
        mirror_configs[self._option_source_user] = Component._parse_or_prompt_value(
            cmd_args.docker_source_user,
            '',
            'No user specified for Docker registry source, please use'
            ' --docker-registry-source-user to specify one',
            'What is the username for the docker registry whose contents need to be mirrored?',
            'Docker registry source user',
            cmd_args.no_prompt, lower_case_input=False)
        src_passowrd = Component._parse_or_prompt_value(
            cmd_args.docker_source_password,
            '',
            'No password specified for Docker registry source, please use'
            ' --docker-registry-source-password to specify one',
            'What is the password for the docker registry whose contents need to be mirrored?',
            'Docker registry source password',
            cmd_args.no_prompt, lower_case_input=False, password=True,
            apply_password_validation=False)
        mirror_configs[self._option_source_password] = str_base64_encode(src_passowrd)
        self._mark_configured(mirror_configs, config_parser)

    def do_setup(self, cmd_args, config_parser):
        self.load_config(config_parser)
        registry_config_yml = os.path.join(self.get_conf_dir(), 'docker-registry-config.yml')

        # only 1 host is supported currently
        host = self.get_hosts()[0]
        # remote host isn't currently supported, we expect the CLI tool to be running
        # on the same host as the target docker registry host. so we do a local file copy
        shutil.copyfile(os.path.join(get_templates_dir_root(), 'docker-registry-config.yml'),
                        registry_config_yml)
        if not self.is_offline_deployment():
            self.docker_login()
        self.docker_compose_check()
        logger.info('Created Docker registry configuration at ' + registry_config_yml
                    + ' on ' + host)

    def backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        return

    def docker_login(self):
        user_name = self.configs[self._option_source_user]
        host = self.configs[self._option_source_host]
        port = self.configs[self._option_source_port]
        if not user_name:
            return
        password = str_base64_decode(self.configs[self._option_source_password])
        command = 'docker login -u=' + user_name + ' -p=' + password + ' ' + host
        if port:
            command += ':' + str(port)
        run_command(command)

    def get_hosts(self) -> list:
        return self.configs[self._option_host]

    def source_registry_component(self):
        from rdaf.component.dockerregistry import DockerRegistry
        return rdaf.component.dockerregistry.DockerRegistry.from_config(
            self.configs[self._option_source_host], self.configs[self._option_source_user],
            str_base64_decode(self.configs[self._option_source_password]),
            port=self.configs[self._option_source_port])

    def get_source_registry_url(self) -> str:
        url = self.configs[self._option_source_host]
        if self.configs[self._option_source_port]:
            url += ':' + str(self.configs[self._option_source_port])
        if self.configs[self._option_source_project]:
            url += '/' + str(self.configs[self._option_source_project])
        return url

    def get_deployment_file_path(self) -> os.path:
        return os.path.join(self.get_deployment_dir(), 'onprem-registry.yaml')

    def get_deployment_replacements(self, cmd_args: argparse.Namespace) -> dict:
        replacements = dict()
        replacements['DOCKER_REPO'] = self.get_source_registry_url()
        replacements['DOCKER_REGISTRY_DATA'] = self.get_data_dir()
        replacements['CERT_DIR'] = os.path.join(self.get_cert_dir(), 'rdaf')
        replacements['LOG_DIR'] = self.get_log_dir()
        replacements['CONFIG_DIR'] = self.get_conf_dir()
        replacements['IMPORT_DIR'] = self.get_import_dir()
        replacements['TAG'] = cmd_args.tag
        replacements['REGISTRY_HTTP_CERT_FILE'] = os.path.join('/opt', 'rdaf', 'cert', 'rdaf.cert')
        replacements['REGISTRY_HTTP_KEY_FILE'] = os.path.join('/opt', 'rdaf', 'cert', 'rdaf.key')
        return replacements

    def create_compose_file_worker(self, cmd_args: argparse.Namespace):
        compose_file = self.get_deployment_file_path()
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(),
                                 os.path.basename(self.get_deployment_file_path()))
        replacements = self.get_deployment_replacements(cmd_args)
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        values = yaml.safe_load(original_content)

        # environment variables
        proxy_component = rdaf.contextual.COMPONENT_REGISTRY.require(
            rdaf.component.proxy.COMPONENT_NAME)

        # the skopeo tool uses Go standard library for HTTP calls. The Go standard library
        # relies on the standard http_proxy and http_proxy environment
        # variables for proxy selection and expects those values to be URLs,
        # so the value needs to be properly URL encoded if it has special
        # characters in username etc...
        # Hence, we set quote_special_chars as True

        prox_env = proxy_component.get_container_proxy_env_vars(quote_special_chars=True)
        values['services']['docker-registry']['environment'].update(prox_env)
        with open(compose_file, 'w') as f:
            yaml.safe_dump(values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

    def get_ports(self) -> tuple:
        ports = ['5000']
        # TODO intentionally added hostname so that it goes fine without
        # configuring ssh
        hosts = [socket.gethostname()]
        return hosts, ports

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.open_ports(config_parser)
        self.create_compose_file_worker(cmd_args)
        deployment_file = self.get_deployment_file_path()
        if self.is_offline_deployment():
            command = '/usr/local/bin/docker-compose -f {file} up -d '.format(file=deployment_file)
        else:
            command = '/usr/local/bin/docker-compose -f {file} pull && ' \
                      '/usr/local/bin/docker-compose -f {file} up -d '.format(file=deployment_file)
        for host in self.get_hosts():
            # create self-signed cert for the registry
            self._create_cert(host)
            logger.info("Installing docker registry on host {}".format(host))
            run_command(command)

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        # for creating import dir on existing setups
        run_command(f'mkdir -p {self.get_import_dir()}')

        self.create_compose_file_worker(cmd_args)
        deployment_file = self.get_deployment_file_path()
        if self.is_offline_deployment():
            command = '/usr/local/bin/docker-compose -f {file} up -d '.format(file=deployment_file)
        else:
            command = '/usr/local/bin/docker-compose -f {file} pull && ' \
                      '/usr/local/bin/docker-compose -f {file} up -d '.format(file=deployment_file)
        for host in self.get_hosts():
            logger.info("Upgrading docker registry on host {}".format(host))
            run_command(command)

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        compose_file = self.get_deployment_file_path()
        if not os.path.exists(compose_file):
            return

        command = '/usr/local/bin/docker-compose -f ' + compose_file + ' up -d ' \
                  + self.component_name
        run_command(command)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        compose_file = self.get_deployment_file_path()
        if not os.path.exists(compose_file):
            return
        command = '/usr/local/bin/docker-compose -f ' + compose_file \
                  + ' rm -fs ' + self.component_name

        run_command(command)

    def reset(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.down(cmd_args, config_parser)
        dir_path = self._install_root
        if not dir_path.endswith('/'):
            dir_path = dir_path + '/'
        dir_path = dir_path.strip()
        if dir_path == '/' or os.path.realpath(dir_path) == '/':
            # do not delete root filesystem
            return False
        command = 'sudo find ' + dir_path + '. -name . -o -prune -exec rm -rf -- {} +'
        run_command(command)
        command = 'sudo find /opt/rdaf-registry/. -name . -o -prune -exec rm -rf -- {} +'
        run_command(command)

    def _create_cert(self, host: str):
        cert_root_dir = self.get_cert_dir()
        logger.info('Generating cert for docker registry that will run on host ' + host)
        # TODO: i don't think we need/use this in the cert file. I think it gets used
        # in jks file, which docker registry doesn't use or need. Right now the cert manager
        # api that we use to generate the cert mandates this since it creates a jks. we need
        # to make that optional
        keystore_pass = 'abcd1234#'
        rdaf.component.cert.create_self_signed_san_cert([host], cert_root_dir, keystore_pass,
                                                        overwrite_cert=False)

    def mirror(self, tags_to_sync: List, tag_type: str, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if cmd_args.images is not None:
            if 'mc' in cmd_args.images or 'minio' in cmd_args.images:
                cli_err_exit('Please use --minio-tag to fetch minio related images')

        mirror_host = self.get_hosts()[0]
        with Component.new_docker_client(mirror_host, config_parser, timeout=60) as docker_client:
            containers = self.find_component_container_on_host(docker_client)
            if len(containers) == 0:
                cli_err_exit('No docker-registry container running on host ' + mirror_host)
            # pick the first one
            container_id = containers[0]['Id']
            yaml_cfg_file = os.path.join(self.get_conf_dir(), 'registry-sync-skopeo.yaml')
            os.makedirs(os.path.dirname(yaml_cfg_file), exist_ok=True)
            if cmd_args.images:
                image_names = cmd_args.images
            else:
                image_names = self._get_image_names()
                image_names.remove('minio')
                image_names.remove('mc')
            self._create_skopeo_sync_yaml(yaml_cfg_file, image_names, tags_to_sync, tag_type)

            yaml_file_in_container_path = '/opt/rdaf/conf/registry-sync-skopeo.yaml'
            # skopeo sync -s yaml -d docker --dest-no-creds --dest-tls-verify=false
            # <path-to-yaml-file> <dest-host>:<dest-port>
            sync_cmd = 'skopeo sync --keep-going -s yaml -d docker --dest-no-creds' \
                       ' --dest-tls-verify=false ' \
                       + yaml_file_in_container_path \
                       + ' ' + mirror_host + ':5000'
            logger.info('Initiating mirroring of ' + str(len(image_names)) + ' images'
                        + ' from container ' + container_id
                        + ' on host ' + mirror_host)
            stream = docker_client.docker_exec_stream(container_id, sync_cmd)
            for entry in stream:
                logger.info(entry.decode('UTF-8'))
        logger.info('Docker registry mirroring completed.')

    def _create_skopeo_sync_yaml(self, target_file: os.path, image_names: List, tags: List,
                                 tag_type: str):
        template = """
${REGISTRY_SERVER}:
  tls-verify: ${TLS_VERIFY}
  credentials:
    username: ${USERNAME}
    password: ${PASSWORD}
  images-by-tag-regex:
    ${IMAGE_TAG}
quay.io:
  tls-verify: ${TLS_VERIFY}
  images-by-tag-regex:
    minio/mc: RELEASE.2024-11-21T17-21-54Z
    ${MINIO_TAG}
        """
        replacements = dict()
        replacements['USERNAME'] = self.configs[self._option_source_user]
        replacements['PASSWORD'] = str_base64_decode(self.configs[self._option_source_password])
        replacements['TLS_VERIFY'] = 'false'
        if self.configs[self._option_source_port]:
            replacements['REGISTRY_SERVER'] = self.configs[self._option_source_host] \
                                          + ':' + self.configs[self._option_source_port]
        else:
            replacements['REGISTRY_SERVER'] = self.configs[self._option_source_host]

        # image tag regex of the form:
        # fooimage: (^tag1$ | ^tag2$)
        # barimage: (^tag1$ | ^tag2$)
        minio_tag_regex_lines = ''
        image_tag_regex_lines = ''
        tag_regex = ''
        if tag_type == 'minio':
            for t in tags:
                if len(tag_regex) == 0:
                    tag_regex += '('
                else:
                    tag_regex += '|'
                tag_regex += '^' + t + '$'
            if len(tag_regex) > 0:
                tag_regex += ')'
                minio_tag_regex_lines += 'minio/minio' + ': ' + tag_regex
            else:
                minio_tag_regex_lines += 'minio/minio' + ': .*'
        else:
            for image_name in image_names:
                if self.configs[self._option_source_project]:
                    image_name = self.configs[self._option_source_project] + '/' + image_name
                if len(image_tag_regex_lines) != 0:
                    image_tag_regex_lines += '\n    '
                tag_regex = ''
                for t in tags:
                    if len(tag_regex) == 0:
                        tag_regex += '('
                    else:
                        tag_regex += '|'
                    tag_regex += '^' + t + '$'
                if len(tag_regex) > 0:
                    tag_regex += ')'
                    image_tag_regex_lines += image_name + ': ' + tag_regex
                else:
                    image_tag_regex_lines += image_name + ': .*'

        replacements['IMAGE_TAG'] = image_tag_regex_lines
        replacements['MINIO_TAG'] = minio_tag_regex_lines
        substituted_content = string.Template(template).substitute(replacements)
        # write it to the config file
        with open(target_file, 'w') as f:
            f.write(substituted_content)
        logger.info('Generated skopeo yaml at ' + target_file)

    def _get_image_names(self) -> List:
        image_names = list()
        for deployment in ['infra.yaml', 'platform.yaml', 'worker.yaml', 'telegraf.yaml', 'event-gateway.yaml', 'nginx.yaml',
                           'self-monitoring.yaml', 'bulkstats.yaml', 'logstash.yaml', 'filebeat.yaml', 'file-object.yaml']:
            yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), deployment)
            with open(yaml_path, 'r') as f:
                template_content = f.read()
            values = yaml.safe_load(template_content)
            for service in values['services']:
                image = values['services'][service]['image']
                item = image.rsplit(':', 1)[0]
                item = item.split('/', 1)[1]
                image_names.append(item)
        image_names.extend(['rda-platform-arangodb','cfx-onprem-portal-dbinit', 'ubuntu-cfxdx-nb-nginx-all',
                            'cfxcollector', 'ubuntu-rda-event-gateway', 'ubuntu-rdac', 'rda-platform-k8s-haproxy',
                            'ubuntu-rdac-full', 'mc', 'rda-platform-busybox', 'rda-platform-nats-boot-config',
                            'rda-platform-nats-box', 'rda-platform-nats-server-config-reloader',
                            'rda-platform-prometheus-nats-exporter', 'rda-platform-kubectl', 'rda-platform-kube-arangodb'])
        app_entries = set()
        for deployment in ['oia.yaml']:
            yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), deployment)
            with open(yaml_path, 'r') as f:
                template_content = f.read()
            values = yaml.safe_load(template_content)
            for service in values['services']:
                image = values['services'][service]['image']
                entry = image.rsplit(':', 1)[0]
                entry = entry.split('/', 1)[1]
                app_entries.add(entry)

        image_names = list(set(image_names + list(app_entries)))
        return image_names

    def delete_images(self, tags, config_parser):
        logger.info("Deleting images with tags: " + str(tags))
        host = self.get_hosts()[0]
        images = self._get_image_names()
        with Component.new_docker_client(host, config_parser, timeout=60) as docker_client:
            containers = self.find_component_container_on_host(docker_client)
            if len(containers) == 0:
                cli_err_exit('No docker-registry container running on host ' + host)
            # pick the first one
            container_id = containers[0]['Id']
            for tag in tags:
                for image in images:
                    skopeo_delete = 'skopeo delete docker://{}:5000/{}:{} --no-creds' \
                       ' --tls-verify=false'.format(host, image, tag)
                    output, error = docker_client.docker_exec(container_id, skopeo_delete)
                    if output is not None:
                        logger.info(str(output, encoding='utf-8'))
                    if error is not None:
                        logger.warning(str(error, encoding='utf-8'))

            logger.info('Initiating cleanup of disk space...')
            cleanup_cmd = '/bin/registry garbage-collect /etc/docker/registry/config.yml --delete-untagged'
            output, error = docker_client.docker_exec(container_id, cleanup_cmd)
            if output is not None:
                logger.info('\n' + str(output, encoding='utf-8'))
            if error is not None:
                err_msg = str(error, encoding='utf-8')
                logger.warning('Failed to cleanup disk space, due to: ' + err_msg)

        # update the current config_parser instance too
        self.store_config(config_parser)
        self.write_configs(config_parser, config_file=os.path.join('/opt/rdaf/', 'rdaf-registry.cfg'))

    def import_file(self, cmd_args, config_parser: ConfigParser):
        tar_path = os.path.join(self.get_import_dir(), cmd_args.file)
        if not os.path.exists(tar_path):
            cli_err_exit(f"{cmd_args.file} is not found at {self.get_import_dir()}")

        # read manifest from tar and get entries list
        try:
            with tarfile.open(tar_path, 'r') as tar:
                manifest_member = tar.getmember('manifest.json')
                manifest_file = tar.extractfile(manifest_member)
                if manifest_file:
                    manifest = json.loads(manifest_file.read().decode('utf-8'))
        except KeyError:
            cli_err_exit("No manifest.json found in tar")
        except Exception as e:
            cli_err_exit(f"Failed to read manifest from tar: {e}")

        for image in manifest:
            if 'RepoTags' in image and image['RepoTags']:
                for tag in image['RepoTags']:
                    self.import_image_from_tar(config_parser, cmd_args.file, tag)

    def import_image_from_tar(self, config_parser, tar_file, src_tag):
        tar_path = os.path.join('/opt/rdaf/import/', tar_file)
        dest_tag = src_tag.split('/')[-1]
        mirror_host = self.get_hosts()[0]
        with Component.new_docker_client(mirror_host, config_parser, timeout=60) as docker_client:
            containers = self.find_component_container_on_host(docker_client)
            if len(containers) == 0:
                cli_err_exit('No docker-registry container running on host ' + mirror_host)
            # pick the first one
            container_id = containers[0]['Id']
            # skopeo copy docker-archive:images.tar:cfxregistry.cloudfabrix.io:443/ubuntu-rda-scheduler:3.6
            # docker://10.95.122.114:5000/ubuntu-rda-scheduler:3.6 --tls-verify=false
            import_cmd = f'skopeo copy docker-archive:{tar_path}:{src_tag} docker://{mirror_host}:5000/{dest_tag}  --tls-verify=false'
            stream = docker_client.docker_exec_stream(container_id, import_cmd)
            for entry in stream:
                logger.info(entry.decode('UTF-8'))
        logger.info(f'Docker import of {dest_tag} completed.')

    def list_tags(self):
        logger.info("querying for tags of all images...")
        host = self.get_hosts()[0]
        images = self._get_image_names()
        image_tag_dict = OrderedDict()
        with requestsutil.new_session():
            requests.packages.urllib3.disable_warnings(
                requests.packages.urllib3.exceptions.InsecureRequestWarning)
            for image in images:
                url = "https://" + host + ":5000" + "/v2/" \
                      + image + "/tags/list?n=1000"
                response = requests.get(url, verify=False)
                if response.status_code == 404:
                    image_tag_dict[image] = ''
                    continue
                tags = response.json()
                tags_list = tags.get('tags')
                image_tag_dict[image] = tags_list
        DockerRegistryMirror._print_image_tag_list(image_tag_dict)

    @staticmethod
    def _print_image_tag_list(image_tags_dict, target_file=None):
        rows = []
        if image_tags_dict is None:
            rdafutils.print_tabular(DockerRegistryMirror._image_tag_listing_header,
                                    [], target_file=target_file)
            return
        for image, tags in image_tags_dict.items():
            tags_csv = ', '.join(tags) if tags else ''
            rows.append([image] + [tags_csv])
        rdafutils.print_tabular(DockerRegistryMirror._image_tag_listing_header,
                                rows, show_row_separator=True, target_file=target_file)

    @staticmethod
    def docker_compose_check():
        docker_compose_file = os.path.join(rdaf.get_scripts_dir_root(), 'docker-compose')
        docker_compose_file_path = os.path.join('/usr', 'local', 'bin', 'docker-compose')
        copy_command = 'sudo cp ' + docker_compose_file + ' ' + docker_compose_file_path
        chmod_cmd = 'sudo chmod +x /usr/local/bin/docker-compose'
        run_command(copy_command)
        run_command(chmod_cmd)
