import argparse
import os
import logging

import string
from rdaf.contextual import COMPONENT_REGISTRY
from typing import Callable, Any, List
from rdaf.component import dockerregistry, run_command
import configparser
import rdaf
import yaml

from rdaf.component import Component, OtherCategoryOrder, _comma_delimited_to_list, _list_to_comma_delimited, \
    do_potential_scp, run_potential_ssh_command

logger = logging.getLogger(__name__)
COMPONENT_NAME = 'rda_file_object'


class FileObject(Component):
    _option_host = 'host'

    def __init__(self):
        super().__init__(COMPONENT_NAME, COMPONENT_NAME,  category='other',
                         category_order=OtherCategoryOrder.FILE_OBJECT.value)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host:
            return _comma_delimited_to_list
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host:
            return _list_to_comma_delimited
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_host] = None
        return default_configs

    def get_hosts(self) -> List:
        if not self.configs[self._option_host]:
            return []
        return self.configs[self._option_host]

    def get_deployment_file_path(self) -> os.path:
        return os.path.join("/opt/rdaf/deployment-scripts", 'file-object.yaml')

    def get_deployment_replacements(self, cmd_args: argparse.Namespace) -> dict:
        replacements = self._get_docker_repo()
        replacements['TAG'] = cmd_args if isinstance(cmd_args, str) else cmd_args.tag
        return replacements

    def create_compose_file(self, cmd_args: argparse.Namespace):
        compose_file = self.get_deployment_file_path()
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), os.path.basename(compose_file))
        replacements = self.get_deployment_replacements(cmd_args)
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        original_values = yaml.safe_load(original_content)
        self.apply_user_inputs('rda-file-object', original_values['services']['rda-file-object'])
        with open(compose_file, 'w') as f:
            yaml.safe_dump(original_values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        for host in self.get_hosts():
            if not Component.is_local_host(host):
                do_potential_scp(host, compose_file, compose_file)

    def pull_images(self, cmd_args, config_parser):
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        if self.configs.get(self._option_host):
            hosts = self.get_hosts()
        else:
            hosts = cmd_args.file_object_host
        for host in hosts:
            logger.info(f'Pulling file_object image on host {host}')
            docker_pull_command = f'docker pull {docker_repo}/ubuntu-rda-file-object:{cmd_args.tag}'
            run_potential_ssh_command(host, docker_pull_command, config_parser)

    def setup_install_root_dir_hierarchy(self, config_parser: configparser.ConfigParser):
        known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME, COMPONENT_NAME])
        install_root = self.get_rdaf_install_root(config_parser)
        uid = os.getuid()
        gid = os.getgid()
        dirs = [install_root, os.path.join(install_root, 'config'),
                os.path.join(install_root, 'data'),
                os.path.join(install_root, 'logs'),
                os.path.join(install_root, 'deployment-scripts')]
        for host in self.get_hosts():
            if host in known_hosts:
                continue
            for path in dirs:
                logger.info(f'Creating directory {path} and setting ownership to user {uid} and group to group {gid} on host {host}')
                command = ('sudo mkdir -p ' + path + ' && sudo chown -R ' + str(uid) + ' ' + path
                           + ' && sudo chgrp -R ' + str(gid) + ' ' + path)
                run_potential_ssh_command(host, command, config_parser)

    def docker_compose_check(self, config_parser: configparser.ConfigParser):
        docker_compose_file = os.path.join(rdaf.get_scripts_dir_root(), 'docker-compose')
        docker_compose_file_path = os.path.join('/usr', 'local', 'bin', 'docker-compose')
        remote_tmp_path = '/tmp/docker-compose'
        chmod_cmd = 'sudo chmod +x /usr/local/bin/docker-compose'
        copy_command = 'sudo cp ' + docker_compose_file + ' ' + docker_compose_file_path
        ssh_copy_command = 'sudo cp ' + remote_tmp_path + ' ' + docker_compose_file_path

        for host in self.get_hosts():
            if Component.is_local_host(host):
                run_command(copy_command)
                run_command(chmod_cmd)
            else:
                do_potential_scp(host, docker_compose_file, remote_tmp_path, sudo=True)
                run_potential_ssh_command(host, ssh_copy_command, config_parser)
                run_potential_ssh_command(host, chmod_cmd, config_parser)


    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.setup_install_root_dir_hierarchy(config_parser)
        self.docker_compose_check(config_parser)
        network_config = os.path.join('/opt', 'rdaf', 'config', 'network_config', 'config.json')
        for host in self.get_hosts():
            if Component.is_local_host(host):
                continue
            logger.info(f'Copying  {network_config} to {host}' )
            do_potential_scp(host, network_config, network_config)
    
        self.create_compose_file(cmd_args)
        command = '/usr/local/bin/docker-compose --project-name file_object -f {file} up -d '\
            .format(file=self.get_deployment_file_path())
        for host in self.get_hosts():
            self.copy_logging_config(config_parser, host, 'rda_file_object')
            logger.info("Installing file_object on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

        logger.info('Successfully installed rdaf file_object service')

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.create_compose_file(cmd_args)
        command = '/usr/local/bin/docker-compose --project-name file_object -f {file} up -d ' \
            .format(file=self.get_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Upgrading file_object on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_deployment_file_path()):
            return

        command = '/usr/local/bin/docker-compose --project-name file_object -f {} up -d ' \
            .format(self.get_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Creating file_object services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_deployment_file_path()):
            return

        command = '/usr/local/bin/docker-compose --project-name file_object -f {file} rm -fsv'\
            .format(file=self.get_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Deleting file_object service on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_deployment_file_path()):
            return

        command = '/usr/local/bin/docker-compose --project-name file_object -f {} start ' \
            .format(self.get_deployment_file_path())
        for host in self.get_hosts():
            logger.info("Starting file_object services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        if not os.path.exists(self.get_deployment_file_path()):
            return
        command = '/usr/local/bin/docker-compose --project-name file_object -f {} stop ' \
            .format(self.get_deployment_file_path())
        for host in reversed(self.get_hosts()):
            logger.info("Stopping file_object services on host {}".format(host))
            run_potential_ssh_command(host, command, config_parser)

    def get_container_identification_labels(self) -> dict:
        labels = {'com.docker.compose.service': 'rda-file-object'}
        return labels