import argparse
import configparser
import logging
import os
import string
import shutil
import json
import tempfile
import time
import yaml
from typing import Callable, Any, List, Tuple
import requests
import termcolor

import rdaf
from rdaf import rdafutils
from rdaf import get_templates_dir_root
from rdaf.component import (Component, InfraCategoryOrder, _apply_data_dir_defaults,
                            run_potential_ssh_command, execute_command, execute_command_ssh, run_command,
                            _host_dir_storer, _host_dir_loader, do_potential_scp, do_potential_scp_fetch,
                            check_potential_remote_file_exists)
from rdaf.component import _comma_delimited_to_list, _list_to_comma_delimited
from rdaf.rdafutils import str_base64_decode, str_base64_encode, cli_err_exit
from rdaf.util import requestsutil
from rdaf.dockerutils import cliDockerSession

logger = logging.getLogger(__name__)
COMPONENT_NAME = 'graphdb'


class GraphDB(Component):
    _option_host = 'host'
    _option_user = 'user'
    _option_password = 'password'
    _default_data_dirs = ['/graphdb']
    _option_data_dir = 'datadir'

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'graphdb', 'infra', InfraCategoryOrder.GRAPHDB.value)

    def get_k8s_component_name(self):
        return 'rda-arangodb'

    def get_k8s_chart_name(self):
        return 'rda_arangodb'

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host:
            return _list_to_comma_delimited
        if config_name == self._option_data_dir:
            # convert the list of tuple to a comma separate value
            return _host_dir_storer
        return None

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host:
            return _comma_delimited_to_list
        if config_name == self._option_data_dir:
            # convert the comma separated value to list of tuples
            return _host_dir_loader
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_host] = None
        default_configs[self._option_user] = None
        default_configs[self._option_password] = None
        default_configs[self._option_data_dir] = None
        return default_configs

    def gather_minimal_setup_inputs(self, cmd_args, config_parser):
        graphdb_configs = self._init_default_configs()
        graphdb_configs[self._option_host] = [self.get_default_host()]
        graphdb_configs[self._option_user] = str_base64_encode('rdafadmin')
        graphdb_configs[self._option_data_dir] = [(self.get_default_host(), self._default_data_dirs)]
        graphdb_configs[self._option_password] = str_base64_encode(rdafutils.gen_password(num_chars=8))
        self._mark_configured(graphdb_configs, config_parser)

    def gather_setup_inputs(self, cmd_args, config_parser):
        graphdb_configs = self._init_default_configs()
        default_host_name = Component.get_default_host()
        no_prompt_err_msg = 'No GraphDB host specified. Use --graphdb_host ' \
                            'to specify a GraphDB host'
        graphdb_host_desc = 'What is the "host/path-on-host" on which you want the ' \
                            'GraphDB to be deployed?'
        host_dirs = Component._parse_or_prompt_host_dirs(cmd_args.graphdb_host,
                                                         default_host_name,
                                                         no_prompt_err_msg, graphdb_host_desc,
                                                         'GraphDB host/path',
                                                         cmd_args.no_prompt)
        graphdb_configs[self._option_data_dir] = \
            _apply_data_dir_defaults(host_dirs, self._default_data_dirs)
        graphdb_hosts = []
        for host, data_dirs in graphdb_configs[self._option_data_dir]:
            graphdb_hosts.append(host)
        graphdb_configs[self._option_host] = graphdb_hosts

        graphdb_user_desc = 'What is the user name you want to give ' \
                            'for GraphDB user that will be created and used by the RDAF platform?'
        user_no_prompt_err_msg = 'No GraphDB user specified. Use ' \
                                 '--graphdb-user to specify one'
        user = rdaf.component.Component._parse_or_prompt_value(cmd_args.graphdb_user,
                                                               'rdafadmin',
                                                               user_no_prompt_err_msg,
                                                               graphdb_user_desc,
                                                               'GraphDB user',
                                                               cmd_args.no_prompt)
        graphdb_configs[self._option_user] = str_base64_encode(user)

        pass_desc = 'What is the password you want to use for the newly created' \
                    ' GraphDB user?'
        pass_no_prompt_err_msg = 'No GraphDB password specified. Use --graphdb-password' \
                                 ' to specify one'
        default_autogen_password = rdafutils.gen_password(num_chars=8)
        passwd = rdaf.component.Component._parse_or_prompt_value(cmd_args.graphdb_password,
                                                                 default_autogen_password,
                                                                 pass_no_prompt_err_msg,
                                                                 pass_desc, 'GraphDB password',
                                                                 cmd_args.no_prompt,
                                                                 password=True, password_min_length=8)
        graphdb_configs[self._option_password] = str_base64_encode(passwd)
        self._mark_configured(graphdb_configs, config_parser)

    def get_hosts(self) -> list:
        hosts = self.configs[self._option_host]
        if hosts:
            return hosts
        return []

    def get_ports(self) -> tuple:
        hosts = self.get_hosts()
        ports = ['8528', '8529', '8530', '8531'] if len(hosts) > 1 else ['8529']
        return hosts, ports

    def _get_host_data_dirs(self) -> List[Tuple[str, List[str]]]:
        return self.configs[self._option_data_dir]

    def get_user(self) -> str:
        return str_base64_decode(self.configs[self._option_user])

    def get_password(self) -> str:
        return str_base64_decode(self.configs[self._option_password])

    def get_escaped_password(self) -> str:
        return self.get_password().replace("$", "$$")

    def get_images_involved(self):
        return ['rda-platform-arangodb-starter', 'rda-platform-arangodb']
    
    def check_avx_support(self, config_parser: configparser.ConfigParser) -> list:
        if self.get_deployment_type(config_parser) == "aws":
           return
        avx_disabled_hosts = []
        for host in self.get_hosts():
            check_avx_command = "cat /proc/cpuinfo | grep -q avx"
            if Component.is_local_host(host):
                exit_code, stdout, stderr = execute_command(check_avx_command)
            else:
                exit_code, stdout, stderr = execute_command_ssh(check_avx_command, host, config_parser)
            logger.debug(f"Checking AVX support on host: {host}")
            if exit_code != 0:
                logger.warning(f"AVX is not enabled on host {host}.")
                avx_disabled_hosts.append(host)
        return avx_disabled_hosts

    def do_setup(self, cmd_args, config_parser):
        command = 'mkdir -p ' + self.get_logs_dir()
        for host in self.get_hosts():
            run_potential_ssh_command(host, command, config_parser)

        for host, data_dirs in self._get_host_data_dirs():
            data_dir = data_dirs[0]
            command = f'sudo mkdir -p {data_dir} && sudo chown -R {str(os.getuid())} {data_dir} ' \
                      f'&& sudo chgrp -R {str(os.getuid())} {data_dir}'
            run_potential_ssh_command(host, command, config_parser)

        # create jwt secret used by all nodes
        self._create_jwt_secret(config_parser)

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        avx_disabled_hosts = self.check_avx_support(config_parser)
        if avx_disabled_hosts:
            return
        self.open_ports(config_parser)
        self.pull_images(cmd_args, config_parser)
        # TODO we will remove this in the coming releases.. just an extra level of check
        # create jwt secret used by all nodes
        self._create_jwt_secret(config_parser)
        for host in self.get_hosts():
            compose_file_path = self.create_compose_file(cmd_args, host)
            command = '/usr/local/bin/docker-compose -f {file} pull {service} && ' \
                      '/usr/local/bin/docker-compose --project-name infra -f {file} up -d {service} ' \
                .format(file=compose_file_path, service=self.component_name)
            run_potential_ssh_command(host, command, config_parser)

        # create user
        self._create_user(config_parser)
        self._apply_user_inputs(config_parser)

        if len(self.get_hosts()) > 1:
            self.add_recover_script(config_parser)

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        avx_disabled_hosts = self.check_avx_support(config_parser)
        if avx_disabled_hosts:
            return
        self.pull_images(cmd_args, config_parser)
        for host in self.get_hosts():
            command = "docker rm -f $(docker ps -aq --filter name=^/infra-graphdb-1) || true"
            try:
                execute_command_ssh(command, host, config_parser)
            except Exception:
                logger.error('Failed to upgrade Graphdb service on host' + host)
            time.sleep(3)
            compose_file_path = self.create_compose_file(cmd_args, host)
            command = '/usr/local/bin/docker-compose -f {file} pull {service} && ' \
                      '/usr/local/bin/docker-compose --project-name infra -f {file} up -d {service} ' \
                .format(file=compose_file_path, service=self.component_name)
            run_potential_ssh_command(host, command, config_parser)
        logger.info("Waiting for GraphDB to be up and running...")
        self._apply_user_inputs(config_parser)

    def up(self, cmd_args, config_parser):
        super().up(cmd_args, config_parser)
        self._apply_user_inputs(config_parser)
        
    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.up(cmd_args, config_parser)
    
    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.down(cmd_args, config_parser)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in reversed(self.get_hosts()):
            logger.info("Removing GraphDB service on host: {}".format(host))
            command = "docker rm -f $(docker ps -aq --filter name=^/infra-graphdb-1) || true"
            try:
              execute_command_ssh(command, host, config_parser)
            except Exception:
              logger.error('Failed to remove Graphdb service on host' + host)

    def create_compose_file(self, cmd_args: argparse.Namespace, host: str, component_name=None) -> os.path:
        if not component_name:
            component_name = self.component_name
        compose_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', host,
                                    self.get_deployment_file_name())
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(),
                                 self.get_deployment_file_name())
        replacements = self._get_infra_component_replacements(cmd_args, host)
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        values = yaml.safe_load(original_content)
        component_value = values['services'][component_name]
        if 'environment' in component_value.keys():
            self.update_cluster_env(component_value, host)
        # override with values.yaml inputs if provided any
        self.apply_user_inputs(component_name, component_value)
        if len(self.get_hosts()) > 1 and self.get_hosts()[0] != host:
            component_value['command'].append(f"--starter.join={self.get_hosts()[0]}")
        if not os.path.exists(compose_file):
            content = {"services": {component_name: component_value}}
        else:
            with open(compose_file, 'r') as f:
                content = yaml.safe_load(f)
                content['services'][component_name] = component_value

        run_command('mkdir -p ' + os.path.dirname(compose_file))
        with open(compose_file, 'w') as f:
            yaml.safe_dump(content, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        if not Component.is_local_host(host):
            do_potential_scp(host, compose_file, compose_file)
        return compose_file

    def get_deployment_env(self, host: str):
        env = dict()
        env['HOST_IP'] = host
        if len(self.get_hosts()) == 1:
            env['DEPLOYMENT_MODE'] = 'single'
        else:
            env['DEPLOYMENT_MODE'] = 'cluster'
            # env['STARTER_JOIN'] = '""' if self.get_hosts()[0] == host  else '"--starter.join={}"'.format(self.get_hosts()[0])
        return env

    @staticmethod
    def _find_component_container_on_host(docker_client: cliDockerSession,
                                          all_states: bool = False) -> List:
        labels = {'created-by': 'arangodb-starter'}
        component_filter_labels = []
        for k, v in labels.items():
            component_filter_labels.append(str(k + '=' + v))
        return docker_client.client.containers(all=all_states,
                                               filters={'label': component_filter_labels})

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        statuses = []
        for host in self.get_hosts():
            component_status = dict()
            statuses.append(component_status)
            component_status['component_name'] = 'graphdb[operator]'
            component_status['host'] = host
            try:
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = self.find_component_container_on_host(
                        docker_client, all_states=True)
                    if len(containers) == 0:
                        logger.debug(
                            'No container found for graphdb on host ' + host)
                        component_status['containers'] = []
                    else:
                        component_status['containers'] = containers
            except Exception:
                logger.debug('Failed to get status of graphdb on host ' + host,
                             exc_info=1)
                # set the status as error
                component_status['status'] = {'error': True, 'message': 'Unknown'}

            # get the status of the agent/server/coordinator
            try:
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = self._find_component_container_on_host(
                        docker_client, all_states=True)
                    if len(containers) == 0:
                        logger.debug(
                            'No container found for graphdb components on host ' + host)
                    self.__construct_component_status(statuses, host, containers)
            except Exception:
                logger.debug('Failed to get status of graphdb components on host ' + host,
                             exc_info=1)
                self.__construct_component_status(statuses, host, [], error=True)

        return statuses

    def __construct_component_status(self, statuses, host, containers, error=False):
        status_names = ['graphdb[agent]', 'graphdb[server]', 'graphdb[coordinator]'] if len(self.get_hosts()) > 1 else [
            'graphdb[server]']
        components_size = len(self.get_hosts())
        if error:
            for index in range(components_size):
                component_status = dict()
                statuses.append(component_status)
                component_status['component_name'] = status_names[index]
                component_status['host'] = host
                component_status['status'] = {'error': True, 'message': 'Unknown'}
            return
        # case where the container are removed or not present
        if not containers:
            for index in range(components_size):
                component_status = dict()
                statuses.append(component_status)
                component_status['component_name'] = status_names[index]
                component_status['host'] = host
                component_status['containers'] = []
            return

        # there will be only one container running in standalone
        if components_size == 1:
            component_status = dict()
            statuses.append(component_status)
            component_status['component_name'] = status_names[0]
            component_status['host'] = host
            component_status['containers'] = containers
            return

        for index in range(components_size):
            component_status = dict()
            statuses.append(component_status)
            component_status['component_name'] = status_names[index]
            component_status['host'] = host
            component_status['containers'] = self._get_component_containers(containers, index)

    def _get_component_containers(self, containers, index):
        components = ['agent', 'server', 'coordinator']
        for container in containers:
            for name in container['Names']:
                if components[index] in name:
                    return [container]
        return []
    
    def reset(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.down(cmd_args, config_parser)
        self._delete_data(config_parser)

    def do_k8s_setup(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != 'aws':
            for host, data_dirs in self._get_host_data_dirs():
                data_dir = data_dirs[0]
                command = f'sudo mkdir -p {data_dir} && sudo chown -R root {data_dir} ' \
                          f'&& sudo chgrp -R root {data_dir}'
                run_potential_ssh_command(host, command, config_parser)
            
        replacements = self._get_docker_repo()
        replacements['REPLICAS'] = 1 if len(self.get_hosts()) == 1 else 2

        template_path = os.path.join(get_templates_dir_root(), 'k8s-local', 'arangodb-operator-values.yaml')
        if self.get_deployment_type(config_parser) == "aws":
            template_path = os.path.join(get_templates_dir_root(), 'k8s-aws', 'arangodb-operator-values.yaml')
        dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'arangodb-operator-values.yaml')
        with open(template_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        values = yaml.safe_load(original_content)
        with open(dest_path, 'w') as f:
            yaml.safe_dump(values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        avx_disabled_hosts = self.check_avx_support(config_parser)
        if avx_disabled_hosts:
            return
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        # creating pv
        if self.get_deployment_type(config_parser) != 'aws':
            arangodb_pv_file = 'arangodb-pv-cluster.yaml' if len(self.get_hosts()) > 1 else 'arangodb-pv-single.yaml'

            arango_pvs = 'kubectl apply -f {} -n {}'\
                .format(os.path.join(get_templates_dir_root(), 'k8s-local', arangodb_pv_file), namespace)
            run_command(arango_pvs)

        operator_values = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'arangodb-operator-values.yaml')
        install_command = 'helm install --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, operator_values, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(install_command)
        time.sleep(10)
        replacements = self._get_docker_repo()
        replacements['REPLICAS'] = 1 if len(self.get_hosts()) == 1 else 2
        arangodb_deployment_file = 'arangodb-cluster.yaml' if len(self.get_hosts()) > 1 else 'arangodb-single-server.yaml'
        deployment_source_path = os.path.join(get_templates_dir_root(), 'k8s-local', arangodb_deployment_file)
        if self.get_deployment_type(config_parser) == "aws":
            deployment_source_path = os.path.join(get_templates_dir_root(), 'k8s-aws', arangodb_deployment_file)
        deployment_destination_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', arangodb_deployment_file)
        with open(deployment_source_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        values = yaml.safe_load(original_content)
        with open(deployment_destination_path, 'w') as f:
            yaml.safe_dump(values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)
        arangodb_deployments = f'kubectl apply -f {deployment_destination_path} -n {namespace}'
        run_command(arangodb_deployments)
        logger.info("Waiting for arangodb operator pod to be up and running...")
        arangodb_operator_status = 'kubectl wait --for=condition=Ready pod --timeout=600s --all -n {} ' \
                             '-l app_component=rda-arangodb'.format(namespace)
        run_command(arangodb_operator_status)
        retry = 0
        while retry < 6:
            # Run kubectl command to get pods with the specified label in the namespace
            retry += 1
            if len(self.get_hosts()) == 1:
              status_command = 'kubectl get pods -n {} -l {} --no-headers'.format(namespace, 'role=single')
            else:
              status_command = 'kubectl get pods -n {} -l {} --no-headers'.format(namespace, 'role=coordinator')
            ret, stdout, stderr = execute_command(status_command)
            if ret == 0:
              if "No resources found" in stderr:
                logger.info(f"Waiting for arangodb pod to be up and running.... retry {retry}")
                time.sleep(15)
              elif stdout.strip():
                break
            else:
              cli_err_exit("Unable to retrieve arangodb pod information")
        pod_status_command = 'kubectl wait --for=condition=Ready pod --timeout=600s --all -n {} ' \
                             '-l arango_deployment=rda-arangodb'.format(namespace)
        run_command(pod_status_command)
        time.sleep(10)
        status_command = 'kubectl get pods -n {} -l {} -o json'.format(namespace, 'arango_deployment=rda-arangodb')
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of component {}, due to: {}."
                         .format(self.get_k8s_component_name(), str(stderr)))

        result = json.loads(str(stdout))
        items = result['items']
        for item in items:
            pod = dict()
            pod['component_name'] = item['metadata']['labels']['app_component']
            pod['component_role'] = item['metadata']['labels']['role']
            if pod['component_name'] == 'rda-arangodb' and pod['component_role'] in ['single', 'coordinator']:
                pod['pod_name'] = item['metadata']['name']
                break
        logger.info('configuring arangodb user...')
        arangodb_user_creation = (
            'kubectl exec -it {} -n {} -- sh -c '
            '"arangosh --server.endpoint http://rda-arangodb.{}.svc.cluster.local:8529 '
            '--server.username root --server.password \\"\\" --javascript.execute-string \''
            'const users = require(\\"@arangodb/users\\"); const username = \\"{}\\"; '
            'const password = \\"{}\\"; if (!users.exists(username)) {{ '
            'users.save(username, password); console.log(\\"User \\" + username + \\" created successfully.\\"); }} '
            'users.grantDatabase(username, \\"*\\", \\"rw\\"); users.grantCollection(username, \\"*\\", \\"*\\", \\"rw\\"); '
            'console.log(\\"User \\" + username + \\" granted permissions.\\");\'"'
        ).format(pod['pod_name'], namespace, namespace, self.get_user(), self.get_password())
        run_command(arangodb_user_creation)

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        avx_disabled_hosts = self.check_avx_support(config_parser)
        if avx_disabled_hosts:
            return
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'arangodb-operator-values.yaml')
        upgrade_command = 'helm upgrade --install --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(upgrade_command)
        if self.get_deployment_type(config_parser) != 'aws':
            arangodb_pv_file = 'arangodb-pv-cluster.yaml' if len(self.get_hosts()) > 1 else 'arangodb-pv-single.yaml'

            arango_pvs = 'kubectl apply -f {} -n {}'\
                .format(os.path.join(get_templates_dir_root(), 'k8s-local', arangodb_pv_file), namespace)
            run_command(arango_pvs)
        arangodb_deployment_file = 'arangodb-cluster.yaml' if len(self.get_hosts()) > 1 else 'arangodb-single-server.yaml'
        deployment_destination_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', arangodb_deployment_file)
        arangodb_deployments = f'kubectl apply -f {deployment_destination_path} -n {namespace}'
        run_command(arangodb_deployments)

        status_cmd = f'\033[1;4m kubectl get pods -n {namespace} -l app_category=rdaf-infra \033[0m'
        logger.info('Please check pods status using - ' + status_cmd)

    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        logger.info('Deleting arangodb components...')
        namespace = self.get_namespace(config_parser)
        arangodb_deployment_file = 'arangodb-cluster.yaml' if len(self.get_hosts()) > 1 else 'arangodb-single-server.yaml'
        arangodb_deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', arangodb_deployment_file)
        if not os.path.exists(arangodb_deployment_path):
            return

        arangodb_deployments = f'kubectl delete -f {arangodb_deployment_path} -n {namespace}'
        execute_command(arangodb_deployments)
        # deleting pv's
        if len(self.get_hosts()) > 1:
            delete_pv = ('kubectl delete pv rda-arangodb-agent-pv-0 rda-arangodb-agent-pv-1 rda-arangodb-agent-pv-2 '
                         'rda-arangodb-server-pv-0 rda-arangodb-server-pv-1 rda-arangodb-server-pv-2')
        else:
            delete_pv = 'kubectl delete pv rda-arangodb-sngl-pv-0'
        execute_command(delete_pv)

        command = f'kubectl scale deployment.apps/arango-rda-arangodb-operator -n {namespace} --replicas=0'
        run_command(command)


    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        arangodb_deployment_file = 'arangodb-cluster.yaml' if len(self.get_hosts()) > 1 else 'arangodb-single-server.yaml'
        arangodb_deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', arangodb_deployment_file)
        if not os.path.exists(arangodb_deployment_path):
            return

        if self.get_deployment_type(config_parser) != 'aws':
            arangodb_pv_file = 'arangodb-pv-cluster.yaml' if len(self.get_hosts()) > 1 else 'arangodb-pv-single.yaml'

            nats_pvs = 'kubectl apply -f {} -n {}'\
                .format(os.path.join(get_templates_dir_root(), 'k8s-local', arangodb_pv_file), namespace)
            run_command(nats_pvs)

        replicas = 1 if len(self.get_hosts()) == 1 else 2
        command = f'kubectl scale deployment.apps/arango-rda-arangodb-operator -n {namespace} --replicas={replicas}'
        run_command(command)

        arangodb_deployments = f'kubectl apply -f {arangodb_deployment_path} -n {namespace}'
        run_command(arangodb_deployments)


    def k8s_reset(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        arangodb_deployment_file = 'arangodb-cluster.yaml' if len(self.get_hosts()) > 1 else 'arangodb-single-server.yaml'
        arangodb_deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', arangodb_deployment_file)
        namespace = self.get_namespace(config_parser)
        if not os.path.exists(arangodb_deployment_path):
            return
        logger.info('Deleting ArangoDeployment...')
        arangodb_deployments = f'kubectl delete -f {arangodb_deployment_path} -n {namespace}'
        execute_command(arangodb_deployments)
        time.sleep(5)
        # deleting pv's
        if len(self.get_hosts()) > 1:
            delete_pv = ('kubectl delete pv rda-arangodb-agent-pv-0 rda-arangodb-agent-pv-1 rda-arangodb-agent-pv-2 '
                         'rda-arangodb-server-pv-0 rda-arangodb-server-pv-1 rda-arangodb-server-pv-2')
        else:
            delete_pv = 'kubectl delete pv rda-arangodb-sngl-pv-0'
        execute_command(delete_pv)
        self._delete_data(config_parser)

    def get_k8s_install_args(self, cmd_args):
        return ''

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        namespace = self.get_namespace(config_parser)
        statuses = []
        role_mappings = {
            'single': 'rda-graphdb[server]',
            'agent': 'rda-graphdb[agent]',
            'coordinator': 'rda-graphdb[coordinator]',
            'dbserver': 'rda-graphdb[server]'
        }
        host_pod_map = {}
        status_command = 'kubectl get pods -n {} -l {} -o json'.format(namespace, self.get_k8s_component_label())
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of component {}, due to: {}."
                         .format(self.get_k8s_component_name(), str(stderr)))

        result = json.loads(str(stdout))
        items = result['items']
        for item in items:
            pod = dict()
            pod['component_name'] = item['metadata']['labels']['app_component']
            metadata_labels = item['metadata'].get('labels', {})
            role = metadata_labels.get('role', '')
            if pod['component_name'] == 'rda-arangodb' and metadata_labels.get(
                    'app.kubernetes.io/instance') == 'rda-arangodb':
                pod['component_name'] = 'rda-graphdb[operator]'
            if pod['component_name'] == 'rda-arangodb' and role in role_mappings:
                pod['component_name'] = role_mappings[role]
            pod['host'] = item['status'].get('hostIP', 'Unknown')
            pod['containers'] = []
            if 'containers' in item['spec']:
              for entry in item['spec']['containers']:
                if 'image' in entry and 'kube-arangodb' in entry['image']:
                    image = entry['image']
                    break
            if 'containerStatuses' in item['status']:
                for entry in item['status']['containerStatuses']:
                    if 'containerID' in entry.keys():
                        container = dict()
                        container['Id'] = entry['containerID']
                        container['Image'] = image
                        container['name'] = entry['name']
                        for key in entry['state'].keys():
                            container['State'] = key
                            if key == 'running':
                                container['Status'] = self.get_container_age(entry['state'][key]['startedAt'])
                            else:
                                container['Status'] = key
                            break
                        pod['containers'].append(container)
            if pod['host'] not in host_pod_map.keys():
                host_pod_map[pod['host']] = []
            host_pod_map[pod['host']].append(pod)

        for host in self.get_hosts():
            if host in host_pod_map.keys():
                statuses.extend(host_pod_map.get(host))
            else:
                entries = ['rda-graphdb[server]'] if len(self.get_hosts()) == 1 else \
                    ['rda-graphdb[agent]', 'rda-graphdb[coordinator]', 'rda-graphdb[server]']
                for item in entries:
                    entry = dict()
                    statuses.append(entry)
                    entry['component_name'] = item
                    entry['host'] = host
                    entry['containers'] = []
        return statuses

    def healthcheck(self, component_name, host, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        logger.debug("Running Health check for GraphDB...")
        headers = {'Content-type': 'application/json'}
        try:
            with requestsutil.new_session():
                requests.packages.urllib3.disable_warnings(
                    requests.packages.urllib3.exceptions.InsecureRequestWarning)
                register_url = "http://{host}:8529".format(host=host)
                response = requests.get(register_url, verify=False, headers=headers)
                response.raise_for_status()
        except Exception as e:
            return [component_name, "Service Status", termcolor.colored("Failed", color='red'), str(e)]
        return [component_name, "Service Status", "OK", "N/A"]

    def backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        pass

    def _create_user(self, config_parser):
        logger.info("Waiting for GraphDB to be up and running...")
        self.wait_for_graphdb_to_be_up(host=self.get_hosts()[0])
        logger.info("Creating GraphDB user...")
        try:
            with Component.new_docker_client(self.get_hosts()[0], config_parser, timeout=60) as docker_client:
                containers = self._find_component_container_on_host(docker_client, all_states=True)
                if len(containers) == 0:
                    cli_err_exit('No container found for arangodb on host ' + self.get_hosts()[0])
                if len(self.get_hosts()) == 1:
                    container = containers[0]
                else:
                    container = self._get_component_containers(containers, 2)[0]
                user_creation=("arangosh --server.password \"\" --javascript.execute-string "
                               "'var users=require(\"@arangodb/users\");var rootuser=\"root\";var username=\"{}\";"
                               "var password=\"{}\";if(!users.exists(username)){{users.save(username, password, true);"
                               "users.grantDatabase(username, \"*\", \"rw\");users.update(rootuser, password);}}'"
                               ).format( self.get_user(), self.get_password())
                output, error = docker_client.docker_exec(container['Id'], user_creation)
                logger.debug(output)
        except Exception:
            logger.exception('Failed to create user for graphdb on host ' + self.get_hosts()[0], exc_info=1)

    def wait_for_graphdb_to_be_up(self, host=None):
        retry = 0
        headers = {'Content-type': 'application/json'}
        if host is None:
            host = self.get_hosts()[len(self.get_hosts()) - 1]
        while retry < 15:
            try:
                with requestsutil.new_session():
                    requests.packages.urllib3.disable_warnings(
                        requests.packages.urllib3.exceptions.InsecureRequestWarning)
                    logger.debug("waiting for graphdb to be up and running... " + str(retry))
                    register_url = f"http://{host}:8529"
                    response = requests.get(register_url, verify=False, headers=headers)
                    response.raise_for_status()
                    break
            except Exception:
                retry += 1
                time.sleep(15)

        if retry >= 15:
            raise Exception("GraphDB is not up and running in specified time.")

    def _create_jwt_secret(self, config_parser):
        # arangodb create jwt-secret --secret=arangodb.
        host = self.get_hosts()[0]
        if check_potential_remote_file_exists(host, '/graphdb/arango.secret'):
            logger.debug("arango.secret already exists on host " + host)
            return

        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            compose_file = os.path.join(tmp, 'docker-compose.yaml')
            image_name = self._get_docker_repo()['DOCKER_REPO'] + '/rda-platform-arangodb-starter:1.0.4'
            content = {"services": {'arango-secret': {
                "image": image_name,
                "network_mode": "host",
                "command": ["create", "jwt-secret", "--secret=/data/arango.secret"],
                "volumes": ["/graphdb:/data"]}}}

            with open(compose_file, 'w+') as f:
                yaml.safe_dump(content, f, default_flow_style=False, explicit_start=True,
                               allow_unicode=True, encoding='utf-8', sort_keys=False)
            if not Component.is_local_host(host):
                do_potential_scp(host, tmp, tmp)
            pull_cmd = '/usr/local/bin/docker-compose -f {file} pull  ' \
                       '&& /usr/local/bin/docker-compose -f {file} up ' \
                       '&& /usr/local/bin/docker-compose -f {file} rm -fsv'.format(file=compose_file)
            run_potential_ssh_command(host, pull_cmd, config_parser)
            run_potential_ssh_command(host, "sudo chmod 777 /graphdb/arango.secret", config_parser)
            do_potential_scp_fetch(host, '/graphdb/arango.secret', os.path.join(tmp, 'arango.secret'),
                                   is_dir=False)
            run_potential_ssh_command(host, "sudo chmod 400 /graphdb/arango.secret", config_parser)
            for entry in self.get_hosts():
                if entry == host:
                    continue

                do_potential_scp(entry, os.path.join(tmp, 'arango.secret'), '/graphdb/arango.secret')
                run_potential_ssh_command(host, "sudo chmod 400 /graphdb/arango.secret", config_parser)

            logger.info("Created arango secret...")

    def _apply_user_inputs(self, config_parser):
        values_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if not os.path.exists(values_path):
            return
        self.wait_for_graphdb_to_be_up()
        with open(values_path, 'r') as f:
            values = yaml.safe_load(f)
        for host in self.get_hosts():
            logger.info("Updating docker memory limits for graphdb components on host " + host)
            try:
                with Component.new_docker_client(host, config_parser, timeout=60) as docker_client:
                    containers = self._find_component_container_on_host(docker_client)
                    if len(containers) == 0:
                        cli_err_exit('No container found for arangodb on host ' + host)
                    if len(self.get_hosts()) == 1:
                        container = containers[0]
                        if 'graphdb-server' not in values['services']:
                            return
                        mem_limit = values['services']['graphdb-server'].get('mem_limit', None)
                        memswap_limit = values['services']['graphdb-server'].get('memswap_limit', None)
                        restart_policy = {"Name": "unless-stopped"}
                        docker_client.client.update_container(container['Id'], mem_limit=mem_limit,
                                                              memswap_limit=memswap_limit,restart_policy=restart_policy)
                    else:
                        for index in range(3):
                            container = self._get_component_containers(containers, index)[0]
                            if index == 0:
                                service = 'graphdb-agent'
                            elif index == 1:
                                service = 'graphdb-server'
                            else:
                                service = 'graphdb-coordinator'
                            if service not in values['services']:
                                continue
                            mem_limit = values['services'][service].get('mem_limit', None)
                            memswap_limit = values['services'][service].get('memswap_limit', None)
                            restart_policy = {"Name": "unless-stopped"}
                            docker_client.client.update_container(container['Id'], mem_limit=mem_limit,
                                                                  memswap_limit=memswap_limit,restart_policy=restart_policy)
            except Exception:
                logger.exception('Failed to update graphdb container on host ' + host, exc_info=1)


    def add_recover_script(self, config_parser):
        # copy the graphdb recover script on each of these host
        for host in self.get_hosts():
            # adding the receovery log file
            recovery_log_file = os.path.join(self.get_logs_dir(), 'recovery.log')
            command = 'sudo touch ' + recovery_log_file + ' && sudo chown -R ' \
                      + str(os.getuid()) + ' ' + recovery_log_file + ' && sudo chgrp -R ' \
                      + str(os.getuid()) + ' ' + recovery_log_file
            run_potential_ssh_command(host, command, config_parser)
            template_root = get_templates_dir_root()
            script_template = os.path.join(template_root, 'graphdb-recover.sh')
            # copy to the conf dir in the install root of the host
            script_file = os.path.join(self.get_conf_dir(), 'graphdb-recover.sh')
            logger.info('Creating ' + script_file + ' on host ' + host)
            run_potential_ssh_command(host, 'mkdir -p ' + self.get_conf_dir(), config_parser)
            do_potential_scp(host, script_template, script_file)

            command = 'chmod +x ' + script_file
            run_potential_ssh_command(host, command, config_parser)
            self.update_crontab_job(host, config_parser, script_file, recovery_log_file)
    
    @staticmethod
    def update_crontab_job(host, config_parser, script_file, log_file):
        command = "crontab -l | grep -v '{}' | crontab -".format(script_file)
        run_potential_ssh_command(host, command, config_parser)
        command = '(crontab -l 2>/dev/null || true; echo "@reboot sleep 60 && ' \
                  '/bin/bash {} >> {}") | crontab -'.format(script_file, log_file)
        run_potential_ssh_command(host, command, config_parser)