import argparse
import configparser
import json
import re
import socket
import subprocess
import tempfile
import logging
import os
import string
import time
from typing import Callable, Any
import termcolor
import yaml
from pynats import NATSClient

import rdaf
from rdaf import get_templates_dir_root, rdafutils
from rdaf.component import Component, InfraCategoryOrder, run_potential_ssh_command, check_potential_remote_file_exists, \
    run_command, execute_command
from rdaf.component import _comma_delimited_to_list, _list_to_comma_delimited, \
    do_potential_scp, do_potential_scp_fetch
from rdaf.rdafutils import cli_err_exit

logger = logging.getLogger(__name__)
COMPONENT_NAME = 'nats'


class Nats(Component):
    _option_host = 'host'

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'nats', 'infra', InfraCategoryOrder.NATS.value)

    def get_k8s_component_name(self):
        return 'rda-nats'

    def get_k8s_chart_name(self):
        return 'rda_nats'

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host:
            return _list_to_comma_delimited
        return None

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host:
            return _comma_delimited_to_list
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_host] = None
        return default_configs

    def gather_minimal_setup_inputs(self, cmd_args, config_parser):
        nats_configs = self._init_default_configs()
        nats_configs[self._option_host] = [self.get_default_host()]
        self._mark_configured(nats_configs, config_parser)

    def gather_setup_inputs(self, cmd_args, config_parser):
        nats_configs = self._init_default_configs()
        default_host_name = Component.get_default_host()
        no_prompt_err_msg = 'No Nats host specified. Use --nats_host ' \
                            'to specify a Nats host'
        nats_host_desc = 'What is the "host/path-on-host" on which you want the ' \
                         'Nats to be deployed?'
        host_dirs = Component._parse_or_prompt_host_dirs(cmd_args.nats_host,
                                                         default_host_name,
                                                         no_prompt_err_msg, nats_host_desc,
                                                         'Nats host/path',
                                                         cmd_args.no_prompt)

        nats_hosts = []
        for host, data_dirs in host_dirs:
            nats_hosts.append(host)
        nats_configs[self._option_host] = nats_hosts
        self._mark_configured(nats_configs, config_parser)

    def get_hosts(self) -> list:
        return self.configs[self._option_host]

    def get_ports(self) -> tuple:
        hosts = self.get_hosts()
        ports = ['4222', '6222', '8222'] if len(hosts) > 1 else ['4222', '8222']
        return hosts, ports

    def do_setup(self, cmd_args, config_parser):
        for host in self.get_hosts():
            self.copy_nsc_tool(host, config_parser)
            content = self._create_nats_conf(host)
            nats_conf = os.path.join(self.get_conf_dir(), 'nats-server.conf')
            created_location = rdaf.component.create_file(host, content.encode(encoding='UTF-8'),
                                                          nats_conf)
            logger.info('Created Nats configuration at ' + created_location + ' on ' + host)

            # setup account
            self._setup_account(host, config_parser)

    def do_k8s_setup(self, cmd_args, config_parser):
        self.copy_nsc_tool(socket.gethostname(), config_parser)
        data_dir = 'sudo mkdir -p ' + self.get_data_dir() + ' && sudo chown -R ' + str(os.getuid()) + ' ' \
                   + self.get_data_dir() + ' && sudo chgrp -R ' + str(os.getgid()) \
                   + ' ' + self.get_data_dir()
        run_command(data_dir)

        nkeys_path = os.path.join(self.get_data_dir(), '.nkeys')
        nats_setup = 'export NKEYS_PATH=' + nkeys_path + ' && nsc env -s ' + self.get_data_dir() + \
                     ' && nsc add operator noperator && nsc add account --name SYS  && ' \
                     'nsc add user --name sys -a SYS && nsc env -o noperator && nsc env -a SYS'
        run_command(nats_setup)

        nats_config = 'nsc generate config --sys-account SYS --nats-resolver'
        ret, stdout, stderr = execute_command(nats_config)
        if ret != 0:
            cli_err_exit("Unable to generate nats configuration.")

        operator = re.search('operator:(.*)\n#', stdout).group(1)
        account = re.search('system_account:(.*)\n', stdout).group(1)
        stdout = stdout.replace('\n', '')
        resolver = re.search('resolver_preload: {(.*),', stdout).group(1).strip()

        # create values-nats.yaml
        logger.debug('Generate values yaml for nats')
        replacements = self._get_docker_repo()
        replacements['OPERATOR'] = operator.strip()
        replacements['SYS_ACCOUNT'] = account.strip()
        replacements['PRELOAD_VALUE'] = resolver.split(':')[1].strip()

        template_dir = 'k8s-local'
        if self.get_deployment_type(config_parser) == "aws":
            template_dir = 'k8s-aws'
        template_path = os.path.join(get_templates_dir_root(), template_dir, 'nats-values.yaml')
        dest_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'nats-values.yaml')
        with open(template_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)

        values = yaml.safe_load(original_content)
        values['config']['cluster']['replicas'] = len(self.get_hosts())
        values['config']['cluster']['enabled'] = True if len(self.get_hosts()) > 1 else False
        with open(dest_path, 'w') as f:
            yaml.safe_dump(values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        self.create_cert_configs(config_parser)
        # setup a cronjob for nats reachability
        if len(self.get_hosts()) > 1:
            self._configure_cron_job()

    def create_cert_configs(self,config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        # create the kubectl secrets
        logger.debug("Creating tls cert secret for nats")
        cert_secret = 'kubectl create secret generic -n {} rdaf-certs ' \
                      '--from-file=ca.cert=/opt/rdaf/cert/ca/ca.crt ' \
                      '--from-file=rdaf.key=/opt/rdaf/cert/rdaf/rdaf.key ' \
                      '--from-file=rdaf.cert=/opt/rdaf/cert/rdaf/rdaf.cert ' \
                      '--save-config --dry-run=client -o yaml |  kubectl apply -f -'.format(namespace)
        run_command(cert_secret)

    def get_k8s_install_args(self, cmd_args):
        from rdaf.component.dockerregistry import COMPONENT_NAME
        from rdaf.contextual import COMPONENT_REGISTRY
        docker_registry = COMPONENT_REGISTRY.require(COMPONENT_NAME)
        image = docker_registry.get_docker_registry_url() + '/rda-platform-nats'
        args = f'--set container.image.repository={image} --set container.image.tag={cmd_args.tag}'
        return args

    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != 'k8s':
            return
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            for image in ['rda-platform-nats', 'rda-platform-nats-server-config-reloader', 'rda-platform-nats-box']:
                logger.info(f'Pulling {image} on host {host}')
                docker_pull_command = f'docker pull {docker_repo}/{image}:{cmd_args.tag}'
                run_potential_ssh_command(host, docker_pull_command, config_parser)

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        if self.get_deployment_type(config_parser) != "aws":
            nats_pv_file = 'nats-pv-cluster.yaml' if len(self.get_hosts()) > 1 else 'nats-pv.yaml'
            nats_pvs = 'kubectl apply -f {} -n {}'\
                .format(os.path.join(get_templates_dir_root(), 'k8s-local', nats_pv_file), namespace)
            run_command(nats_pvs)

        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'nats-values.yaml')
        install_command = 'helm install  --create-namespace -n {} -f {} {} {} {}' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(), deployment_path)
        run_command(install_command)
    
    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name())
        deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name())
        self.copy_helm_chart(chart_template_path, deployment_path)
        values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'nats-values.yaml')
        upgrade_command = 'helm upgrade --install --create-namespace -n {} -f {} {} {} {} ' \
            .format(namespace, values_yaml, self.get_k8s_install_args(cmd_args), self.get_k8s_component_name(),
                    deployment_path)
        run_command(upgrade_command)
        
    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        comp_name = self.get_k8s_component_name()
        cmd = 'kubectl get pods -l {} -n {} -o json'.format(self.get_k8s_component_label(), namespace)
        ret, stdout, stderr = execute_command(cmd)
        component = json.loads(stdout)
        for items in component["items"]:
            metadata = items["metadata"]["ownerReferences"]
            for pod in metadata:
                kind = pod['kind']
                command = f'kubectl scale {kind}.apps/{comp_name} -n {namespace} --replicas=0'
                run_command(command)
                command = f'kubectl scale deployment.apps/rda-nats-box -n {namespace} --replicas=0'
                run_command(command)
                return
    
    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        comp_name = self.get_k8s_component_name()
        cmd = 'kubectl get all -l {} -n {} -o json'.format(self.get_k8s_component_label(), namespace)
        ret, stdout, stderr = execute_command(cmd)
        component = json.loads(stdout)
        replicas = len(self.get_hosts())
        for comp in component["items"]:
            if comp["kind"] == 'StatefulSet':
                command = f'kubectl scale statefulset.apps/{comp_name} -n {namespace} --replicas={replicas}'
                run_command(command)     
                command = f'kubectl scale deployment.apps/rda-nats-box -n {namespace} --replicas={replicas}'
                run_command(command)
                return
            
    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        super().install(cmd_args, config_parser)
        logger.info('Pushing the Nats accounts..')
        time.sleep(5)
        host = self.get_hosts()[0]
        nsc = os.path.join('/usr', 'local', 'bin', 'nsc')
        nkeys_path = os.path.join(self.get_data_dir(), '.nkeys')
        command = 'export NKEYS_PATH=' + nkeys_path \
                  + " && export NATS_CA='/opt/rdaf/cert/ca/ca.crt'" \
                  + ' && ' + nsc + ' push -a SYS --account-jwt-server-url nats://' + host + ':4222'
        run_potential_ssh_command(host, command, config_parser)

    def _setup_account(self, host: str, config_parser: configparser.ConfigParser, port='4222'):
        data_dir = self.get_data_dir()
        nkeys_path = os.path.join(data_dir, '.nkeys')
        jwt_file_path = self.get_operator_jwt_path()
        # check for the jwt presence
        if check_potential_remote_file_exists(host, jwt_file_path):
            logger.info('Skipping Operator and sys account creation for Nats server. Using existing')
            return

        nsc = os.path.join('/usr', 'local', 'bin', 'nsc')
        service_url = 'nats://' + host + ':' + port
        if host == self.get_hosts()[0]:
            command = 'export NKEYS_PATH=' + nkeys_path + ' && mkdir -p ' + data_dir \
                      + ' && ' + nsc + ' env -s ' + data_dir \
                      + ' && ' + nsc + ' add operator -n noperator --sys' \
                      + ' && ' + nsc + ' env -o noperator' \
                      + ' && ' + nsc + ' edit operator  --account-jwt-server-url ' + service_url \
                      + ' && ' + nsc + ' edit operator  --service-url ' + service_url
        else:
            if not os.path.exists(data_dir):
                with tempfile.TemporaryDirectory(prefix='nats') as tmp:
                    do_potential_scp_fetch(self.get_hosts()[0], data_dir, tmp)
                    do_potential_scp(host, tmp, data_dir)
            else:
                do_potential_scp(host, data_dir, data_dir)
            command = 'export NKEYS_PATH=' + nkeys_path \
                      + ' && ' + nsc + ' env -s ' + data_dir \
                      + ' && ' + nsc + ' env -o noperator' \
                      + ' && ' + nsc + ' edit operator  --account-jwt-server-url ' + service_url \
                      + ' && ' + nsc + ' edit operator  --service-url ' + service_url

        logger.info("Creating nats operator and accounts on host " + host)
        # execute a remote command
        run_potential_ssh_command(host, command, config_parser)

    def setup_tenant_users(self, uid, config_parser: configparser.ConfigParser, k8s=False):
        namespace = self.get_namespace(config_parser)
        logger.info("Creating Tenant User for Nats...")
        account = uid + 'Account'
        user = uid + 'rdauser'
        stream_user = uid + 'streamsuser'
        host = self.get_hosts()[0]
        nsc = os.path.join('/usr', 'local', 'bin', 'nsc')
        nkeys_path = os.path.join(self.get_data_dir(), '.nkeys')
        add_account = '{nsc} add account {Acc}'.format(nsc=nsc, Acc=account)
        add_user = '{nsc} add user -a {Acc} -n {User} --allow-pub "tenants.{UUID}.>" ' \
                   '--allow-sub "tenants.{UUID}.>"'.format(nsc=nsc, Acc=account, User=user, UUID=uid)
        edit_user = '{nsc} edit user -a {Acc} -n {User} --allow-pub "_INBOX.>"' \
                    ' --allow-sub "_INBOX.>"'.format(nsc=nsc, Acc=account, User=user)
        add_stream_user = '{nsc} add user -a {Acc} -n {User} --allow-pub "tenants.{UUID}.streams.>" ' \
                          '--allow-sub "tenants.{UUID}.streams.>"' \
            .format(nsc=nsc, Acc=account, User=stream_user, UUID=uid)

        if k8s:
            push_command = '{nsc} push -a {Acc} --system-account SYS -u nats://127.0.0.1:4222 ' \
                .format(nsc=nsc, Acc=account)
        else:
            push_command = '{nsc} push -a {Acc} '.format(nsc=nsc, Acc=account)

        command = "export NKEYS_PATH=" + nkeys_path \
                  + " && export NATS_CA='/opt/rdaf/cert/ca/ca.crt'" \
                  + " && " + add_account + " && " + add_user \
                  + " && " + edit_user + " && " \
                  + add_stream_user + " && " + push_command
        if k8s:
            pf_proc = subprocess.Popen('exec kubectl port-forward pods/rda-nats-0 -n {} '
                                       '4222:4222'.format(namespace), shell=True)
            time.sleep(10)

            completed_process = subprocess.run([command], shell=True)
            pf_proc.kill()
            if completed_process.stdout:
                logger.info(completed_process.stdout)
            if completed_process.returncode != 0:
                rdafutils.cli_err_exit('Command execution failed ' + str(completed_process.stderr))
        else:
            run_potential_ssh_command(host, command, config_parser)

    def _create_nats_conf(self, host: str):
        template_root = get_templates_dir_root()
        nats_file_name = 'nats-server-cluster.conf' if len(self.get_hosts()) > 1 else 'nats-server.conf'
        nats_conf = os.path.join(template_root, nats_file_name)
        with open(nats_conf, 'r') as f:
            template_content = f.read()

        replacements = dict()
        replacements['NATS_HOST'] = host
        replacements['NATS_CERT_FILE'] = os.path.join('/opt/rdaf/certs/rdaf/rdaf.cert')
        replacements['NATS_KEY_FILE'] = os.path.join('/opt/rdaf/certs/rdaf/rdaf.key')
        replacements['NATS_CA_FILE'] = os.path.join('/opt/rdaf/certs/', 'ca', 'ca.crt')
        replacements['NATS_ROUTES'] = self._get_routes(host)
        original_content = string.Template(template_content).substitute(replacements)
        return original_content

    def get_deployment_env(self, host: str):
        env = dict()
        env['NATS_CONFIG_MOUNT'] = self.get_conf_dir()
        env['NATS_DATA_MOUNT'] = self.get_data_dir()
        env['NATS_LOG_MOUNT'] = self.get_logs_dir()
        return env

    def get_operator_jwt_path(self) -> os.path:
        data_dir = os.path.join(self.get_data_dir())
        jwt_file_path = os.path.join(data_dir, "noperator", "noperator.jwt")
        return jwt_file_path

    def get_tenant_token(self, uid):
        account = uid + 'Account'
        user = uid + 'rdauser.creds'

        noperator_dir = os.path.join(self.get_data_dir(), '.nkeys', 'creds', 'noperator')
        file_path = os.path.join(noperator_dir, account, user)
        if os.path.exists(file_path):
            with open(file_path, 'r') as file:
                data = file.read().replace('\n', '\\n')
            return data
        with tempfile.NamedTemporaryFile(suffix='rdauser') as tmp:
            do_potential_scp_fetch(self.get_hosts()[0], file_path, tmp.name, False)
            with open(tmp.name, 'r') as file:
                data = file.read().replace('\n', '\\n')
        return data

    def get_cert(self):
        cert_path = os.path.join(self.get_rdaf_install_root(), 'cert', 'ca', 'ca.crt')
        with open(cert_path, 'r') as file:
            data = file.read().replace('\n', '\\n')
        return data

    def get_public_cert(self) -> str:
        return "-----BEGIN CERTIFICATE-----\\n" \
               "MIIE0DCCA7igAwIBAgIBBzANBgkqhkiG9w0BAQsFADCBgzELMAkGA1UEBhMCVVMx\\n" \
               "EDAOBgNVBAgTB0FyaXpvbmExEzARBgNVBAcTClNjb3R0c2RhbGUxGjAYBgNVBAoT\\n" \
               "EUdvRGFkZHkuY29tLCBJbmMuMTEwLwYDVQQDEyhHbyBEYWRkeSBSb290IENlcnRp\\n" \
               "ZmljYXRlIEF1dGhvcml0eSAtIEcyMB4XDTExMDUwMzA3MDAwMFoXDTMxMDUwMzA3\\n" \
               "MDAwMFowgbQxCzAJBgNVBAYTAlVTMRAwDgYDVQQIEwdBcml6b25hMRMwEQYDVQQH\\n" \
               "EwpTY290dHNkYWxlMRowGAYDVQQKExFHb0RhZGR5LmNvbSwgSW5jLjEtMCsGA1UE\\n" \
               "CxMkaHR0cDovL2NlcnRzLmdvZGFkZHkuY29tL3JlcG9zaXRvcnkvMTMwMQYDVQQD\\n" \
               "EypHbyBEYWRkeSBTZWN1cmUgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IC0gRzIwggEi\\n" \
               "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC54MsQ1K92vdSTYuswZLiBCGzD\\n" \
               "BNliF44v/z5lz4/OYuY8UhzaFkVLVat4a2ODYpDOD2lsmcgaFItMzEUz6ojcnqOv\\n" \
               "K/6AYZ15V8TPLvQ/MDxdR/yaFrzDN5ZBUY4RS1T4KL7QjL7wMDge87Am+GZHY23e\\n" \
               "cSZHjzhHU9FGHbTj3ADqRay9vHHZqm8A29vNMDp5T19MR/gd71vCxJ1gO7GyQ5HY\\n" \
               "pDNO6rPWJ0+tJYqlxvTV0KaudAVkV4i1RFXULSo6Pvi4vekyCgKUZMQWOlDxSq7n\\n" \
               "eTOvDCAHf+jfBDnCaQJsY1L6d8EbyHSHyLmTGFBUNUtpTrw700kuH9zB0lL7AgMB\\n" \
               "AAGjggEaMIIBFjAPBgNVHRMBAf8EBTADAQH/MA4GA1UdDwEB/wQEAwIBBjAdBgNV\\n" \
               "HQ4EFgQUQMK9J47MNIMwojPX+2yz8LQsgM4wHwYDVR0jBBgwFoAUOpqFBxBnKLbv\\n" \
               "9r0FQW4gwZTaD94wNAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhhodHRwOi8v\\n" \
               "b2NzcC5nb2RhZGR5LmNvbS8wNQYDVR0fBC4wLDAqoCigJoYkaHR0cDovL2NybC5n\\n" \
               "b2RhZGR5LmNvbS9nZHJvb3QtZzIuY3JsMEYGA1UdIAQ/MD0wOwYEVR0gADAzMDEG\\n" \
               "CCsGAQUFBwIBFiVodHRwczovL2NlcnRzLmdvZGFkZHkuY29tL3JlcG9zaXRvcnkv\\n" \
               "MA0GCSqGSIb3DQEBCwUAA4IBAQAIfmyTEMg4uJapkEv/oV9PBO9sPpyIBslQj6Zz\\n" \
               "91cxG7685C/b+LrTW+C05+Z5Yg4MotdqY3MxtfWoSKQ7CC2iXZDXtHwlTxFWMMS2\\n" \
               "RJ17LJ3lXubvDGGqv+QqG+6EnriDfcFDzkSnE3ANkR/0yBOtg2DZ2HKocyQetawi\\n" \
               "DsoXiWJYRBuriSUBAA/NxBti21G00w9RKpv0vHP8ds42pM3Z2Czqrpv1KrKQ0U11\\n" \
               "GIo/ikGQI31bS/6kA1ibRrLDYGCD+H1QQc7CoZDDu+8CL9IVVO5EFdkKrqeKM+2x\\n" \
               "LXY2JtwE65/3YR8V3Idv7kaWKK2hJn0KCacuBKONvPi8BDAB\\n-----END CERTIFICATE-----\\n\\n"

    def get_servers_url(self, port='4222'):
        return ','.join(['nats://' + i + ':' + port for i in self.get_hosts()])

    def _get_routes(self, host):
        hosts = self.get_hosts()
        if len(hosts) == 1:
            return ''
        routes = []
        for entry in hosts:
            if entry == host:
                continue
            routes.append('nats-route://cfxruser:T0pS3cr3t@{}:6222'.format(entry))
        return '{}'.format(','.join(routes))

    @staticmethod
    def copy_nsc_tool(host, config_parser):
        nsc_file = os.path.join(rdaf.get_scripts_dir_root(), 'nsc')
        nsc_path = os.path.join('/usr', 'local', 'bin', 'nsc')
        remote_tmp_path = os.path.join('/tmp', 'nsc')
        chmod_cmd = 'sudo chmod +x ' + nsc_path
        copy_command = 'sudo cp ' + nsc_file + ' ' + nsc_path
        ssh_copy_command = 'sudo cp ' + remote_tmp_path + ' ' + nsc_path
        if Component.is_local_host(host):
            run_command(copy_command)
            run_command(chmod_cmd)
        else:
            do_potential_scp(host, nsc_file, remote_tmp_path, sudo=True)
            run_potential_ssh_command(host, ssh_copy_command, config_parser)
            run_potential_ssh_command(host, chmod_cmd, config_parser)

    def healthcheck(self, component_name, host, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        logger.debug("Running Health check for Nats...")
        try:
            ca_cert = os.path.join('/opt/rdaf/cert/', 'ca', 'ca.crt')
            with NATSClient("tls://{}:4222".format(host),
                            tls_cacert=ca_cert) as client:
                # Connect
                client.connect()
        except Exception as e:
            return [component_name, "Service Status", termcolor.colored("Failed", color='red'), str(e)]

        return [component_name, "Service Status", "OK", "N/A"]

    def backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        component_data_dir = self.get_data_dir()

        # copy/backup from each component specific host
        for host in self.get_hosts():
            if not check_potential_remote_file_exists(host, component_data_dir):
                return
            command = 'sudo chown -R ' + str(os.getuid()) + ' ' + component_data_dir \
                      + ' && sudo chgrp -R ' + str(os.getgid()) + ' ' + component_data_dir
            run_potential_ssh_command(host, command, config_parser)

            dest_data_dir = os.path.join(backup_dir_root, 'data', host, self.get_name())
            logger.info('Backing up ' + component_data_dir + ' from host ' + host + ' to '
                        + dest_data_dir)
            os.makedirs(dest_data_dir, exist_ok=True)
            do_potential_scp_fetch(host, component_data_dir, dest_data_dir)

    def k8s_backup_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_state: configparser.ConfigParser, backup_dir_root: os.path):
        component_data_dir = self.get_data_dir()

        if not os.path.exists(component_data_dir):
            return

        command = 'sudo chown -R ' + str(os.getuid()) + ' ' + component_data_dir \
                  + ' && sudo chgrp -R ' + str(os.getgid()) + ' ' + component_data_dir
        run_command(command)
        dest_data_dir = os.path.join(backup_dir_root, 'data', self.get_name())
        logger.info('Backing up ' + component_data_dir + ' to ' + dest_data_dir)
        os.makedirs(dest_data_dir, exist_ok=True)
        run_command('cp -rT {} {} '.format(component_data_dir, dest_data_dir))

    def k8s_restore_conf(self, config_parser: configparser.ConfigParser,
                         backup_content_root_dir: os.path):
        self.create_cert_configs(config_parser)

    def check_pod_state_for_restore(self,config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = 'kubectl get pods -n ' + namespace + ' --selector=app_component=rda-nats ' \
                  '-o jsonpath="{.items[*].status.phase}"'
        exit_code, stdout, stderr = execute_command(command)
        if 'Running' not in str(stdout):
            cli_err_exit('Nats pod should be Running for restore')

    def k8s_restore_data(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                         backup_content_root_dir: os.path, backup_cfg_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        nats_backup_dir = os.path.join(backup_content_root_dir, 'data', self.get_name())
        if os.path.exists(nats_backup_dir):
            logger.info('Restoring ' + nats_backup_dir + ' to ' + self.get_data_dir())
            command = 'sudo cp -rT "{}" "{}"'.format(nats_backup_dir, self.get_data_dir())
            run_command(command)

        nkeys_path = os.path.join(self.get_data_dir(),  '.nkeys')
        push_command = 'nsc push -A --system-account SYS -u nats://127.0.0.1:4222 '
        command = "export NKEYS_PATH=" + nkeys_path \
                  + " && export NATS_CA='/opt/rdaf/cert/ca/ca.crt'" \
                  + " && " + push_command
        pf_proc = subprocess.Popen('exec kubectl port-forward pods/rda-nats-0 -n {} 4222:4222'.format(namespace),
                                   shell=True)
        time.sleep(10)

        completed_process = subprocess.run([command], shell=True)
        if completed_process.stdout:
            logger.info(completed_process.stdout)
        if pf_proc:
            pf_proc.kill()
        if completed_process.returncode != 0:
            rdafutils.cli_err_exit('Command execution failed ' + str(completed_process.stderr))

    @ staticmethod
    def _configure_cron_job():
        log_file = '/opt/rdaf/logs/nats_check_pods.log'
        os.makedirs(os.path.dirname(log_file), exist_ok=True)
        replacements = dict()
        replacements['KUBE_CONFIG'] = os.getenv('KUBECONFIG', os.path.expanduser('~/.kube/config'))
        replacements['LOG_FILE'] = log_file
        template_path = os.path.join(get_templates_dir_root(), 'k8s-local', 'check_nats_pods.py')
        destination_path = os.path.join('/opt', 'rdaf', 'config', 'nats', 'check_nats_pods.py')

        with open(template_path, 'r') as f:
            template_content = f.read()
        content = string.Template(template_content).safe_substitute(replacements)
        os.makedirs(os.path.dirname(destination_path), exist_ok=True)
        with open(destination_path, 'w') as f:
            f.write(content)
        command = "crontab -l | grep -v '{}' | crontab -".format(destination_path)
        execute_command(command)
        command = '(crontab -l 2>/dev/null || true; echo "* * * * *     python {}"; echo "* * * * *   ( sleep 30; python {})") | crontab -'.format(destination_path, destination_path)
        execute_command(command)
