import argparse
import configparser
import json
import logging
import os
import string
from rdaf import rdafutils
import time
from datetime import datetime
import yaml
from typing import Callable, Any, List
from rdaf.component import Component, run_potential_ssh_command, \
    do_potential_scp, AppCategoryOrder, run_command, execute_command
from rdaf.component import _comma_delimited_to_list, _list_to_comma_delimited
from rdaf.dockerutils import cliDockerSession
import rdaf.component.haproxy as haproxy_comp
from rdaf.contextual import COMPONENT_REGISTRY
import rdaf
from rdaf.rdafutils import cli_err_exit

logger = logging.getLogger(__name__)
COMPONENT_NAME = 'oia'


class OIA(Component):
    _service_host = 'service_host'

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'common', 'apps', AppCategoryOrder.OIA.value)

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._service_host:
            return _list_to_comma_delimited
        return None

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._service_host:
            return _comma_delimited_to_list
        return None

    def _init_default_configs(self):
        default_configs = dict()
        return default_configs

    def get_hosts(self) -> list:
        return self.configs[self._service_host]

    def get_ports(self) -> tuple:
        ports = ['8888', '8456']
        hosts = self.get_hosts()
        return hosts, ports

    def get_k8s_service_args(self, service, cmd_args):
        args = '--set tag={} '.format(cmd_args.tag)

        # adding the timestamp for daily tags which will work for upgrade scenarios to pull the
        # latest image, with a change to the chart templates.
        if cmd_args.tag == 'daily':
            timestamp = datetime.now().strftime("%Y-%m-%d-%H:%M:%S")
            args += ' --set {service}.extraEnvs[0].name=timestamp,{service}.extraEnvs[0].value={value}' \
                .format(service=self.get_k8s_services_map()[service], value=timestamp)
        return args

    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != 'k8s':
            return
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        services = self.get_k8s_involved_services() if cmd_args.services is None \
            else self.filter_k8s_involved_services(cmd_args.services)
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            for service in services:
                image = self.get_k8s_involved_images()[service]
                logger.info(f'Pulling {image} for service {service} on host {host}')
                docker_pull_command = f'docker pull {docker_repo}/{image}:{cmd_args.tag}'
                run_potential_ssh_command(host, docker_pull_command, config_parser)

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        services = self.get_k8s_involved_services() if cmd_args.services is None \
            else self.filter_k8s_involved_services(cmd_args.services)
        for service in services:
            chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name(), service)
            deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name(),
                                           service)
            self.copy_helm_chart(chart_template_path, deployment_path)
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
            install_command = 'helm install --create-namespace -n {} -f {} {} {} {} ' \
                .format(namespace, values_yaml, self.get_k8s_service_args(service, cmd_args), service, deployment_path)
            run_command(install_command)

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        services = self.get_k8s_involved_services() if cmd_args.services is None \
            else self.filter_k8s_involved_services(cmd_args.services)
        for service in services:
            chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name(), service)
            deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm', self.get_k8s_component_name(),
                                           service)
            self.copy_helm_chart(chart_template_path, deployment_path)
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
            upgrade_command = 'helm upgrade --install --create-namespace -n {} -f {} {} {} {} ' \
                .format(namespace, values_yaml, self.get_k8s_service_args(service, cmd_args), service, deployment_path)
            run_command(upgrade_command)

    @staticmethod
    def get_deployment_file_path(host) -> os.path:
        return os.path.join('/opt', 'rdaf', 'deployment-scripts', host, 'oia.yaml')

    def validate_input_services(self, cmd_args):
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        if 'services' not in cmd_args or not cmd_args.services:
            return
        for service in cmd_args.services:
            if service not in self.get_involved_services(template_path):
                cli_err_exit(service + ' service is not part of ' + self.get_name() + ' components')

    def create_compose_file_oia(self, host, service, cmd_args: argparse.Namespace):
        compose_file = self.get_deployment_file_path(host)
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        replacements = self.get_deployment_replacements(cmd_args)
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        values = yaml.safe_load(original_content)

        service_value = values['services'][service]
        # override with values.yaml inputs if provided any
        self.apply_user_inputs(service, service_value)

        if not os.path.exists(compose_file):
            content = {"services": {service: service_value}}
        else:
            with open(compose_file, 'r') as f:
                content = yaml.safe_load(f)
                content['services'][service] = service_value

        os.makedirs(os.path.dirname(compose_file), exist_ok=True)

        ipv6_network_block = self.generate_ipv6_network_block()
        if ipv6_network_block:
            if 'networks' not in content['services'][service]:
                content['services'][service]['networks'] = ['rda_app_oia']
            if 'networks' not in content:
                content['networks'] = {'rda_app_oia': ipv6_network_block}

        with open(compose_file, 'w') as f:
            yaml.safe_dump(content, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        if not Component.is_local_host(host):
            do_potential_scp(host, compose_file, compose_file)
        return compose_file

    def get_deployment_replacements(self, cmd_args: argparse.Namespace) -> dict:
        replacements = self._get_docker_repo()
        replacements['TAG'] = cmd_args.tag
        haproxy_advertised_host = COMPONENT_REGISTRY.require(haproxy_comp.COMPONENT_NAME).get_internal_access_host()
        replacements['ADVERTISED_HOST'] = haproxy_advertised_host
        return replacements

    def get_images_involved(self, services=None):
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        involved_services = self.get_involved_services(template_path) if services is None else services
        with open(template_path, 'r') as f:
            template_content = f.read()
        values = yaml.safe_load(template_content)
        images_involved = []
        # If service_name is provided, find the image associated with that service
        for service in involved_services:
            image = values['services'][service]['image']
            item = image.rsplit(':', 1)[0].split('/', 1)[1]
            images_involved.append(item)
        return images_involved

    def pull_images(self, cmd_args, config_parser):
        self.validate_input_services(cmd_args)
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        services = self.get_involved_services(template_path) if cmd_args.services is None \
            else cmd_args.services
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        images_involved = self.get_images_involved(services)
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            for image in images_involved:
                logger.info(f'Pulling {image} image on host {host}')
                docker_pull_command = f'docker pull {docker_repo}/{image}:{cmd_args.tag}'
                run_potential_ssh_command(host, docker_pull_command, config_parser)

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.open_ports(config_parser)
        self.validate_input_services(cmd_args)

        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        services = self.get_involved_services(template_path) if cmd_args.services is None \
            else cmd_args.services
        for index, service in enumerate(services):
            hosts = self.get_qualified_service_hosts(index, service, config_parser)
            for host in hosts:
                logger.info("Installing {} on host {}".format(service, host))
                compose_file_path = self.create_compose_file_oia(host, service, cmd_args)
                self.copy_logging_config(config_parser, host, service)
                command = '/usr/local/bin/docker-compose --project-name oia -f {file} up -d {service} ' \
                    .format(file=compose_file_path, service=service)
                run_potential_ssh_command(host, command, config_parser)

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        services = self.get_involved_services(template_path) if cmd_args.services is None \
            else cmd_args.services
        self.validate_input_services(cmd_args)
        if cmd_args.rolling:
            if len(self.get_hosts()) == 1:
                logger.info("OIA app is deployed on single host.. continuing with normal upgrade.")
                cmd_args.rolling = False
                self.upgrade(cmd_args, config_parser)
                return
            else:
                self.handle_rolling_upgrade(cmd_args, services, config_parser)
        else:
            for index, service in enumerate(services):
                hosts = self.get_qualified_service_hosts(index, service, config_parser)
                for host in hosts:
                    logger.info("Upgrading {} on host {}".format(service, host))
                    compose_file_path = self.create_compose_file_oia(host, service, cmd_args)
                    self.copy_logging_config(config_parser, host, service)
                    command = '/usr/local/bin/docker-compose --project-name oia -f {file} up -d {service} ' \
                        .format(file=compose_file_path, service=service)
                    run_potential_ssh_command(host, command, config_parser)

    def handle_rolling_upgrade(self, cmd_args, services, config_parser):
        if cmd_args.rolling and len(self.get_hosts()) == 1:
            logger.info("OIA app is deployed on single host.. continuing with normal upgrade.")
            cmd_args.rolling = False
            self.upgrade(cmd_args, config_parser)
            return
        
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        services = self.get_involved_services(template_path) if cmd_args.services is None \
                else cmd_args.services
        deployed_services = []
        for host in self.get_hosts():
            deployed_services += self.get_involved_services(self.get_deployment_file_path(host))

        new_services = list(set(services) - set(deployed_services))
        logger.info("Gathering OIA app container details.")
        service_host_container = {}
        for service in services:
            if service in new_services:
                continue
            for host in self.get_hosts():
                try:
                    with Component.new_docker_client(host, config_parser) as docker_client:
                        containers = self._find_component_container_on_host(docker_client, service, all_states=True)
                        if len(containers) == 0:
                            logger.debug('No container found for ' + service + ' on host ' + host)
                        else:
                            entry = service_host_container.get(service, {})
                            entry[host] = containers[0]['Id'][:12]
                            service_host_container[service] = entry
                except Exception:
                    logger.warning('Failed to get status of ' + service + ' on host ' + host,
                                   exc_info=1)
            if len(service_host_container[service]) < 2:
                if not rdafutils.query_yes_no(f"Warning: Only {len(service_host_container[service])} container(s) running for service {service}. "
                                             f"Minimum of two containers recommended for rolling upgrade. Continue anyway?"):
                    cli_err_exit(f"Exiting rolling upgrade, minimum of two containers need to be running for service {service}.")

        involved_containers = [value for hc in service_host_container.values() for value in hc.values()]
        # check if rdac pods are present
        logger.info("Gathering rdac pod details.")
        return_code, stdout, stderr = execute_command('rdac pods --json --show_maintenance')
        sanitized_output = rdafutils.cleanup_stdout(stdout)
        rdac_pods = json.loads(sanitized_output)
        container_id_pods = {}
        for pod in rdac_pods:
            # hostname is docker container id
            if pod.get("hostname") in involved_containers:
                container_id_pods[pod.get("hostname")] = pod

        missing = False
        if len(container_id_pods) != len(involved_containers):
            for entry in involved_containers:
                if entry not in container_id_pods.keys():
                    missing = True
                    logger.warning(f"Container {entry} not in rdac pods.")
        if missing:
            if not rdafutils.query_yes_no("Warning: Some containers are missing in rdac pods. This may affect the rolling upgrade. Continue anyway?"):
                cli_err_exit("Exiting...few containers are missing in rdac pods.")

        for host in self.get_hosts():
            containers_on_host = [hc[host] for hc in service_host_container.values() if host in hc]
            pre_maint_pods = {}
            maint_pod = {}
            table = [["Pod ID", "Pod Type", "Version", "Age", "Hostname", "Maintenance", "Pod Status"]]
            for entry in containers_on_host:
                # Skip containers that are missing from rdac pods
                if entry not in container_id_pods:
                    if not rdafutils.query_yes_no(f"Warning: Container {entry} is missing from rdac pods. Skip this container and continue?"):
                        cli_err_exit(f"Exiting due to missing container {entry} in rdac pods.")
                    logger.warning(f"Skipping container {entry} as it's missing from rdac pods")
                    continue
                pod = container_id_pods[entry]
                if pod.get('is_maintenance'):
                    logger.info(f'Pod {pod.get("pod_type", "")} is already in maintenance mode on {host}')
                    maint_pod[pod.get("hostname", "")] = pod
                    continue
                
                age = datetime.fromisoformat(pod.get("now")) - datetime.fromisoformat(pod.get("started_at"))
                age = str(age).rsplit(".", 1)[0]
                table.append(
                    [pod.get("pod_id", ""), pod.get("pod_type", ""), pod.get("build_tag", ""), str(age),
                     pod.get("hostname", ""), pod.get("is_maintenance"), pod.get("pod_ready")])
                pre_maint_pods[pod.get("hostname", "")] = pod
            if len(table) == 1 and not maint_pod:
                logger.info(f'{host} all pods are already running with tag {cmd_args.tag}')
                continue
            if pre_maint_pods:
                column_headers = table[0]
                rows = table[1:]
                rdafutils.print_tabular(column_headers, rows)
                if not rdafutils.query_yes_no("Continue moving above pods to maintenance mode?"):
                    return
                pod_ids = ",".join([pod.get("pod_id") for pod in pre_maint_pods.values()])
                maint_command = f"rdac maintenance start --ids {pod_ids}"
                logger.info('Initiating Maintenance Mode...')
                execute_command(maint_command)

            # checking for pods in maintenance mode
            maint_pod = {}
            retry = 0
            while retry < 10:
                retry += 1
                return_code, stdout, stderr = execute_command('rdac pods --show_maintenance --json')
                sanitized_output = rdafutils.cleanup_stdout(stdout)
                rdac_pods = json.loads(sanitized_output)
                for pod in rdac_pods:
                    if pod.get("hostname") in involved_containers and pod.get('is_maintenance'):
                        maint_pod[pod.get("hostname")] = pod
                if len(maint_pod) >= len(pre_maint_pods):
                    break
                logger.info(f"Waiting for services to be moved to maintenance.")
                time.sleep(20)
            if not maint_pod:
                if not rdafutils.query_yes_no(f"Warning: Failed to move services on host {host} to maintenance mode. Continue anyway?"):
                    cli_err_exit(f'Failed to move services on host {host} to maintenance.')

            logger.info("Following container are in maintenance mode")
            table = [["Pod ID", "Pod Type", "Version", "Age", "Hostname", "Maintenance", "Pod Status"]]
            for pod in maint_pod.values():
                age = datetime.fromisoformat(pod.get("now")) - datetime.fromisoformat(pod.get("started_at"))
                age = str(age).rsplit(".", 1)[0]
                table.append([pod.get("pod_id", ""), pod.get("pod_type", ""), pod.get("build_tag", ""),
                              str(age), pod.get("hostname", ""), pod.get("is_maintenance"), pod.get("pod_ready")])
            column_headers = table[0]
            rows = table[1:]
            rdafutils.print_tabular(column_headers, rows)
            timeout = cmd_args.timeout
            logger.info(f"Waiting for timeout of {timeout} seconds...")
            time.sleep(timeout)
            # upgrading the pods in maintenance
            involved_services = [key for key, host_containers in service_host_container.items() for subkey, value in host_containers.items() if value in maint_pod.keys()]
            involved_services.extend(new_services)
            for service in involved_services:
                logger.info(f"Upgrading {service} on host {host}")
                compose_file_path = self.create_compose_file_oia(host, service, cmd_args)
                self.copy_logging_config(config_parser, host, service)
                command = '/usr/local/bin/docker-compose --project-name oia -f {file} up -d {service} ' \
                    .format(file=compose_file_path, service=service)
                run_potential_ssh_command(host, command, config_parser)
            logger.info("Waiting for upgraded containers to join rdac pods")
            self.wait_for_component_to_join_pods(host, services, config_parser)

    def wait_for_component_to_join_pods(self, host, services, config_parser):
        logger.info(f"Checking if the upgraded components '{services}' has joined the rdac pods...")
        container_service = {}  # Initialize container to None
        for service in services:
            try:
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = self._find_component_container_on_host(docker_client, service, all_states=True)
                    if len(containers) > 0:
                        container_service[containers[0]['Id'][:12]] = service
            except Exception:
                logger.warning(f'Failed to get status of {service} on host {host}', exc_info=1)

        retry = 0
        while retry < 10:
            time.sleep(10)
            retry += 1
            return_code, stdout, stderr = execute_command('rdac pods --json')
            sanitized_output = rdafutils.cleanup_stdout(stdout)
            rdac_pods = json.loads(sanitized_output)
            for pod in rdac_pods:
                if pod.get("hostname") in container_service.keys():
                    container_service.pop(pod.get("hostname"), "")
            if container_service:
                logger.info(f"Waiting for oia components to be up and running... retry {retry}")
                continue

        if container_service:
            failed_components = list(container_service.values())
            if not rdafutils.query_yes_no(f"Warning: Components '{failed_components}' on host {host} failed to join rdac pods. Continue anyway?"):
                cli_err_exit(f"Components '{failed_components}' on host {host} has not been upgraded properly.")

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.validate_input_services(cmd_args)
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        services = self.get_involved_services(template_path) if cmd_args.services is None \
            else cmd_args.services

        if cmd_args.services is None:
            for host in self.get_hosts():
                deployment_file_path = self.get_deployment_file_path(host)
                if os.path.exists(deployment_file_path):
                    logger.info("Creating oia services on host {}".format(host))
                    command = '/usr/local/bin/docker-compose --project-name oia -f ' \
                              + self.get_deployment_file_path(host) + ' up -d '
                    run_potential_ssh_command(host, command, config_parser)
        else:
            for index, service in enumerate(services):
                hosts = self.get_qualified_service_hosts(index, service, config_parser)
                for host in hosts:
                    logger.info("Creating service: {} on host {}".format(service, host))
                    command = '/usr/local/bin/docker-compose --project-name oia -f {file} up -d {service} ' \
                        .format(file=self.get_deployment_file_path(host), service=service)
                    run_potential_ssh_command(host, command, config_parser)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.validate_input_services(cmd_args)
        if hasattr(cmd_args, 'services') and cmd_args.services is not None:
            for service in cmd_args.services:
                for host in self.get_all_service_hosts(service, config_parser):
                    logger.info("Deleting service: {} on host {}".format(service, host))
                    command = '/usr/local/bin/docker-compose --project-name oia -f ' + \
                              self.get_deployment_file_path(host) + ' rm -fs ' + service
                    run_potential_ssh_command(host, command, config_parser)
        else:
            for host in reversed(self.get_hosts()):
                deployment_file_path = self.get_deployment_file_path(host)
                if os.path.exists(deployment_file_path):
                    logger.info("Deleting oia services on host {}".format(host))
                    command = '/usr/local/bin/docker-compose --project-name oia -f ' \
                              + self.get_deployment_file_path(host) + ' rm -fsv'
                    run_potential_ssh_command(host, command, config_parser)

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.validate_input_services(cmd_args)
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        services = self.get_involved_services(template_path) if cmd_args.services is None \
            else cmd_args.services

        if cmd_args.services is None:
            for host in self.get_hosts():
                deployment_file_path = self.get_deployment_file_path(host)
                if os.path.exists(deployment_file_path):
                    logger.info("Starting oia services on host {}".format(host))
                    command = '/usr/local/bin/docker-compose --project-name oia -f ' \
                              + self.get_deployment_file_path(host) + ' start '
                    run_potential_ssh_command(host, command, config_parser)
        else:
            for index, service in enumerate(services):
                hosts = self.get_qualified_service_hosts(index, service, config_parser)
                for host in hosts:
                    logger.info("Starting service: {} on host {}".format(service, host))
                    command = '/usr/local/bin/docker-compose --project-name oia -f {file} start  {service} ' \
                        .format(file=self.get_deployment_file_path(host), service=service)
                    run_potential_ssh_command(host, command, config_parser)

    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.validate_input_services(cmd_args)
        if hasattr(cmd_args, 'services') and cmd_args.services is not None:
            for service in cmd_args.services:
                for host in self.get_all_service_hosts(service, config_parser):
                    logger.info("Stopping service: {} on host {}".format(service, host))
                    command = '/usr/local/bin/docker-compose --project-name oia -f ' + \
                              self.get_deployment_file_path(host) + ' stop ' + service
                    run_potential_ssh_command(host, command, config_parser)
        else:
            for host in reversed(self.get_hosts()):
                deployment_file_path = self.get_deployment_file_path(host)
                if os.path.exists(deployment_file_path):
                    logger.info("Stopping oia services on host {}".format(host))
                    command = '/usr/local/bin/docker-compose --project-name oia -f ' \
                              + self.get_deployment_file_path(host) + ' stop'
                    run_potential_ssh_command(host, command, config_parser)
    
    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = f'kubectl get deployments -n {namespace} -l app_name=oia -o json'
        ret, stdout, stderr = execute_command(command)
        components = json.loads(stdout)

        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
        service_map = self.get_k8s_services_map()
        if cmd_args.services is None:
            for component in components["items"]:
                comp = component["metadata"]["labels"]["app_component"]
                replicas = data[service_map[comp]]['replicas']
                command = f'kubectl scale deployment.apps/{comp} -n {namespace} --replicas={replicas}'
                run_command(command)
                logger.info(f"Creating service: {comp} ")
        else:
            for service in cmd_args.services:
                replicas = data[service_map[service]]['replicas']
                command = f'kubectl scale deployment.apps/{service} -n {namespace} --replicas={replicas}'
                run_command(command)
                logger.info(f"Creating service: {service} ")

    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = f'kubectl get deployments -n {namespace} -l app_name=oia -o json'
        ret, stdout, stderr = execute_command(command)
        components = json.loads(stdout)
        if cmd_args.services is None:
            for component in components["items"]:
                comp = component["metadata"]["labels"]["app_component"]
                command = f'kubectl scale deployment.apps/{comp} -n {namespace} --replicas=0'
                run_command(command)
                logger.info(f"Deleting service: {comp} ")
                if cmd_args.force:
                    delete_pods_command = f'kubectl delete pods -n {namespace} -l app_component={comp} --grace-period=0 --force'
                    run_command(delete_pods_command)
        else:
            for service in cmd_args.services:
                oia_services = self.get_k8s_involved_services()
                if not service in oia_services:
                   cli_err_exit(f"{service} does not belong to app services")
                command = f'kubectl scale deployment.apps/{service} -n {namespace} --replicas=0'
                run_command(command)
                logger.info(f"Deleting service: {service} ")
                if cmd_args.force:
                    delete_pods_command = f'kubectl delete pods -n {namespace} -l app_component={service} --grace-period=0 --force'
                    run_command(delete_pods_command)

    def _find_component_container_on_host(self, docker_client: cliDockerSession, service,
                                          all_states: bool = False) -> List:
        labels = {'com.docker.compose.service': service}
        component_filter_labels = []
        for k, v in labels.items():
            component_filter_labels.append(str(k + '=' + v))
        return docker_client.client.containers(all=all_states,
                                               filters={'label': component_filter_labels})

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        statuses = []
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'oia.yaml')
        services = self.get_involved_services(template_path)

        for service in services:
            for host in self.get_all_service_hosts(service, config_parser):
                deployment_file_path = self.get_deployment_file_path(host)
                if not os.path.exists(deployment_file_path):
                    logger.debug("No OIA app services deployed.")
                    return statuses
                component_status = dict()
                component_status['component_name'] = service
                component_status['host'] = host
                try:
                    with Component.new_docker_client(host, config_parser) as docker_client:
                        containers = self._find_component_container_on_host(docker_client, service,
                                                                            all_states=True)
                        if len(containers) == 0:
                            logger.debug(
                                'No container found for ' + service + ' on host ' + host)
                            component_status['containers'] = []
                        else:
                            component_status['containers'] = containers
                except Exception:
                    logger.debug('Failed to get status of ' + service + ' on host ' + host,
                                 exc_info=1)
                    # set the status as error
                    component_status['status'] = {'error': True, 'message': 'Unknown'}
                statuses.append(component_status)
        return statuses

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        namespace = self.get_namespace(config_parser)
        statuses = []
        status_command = f'kubectl get pods -n {namespace} -l app_name=oia -o json'
        ret, stdout, stderr = execute_command(status_command)
        if ret != 0:
            cli_err_exit(f"Failed to get status of OIA app, due to: {str(stderr)}.")
        
        result = json.loads(str(stdout))
        items = result.get('items', [])
        hosts = self.get_hosts()
        
        if not hosts:
            cli_err_exit("No hosts defined for App.")
        for host in hosts:
            deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm',
                                               self.get_k8s_component_name())
            if os.path.exists(deployment_path):
                if not items:
                    logger.debug("No Pods found for OIA...")
                    for host in hosts:
                        for service in self.get_k8s_involved_services():
                            statuses.append({'component_name': service, 'host': host, 'containers': []})
                    return statuses
         
                for item in items:
                    pod = {
                        'component_name': item['metadata']['labels']['app_component'],
                        'host': item['status'].get('hostIP', 'Unknown'),
                        'containers': []
                    }
                    if 'containerStatuses' in item['status']:
                        for entry in item['status']['containerStatuses']:
                            if 'containerID' in entry:
                                container = {
                                    'Id': entry['containerID'],
                                    'Image': entry['image'],
                                    'State': next(iter(entry['state'].keys())),
                                    'Status': (self.get_container_age(entry['state']['running']['startedAt']) 
                                               if 'running' in entry['state'] else next(iter(entry['state'].keys())))
                                }
                                pod['containers'].append(container)
                    statuses.append(pod)
          
                for host in hosts:
                    for service in self.get_k8s_involved_services():
                        if not any(status['component_name'] == service for status in statuses):
                            statuses.append({'component_name': service, 'host': host, 'containers': []})
                
                return statuses

    def update_config(self, config_parser: configparser.ConfigParser):
        content = \
            """
backend webhook
    mode http
    balance roundrobin
    stick-table type ip size 10k expire 10m
    stick on src
    option httpchk GET /healthcheck
    http-check expect rstatus (2|3)[0-9][0-9]
    http-check disable-on-404
    http-response set-header Cache-Control no-store
    http-response set-header Pragma no-cache
    default-server inter 10s downinter 5s fall 3 rise 2
    cookie SERVERID insert indirect nocache maxidle 30m maxlife 24h httponly secure
    ${BACKEND_WEBHOOK_SERVER_DIRECTIVE}

frontend smtp
    bind 0.0.0.0:25
    mode tcp
    timeout client 1m
    log global
    option tcplog
    default_backend smtp

backend smtp
    mode tcp
    log global
    option tcplog
    timeout server 1m
    timeout connect 7s
    ${BACKEND_SMTP_SERVER_DIRECTIVE}

"""
        replacements = dict()
        server_id = 0
        webhook_server_directives = []
        for host in self.get_all_service_hosts('cfx-rda-webhook-server', config_parser):
            server_id += 1
            server_name = 'rdaf-webhook-' + str(server_id)
            webhook_server_directives.append('server ' + server_name + ' ' + host + ':8888' +
                                             ' ' + 'check' + ' ' + 'cookie ' + server_name)

        replacements['BACKEND_WEBHOOK_SERVER_DIRECTIVE'] = '\n    '.join(webhook_server_directives)

        server_id = 0
        smtp_server_directives = []
        for host in self.get_all_service_hosts('cfx-rda-smtp-server', config_parser):
            server_id += 1
            server_name = 'rdaf-smtp-' + str(server_id)
            smtp_server_directives.append('server ' + server_name + ' ' + host + ':8456')

        replacements['BACKEND_SMTP_SERVER_DIRECTIVE'] = '\n    '.join(smtp_server_directives)

        replaced_content = string.Template(content).substitute(replacements)
        haproxy = COMPONENT_REGISTRY.require(haproxy_comp.COMPONENT_NAME)
        haproxy.append_config(replaced_content, config_parser)

    def update_k8s_config(self, config_parser):
        logger.info('Updated HAProxy configuration')

    def get_all_service_hosts(self, service: str, config_parser):
        # load the values.yaml if any
        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
            if service in data['services'] and data['services'][service].get('hosts'):
                return data['services'][service].get('hosts')

        qualified_hosts = self.get_service_hosts(config_parser)
        return qualified_hosts

    def get_qualified_service_hosts(self, index: int, service: str, config_parser):
        # load the values.yaml if any
        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
            if service in data['services'] and data['services'][service].get('hosts'):
                # to remove the duplicates
                return list(dict.fromkeys(data['services'][service].get('hosts')))

        qualified_hosts = self.get_service_hosts(config_parser)
        selected = index % len(qualified_hosts)
        return [qualified_hosts[selected]]

    @staticmethod
    def get_k8s_services_map():
        return {'rda-alert-ingester': 'alert_ingester', 'rda-alert-processor': 'alert_processor',
                'rda-app-controller': 'app_controller', 'rda-collaboration': 'collaboration',
                'rda-configuration-service': 'configuration_service', 'rda-event-consumer': 'event_consumer',
                'rda-file-browser': 'file_browser', 'rda-ingestion-tracker': 'ingestion_tracker',
                'rda-irm-service': 'irm_service', 'rda-ml-config': 'ml_config',
                'rda-alert-processor-companion': 'alert_processor_companion', 'rda-alert-correlator': 'alert_correlator',
                'rda-notification-service': 'notification_service', 'rda-reports-registry': 'reports_registry',
                'rda-smtp-server': 'smtp_server', 'rda-webhook-server': 'webhook_server'}

    @staticmethod
    def get_k8s_involved_images():
        return {
                'rda-alert-ingester': 'cfx-rda-alert-ingester',
                'rda-alert-processor': 'cfx-rda-alert-processor',
                'rda-alert-correlator': 'cfx-rda-alert-processor',
                'rda-alert-processor-companion': 'cfx-rda-alert-processor-companion',
                'rda-app-controller': 'cfx-rda-app-controller',
                'rda-collaboration': 'cfx-rda-collaboration',
                'rda-configuration-service': 'cfx-rda-configuration-service',
                'rda-event-consumer': 'cfx-rda-event-consumer',
                'rda-file-browser': 'cfx-rda-file-browser',
                'rda-ingestion-tracker': 'cfx-rda-ingestion-tracker',
                'rda-irm-service': 'cfx-rda-irm-service',
                'rda-ml-config': 'cfx-rda-ml-config',
                'rda-notification-service': 'cfx-rda-notification-service',
                'rda-reports-registry': 'cfx-rda-reports-registry',
                'rda-smtp-server': 'cfx-rda-smtp-server',
                'rda-webhook-server': 'cfx-rda-webhook-server'
        }