import argparse
import configparser
from datetime import datetime, timedelta
import json
import logging
import os
import secrets
import socket
import string
import subprocess
import tempfile
import time
import uuid
from typing import Callable, Any, List
from rdaf import rdafutils
import jwt
import requests
import termcolor
import yaml
from OpenSSL import crypto


import rdaf
import rdaf.component.haproxy as haproxy
import rdaf.component.kafka as kafka_comp
import rdaf.component.mariadb as mariadb_comp
import rdaf.component.minio as minio_comp
import rdaf.component.nats as nats_comp
import rdaf.component.opensearch as opensearch_comp
import rdaf.component.graphdb as graphdb_comp
from rdaf import get_templates_dir_root
from rdaf.component import Component, PlatformCategoryOrder, \
    run_potential_ssh_command, do_potential_scp, run_command_exitcode, execute_command, do_potential_scp_fetch, \
    check_potential_remote_file_exists
from rdaf.component import _comma_delimited_to_list, _list_to_comma_delimited, \
    create_file, run_command
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.dockerutils import cliDockerSession
from rdaf.rdafutils import gen_password, cli_err_exit, str_base64_decode
from rdaf.util import requestsutil

logger = logging.getLogger(__name__)
COMPONENT_NAME = 'platform'


class RDAPlatform(Component):
    _platform_host = 'platform_service_host'
    _service_hosts = 'service_host'
    _portal_token = 'portal_token'
    _api_key = 'api_key'
    _tenant_id = 'tenant_id'
    _option_admin_organization = 'admin_organization'

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'common', 'platform',
                         PlatformCategoryOrder.PLATFORM.value)

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._platform_host or \
                config_name == self._service_hosts:
            return _list_to_comma_delimited
        return None

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._platform_host or \
                config_name == self._service_hosts:
            return _comma_delimited_to_list
        return None

    def _init_default_configs(self):
        default_configs = dict()
        return default_configs

    def get_hosts(self) -> list:
        return self.configs[self._platform_host]

    def get_admin_organization(self) -> str:
        return self.configs[self._option_admin_organization]

    def get_conf_dir(self) -> os.path:
        return os.path.join(self.get_rdaf_install_root(), 'config')

    def get_ports(self) -> tuple:
        ports = ['8807', '8080', '7780', '8001']
        hosts = self.get_hosts()
        return hosts, ports

    def get_involved_hosts(self, service: str):
        # we should bundle the frontend and backend together
        if service == 'portal-backend':
            service = 'portal-frontend'

        # load the values.yaml if any
        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
            if service in data['services'] and data['services'][service].get('hosts'):
                return data['services'][service].get('hosts')
        return self.get_hosts()
    
    def get_qualified_platform_hosts(self, index: int, service: str):
        # we should bundle the frontend and backend together
        if service == 'portal-backend':
            service = 'portal-frontend'

        # load the values.yaml if any
        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
            if service in data['services'] and data['services'][service].get('hosts'):
                # to remove the duplicates
                return list(dict.fromkeys(data['services'][service].get('hosts')))

        qualified_hosts = self.get_hosts()
        selected = index % len(qualified_hosts)
        return [qualified_hosts[selected]]
    
    def get_deployment_file_path(self, host) -> os.path:
        return os.path.join('/opt', 'rdaf', 'deployment-scripts', host, 'platform.yaml')

    def validate_input_services(self, cmd_args):
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'platform.yaml')
        if 'services' not in cmd_args or not cmd_args.services:
            return
        for service in cmd_args.services:
            if service not in self.get_involved_services(template_path):
                cli_err_exit(service + ' service is not part of ' + self.get_name() + ' components')

    def get_deployment_replacements(self, cmd_args: argparse.Namespace) -> dict:
        es = COMPONENT_REGISTRY.require(opensearch_comp.COMPONENT_NAME)
        replacements = self._get_docker_repo()
        replacements['TAG'] = cmd_args.tag
        replacements['RDA_JWT_SECRET_KEY'] = self.configs[self._api_key]
        haproxy_advertised_host = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME) \
            .get_internal_access_host()
        replacements['PORTAL_HOST'] = haproxy_advertised_host
        replacements['RDAC_ENDPOINT'] = f'http://{haproxy_advertised_host}:8808'
        replacements['OPENSEARCH_ADMIN_USER'] = es.get_user()
        replacements['OPENSEARCH_ADMIN_PASSWORD'] = es.get_password()
        return replacements

    def create_compose_file_platform(self, service, host, cmd_args: argparse.Namespace):
        compose_file = self.get_deployment_file_path(host)
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'platform.yaml')
        replacements = self.get_deployment_replacements(cmd_args)
        with open(template_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        values = yaml.safe_load(original_content)

        service_value = values['services'][service]
        # override with values.yaml inputs if provided any
        self.apply_user_inputs(service, service_value)
        if not os.path.exists(compose_file):
            content = {"services": {service: service_value}}
        else:
            with open(compose_file, 'r') as f:
                content = yaml.safe_load(f)
                content['services'][service] = service_value

        ipv6_network_block = self.generate_ipv6_network_block()
        if ipv6_network_block and 'network_mode' not in content['services'][service]:
            if 'networks' not in content['services'][service]:
                content['services'][service]['networks'] = ['rda_platform']
            if 'networks' not in content:
                content['networks'] = {'rda_platform': ipv6_network_block}
                
        os.makedirs(os.path.dirname(compose_file), exist_ok=True)
        with open(compose_file, 'w') as f:
            yaml.safe_dump(content, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)
        if not Component.is_local_host(host):
            do_potential_scp(host, compose_file, compose_file)
        return compose_file

    def get_images_involved(self, services=None):
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'platform.yaml')
        involved_services = self.get_involved_services(template_path) if services is None else services
        with open(template_path, 'r') as f:
            template_content = f.read()
        values = yaml.safe_load(template_content)
        images_involved = []
        # If service_name is provided, find the image associated with that service
        for service in involved_services:
            image = values['services'][service]['image']
            item = image.rsplit(':', 1)[0].split('/', 1)[1]
            images_involved.append(item)
        return images_involved

    def pull_images(self, cmd_args, config_parser):
        self.validate_input_services(cmd_args)
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'platform.yaml')

        services = self.get_involved_services(template_path) if cmd_args.services is None else cmd_args.services
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        images_involved = self.get_images_involved(services)
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            for image in images_involved:
                logger.info(f'Pulling {image} image on host {host}')
                docker_pull_command = f'docker pull {docker_repo}/{image}:{cmd_args.tag}'
                run_potential_ssh_command(host, docker_pull_command, config_parser)

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.validate_input_services(cmd_args)
        if self.get_deployment_type(config_parser) == 'rda-edge':
            self._generate_edge_tenant_configuration(config_parser)
        else:
            self._generate_tenant_configuration(config_parser)
        self._generate_data_plane_policy(config_parser)
        self._generate_portal_configuration(config_parser)
        self._generate_geodr_config(config_parser)
        self.open_ports(config_parser)

        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'platform.yaml')

        services = self.get_involved_services(template_path) if cmd_args.services is None else cmd_args.services
        # we initialize schema when ever portal back-end is installed
        if 'portal-backend' in services:
            #self._copy_nats_keys(config_parser)
            self._init_portal_schema(config_parser, cmd_args.tag)

        for index, service in enumerate(services):
            hosts = self.get_qualified_platform_hosts(index, service)
            for host in hosts:
                logger.info("Installing service: {} on host {}".format(service, host))
                if service != 'portal-frontend':
                    self.copy_logging_config(config_parser, host, service)
                compose_file_path = self.create_compose_file_platform(service, host, cmd_args)
                command = '/usr/local/bin/docker-compose --project-name platform -f {file} up -d {service} ' \
                    .format(file=compose_file_path, service=service)
                run_potential_ssh_command(host, command, config_parser)

        if 'portal-frontend' in services:
            self._create_user(config_parser)
            haproxy_advertised_host = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME) \
                .get_internal_access_host()
            ui_url = '\033[1;4m' + "https://" + haproxy_advertised_host + '\033[0m'
            print('\n')
            logger.info('\033[1m UI can be accessed at - \033[0m' + ui_url
                        + '\033[1m with default user - \033[0m' + '\033[1;4madmin@cfx.com\033[0m')

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.validate_input_services(cmd_args)
        if cmd_args.rolling:
            self.handle_rolling_upgrade(cmd_args, config_parser)
            return

        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'platform.yaml')

        services = self.get_involved_services(template_path) if cmd_args.services is None \
            else cmd_args.services
        if 'portal-backend' in services:
            self._handle_portal_db_upgrades(cmd_args.tag, config_parser)

        for service in services:
            for host in self.get_hosts():
                logger.info("Upgrading service: {} on host {}".format(service, host))
                if service != 'portal-frontend':
                    self.copy_logging_config(config_parser, host, service)
                compose_file_path = self.create_compose_file_platform(service, host, cmd_args)
                command = '/usr/local/bin/docker-compose --project-name platform -f {file} up -d {service} ' \
                    .format(file=compose_file_path, service=service)
                run_potential_ssh_command(host, command, config_parser)

    def wait_for_component_to_join_pods(self, host, services, config_parser):
        logger.info(f"Checking if the upgraded components '{services}' has joined the rdac pods...")
        container_service = {}  # Initialize container to None
        for service in services:
            # if service in ['portal-backend', 'portal-frontend', 'rda_chat_helper']:
            if service in ['portal-backend', 'portal-frontend']:
                continue
            try:
                with Component.new_docker_client(host, config_parser) as docker_client:
                    containers = self._find_component_container_on_host(docker_client, service, all_states=True)
                    if len(containers) > 0:
                        container_service[containers[0]['Id'][:12]] = service
            except Exception:
                logger.warning(f'Failed to get status of {service} on host {host}', exc_info=1)

        retry = 0
        while retry < 10:
            time.sleep(10)
            retry += 1
            return_code, stdout, stderr = execute_command('rdac pods --json')
            sanitized_output = rdafutils.cleanup_stdout(stdout)
            rdac_pods = json.loads(sanitized_output)
            for pod in rdac_pods:
                if pod.get("hostname") in container_service.keys():
                    container_service.pop(pod.get("hostname"), "")
            if container_service:
                logger.info(f"Waiting for platform components to be up and running... retry {retry}")
                continue

        if container_service:
            failed_components = list(container_service.values())
            if not rdafutils.query_yes_no(f"Warning: Components '{failed_components}' on host {host} failed to join rdac pods. Continue anyway?"):
                cli_err_exit(f"Components '{failed_components}' on host {host} has not been upgraded properly.")

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.validate_input_services(cmd_args)
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'platform.yaml')
        services = self.get_involved_services(template_path) if cmd_args.services is None \
            else cmd_args.services

        if cmd_args.services is None:
            for host in self.get_hosts():
                deployment_file_path = self.get_deployment_file_path(host)
                if os.path.exists(deployment_file_path):
                    logger.info("Creating platform services on host {}".format(host))
                    command = '/usr/local/bin/docker-compose --project-name platform -f ' \
                              + self.get_deployment_file_path(host) + ' up -d '
                    run_potential_ssh_command(host, command, config_parser)
        else:
            for index, service in enumerate(services):
                for host in self.get_involved_hosts(service):
                    logger.info("Creating service: {} on host {}".format(service, host))
                    command = '/usr/local/bin/docker-compose --project-name platform -f {file} up -d {service} ' \
                        .format(file=self.get_deployment_file_path(host), service=service)
                    run_potential_ssh_command(host, command, config_parser)


    def k8s_up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = f'kubectl get deployments -n {namespace} -l app_category=rdaf-platform -o json'
        ret, stdout, stderr = execute_command(command)
        components = json.loads(stdout)

        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
        service_map = self.get_k8s_services_map()
        if cmd_args.services is None:
            for component in components["items"]:
                comp = component["metadata"]["labels"]["app_component"]
                replicas = data[service_map[comp]]['replicas']
                command = f'kubectl scale deployment.apps/{comp} -n {namespace} --replicas={replicas}'
                run_command(command)
                logger.info(f"Creating service: {comp} ")
        else:
            for service in cmd_args.services:
                replicas = data[service_map[service]]['replicas']
                command = f'kubectl scale deployment.apps/{service} -n {namespace} --replicas={replicas}'
                run_command(command)
                logger.info(f"Creating service: {service} ")

    def k8s_down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        command = f'kubectl get deployments -n {namespace} -l app_category=rdaf-platform -o json'
        ret, stdout, stderr = execute_command(command)
        components = json.loads(stdout)
        if cmd_args.services is None:
            for component in components["items"]:
                comp = component["metadata"]["labels"]["app_component"]
                command = f'kubectl scale deployment.apps/{comp} -n {namespace} --replicas=0'
                run_command(command)
                logger.info(f"Deleting service: {comp}")
                if cmd_args.force:
                    delete_pods_command = f'kubectl delete pods -n {namespace} -l app_component={comp} --grace-period=0 --force'
                    run_command(delete_pods_command)
        else:
            for service in cmd_args.services:
                platform_services = self.get_k8s_involved_services()
                if not service in platform_services:
                   cli_err_exit(f"{service} does not belong to platform services")
                command = f'kubectl scale deployment.apps/{service} -n {namespace} --replicas=0'
                run_command(command)
                logger.info(f"Deleting service: {service} ")
                if cmd_args.force:
                    delete_pods_command = f'kubectl delete pods -n {namespace} -l app_component={service} --grace-period=0 --force'
                    run_command(delete_pods_command)
                    
    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.validate_input_services(cmd_args)
        if hasattr(cmd_args, 'services') and cmd_args.services is not None:
            for service in cmd_args.services:
                for host in self.get_involved_hosts(service):
                    logger.info("Deleting service: {} on host {}".format(service, host))
                    command = '/usr/local/bin/docker-compose --project-name platform -f ' + \
                              self.get_deployment_file_path(host) + ' rm -fs ' + service
                    try:
                        run_potential_ssh_command(host, command, config_parser)
                    except Exception as e:
                        logger.debug("Failed to delete service {} on host {}: {}".format(service, host, e))
                        continue
        else:
            for host in reversed(self.get_hosts()):
                deployment_file_path = self.get_deployment_file_path(host)
                if os.path.exists(deployment_file_path):
                    logger.info("Deleting platform services on host {}".format(host))
                    command = '/usr/local/bin/docker-compose --project-name platform -f ' \
                              + self.get_deployment_file_path(host) + ' rm -fsv'
                    run_potential_ssh_command(host, command, config_parser)

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.validate_input_services(cmd_args)
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'platform.yaml')
        services = self.get_involved_services(template_path) if cmd_args.services is None \
            else cmd_args.services

        if cmd_args.services is None:
            for host in self.get_hosts():
                deployment_file_path = self.get_deployment_file_path(host)
                if os.path.exists(deployment_file_path):
                    logger.info("Starting platform services on host {}".format(host))
                    command = '/usr/local/bin/docker-compose --project-name platform -f ' + self.get_deployment_file_path(host) \
                              + ' start '
                    run_potential_ssh_command(host, command, config_parser)
        else:
            for index, service in enumerate(services):
                hosts = self.get_involved_hosts(service)
                for host in hosts:
                    logger.info("Starting service: {} on host {}".format(service, host))
                    command = '/usr/local/bin/docker-compose --project-name platform -f ' + self.get_deployment_file_path(host) \
                              + ' start ' + service
                    run_potential_ssh_command(host, command, config_parser)

    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.validate_input_services(cmd_args)
        if hasattr(cmd_args, 'services') and cmd_args.services is not None:
            for service in cmd_args.services:
                for host in self.get_involved_hosts(service):
                    logger.info("Stopping service: {} on host {}".format(service, host))
                    command = '/usr/local/bin/docker-compose --project-name platform -f ' + self.get_deployment_file_path(host) + \
                              ' stop ' + service
                    run_potential_ssh_command(host, command, config_parser)
        else:
            for host in reversed(self.get_hosts()):
                deployment_file_path = self.get_deployment_file_path(host)
                if os.path.exists(deployment_file_path):
                    logger.info("Stopping platform services on host {}".format(host))
                    command = '/usr/local/bin/docker-compose --project-name platform -f ' + self.get_deployment_file_path(host) + \
                              ' stop'
                    run_potential_ssh_command(host, command, config_parser)

    def _find_component_container_on_host(self, docker_client: cliDockerSession, service,
                                          all_states: bool = False) -> List:
        labels = {'com.docker.compose.service': service}
        component_filter_labels = []
        for k, v in labels.items():
            component_filter_labels.append(str(k + '=' + v))
        return docker_client.client.containers(all=all_states,
                                               filters={'label': component_filter_labels})

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'platform.yaml')
        services = self.get_involved_services(template_path)
        statuses = []
        for service in services:
            for host in self.get_involved_hosts(service):
                deployment_file_path = self.get_deployment_file_path(host)
                if not os.path.exists(deployment_file_path):
                    logger.debug("No platform services deployed.")
                    return statuses
                component_status = dict()
                statuses.append(component_status)
                component_status['component_name'] = service
                component_status['host'] = host
                try:
                    with Component.new_docker_client(host, config_parser) as docker_client:
                        containers = self._find_component_container_on_host(docker_client,
                                                                            service,
                                                                            all_states=True)
                        if len(containers) == 0:
                            logger.debug(
                                'No container found for ' + service + ' on host ' + host)
                            component_status['containers'] = []
                        else:
                            component_status['containers'] = containers
                except Exception:
                    logger.debug('Failed to get status of ' + service + ' on host ' + host,
                                 exc_info=1)
                    # set the status as error
                    component_status['status'] = {'error': True, 'message': 'Unknown'}
        return statuses

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        namespace = self.get_namespace(config_parser)
        statuses = []
        service_host_map = {}
        
        status_command = f'kubectl get pods -n {namespace} -l app_category=rdaf-platform -o json'
        ret, stdout, stderr = execute_command(status_command)
        
        if ret != 0:
            cli_err_exit(f"Failed to get status of platform, due to: {str(stderr)}.")
        
        result = json.loads(str(stdout))
        items = result['items']
        hosts = self.get_hosts()
        if not hosts:
            cli_err_exit("No hosts defined for platform.")
        for host in hosts:
            deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm',
                                               self.get_k8s_component_name())
            if os.path.exists(deployment_path):
                if not items:
                    logger.debug("No Pods found for platform...")
                    # Handle no items found
                    for service in self.get_k8s_involved_services():
                        if service == 'rda-portal':
                            statuses.append({'component_name': 'rda-portal-backend', 'host': host, 'containers': []})
                            statuses.append({'component_name': 'rda-portal-frontend', 'host': host, 'containers': []})
                        else:
                            statuses.append({'component_name': service, 'host': host, 'containers': []})
                    return statuses
            
                # Process items if not empty
                for item in items:
                    pod = dict()
                    pod['component_name'] = item['metadata']['labels']['app_component']
                    pod['host'] = item['status'].get('hostIP', 'Unknown')
                    pod['containers'] = []
                    
                    if 'containerStatuses' in item['status']:
                        for entry in item['status']['containerStatuses']:
                            if 'containerID' in entry.keys():
                                container = dict()
                                container['Id'] = entry['containerID']
                                container['Image'] = entry['image']
                                container['name'] = entry['name']
                                for key in entry['state'].keys():
                                    container['State'] = key
                                    if key == 'running':
                                        container['Status'] = self.get_container_age(entry['state'][key]['startedAt'])
                                    else:
                                        container['Status'] = key
                                    break
                                pod['containers'].append(container)
                    service_host_map.setdefault(pod['component_name'], {})
                    service_host_map[pod['component_name']][pod['host']] = pod
            
                for service in self.get_k8s_involved_services():
                    entries = service_host_map.get(service, {})
                    if entries:
                        for host in entries.keys():
                            if service == 'rda-portal':
                                backend_container = None
                                frontend_container = None
                                for container in entries[host]['containers']:
                                    if container['name'] == 'rda-portal-backend':
                                        backend_container = container
                                    elif container['name'] == 'rda-portal-frontend':
                                        frontend_container = container
                                backend_entry = {'component_name': 'rda-portal-backend', 'host': host, 'containers': []}
                                frontend_entry = {'component_name': 'rda-portal-frontend', 'host': host, 'containers': []}
                                if backend_container:
                                    backend_entry['containers'].append(backend_container)
                                if frontend_container:
                                    frontend_entry['containers'].append(frontend_container)
                                statuses.append(backend_entry)
                                statuses.append(frontend_entry)
                            else:
                                statuses.append(entries[host])
                    else:
                        if service == 'rda-portal':
                            statuses.append({'component_name': 'rda-portal-backend', 'host': 'host', 'containers': []})
                            statuses.append({'component_name': 'rda-portal-frontend', 'host': 'host', 'containers': []})
                        else:
                            statuses.append({'component_name': service, 'host': host, 'containers': []})
        
                return statuses

    def get_k8s_platform_service_args(self, config_parser, service, cmd_args):
        namespace = self.get_namespace(config_parser)
        if service == 'rda-registry':
            if self.get_deployment_type(config_parser) == "aws":
                portal_host = 'rda-portal-lb.{}.svc.cluster.local'.format(namespace)
            else:
                portal_host = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME) \
                    .get_internal_access_host()
            args = '--set tag={} --set rda_registry.env.PORTAL_HOST={} ' \
                .format(cmd_args.tag, portal_host)
        elif service == 'rda-api-server':
            args = '--set tag={} --set rda_api_server.env.RDA_JWT_SECRET_KEY={}' \
                .format(cmd_args.tag, self.configs[self._api_key])
        else:
            args = '--set tag={} '.format(cmd_args.tag)

        # adding the timestamp for daily tags which will work for upgrade scenarios to pull the
        # latest image, with a change to the chart templates.
        if cmd_args.tag == 'daily':
            timestamp = datetime.now().strftime("%Y-%m-%d-%H:%M:%S")
            if service == 'rda-portal':
                args += ' --set {service}.extraEnvs[0].name=timestamp,{service}.extraEnvs[0].value={value}' \
                    .format(service='rda_portal.portal_frontend', value=timestamp)
                args += ' --set {service}.extraEnvs[0].name=timestamp,{service}.extraEnvs[0].value={value}' \
                    .format(service='rda_portal.portal_backend', value=timestamp)
            else:
                args += ' --set {service}.extraEnvs[0].name=timestamp,{service}.extraEnvs[0].value={value}' \
                    .format(service=self.get_k8s_services_map()[service], value=timestamp)
        return args

    def k8s_pull_images(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) != 'k8s':
            return
        services = self.get_k8s_involved_services() if cmd_args.services is None \
            else self.filter_k8s_involved_services(cmd_args.services)
        docker_repo = self._get_docker_repo()['DOCKER_REPO']
        for host in self.get_hosts():
            logger.info(f'Pulling {self.component_name} images on host {host}')
            for service in services:
                if service == 'rda-portal':
                    for image in self.get_k8s_involved_images()[service]:
                        logger.info(f'Pulling {image} for service {service} on host {host}')
                        docker_pull_command = f'docker pull {docker_repo}/{image}:{cmd_args.tag}'
                        run_potential_ssh_command(host, docker_pull_command, config_parser)
                else:
                    image = self.get_k8s_involved_images()[service]
                    logger.info(f'Pulling {image} for service {service} on host {host}')
                    docker_pull_command = f'docker pull {docker_repo}/{image}:{cmd_args.tag}'
                    run_potential_ssh_command(host, docker_pull_command, config_parser)

    def k8s_install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        self._generate_tenant_configuration(config_parser, k8s=True)
        self._generate_data_plane_policy(config_parser, k8s=True)
        self._generate_portal_configuration(config_parser, k8s=True)
        self._configure_subscription_yaml(config_parser)

        services = self.get_k8s_involved_services() if cmd_args.services is None \
            else self.filter_k8s_involved_services(cmd_args.services)
        # we initialize schema when ever portal back-end is installed
        if 'rda-portal' in services:
            self._init_k8s_portal_schema(config_parser, cmd_args.tag)
            # if self.get_deployment_type(config_parser) == "aws":
            #     self._init_efs_pod(config_parser)
        for service in services:
            logger.info("Installing service: " + service)
            chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name(), service)
            deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm',
                                           self.get_k8s_component_name(), service)
            self.copy_helm_chart(chart_template_path, deployment_path)
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
            install_command = 'helm upgrade --install --wait --create-namespace -n {} -f {} {} {} {} ' \
                .format(namespace, values_yaml, self.get_k8s_platform_service_args(config_parser, service, cmd_args),
                        service, deployment_path)
            run_command(install_command)

        if 'rda-portal' in services:
            self._create_user(config_parser, k8s=True)
            if self.get_deployment_type(config_parser) == "aws":
                ret, stdout, stderr = execute_command(f"kubectl get svc/rda-portal-lb -n {namespace} -o json")
                if ret != 0:
                    cli_err_exit("Failed to get portal load balancer service, due to: {}.".format(str(stderr)))
                result = json.loads(str(stdout))
                if 'hostname' in result['status']['loadBalancer']['ingress'][0]:
                    platform_host = result['status']['loadBalancer']['ingress'][0]['hostname']
                else:
                    platform_host = result['status']['loadBalancer']['ingress'][0]['ip']
                # self._copy_nats_keys_aws(config_parser)
            else:
                platform_host = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME).get_internal_access_host()
            ui_url = '\033[1;4m' + "https://" + platform_host + '\033[0m'
            print('\n')
            logger.info('\033[1m UI can be accessed at - \033[0m' + ui_url
                        + '\033[1m with default user - \033[0m' + '\033[1;4madmin@cfx.com\033[0m')

    def k8s_upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        services = self.get_k8s_involved_services() if cmd_args.services is None \
            else self.filter_k8s_involved_services(cmd_args.services)
        if 'rda-portal' in services:
            self._handle_k8s_portal_db_upgrades(config_parser, cmd_args.tag)
        for service in services:
            chart_template_path = os.path.join(rdaf.get_helm_charts_dir(), self.get_k8s_component_name(), service)
            deployment_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'helm',
                                           self.get_k8s_component_name(), service)
            self.copy_helm_chart(chart_template_path, deployment_path)
            values_yaml = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
            upgrade_command = 'helm upgrade --install --create-namespace -n {} -f {} {} {} {} ' \
                .format(namespace, values_yaml, self.get_k8s_platform_service_args(config_parser, service, cmd_args),
                        service, deployment_path)
            run_command(upgrade_command)

    def _generate_tenant_configuration(self, config_parser: configparser.ConfigParser, k8s=False):
        network_config = os.path.join(self.get_conf_dir(), 'network_config', 'config.json')
        if os.path.exists(network_config):
            logger.debug("tenant configuration already exists...")
            return
        peer_configs = None
        uid = uuid.uuid4().hex
        peer_cfg = os.path.join('/opt', 'rdaf', 'rdaf-peer.cfg')
        if os.path.exists(peer_cfg):
            peer_configs = configparser.ConfigParser(allow_no_value=True)
            with open(peer_cfg, 'r') as f:
                peer_configs.read_file(f)
            uid = peer_configs.get('common', 'tenant_id', fallback=uuid.uuid4().hex)

        nats = COMPONENT_REGISTRY.require(nats_comp.COMPONENT_NAME)
        nats.setup_tenant_users(uid, config_parser, k8s)
        minio_user = uid + "rdauser"
        minio_password = rdafutils.gen_password_with_uuid(uid)
        minio = COMPONENT_REGISTRY.require(minio_comp.COMPONENT_NAME)
        haproxy_comp = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
        namespace = self.get_namespace(config_parser)
        if k8s:
            minio_host = f'rda-minio.{namespace}.svc.cluster.local:9000'
            minio_secure = 'false'
            minio.setup_tenant_user(uid, minio_user, minio_password, config_parser, port='30443')
        else:
            minio_host = haproxy_comp.get_internal_access_host() + ':9443'
            minio_secure = 'true'
            minio.setup_tenant_user(uid, minio_user, minio_password, config_parser)

        mariadb = COMPONENT_REGISTRY.require(mariadb_comp.COMPONENT_NAME)
        db_user, db_password = mariadb.get_db_credentials()
        if k8s:
            mariadb_host = f'rda-mariadb-mariadb-galera.{namespace}.svc.cluster.local'
            mariadb_port = 3306
        else:
            mariadb_host = haproxy_comp.get_internal_access_host()
            mariadb_port = 3307

        es = COMPONENT_REGISTRY.require(opensearch_comp.COMPONENT_NAME)
        es_port = '9200'
        es_hosts = es.get_hosts()
        if k8s:
            if self.get_deployment_type(config_parser) != 'aws':
                es_port = self.get_service_node_port('opensearch-cluster-master', config_parser)
            es_hosts = [
                    'opensearch-cluster-master-{}.opensearch-cluster-master-headless.{}.svc.cluster.local'
                    .format(str(i), namespace)
                    for i in range(len(es.get_hosts()))]
        es_user, es_password = es.setup_tenant(config_parser, uid, es_port)

        if k8s:
            nats_servers = f'nats://rda-nats.{namespace}.svc.cluster.local:4222'
        else:
            nats_servers = nats.get_servers_url('4222')
        token = nats.get_tenant_token(uid)
        cert = nats.get_cert()

        kafka = COMPONENT_REGISTRY.require(kafka_comp.COMPONENT_NAME)
        if k8s:
            kafka_ext_user, kafka_ext_password = kafka.setup_k8s_tenant(config_parser, uid)
            kafka_host = ','.join([
                                      'rda-kafka-controller-{}.rda-kafka-controller-headless.{}.svc.cluster.local:9093'.format(
                                          str(i), namespace)
                                      for i in range(len(kafka.get_hosts()))])
        else:
            kafka_ext_user, kafka_ext_password = kafka.setup_tenant(uid, config_parser)
            kafka_host = ','.join([i + ':9093' for i in kafka.get_hosts()])

        graphdb = COMPONENT_REGISTRY.require(graphdb_comp.COMPONENT_NAME)
        if k8s:
            graphdb_hosts = f'rda-arangodb.{namespace}.svc.cluster.local:8529'
        else:
            graphdb_hosts = ','.join([i + ':8529' for i in graphdb.get_hosts()])
        output = """{
                    "tenant_id":"%s",
                    "nats": {
                        "servers": "%s",
                        "token": "%s",
                        "server_cert_data": "%s"
                    },
                    "minio": {
                        "host": "%s",
                        "secure": %s,
                        "access_key": "%s",
                        "secret_key": "%s",
                        "bucket_name": "tenants.%s"
                    },
                    "service_config": {
                        "platform_mysql": {
                            "host": "%s",
                            "port": "%s",
                            "username": "%s",
                            "password": "%s"
                        },
                        "dimensions_services": {
                            "host": "%s",
                            "port": "%s",
                            "username": "%s",
                            "password": "%s"
                        }
                    },
                    "kafka": {
                        "bootstrap.servers": "%s",
                        "security.protocol": "SASL_SSL",
                        "sasl.mechanism": "SCRAM-SHA-256",
                        "ssl.ca.pem": "%s",
                        "sasl.username" : "%s.user",
                        "sasl.password": "%s.secret"
                    },
                    "kafka-external": {
                        "bootstrap.servers": "%s",
                        "security.protocol": "SASL_SSL",
                        "sasl.mechanism": "SCRAM-SHA-256",
                        "ssl.ca.pem": "%s",
                        "sasl.username" : "%s",
                        "sasl.password": "%s"
                    },
                    "es": {
                        "hosts": %s,
                        "user": "%s",
                        "password": "%s",
                        "port": 9200,
                        "scheme": "https",
                        "ssl_verify": false
                    },
                    "graphdb": {
                        "hosts": "%s",
                        "username": "%s",
                        "password": "%s"
                    }}""" % (uid, nats_servers, token, cert, minio_host, minio_secure, minio_user,
                             minio_password, uid, mariadb_host, mariadb_port, db_user, db_password,
                             mariadb_host, mariadb_port, db_user, db_password, kafka_host, cert, uid, uid,
                             kafka_host, cert, kafka_ext_user, kafka_ext_password, json.dumps(es_hosts), es_user,
                             es_password, graphdb_hosts, graphdb.get_user(), graphdb.get_password())
        os.makedirs(os.path.dirname(network_config), exist_ok=True)
        output = json.loads(output)
        self.encrypt_rda_config(output)
        output = json.dumps(output, indent=4)
        with open(network_config, "w") as f:
            f.write(output)
            f.flush()

        # create external network config.json
        if k8s:
            if self.get_deployment_type(config_parser) in ['k8s', 'ctr']:
                nats_port = self.get_service_node_port('rdaf-nats-public', config_parser)
                nats_servers = nats.get_servers_url(nats_port)
                kafka_servers = kafka.get_k8s_public_endpoint(config_parser)
                os_hosts = json.dumps([f"{haproxy_comp.get_internal_access_host()}"])
                output = """{
                                "tenant_id":"%s",
                                "nats": {
                                    "servers": "%s",
                                    "token": "%s",
                                    "server_cert_data": "%s"
                                },
                                "minio": {
                                    "host": "%s:9443",
                                    "secure": true,
                                    "access_key": "%s",
                                    "secret_key": "%s",
                                    "bucket_name": "tenants.%s"
                                },
                                "kafka": {
                                    "bootstrap.servers": "%s",
                                    "security.protocol": "SASL_SSL",
                                    "sasl.mechanism": "SCRAM-SHA-256",
                                    "ssl.ca.pem": "%s",
                                    "sasl.username" : "%s.user",
                                    "sasl.password": "%s.secret"
                                },
                                "es": {
                                    "hosts": %s,
                                    "user": "%s",
                                    "password": "%s",
                                    "port": 9200,
                                    "scheme": "https",
                                    "ssl_verify": false
                                }
                            }""" % (uid, nats_servers, token, cert, haproxy_comp.get_internal_access_host(),
                                    minio_user, minio_password, uid, kafka_servers, cert, uid, uid, os_hosts,
                                    es_user, es_password)
            else:
                ret, stdout, stderr = execute_command(f"kubectl get svc/rda-nats-lb -n {namespace} -o json")
                if ret != 0:
                    cli_err_exit("Failed to get status of nats service, due to: {}.".format(str(stderr)))
                result = json.loads(str(stdout))
                if 'hostname' in result['status']['loadBalancer']['ingress'][0]:
                    host = result['status']['loadBalancer']['ingress'][0]['hostname']
                else:
                    host = result['status']['loadBalancer']['ingress'][0]['ip']
                nats_servers = 'nats://{}:4222'.format(host)
                ret, stdout, stderr = execute_command(f"kubectl get svc/rda-minio-lb -n {namespace} -o json")
                if ret != 0:
                    cli_err_exit("Failed to get status minio service, due to: {}.".format(str(stderr)))
                result = json.loads(str(stdout))
                if 'hostname' in result['status']['loadBalancer']['ingress'][0]:
                    minio_host = result['status']['loadBalancer']['ingress'][0]['hostname']
                else:
                    minio_host = result['status']['loadBalancer']['ingress'][0]['ip']

                kafka_servers = kafka.get_k8s_public_endpoint(config_parser)
                opensearch_servers, os_port = es.get_aws_opensearch_endpoint(config_parser)
                os_hosts = json.dumps([f"{opensearch_servers}"])
                output = """{
                                "tenant_id":"%s",
                                "nats": {
                                    "servers": "%s",
                                    "token": "%s",
                                    "server_cert_data": "%s"
                                },
                                "minio": {
                                    "host": "%s:443",
                                    "secure": true,
                                    "access_key": "%s",
                                    "secret_key": "%s",
                                    "bucket_name": "tenants.%s"
                                },
                                "kafka": {
                                    "bootstrap.servers": "%s",
                                    "security.protocol": "SASL_SSL",
                                    "sasl.mechanism": "SCRAM-SHA-256",
                                    "ssl.ca.pem": "%s",
                                    "sasl.username" : "%s.user",
                                    "sasl.password": "%s.secret"
                                },
                                "es": {
                                    "hosts": %s,
                                    "user": "%s",
                                    "password": "%s",
                                    "port": %s,
                                    "scheme": "https",
                                    "ssl_verify": false
                                }
                            }""" % (uid, nats_servers, token, cert, minio_host, minio_user,
                                    minio_password, uid, kafka_servers, cert, uid, uid, os_hosts, es_user,
                                    es_password, os_port)
            external_config = os.path.join(self.get_conf_dir(), 'network_config', 'external-config.json')
            os.makedirs(os.path.dirname(external_config), exist_ok=True)
            output = json.loads(output)
            self.encrypt_rda_config(output)
            output = json.dumps(output, indent=4)
            with open(external_config, "w") as f:
                f.write(output)
                f.flush()
        if k8s:
            config = f'kubectl create configmap rda-network-config -n {namespace} --from-file=config.json={network_config}'
            run_command(config)
        else:
            self._copy_configs_known_hosts(config_parser)
        if peer_configs and peer_configs.has_option('common', 'api_key'):
            self.configs[self._api_key] = peer_configs.get('common', 'api_key')
        else:
           self.configs[self._api_key] = self._generate_api_secret_key(uid)
        self.configs[self._tenant_id] = uid
        self._mark_configured(self.configs, config_parser)
        self.write_configs(config_parser)
        logger.info('Tenant configuration generated.')

    def _copy_configs_known_hosts(self, config_parser):
        hosts = set(self.get_hosts() + self.configs[self._service_hosts] +
                    self.get_worker_hosts(config_parser))
        network_config = os.path.join(self.get_conf_dir(), 'network_config', 'config.json')
        command = 'mkdir -p ' + os.path.dirname(network_config)
        for host in hosts:
            if Component.is_local_host(host):
                continue
            logger.info('Creating directory ' + os.path.dirname(network_config))
            run_potential_ssh_command(host, command, config_parser)
            do_potential_scp(host, network_config, network_config)

    def _generate_portal_configuration(self, config_parser, k8s=False):
        pconfig_yek = os.path.join(self.get_conf_dir(), 'portal', 'pconfig.yek')
        portal_settings = os.path.join(self.get_conf_dir(), 'portal', 'settings.json')
        config_host = socket.gethostname() if k8s else self.get_hosts()[0]
        if check_potential_remote_file_exists(config_host, portal_settings):
            logger.debug("Portal settings already exists...")
            return

        logger.info("Generating portal settings...")
        template_root = get_templates_dir_root()
        portal_template = os.path.join(template_root, 'portal_settings.json')
        with open(portal_template, 'r') as f:
            template_content = f.read()

        jwt_token, public, private = self._generate_portal_keys()
        # secret key can be used for flask app secret key
        secret_key = secrets.token_hex(32)

        replacement_values = dict()
        replacement_values['SECRET_KEY'] = secret_key
        replacement_values['JWT_PUBLIC_KEY'] = public.replace('\n', '\\n')
        replacement_values['JWT_PRIVATE_KEY'] = private.replace('\n', '\\n')

        mariadb = COMPONENT_REGISTRY.require(mariadb_comp.COMPONENT_NAME)
        db_user, db_password = mariadb.get_db_credentials()
        namespace = self.get_namespace(config_parser)
        if k8s:
            replacement_values['DATABASE_URI'] = \
                f'mysql+mysqldb://{db_user}:{db_password}@rda-mariadb-mariadb-galera.{namespace}.svc.cluster.local/saasportal'
        else:
            haproxy_advertised_host = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME).get_internal_access_host()
            replacement_values['DATABASE_URI'] = f'mysql+pymysql://{db_user}:{db_password}@{haproxy_advertised_host}:3307/saasportal'

        substituted_content = string.Template(template_content).substitute(replacement_values)
        # pconfig.yek
        private_yek = self._get_private_yek()
        encrypted_content = self.encrypt(substituted_content, str(private_yek).strip())
        if k8s:
            create_file(socket.gethostname(), encrypted_content.encode(encoding='UTF-8'), portal_settings)
            create_file(socket.gethostname(), private_yek.encode(encoding='UTF-8'), pconfig_yek)
        else:
            for host in self.get_hosts():
                create_file(host, encrypted_content.encode(encoding='UTF-8'), portal_settings)
                create_file(host, private_yek.encode(encoding='UTF-8'), pconfig_yek)
                logger.info('Portal settings created on host ' + host)

        self.configs[self._portal_token] = jwt_token
        self._mark_configured(self.configs, config_parser)
        self.write_configs(config_parser)

        if k8s:
            config = 'kubectl create configmap rda-portal-backend-settings -n {} ' \
                     '--from-file=settings.json={}'.format(namespace, portal_settings)
            run_command(config)
            config = 'kubectl create configmap rda-portal-backend-pconfig -n {} ' \
                     '--from-file=pconfig.yek={}'.format(namespace, pconfig_yek)
            run_command(config)

    @staticmethod
    def _generate_portal_keys():
        logger.info("Generating portal keys.")
        key = crypto.PKey()
        key.generate_key(crypto.TYPE_RSA, 2048)
        private_key = crypto.dump_privatekey(crypto.FILETYPE_PEM, key).decode("utf-8")
        public_key = crypto.dump_publickey(crypto.FILETYPE_PEM, key).decode("utf-8")

        iat = int(time.time())
        payload = {'fresh': False, 'iat': iat,
                   'jti': 'd59a7954-cc8b-409a-b91c-f18952ecddee', 'type': 'access',
                   'onprem': True, 'sub': None, 'nbf': 1636361575,
                   'post_data': '93zAvw2bncoP1DcR3RSU4Q8z0tiyHUUGafS7Bx1TzfE'}

        # jwt_encoded_token will be used by rdac api server to access the apis from rdac api server
        jwt_encoded_token = jwt.encode(payload, private_key, algorithm='RS256')
        return jwt_encoded_token, public_key, private_key

    def _init_portal_schema(self, config_parser, tag):
        logger.info("Checking for portal schema.")
        mariadb = COMPONENT_REGISTRY.require(mariadb_comp.COMPONENT_NAME)
        db_user = mariadb.get_user()
        db_password = mariadb.get_escaped_password()
        haproxy_comp = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
        haproxy_advertised_host = haproxy_comp.get_internal_access_host()

        # TODO hard-coded db name `saasportal`
        schema_cmd = 'sh -c "mysql -u{} -p{} -h {} -P {} -e \'{}\'"'.format(
            db_user, db_password, haproxy_advertised_host, 3307, 'use saasportal')
        ret_code, stdout, stderr = run_command_exitcode(schema_cmd, socket.gethostname(), config_parser)
        if ret_code == 0:
            logger.info("portal schema is already initialized.")
            return

        logger.info("Initializing portal schema...")
        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            compose_file = os.path.join(tmp, 'docker-compose.yaml')
            image_name = self._get_docker_repo()['DOCKER_REPO'] + '/cfx-onprem-portal-dbinit:' + tag
            content = {"services": {'dbinit': {
                "image": image_name, 'network_mode': 'host'}}}

            with open(compose_file, 'w+') as f:
                yaml.safe_dump(content, f, default_flow_style=False, explicit_start=True,
                               allow_unicode=True, encoding='utf-8', sort_keys=False)
            # using this as we need to have docker-compose in both k8s and onprem
            host = haproxy_comp.get_hosts()[0]
            if not Component.is_local_host(host):
                do_potential_scp(host, tmp, tmp)

            start_cmd = '/usr/local/bin/docker-compose -f {file} pull && ' \
                        '/usr/local/bin/docker-compose --project-name platform -f {file} up -d ' \
                .format(file=compose_file)
            run_potential_ssh_command(host, start_cmd, config_parser)

            logger.info("Creating portal schema...")
            with Component.new_docker_client(host, config_parser, timeout=60) as docker_client:
                containers = docker_client.client.containers(all=True,
                                                             filters={'label': ['com.docker.compose.service=dbinit']})
                if len(containers) == 0:
                    cli_err_exit('No container found for dbinit ')
                container = containers[0]
                schema_cmd = 'sh -c "mysql -u{} -p{} -h {} -P {} < {}"' \
                    .format(db_user, db_password, haproxy_advertised_host, 3307,
                            os.path.join('/opt', 'db', 'changes', 'consolidate', 'schema.sql'))
                output, error = docker_client.docker_exec(container['Id'], schema_cmd)
                logger.debug(output)
                if error:
                    logger.error(error)
                time.sleep(5)
                data_init = 'sh -c "mysql -u{} -p{} -h {} -P {} < {}"' \
                    .format(db_user, db_password, haproxy_advertised_host, 3307,
                            os.path.join('/opt', 'db', 'changes', 'consolidate', 'data.sql'))
                output, error = docker_client.docker_exec(container['Id'], data_init)
                logger.debug(output)
                if error:
                    logger.error(error)
            stop_cmd = '/usr/local/bin/docker-compose -f ' + compose_file + ' --project-name platform rm -fsv dbinit'
            run_potential_ssh_command(host, stop_cmd, config_parser)

    @staticmethod
    def _get_private_yek():
        return str(uuid.uuid4())

    def _create_user(self, config_parser, k8s=False):
        namespace = self.get_namespace(config_parser)
        config = os.path.join(self.get_conf_dir(), 'network_config', 'external-config.json') if k8s else \
            os.path.join(self.get_conf_dir(), 'network_config', 'config.json')
        with open(config, "r") as f:
            data = json.load(f)

        time.sleep(10)
        pf_proc = None
        if k8s:
            pod = self.get_pods_names(config_parser, 'app_component=rda-portal')[0]
            pf_proc = subprocess.Popen(f'exec kubectl port-forward {pod} -n {namespace} 7780:7780',
                                       shell=True)
            time.sleep(10)
            host = '127.0.0.1'
            rdac_endpoint = f'http://rdaf-api-server.{namespace}.svc.cluster.local:8808'
        else:
            host = self.get_involved_hosts('portal-frontend')[0]
            haproxy_host = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME).get_internal_access_host()
            rdac_endpoint = f'http://{haproxy_host}:8808'

        token = self.configs[self._portal_token]
        secret_key = self.decrypt(data['tenant_id'], self.configs[self._api_key])

        wait_time = datetime.now() + timedelta(0, 60)
        requests.packages.urllib3.disable_warnings(
            requests.packages.urllib3.exceptions.InsecureRequestWarning)
        while datetime.now() < wait_time:
            response = requests.post(url='http://' + host + ':7780/api/portal/status', verify=False)
            try:
                response.raise_for_status()
                break
            except requests.exceptions.HTTPError:
                logger.info("Waiting for portal to come up...")
                time.sleep(10)
                continue
        time.sleep(5)
        url = 'http://{}:7780/api/portal/onboard/checkOnpremUser?jwt={}'.format(host, token)
        payload = json.dumps({"id": "admin@cfx.com"})
        headers = {'Content-Type': 'application/json', 'Tenant-Secret-Key': secret_key}
        response = requests.post(url, headers=headers, data=payload, verify=False)
        if response.status_code == 500:
            logger.info("admin user already exists...")
            self._update_network_config_on_secondary_deployments(data, host, token, secret_key)
            if pf_proc:
                pf_proc.kill()
            return
        response.raise_for_status()
        time.sleep(20)
        logger.info("Creating default user...")
        workspace = 'http://{}:7780/api/portal/onboard/createOnpremWorkspace?jwt={}'.format(host, token)
        payload = {"firstname": "RDAF", "lastname": "Admin", "id": "admin@cfx.com",
                   "password": "admin1234", "company": self.get_admin_organization(),
                   "rdac_end_point": rdac_endpoint}
        headers = {'Content-Type': 'application/json', 'Tenant-Secret-Key': secret_key}
        retry = 0
        while retry < 5:
            if retry > 0:
                time.sleep(30)
                logger.info("Retrying default user creation..." + str(retry))
            retry += 1
            try:
                response = requests.post(url, headers=headers, data=json.dumps({"id": "admin@cfx.com"}), verify=False)
                if response.status_code == 500:
                    break
                response.raise_for_status()
                response = requests.post(workspace, headers=headers, data=json.dumps(payload),
                                         verify=False)
                if response.status_code == 200:
                    break
                continue
            except Exception:
                continue
        if retry > 4:
            response.raise_for_status()

        output = response.json()
        workspace_id = output['serviceResult']['workspaceid']
        logger.info("Created workspace:" + workspace_id)

        add_config = 'http://{}:7780/api/portal/rdac/addOnpremNetworkConfig?jwt={}'.format(host, token)
        payload = {'tenantid': data['tenant_id'], "workspaceid": workspace_id, 'tenant_secret_key': secret_key,
                   'networkconfig': {'tenant_id': data['tenant_id'], 'nats': data['nats'],
                                     'minio': data['minio'], 'kafka': data['kafka'], 'es': data['es']}}
        response = requests.post(add_config, headers=headers, data=json.dumps(payload), verify=False)
        response.raise_for_status()
        if pf_proc:
            pf_proc.kill()

    def reset_admin_user(self, config_parser: configparser.ConfigParser, k8s):
        namespace = self.get_namespace(config_parser)
        config = os.path.join(self.get_conf_dir(), 'network_config', 'config.json')
        with open(config, "r") as f:
            data = json.load(f)
        secret_key = self.decrypt(data['tenant_id'], self.configs[self._api_key])
        requests.packages.urllib3.disable_warnings(
            requests.packages.urllib3.exceptions.InsecureRequestWarning)

        pf_proc = None
        if k8s:
            pod = self.get_pods_names(config_parser, 'app_component=rda-portal')[0]
            pf_proc = subprocess.Popen(f'exec kubectl port-forward {pod} -n {namespace} 7780:7780', shell=True)
            time.sleep(10)
            host = '127.0.0.1'
        else:
            host = self.get_involved_hosts('portal-frontend')[0]

        url = f'http://{host}:7780/api/portal/rdac/adminPasswordReset'
        payload = json.dumps({"userid": "admin@cfx.com",
                              "password": "admin1234", "is_from_cli": "true"})
        headers = {'Content-Type': 'application/json', 'Tenant-Secret-Key': secret_key}
        response = requests.post(url, headers=headers, data=payload, verify=False)
        response.raise_for_status()
        logger.info("admin user password reset successful.")
        if pf_proc:
            pf_proc.kill()

    def backup_conf(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_dir_root: os.path):
        return

    def _generate_api_secret_key(self, tenant_id: str):
        token = secrets.token_hex(32)
        secret_key = self.encrypt(str(token), tenant_id)
        return secret_key

    def healthcheck(self, component_name, host, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        try:
            with requestsutil.new_session():
                requests.packages.urllib3.disable_warnings(requests.packages.urllib3.exceptions.InsecureRequestWarning)
                status_url = "http://{host}:7780/api/portal/status".format(host=host)
                response = requests.get(status_url, verify=False)
                response.raise_for_status()
        except Exception:
            return [component_name, "Service Status", termcolor.colored("Failed", color='red'), "service unavailable"]
        return [component_name, "Service Status", "OK", "N/A"]

    def _handle_portal_db_upgrades(self, tag, config_parser):
        logger.info("Checking for portal upgrade schema.")
        mariadb = COMPONENT_REGISTRY.require(mariadb_comp.COMPONENT_NAME)
        db_user = mariadb.get_user()
        db_password = mariadb.get_escaped_password()
        haproxy_advertised_host = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME) \
            .get_internal_access_host()

        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            compose_file = os.path.join(tmp, 'docker-compose.yaml')
            image_name = self._get_docker_repo()['DOCKER_REPO'] + '/cfx-onprem-portal-dbinit:' + tag
            content = {"services": {'dbinit': {
                "image": image_name, 'network_mode': 'host'}}}

            with open(compose_file, 'w+') as f:
                yaml.safe_dump(content, f, default_flow_style=False, explicit_start=True,
                               allow_unicode=True, encoding='utf-8', sort_keys=False)
            start_cmd = '/usr/local/bin/docker-compose -f {file} pull && ' \
                        '/usr/local/bin/docker-compose --project-name platform -f {file} up -d ' \
                .format(file=compose_file)
            run_command(start_cmd)

            with Component.new_docker_client(socket.gethostname(), config_parser, timeout=60) as docker_client:
                containers = docker_client.client.containers(all=True,
                                                             filters={'label': ['com.docker.compose.service=dbinit']})
                if len(containers) == 0:
                    cli_err_exit('No container found for dbinit ')
                container = containers[0]
                upgrade_script_path = os.path.join('/opt', 'db', 'changes', 'upgrades', 'upgrades.sql')
                file_exits = 'sh -c "test -f {} && echo \'pass\'"'.format(upgrade_script_path)
                output, error = docker_client.docker_exec(container['Id'], file_exits)
                if output.decode().strip() == 'pass':
                    logger.debug("upgrade scripts found.")
                    data_init = 'sh -c "mysql -u{} -p{} -h {} -P {} < {}"' \
                        .format(db_user, db_password, haproxy_advertised_host, 3307, upgrade_script_path)
                    output, error = docker_client.docker_exec(container['Id'], data_init)
                    logger.debug(output)
                    if error:
                        logger.error(error)

            stop_cmd = '/usr/local/bin/docker-compose -f ' + compose_file + ' --project-name platform rm -fsv dbinit'
            run_command(stop_cmd)

    def _check_and_create_secret_key(self, config_parser):
        if config_parser.has_option('common', 'api_key'):
            return

        config = os.path.join(self.get_conf_dir(), 'network_config', 'config.json')
        with open(config, "r") as f:
            data = json.load(f)
        api_key = self._generate_api_secret_key(data['tenant_id'])

        self.configs[self._api_key] = api_key
        self._mark_configured(self.configs, config_parser)
        self.write_configs(config_parser)

        logger.info("Created tenant secret key successfully.")

    def get_k8s_involved_services(self):
        services = ['rda-api-server', 'rda-registry', 'rda-identity', 'rda-fsm', 'rda-asm', 'rda-chat-helper',
                    'rda-access-manager', 'rda-resource-manager', 'rda-scheduler',
                    'rda-collector', 'rda-user-preferences', 'rda-portal']
        return self.filter_k8s_involved_services(services)

    def _configure_subscription_yaml(self, config_parser):
        namespace = self.get_namespace(config_parser)
        subscription_yaml = os.path.join(self.get_conf_dir(), 'subscription', 'subscription.yml')
        runtime = os.path.join('/opt', 'rdaf', 'config','runtime')
        if os.path.exists(subscription_yaml):
            logger.debug("Subscription configuration exists...")
            return

        os.makedirs(os.path.dirname(subscription_yaml), exist_ok=True)
        run_command(f'mkdir -p {runtime}')
        with open(subscription_yaml, "w") as f:
            pass

        config = 'kubectl create configmap registry-subscription-config -n {} ' \
                 '--from-file=subscription.yml={}'.format(namespace, subscription_yaml)
        run_command(config)
        runtime_cmd = ('kubectl create configmap rdaf-runtime-config '
                      '--from-file=/opt/rdaf/config/runtime '
                      f'--namespace={namespace} '
                      '--dry-run=client -o yaml | kubectl apply -f -')
        run_command(runtime_cmd)
        logger.info("Updated tenant secret key successfully.")

    @staticmethod
    def get_k8s_services_map():
        return {'rda-api-server': 'rda_api_server', 'rda-registry': 'rda_registry',
                'rda-identity': 'rda_identity', 'rda-fsm': 'rda_fsm',  'rda-asm': 'rda_asm', 'rda-chat-helper': 'rda_chat_helper',
                'rda-access-manager': 'rda_access_manager', 'rda-resource-manager': 'rda_resource_manager',
                'rda-scheduler': 'rda_scheduler', 'rda-collector': 'rda_collector',
                'rda-user-preferences': 'rda_user_preferences', 'rda-portal': 'rda_portal'}

    def _init_k8s_portal_schema(self, config_parser: configparser.ConfigParser, tag):
        namespace = self.get_namespace(config_parser)
        db_init_deployment = os.path.join(get_templates_dir_root(), 'portal-dbinit.yaml')
        with open(db_init_deployment, 'r') as f:
            template_content = f.read()
        replacements = self._get_docker_repo()
        replacements['TAG'] = tag
        replacements['NAMESPACE'] = namespace
        content = string.Template(template_content).substitute(replacements)

        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            deployment_file = os.path.join(tmp, 'portal-dbinit.yaml')
            with open(deployment_file, 'w+') as f:
                f.write(content)
                f.flush()

            run_command('kubectl apply -f ' + deployment_file)
            logger.info("Waiting for portal dbinit pod to be up and running...")
            time.sleep(5)
            pod_status_command = 'kubectl wait --for=condition=Ready pod --timeout=600s -n {} ' \
                                 '-l app_component=portal-dbinit'.format(namespace)
            ret, stdout, stderr = execute_command(pod_status_command)
            if ret != 0:
                cli_err_exit("Failed to get status of portal dbinit pod, due to: {}.".format(str(stderr)))

            dbinit_pod = self.get_pods_names(config_parser, 'app_component=portal-dbinit')[0]
            mariadb = COMPONENT_REGISTRY.require(mariadb_comp.COMPONENT_NAME)
            db_user = mariadb.get_user()
            db_password = mariadb.get_escaped_password()
            mariadb_endpoint = 'rda-mariadb-mariadb-galera.{}.svc.cluster.local'.format(namespace)
            schema_check = "kubectl -n {} exec {} -- mysql -h {} -u {} -p{} -e 'use saasportal'" \
                .format(namespace, dbinit_pod, mariadb_endpoint, db_user, db_password)
            ret, stdout, stderr = execute_command(schema_check)
            if ret == 0:
                logger.info("portal schema is already initialized.")
                # deleting the pod
                run_command('kubectl delete -f ' + deployment_file)
                return
            logger.info("Initializing portal schema...")
            schema_cmd = 'kubectl -n {} exec {} -- bash -c "mysql -h {} -u{} -p{} < {}"' \
                .format(namespace, dbinit_pod, mariadb_endpoint, db_user, db_password,
                        os.path.join('/opt', 'db', 'changes', 'consolidate', 'schema.sql'))
            run_command(schema_cmd)
            time.sleep(5)
            data_init = 'kubectl -n {} exec {} -- bash -c "mysql -h {} -u{} -p{} < {}"' \
                .format(namespace, dbinit_pod, mariadb_endpoint, db_user, db_password,
                        os.path.join('/opt', 'db', 'changes', 'consolidate', 'data.sql'))
            run_command(data_init)
            # deleting the pod
            run_command('kubectl delete -f ' + deployment_file)

    def _init_efs_pod(self, config_parser):
        # initializing pod to mount EFS
        template_path = os.path.join(get_templates_dir_root(), 'k8s-aws', "efs-pod.yaml")
        with open(template_path, 'r') as f:
            template_content = f.read()
        namespace = self.get_namespace(config_parser)
        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            deployment_file = os.path.join(tmp, 'efs-pod.yaml')
            with open(deployment_file, 'w+') as f:
                f.write(template_content)
                f.flush()

            run_command(f'kubectl apply -f {deployment_file} -n {namespace}')
            logger.info("Waiting for efs pod to be up and running...")
            time.sleep(10)
            # check if the pod is ready
            pod_status_command = 'kubectl wait --for=condition=Ready pod --timeout=600s -n {} ' \
                                 '-l app_component=efs'.format(namespace)
            ret, stdout, stderr = execute_command(pod_status_command)
            if ret != 0:
                cli_err_exit("Failed to get status of efs pod, due to: {}.".format(str(stderr)))
            # Set up mount-points within EFS by accessing efs-app pod, utilized by worker and portal charts.
            cmd = 'kubectl -n {} -it exec efs-app -- ' \
                  '/bin/sh -c "mkdir -p /opt/efs/rda_packages && mkdir -p /opt/efs/nats"'.format(namespace)
            ret, stdout, stderr = execute_command(cmd)
            if ret != 0:
                cli_err_exit("Failed to create directory {}.".format(str(stderr)))
            # deleting the pod
            run_command(f'kubectl delete -f {deployment_file} -n {namespace}')

    def _handle_k8s_portal_db_upgrades(self, config_parser, tag):
        namespace = self.get_namespace(config_parser)
        db_init_deployment = os.path.join(get_templates_dir_root(), 'portal-dbinit.yaml')
        with open(db_init_deployment, 'r') as f:
            template_content = f.read()
        replacements = self._get_docker_repo()
        replacements['TAG'] = tag
        replacements['NAMESPACE'] = namespace
        content = string.Template(template_content).substitute(replacements)

        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            deployment_file = os.path.join(tmp, 'portal-dbinit.yaml')
            with open(deployment_file, 'w+') as f:
                f.write(content)
                f.flush()

            run_command('kubectl apply -f ' + deployment_file)
            # check if the pod is ready
            pod_status_command = f'kubectl get pods -n {namespace} -l app_component=portal-dbinit -o json'
            retry = 0
            while retry < 8:
                if retry > 0:
                    logger.info("Checking if portal db upgrade pod is ready... " + str(retry))
                    time.sleep(15)
                    ret, stdout, stderr = execute_command(pod_status_command)
                    if ret != 0:
                        cli_err_exit("Failed to get status of portal dbinit, due to: {}.".format(str(stderr)))
                    result = json.loads(str(stdout))
                    items = result['items']
                    if not items:
                        logger.debug("No Pods found for portal-dbinit...")
                        continue
                    item = items[0]
                    conditions = item['status']['conditions']
                    is_ready = False
                    for condition in conditions:
                        if condition['type'] == 'Ready':
                            is_ready = condition['status'] == 'True'
                            break
                    if is_ready:
                        mariadb = COMPONENT_REGISTRY.require(mariadb_comp.COMPONENT_NAME)
                        db_user = mariadb.get_user()
                        db_password = mariadb.get_escaped_password()
                        mariadb_endpoint = f'rda-mariadb-mariadb-galera.{namespace}.svc.cluster.local'
                        upgrade_script_path = os.path.join('/opt', 'db', 'changes', 'upgrades', 'upgrades.sql')
                        logger.info("checking and upgrading portal schema...")
                        upgrading_cmd = 'kubectl -n {} exec {} -- ' \
                                        'bash -c "test -f {} && mysql -h {} -u{} -p{} < {}"' \
                            .format(namespace, item['metadata']['name'], upgrade_script_path,
                                    mariadb_endpoint, db_user, db_password, upgrade_script_path)
                        run_command(upgrading_cmd)
                        # deleting the pod
                        run_command('kubectl delete -f ' + deployment_file)
                        return
                retry += 1
            if retry > 7:
                cli_err_exit("Failed to deploy portal dbinit pod.")

    def _generate_edge_tenant_configuration(self, config_parser):
        network_config = os.path.join(self.get_conf_dir(), 'network_config', 'config.json')
        if os.path.exists(network_config):
            logger.debug("tenant configuration already exists...")
            return

        uid = uuid.uuid4().hex
        nats = COMPONENT_REGISTRY.require(nats_comp.COMPONENT_NAME)
        nats.setup_tenant_users(uid, config_parser)
        minio_user = uuid.uuid4().hex + "rdauser"
        minio_password = gen_password(12)
        minio = COMPONENT_REGISTRY.require(minio_comp.COMPONENT_NAME)
        haproxy_comp = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)

        minio_host = haproxy_comp.get_internal_access_host() + ':9443'
        minio.setup_tenant_user(uid, minio_user, minio_password, config_parser)

        mariadb = COMPONENT_REGISTRY.require(mariadb_comp.COMPONENT_NAME)
        db_user, db_password = mariadb.get_db_credentials()
        mariadb_host = haproxy_comp.get_internal_access_host()
        mariadb_port = 3307

        es = COMPONENT_REGISTRY.require(opensearch_comp.COMPONENT_NAME)
        es_port = '9200'
        es_hosts = es.get_hosts()
        es.wait_for_opensearch_to_be_up()
        es_user, es_password = es.setup_tenant(config_parser, uid, es_port)

        nats_servers = nats.get_servers_url('4222')
        token = nats.get_tenant_token(uid)
        cert = nats.get_cert()

        output = """{
                    "tenant_id":"%s",
                    "nats": {
                        "servers": "%s",
                        "token": "%s",
                        "server_cert_data": "%s"
                    },
                    "minio": {
                        "host": "%s",
                        "secure": true,
                        "access_key": "%s",
                        "secret_key": "%s",
                        "bucket_name": "tenants.%s"
                    },
                    "service_config": {
                        "platform_mysql": {
                            "host": "%s",
                            "port": "%s",
                            "username": "%s",
                            "password": "%s"
                        },
                        "dimensions_services": {
                            "host": "%s",
                            "port": "%s",
                            "username": "%s",
                            "password": "%s"
                        }
                    },
                    "es": {
                        "hosts": %s,
                        "user": "%s",
                        "password": "%s",
                        "port": 9200,
                        "scheme": "https",
                        "ssl_verify": false
                    }}""" % (uid, nats_servers, token, cert, minio_host, minio_user,
                             minio_password, uid, mariadb_host, mariadb_port, db_user, db_password,
                             mariadb_host, mariadb_port, db_user, db_password, json.dumps(es_hosts),
                             es_user, es_password)
        os.makedirs(os.path.dirname(network_config), exist_ok=True)
        output = json.loads(output)
        self.encrypt_rda_config(output)
        output = json.dumps(output, indent=4)
        with open(network_config, "w") as f:
            f.write(output)
            f.flush()

        self.configs[self._api_key] = self._generate_api_secret_key(uid)
        self._mark_configured(self.configs, config_parser)
        self.write_configs(config_parser)
        logger.info('Tenant configuration generated.')

    def _generate_data_plane_policy(self, config_parser, k8s=False):
        namespace = self.get_namespace(config_parser)
        data_plane_policy = os.path.join(self.get_conf_dir(), 'network_config', 'policy.json')
        if os.path.exists(data_plane_policy):
            logger.debug("data plane policy already exists...")
            return

        config = os.path.join(self.get_conf_dir(), 'network_config', 'config.json')
        with open(config, "r") as f:
            data = json.load(f)
        tenant_id = data['tenant_id']

        es = COMPONENT_REGISTRY.require(opensearch_comp.COMPONENT_NAME)
        es_hosts = es.get_hosts()
        if k8s:
            es_hosts = [
                    'opensearch-cluster-master-{}.opensearch-cluster-master-headless.{}.svc.cluster.local'
                    .format(str(i), namespace)
                    for i in range(len(es.get_hosts()))]
        user, password = es.create_data_plane_policy(config_parser, tenant_id, k8s)

        template_root = get_templates_dir_root()
        portal_template = os.path.join(template_root, 'dataplane-policy.json')
        with open(portal_template, 'r') as f:
            template_content = f.read()

        replacement_values = dict()
        replacement_values["TENANT_ID"] = tenant_id
        replacement_values["OS_HOSTS"] = json.dumps(es_hosts)
        replacement_values["OS_USER"] = self.encrypt(user, tenant_id)
        replacement_values["OS_PASSWORD"] = self.encrypt(password, tenant_id)

        substituted_content = string.Template(template_content).substitute(replacement_values)
        os.makedirs(os.path.dirname(data_plane_policy), exist_ok=True)
        with open(data_plane_policy, "w") as f:
            f.write(substituted_content)
            f.flush()

        if k8s:
            config = 'kubectl create configmap rda-dataplane-config -n {} ' \
                     '--from-file=policy.json={}'.format(namespace, data_plane_policy)
            run_command(config)
        else:
            self._copy_dataplane_configs_known_hosts(config_parser)
        logger.info("dataplane policy is created.")

    def _copy_dataplane_configs_known_hosts(self, config_parser):
        hosts = set(self.get_hosts() + self.configs[self._service_hosts])
        data_plane_policy = os.path.join(self.get_conf_dir(), 'network_config', 'policy.json')
        command = 'mkdir -p ' + os.path.dirname(data_plane_policy)
        for host in hosts:
            if Component.is_local_host(host):
                continue
            logger.info('Creating directory ' + os.path.dirname(data_plane_policy))
            run_potential_ssh_command(host, command, config_parser)
            do_potential_scp(host, data_plane_policy, data_plane_policy)

    def _copy_nats_keys(self, config_parser, k8s=False):
        logger.debug("copying nats keys to platform hosts")
        if k8s:
            nats_hosts = [socket.gethostname()]
        else:
            nats = COMPONENT_REGISTRY.require(nats_comp.COMPONENT_NAME)
            nats_hosts = nats.get_hosts()
        data_dir = os.path.join(self.get_rdaf_install_root(), 'data', 'nats')
        for host in self.get_hosts():
            if host in nats_hosts:
                continue
            command = 'sudo chown -R ' + str(os.getuid()) + ' ' + data_dir \
                      + ' && sudo chgrp -R ' + str(os.getgid()) + ' ' + data_dir
            run_potential_ssh_command(nats_hosts[0], command, config_parser)

            with tempfile.TemporaryDirectory(prefix='nats') as tmp:
                do_potential_scp_fetch(nats_hosts[0], data_dir, tmp)
                command = 'sudo mkdir -p ' + data_dir \
                          + ' && sudo chown -R ' + str(os.getuid()) + ' ' + data_dir \
                          + ' && sudo chgrp -R ' + str(os.getgid()) + ' ' + data_dir
                run_potential_ssh_command(host, command, config_parser)
                do_potential_scp(host, tmp, data_dir, sudo=True)

    def _copy_nats_keys_aws(self, config_parser):
        namespace = self.get_namespace(config_parser)
        logger.info("copying nats keys to portal pod")
        data_dir = os.path.join(self.get_rdaf_install_root(), 'data', 'nats', 'noperator')
        pod_status_command = 'kubectl wait --for=condition=Ready pod --timeout=600s -n {} ' \
                             '-l app_component=rda-portal'.format(namespace)
        ret, stdout, stderr = execute_command(pod_status_command)
        if ret != 0:
            cli_err_exit("Failed to get status of portal pod, due to: {}.".format(str(stderr)))
        pod = self.get_pods_names(config_parser, 'app_component=rda-portal')[0]
        pod_name = pod.split('/')[1]
        copy_data_dir = 'kubectl cp {} {}:/opt/nats/ -n {} ' \
            .format(data_dir, pod_name, namespace)
        run_command(copy_data_dir)
        time.sleep(5)
        # initializing pod to copy nats key from nats-pv to portal-pv
        template_path = os.path.join(get_templates_dir_root(), 'k8s-aws', "nats-key.yaml")
        with open(template_path, 'r') as f:
            template_content = f.read()
        namespace = self.get_namespace(config_parser)
        with tempfile.TemporaryDirectory(prefix='rdaf') as tmp:
            deployment_file = os.path.join(tmp, 'nats-key.yaml')
            with open(deployment_file, 'w+') as f:
                f.write(template_content)
                f.flush()

            run_command(f'kubectl apply -f {deployment_file} -n {namespace}')
            logger.info("Waiting for nats-key pod to be up and running...")
            time.sleep(10)
            # check if the pod is ready
            pod_status_command = 'kubectl wait --for=condition=Ready pod --timeout=600s -n {} ' \
                                 '-l app_component=nats-key'.format(namespace)
            ret, stdout, stderr = execute_command(pod_status_command)
            if ret != 0:
                cli_err_exit("Failed to get status of nats-key pod, due to: {}.".format(str(stderr)))
            copy_files_cmd = 'kubectl exec -it nats-key -n {} -- /bin/sh -c ' \
                             '"cp /opt/nats-accounts/*.jwt /opt/nats/"'.format(namespace)
            ret, stdout, stderr = execute_command(copy_files_cmd)
            if ret != 0:
                cli_err_exit("Failed to copy directory {}.".format(str(stderr)))
            # deleting the pod
            run_command(f'kubectl delete -f {deployment_file} -n {namespace}')

    def handle_rolling_upgrade(self, cmd_args, config_parser):
        if cmd_args.rolling and len(self.get_hosts()) == 1:
            logger.info("Platform is deployed on single host.. continuing with normal upgrade.")
            cmd_args.rolling = False
            self.upgrade(cmd_args, config_parser)
            return

        template_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'platform.yaml')
        services = self.get_involved_services(template_path) if cmd_args.services is None else cmd_args.services
        deployed_services = []
        for host in self.get_hosts():
            deployed_services += self.get_involved_services(self.get_deployment_file_path(host))
        new_services = list(set(services) - set(deployed_services))
        # check if only portal container are involved
        if all(item == 'portal-frontend' or item == 'portal-backend' for item in services):
            logger.info("Only portal containers are involved.. continuing with normal upgrade.")
            cmd_args.rolling = False
            self.upgrade(cmd_args, config_parser)
            return

        logger.info("Gathering platform container details.")
        service_host_container = {}
        for service in services:
            # if service in ['portal-backend', 'portal-frontend', 'rda_chat_helper']:
            if service in ['portal-backend', 'portal-frontend']:
                continue
            if service in new_services:
                continue
            for host in self.get_hosts():
                try:
                    with Component.new_docker_client(host, config_parser) as docker_client:
                        containers = self._find_component_container_on_host(docker_client, service, all_states=True)
                        if len(containers) == 0:
                            logger.debug('No container found for ' + service + ' on host ' + host)
                        else:
                            entry = service_host_container.get(service, {})
                            entry[host] = containers[0]['Id'][:12]
                            service_host_container[service] = entry
                except Exception:
                    logger.warning('Failed to get status of ' + service + ' on host ' + host,
                                     exc_info=1)

            if len(service_host_container[service]) < 2:
                if not rdafutils.query_yes_no(f"Warning: Only {len(service_host_container[service])} container(s) running for service {service}. "
                                             f"Minimum of two containers recommended for rolling upgrade. Continue anyway?"):
                    cli_err_exit(f"Exiting rolling upgrade, minimum of two containers need to be running for service {service}.")

        involved_containers = [value for hc in service_host_container.values() for value in hc.values()]
        # check if rdac pods are present
        logger.info("Gathering rdac pod details.")
        return_code, stdout, stderr = execute_command('rdac pods --json --show_maintenance')
        sanitized_output = rdafutils.cleanup_stdout(stdout)
        rdac_pods = json.loads(sanitized_output)
        container_id_pods = {}
        for pod in rdac_pods:
            # hostname is docker container id
            if pod.get("hostname") in involved_containers:
                container_id_pods[pod.get("hostname")] = pod

        missing = False
        if len(container_id_pods) != len(involved_containers):
            for entry in involved_containers:
                if entry not in container_id_pods.keys():
                    missing = True
                    logger.warning(f"Container {entry} not in rdac pods.")
        if missing:
            if not rdafutils.query_yes_no("Warning: Some containers are missing in rdac pods. This may affect the rolling upgrade. Continue anyway?"):
                cli_err_exit("Exiting...few containers are missing in rdac pods.")

        for host in self.get_hosts():
            containers_on_host = [hc[host] for hc in service_host_container.values() if host in hc]
            pre_maint_pods = {}
            maint_pod = {}
            table = [["Pod ID", "Pod Type", "Version", "Age", "Hostname", "Maintenance", "Pod Status"]]
            for entry in containers_on_host:
                 # Skip containers that are missing from rdac pods
                if entry not in container_id_pods:
                    if not rdafutils.query_yes_no(f"Warning: Container {entry} is missing from rdac pods. Skip this container and continue?"):
                        cli_err_exit(f"Exiting due to missing container {entry} in rdac pods.")
                    logger.warning(f"Skipping container {entry} as it's missing from rdac pods")
                    continue
                pod = container_id_pods[entry]
                if pod.get('is_maintenance'):
                    logger.info(f'Pod {pod.get("pod_type", "")} is already in maintenance mode on {host}')
                    maint_pod[pod.get("hostname", "")] = pod
                    continue
                
                age = datetime.fromisoformat(pod.get("now")) - datetime.fromisoformat(pod.get("started_at"))
                age = str(age).rsplit(".", 1)[0]
                table.append(
                    [pod.get("pod_id", ""), pod.get("pod_type", ""), pod.get("build_tag", ""), str(age),
                     pod.get("hostname", ""), pod.get("is_maintenance"), pod.get("pod_ready")])
                pre_maint_pods[pod.get("hostname", "")] = pod
            if len(table) == 1 and not maint_pod:
                logger.info(f'{host} all pods are already running with tag {cmd_args.tag}')
                self._check_portal_upgrade(services, host, cmd_args, config_parser)
                continue
            if pre_maint_pods:
                column_headers = table[0]
                rows = table[1:]
                rdafutils.print_tabular(column_headers, rows)
                if not rdafutils.query_yes_no("Continue moving above pods to maintenance mode?"):
                    return
                pod_ids = ",".join([pod.get("pod_id") for pod in pre_maint_pods.values()])
                maint_command = f"rdac maintenance start --ids {pod_ids}"
                logger.info('Initiating Maintenance Mode...')
                execute_command(maint_command)

            # checking for pods in maintenance mode
            maint_pod = {}
            retry = 0
            while retry < 10:
                retry += 1
                return_code, stdout, stderr = execute_command('rdac pods --show_maintenance --json')
                sanitized_output = rdafutils.cleanup_stdout(stdout)
                rdac_pods = json.loads(sanitized_output)
                for pod in rdac_pods:
                    if pod.get("hostname") in involved_containers and pod.get('is_maintenance'):
                        maint_pod[pod.get("hostname")] = pod
                if len(maint_pod) >= len(pre_maint_pods):
                    break
                logger.info(f"Waiting for services to be moved to maintenance.")
                time.sleep(20)
            if not maint_pod:
                if not rdafutils.query_yes_no(f"Warning: Failed to move services on host {host} to maintenance mode. Continue anyway?"):
                    cli_err_exit(f'Failed to move services on host {host} to maintenance.')

            logger.info("Following container are in maintenance mode")
            table = [["Pod ID", "Pod Type", "Version", "Age", "Hostname", "Maintenance", "Pod Status"]]
            for pod in maint_pod.values():
                age = datetime.fromisoformat(pod.get("now")) - datetime.fromisoformat(pod.get("started_at"))
                age = str(age).rsplit(".", 1)[0]
                table.append([pod.get("pod_id", ""), pod.get("pod_type", ""), pod.get("build_tag", ""),
                              str(age), pod.get("hostname", ""), pod.get("is_maintenance"), pod.get("pod_ready")])
            column_headers = table[0]
            rows = table[1:]
            rdafutils.print_tabular(column_headers, rows)
            timeout = cmd_args.timeout
            logger.info(f"Waiting for timeout of {timeout} seconds...")
            time.sleep(timeout)
            # upgrading the pods in maintenance
            involved_services = [key for key, host_containers in service_host_container.items() for subkey, value in host_containers.items() if value in maint_pod.keys()]
            involved_services.extend(new_services)
            for service in involved_services:
                # if service in ('portal-backend', 'portal-frontend', 'rda_chat_helper'):
                if service in ('portal-backend', 'portal-frontend'):
                    continue
                logger.info(f"Upgrading service: {service} on host {host}")
                compose_file_path = self._create_compose_file_platform_service(service, host, cmd_args)
                self.copy_logging_config(config_parser, host, service)
                command = '/usr/local/bin/docker-compose --project-name platform -f {file} up -d {service} ' \
                    .format(file=compose_file_path, service=service)
                run_potential_ssh_command(host, command, config_parser)

            self._check_portal_upgrade(services, host, cmd_args, config_parser)
            logger.info("Waiting for upgraded containers to join pods")
            self.wait_for_component_to_join_pods(host, services, config_parser)

    def _check_portal_upgrade(self, involved_services, host, cmd_args, config_parser):
        services = []
        if 'portal-backend' in involved_services:
            services.append('portal-backend')
        if 'portal-frontend' in involved_services:
            services.append('portal-frontend')
        # if 'rda_chat_helper' in involved_services:
        #     services.append('rda_chat_helper')
        for service in services:
            if host == self.get_hosts()[0]:
                self._handle_portal_db_upgrades(cmd_args.tag, config_parser)
            logger.info(f"Upgrading service: {service} on host {host}")
            self._create_compose_file_platform_service(service, host, cmd_args)
            command = '/usr/local/bin/docker-compose -f {file} pull {service} && ' \
                      '/usr/local/bin/docker-compose --project-name platform -f {file} up -d {service} ' \
                .format(file=self.get_deployment_file_path(host), service=service)
            run_potential_ssh_command(host, command, config_parser)


    def _create_compose_file_platform_service(self, service, host, cmd_args):
        compose_file = self.get_deployment_file_path(host)
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(), 'platform.yaml')
        replacements = self.get_deployment_replacements(cmd_args)
        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)

        values = yaml.safe_load(original_content)
        service_value = values['services'][service]
        # override with values.yaml inputs if provided any
        self.apply_user_inputs(service, service_value)
        if not os.path.exists(compose_file):
            content = {"services": {service: service_value}}
        else:
            with open(compose_file, 'r') as f:
                content = yaml.safe_load(f)
                content['services'][service] = service_value

        with open(compose_file, 'w') as f:
            yaml.safe_dump(content, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        if not Component.is_local_host(host):
            do_potential_scp(host, compose_file, compose_file)
        return compose_file

    @staticmethod
    def get_k8s_involved_images():
        return {
            'rda-api-server': 'ubuntu-rda-client-api-server',
            'rda-registry': 'ubuntu-rda-registry',
            'rda-scheduler': 'ubuntu-rda-scheduler',
            'rda-access-manager': 'cfx-rda-access-manager',
            'rda-chat-helper': 'ubuntu-rda-chat-helper',
            'rda-collector': 'ubuntu-rda-collector',
            'rda-fsm': 'ubuntu-rda-fsm',
            'rda-asm': 'ubuntu-rda-asm',
            'rda-identity': 'ubuntu-rda-identity',
            'rda-resource-manager': 'cfx-rda-resource-manager',
            'rda-user-preferences': 'cfx-rda-user-preferences',
            'rda-portal': ['onprem-portal-nginx', 'cfx-onprem-portal']
        }

    @staticmethod
    def _update_network_config_on_secondary_deployments(data, host, token, secret_key):
        peer_cfg = os.path.join('/opt', 'rdaf', 'rdaf-peer.cfg')
        if not os.path.exists(peer_cfg):
            return
        requests.packages.urllib3.disable_warnings(
            requests.packages.urllib3.exceptions.InsecureRequestWarning)
        headers = {'Content-Type': 'application/json', 'Tenant-Secret-Key': secret_key}
        add_config = 'http://{}:7780/api/portal/rdac/addOnpremNetworkConfig?jwt={}'.format(host, token)
        payload = {'tenantid': data['tenant_id'], 'tenant_secret_key': secret_key, 'secondary_deployment': True,
                   'networkconfig': {'tenant_id': data['tenant_id'], 'nats': data['nats'],
                                     'minio': data['minio'], 'kafka': data['kafka']}}
        response = requests.post(add_config, headers=headers, data=json.dumps(payload), verify=False)
        response.raise_for_status()

    def _generate_geodr_config(self, config_parser):
        if not config_parser.has_option('rdaf-cli', 'primary'):
            return

        logger.info("Generating geo dr service config")
        preferred_primary = config_parser.getboolean('rdaf-cli', 'primary', fallback=True)
        tenant_id = config_parser.get('common', 'tenant_id')
        haproxy_comp = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME)
        minio_host = haproxy_comp.get_internal_access_host() + ':9443'
        minio_user = config_parser.get('minio', 'user')
        minio_encrypted_access_key = self.encrypt(str(str_base64_decode(minio_user)), tenant_id)
        minio_password = config_parser.get('minio', 'password')
        minio_encrypted_secret_key = self.encrypt(str(str_base64_decode(minio_password)), tenant_id)
        replication_cfg = os.path.join('/opt', 'rdaf', 'rdaf-peer.cfg')
        if not os.path.isfile(replication_cfg):
            logger.info("rdaf-peer.cfg is not present")
            platform_host = ''
            haproxy_host = ''
            peer_mariadb_encrypted_user = ''
            peer_mariadb_encrypted_password = ''
            mariadb_host = ''
            mariadb_port =''
            peer_opensearch_encrypted_user = ''
            peer_opensearch_encrypted_password = ''
            opensearch_host = ''
            opensearch_port =''
            peer_external_opensearch_encrypted_user = ''
            peer_external_opensearch_encrypted_password = ''
            external_opensearch_host = ''
            external_opensearch_port =''
        else:
            replication_configs = configparser.ConfigParser(allow_no_value=True)
            with open(replication_cfg, 'r') as f:
                replication_configs.read_file(f)
            # TODO handle  ha deployments
            host_value = replication_configs.get('haproxy', 'host')
            haproxy_host = host_value.split(',')[0].strip()
            platform_host_value = replication_configs.get('common', 'platform_service_host')
            platform_host = platform_host_value.split(',')[0].strip()
            mariadb_host = replication_configs.get('mariadb', 'host').split(',')[0].strip()
            mariadb_port = 3306
            peer_mariadb_user = replication_configs.get('mariadb', 'user')
            peer_mariadb_encrypted_user = self.encrypt(str(str_base64_decode(peer_mariadb_user)), tenant_id)
            peer_mariadb_password = replication_configs.get('mariadb', 'password')
            peer_mariadb_encrypted_password = self.encrypt(str(str_base64_decode(peer_mariadb_password)), tenant_id)
            opensearch_host_value = replication_configs.get('opensearch', 'host')
            opensearch_host =  opensearch_host_value.split(',')
            opensearch_port = 9200
            peer_opensearch_user = replication_configs.get('opensearch', 'user')
            peer_opensearch_encrypted_user = self.encrypt(str(str_base64_decode(peer_opensearch_user)), tenant_id)
            peer_opensearch_password = replication_configs.get('opensearch', 'password')
            peer_opensearch_encrypted_password = self.encrypt(str(str_base64_decode(peer_opensearch_password)), tenant_id)
            if replication_configs.has_section('os_external'):
                external_opensearch_host_value = replication_configs.get('os_external', 'host')
                external_opensearch_host =  external_opensearch_host_value.split(',')
                external_opensearch_port = 9200
                peer_external_opensearch_user = replication_configs.get('os_external', 'user')
                peer_external_opensearch_encrypted_user = self.encrypt(str(str_base64_decode(peer_external_opensearch_user)), tenant_id)
                peer_external_opensearch_password = replication_configs.get('os_external', 'password')
                peer_external_opensearch_encrypted_password = self.encrypt(str(str_base64_decode(peer_external_opensearch_password)), tenant_id)
            else:
                peer_external_opensearch_encrypted_user = ''
                peer_external_opensearch_encrypted_password = ''
                external_opensearch_host = ''
                external_opensearch_port =''

        geodr_config = os.path.join(self.get_conf_dir(), 'network_config', 'geodr_config.json')
        output_dict = {
            "geodr_enabled": True,
            "preferred_primary": preferred_primary,
            "peer": {
                "backend_ip": platform_host,
                "frontend_ip": haproxy_host,
                "retries": 3,
                "heartbeat_frequency_sec": 10,
                "auto_switchover": False,
                "mariadb_ip": mariadb_host,
                "mariadb_port": mariadb_port,
                "$rda_mariadb_root_user": peer_mariadb_encrypted_user,
                "$rda_mariadb_root_password": peer_mariadb_encrypted_password,
                "opensearch_ip": opensearch_host,
                "opensearch_port": opensearch_port,
                "$rda_opensearch_root_user": peer_opensearch_encrypted_user,
                "$rda_opensearch_root_password": peer_opensearch_encrypted_password,
                "scheme": "https",
                "ssl_verify": False,
                "os_external": {
                    "hosts": external_opensearch_host,
                    "port": external_opensearch_port,
                    "$user": peer_external_opensearch_encrypted_user,
                    "$password": peer_external_opensearch_encrypted_password,
                    "scheme": "https",
                    "ssl_verify": False
                }
            },
            "minio_ip": minio_host,
            "$rda_minio_access_key": minio_encrypted_access_key,
            "$rda_minio_secret_key": minio_encrypted_secret_key
        }

        os.makedirs(os.path.dirname(geodr_config), exist_ok=True)
        output = json.dumps(output_dict, indent=4)
        with open(geodr_config, "w") as f:
            f.write(output)
            f.flush()
        dir_command = 'mkdir -p ' + os.path.dirname(geodr_config)
        service = 'rda_geodr_api_server'
        for host in self.get_hosts():
            command = f"/usr/local/bin/docker-compose --project-name platform -f {self.get_deployment_file_path(host)} restart {service}"
            if not Component.is_local_host(host):
                logger.info(f'Creating geodr config at {geodr_config} on {host}')
                run_potential_ssh_command(host, dir_command, config_parser)
                do_potential_scp(host, geodr_config, geodr_config)

            if os.path.exists(self.get_deployment_file_path(host)):
                 logger.info(f"Restarting service: {service} on host {host}")
                 run_potential_ssh_command(host, command, config_parser)