import argparse
import configparser
import itertools
import logging
import os
import socket
import string
from typing import Callable, Any, List, Union, Tuple
import yaml
import rdaf.component.event_gateway
import rdaf.component.haproxy as haproxy
import rdaf.component as comp
from rdaf.component import run_command, check_potential_remote_file_exists, \
    do_potential_scp_fetch, execute_command
import rdaf
import rdaf.component.cert as cert
from rdaf import rdafutils
from rdaf import get_templates_dir_root, get_docker_compose_scripts_dir, get_k8s_templates_dir_root
from rdaf.component import Component, run_potential_ssh_command, remove_dir_contents, \
    OtherCategoryOrder, dockerregistry, _copytree, do_potential_scp, copy_content_to_root_owned_file
from rdaf.component import _comma_delimited_to_list
from rdaf.component import _list_to_comma_delimited
from rdaf.contextual import COMPONENT_REGISTRY
from rdaf.rdafutils import cli_err_exit
from collections import OrderedDict


logger = logging.getLogger(__name__)


class PseudoComponent(Component):
    COMPONENT_NAME = 'pseudo-component'
    _option_app_deployment_mode = 'is_app_ha_mode'
    _option_service_hosts = 'service_host'
    _option_platform_service_host = 'platform_service_host'
    _option_platform_deployment_mode = 'is_platform_ha_mode'
    _option_admin_organization = 'admin_organization'
    _option_filesystem_id = 'filesystem_id'
    _option_install_root = os.path.join('/opt', 'rdaf')

    def __init__(self):
        super().__init__(PseudoComponent.COMPONENT_NAME, 'common', category='other',
                         category_order=OtherCategoryOrder.PSEUDO_COMPONENT.value)

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_service_hosts \
                or config_name == self._option_platform_service_host:
            return _comma_delimited_to_list
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_service_hosts \
                or config_name == self._option_platform_service_host:
            return _list_to_comma_delimited
        return None

    def _init_default_configs(self):
        default_configs = dict()
        return default_configs

    def get_conf_dir(self) -> os.path:
        return os.path.join(self.get_rdaf_install_root(), 'config')

    def gather_minimal_setup_inputs(self, cmd_args, config_parser):
        default_host_name = socket.gethostname()
        host_desc = 'What is the host ip to bind RDA services?'
        platform_service_hosts = self._parse_or_prompt_hosts(cmd_args.platform_service_host,
                                                             default_host_name, '', host_desc,
                                                             'Host ip to bind ', cmd_args.no_prompt)
        common_configs = self._init_default_configs()
        common_configs[self._option_platform_service_host] = [platform_service_hosts[0]]
        common_configs[self._option_app_deployment_mode] = False
        common_configs[self._option_service_hosts] = [platform_service_hosts[0]]
        common_configs[self._option_platform_deployment_mode] = False
        organization = cmd_args.admin_organization \
            if hasattr(cmd_args, "admin_organization") and cmd_args.admin_organization is not None \
            else 'CloudFabrix'
        common_configs[self._option_admin_organization] = organization
        self._mark_configured(common_configs, config_parser)

    def gather_setup_inputs(self, cmd_args, config_parser):
        default_host_name = socket.gethostname()
        # now gather platform service deployment mode
        desc = 'Will platform services be installed in HA mode?'
        if cmd_args.no_prompt:
            platform_ha_mode = cmd_args.platform_service_ha
        else:
            platform_ha_mode = rdafutils.query_yes_no(desc, default='no')
        common_configs = self._init_default_configs()
        common_configs[self._option_platform_deployment_mode] = platform_ha_mode

        no_prompt_err_msg = 'No platform service host specified. Use --platform-service-host ' \
                            'to specify a platform host'
        host_desc = 'What are the host(s) on which you want the RDAF platform services' \
                    ' to be installed?'
        platform_service_hosts = self._parse_or_prompt_hosts(cmd_args.platform_service_host,
                                                             default_host_name,
                                                             no_prompt_err_msg, host_desc,
                                                             'Platform service host(s)',
                                                             cmd_args.no_prompt)
        common_configs[self._option_platform_service_host] = platform_service_hosts
        if platform_ha_mode and len(platform_service_hosts) < 2:
            rdafutils.cli_err_exit("Minimum two hosts are needed for platform to deploy in HA mode.")

        # now gather app service deployment mode
        desc = 'Will application services be installed in HA mode?'
        if cmd_args.no_prompt:
            app_ha_mode = cmd_args.app_service_ha
        else:
            app_ha_mode = rdafutils.query_yes_no(desc, default='no')
        common_configs[self._option_app_deployment_mode] = app_ha_mode

        # now gather service hosts
        err_msg = 'No application service host specified. Use --service-host ' \
                  + 'to specify one'
        desc = 'What are the host(s) on which you want the application services' \
               ' to be installed?'
        service_hosts = self._parse_or_prompt_hosts(cmd_args.service_host,
                                                    default_host_name,
                                                    err_msg, desc,
                                                    'Application service host(s)',
                                                    cmd_args.no_prompt)
        common_configs[self._option_service_hosts] = service_hosts

        if app_ha_mode and len(service_hosts) < 2:
            rdafutils.cli_err_exit("Minimum two hosts are needed for apps to deploy in HA mode.")

        # default admin user organization
        organization_desc = 'What is the organization you want to use for the admin user created?'
        organization_no_prompt_err_msg = 'No admin user organization specified. Use --admin-organization ' \
                                         'to specify one'
        organization = rdaf.component.Component._parse_or_prompt_value(cmd_args.admin_organization,
                                                                       "CloudFabrix",
                                                                       organization_no_prompt_err_msg,
                                                                       organization_desc,
                                                                       'Admin organization',
                                                                       cmd_args.no_prompt)
        common_configs[self._option_admin_organization] = organization
        self._mark_configured(common_configs, config_parser)

    def gather_k8s_setup_inputs(self, cmd_args, config_parser):
        default_host_name = socket.gethostname()
        no_prompt_err_msg = 'No platform service host specified. Use --platform-service-host ' \
                            'to specify a platform host'
        host_desc = 'What are the host(s) on which you want the RDAF platform services' \
                    ' to be installed?'
        platform_service_hosts = self._parse_or_prompt_hosts(cmd_args.platform_service_host,
                                                             default_host_name,
                                                             no_prompt_err_msg, host_desc,
                                                             'Platform service host(s)',
                                                             cmd_args.no_prompt)
        common_configs = self._init_default_configs()
        common_configs[self._option_platform_service_host] = platform_service_hosts

        # now gather app service deployment mode
        desc = 'Will application services be installed in HA mode?'
        if cmd_args.no_prompt:
            app_ha_mode = cmd_args.app_service_ha
        else:
            app_ha_mode = rdafutils.query_yes_no(desc, default='no')
        common_configs[self._option_app_deployment_mode] = app_ha_mode

        # now gather service hosts
        err_msg = 'No application service host specified. Use --service-host ' \
                  + 'to specify one'
        desc = 'What are the host(s) on which you want the application services' \
               ' to be installed?'
        service_hosts = self._parse_or_prompt_hosts(cmd_args.service_host,
                                                    default_host_name,
                                                    err_msg, desc,
                                                    'Application service host(s)',
                                                    cmd_args.no_prompt)
        common_configs[self._option_service_hosts] = service_hosts

        if app_ha_mode and len(service_hosts) < 2:
            rdafutils.cli_err_exit("Minimum two hosts are needed for apps to deploy in HA mode.")

        # default admin user organization
        organization_desc = 'What is the organization you want to use for the admin user created?'
        organization_no_prompt_err_msg = 'No admin user organization specified. Use --admin-org ' \
                                         'to specify one'
        organization = rdaf.component.Component._parse_or_prompt_value(cmd_args.admin_organization,
                                                                       "CloudFabrix",
                                                                       organization_no_prompt_err_msg,
                                                                       organization_desc,
                                                                       'Admin organization',
                                                                       cmd_args.no_prompt)
        common_configs[self._option_admin_organization] = organization
        self._mark_configured(common_configs, config_parser)

    def do_k8s_setup(self, cmd_args, config_parser):
        if self.get_deployment_type(config_parser) == "k8s":
            # copying docker-compose to local host for db init
            docker_compose_file = os.path.join(rdaf.get_scripts_dir_root(), 'docker-compose')
            docker_compose_file_path = os.path.join('/usr', 'local', 'bin', 'docker-compose')
            copy_command = 'sudo cp ' + docker_compose_file + ' ' + docker_compose_file_path + \
                           ' && sudo chmod +x /usr/local/bin/docker-compose'
            run_command(copy_command)
            # doing a docker login
            docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
            docker_registry.docker_login_known_hosts(config_parser)

        self.check_kubectl_installation()
        self.create_k8s_configs(config_parser)

        known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        if self.get_deployment_type(config_parser) != "aws":
            # creating internal and external network access
            self._create_network_access(config_parser)

            # create the install root on each known host and set that dir with the correct
            # ownership and group permissions
            self.setup_install_root_dir_hierarchy(known_hosts, config_parser)

        # move the generated certs (if any) from the temporary location into under
        # the proper install location
        cert_manager: cert.CertManager = COMPONENT_REGISTRY.require(cert.CertManager.COMPONENT_NAME)
        tmp_cert_dir = cert_manager.get_temp_cert_dir()
        if os.path.exists(tmp_cert_dir):
            cert_dir = cert_manager.get_cert_root_dir()
            logger.info('Moving generated certs from ' + tmp_cert_dir + ' to ' + cert_dir)
            _copytree(tmp_cert_dir, cert_dir)
            # remove the temporary location
            cert_manager.delete_temp_cert_dir()
            # we don't need to copy the certs as Charts are moved to config
            if self.get_deployment_type(config_parser) != "aws":
                for host in known_hosts:
                    do_potential_scp(host, cert_dir, cert_dir)

        # copy the k8s values.yaml
        template_sub_dir = 'k8s-local'
        if self.get_deployment_type(config_parser) == 'aws':
            template_sub_dir = 'k8s-aws'
        template_path = os.path.join(get_templates_dir_root(), template_sub_dir, "values.yaml")
        destination_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        dest_dir = os.path.dirname(destination_path)
        command = 'sudo mkdir -p ' + dest_dir + ' && sudo chown -R ' + str(os.getuid()) \
                  + ' ' + dest_dir + ' && sudo chgrp -R ' + str(os.getgid()) + ' ' + dest_dir
        run_command(command)
        event_gateway = COMPONENT_REGISTRY.require(rdaf.component.event_gateway.COMPONENT_NAME)
        replacements = dict()
        replacements['NAMESPACE'] = self.get_namespace(config_parser)
        replacements['WORKER_REPLICAS'] = len(self.get_worker_hosts(config_parser))
        replacements['EVENT_GATEWAY_REPLICAS'] = len(event_gateway.get_hosts())
        replacements['PLATFORM_REPLICAS'] = 2 if len(self.get_platform_services_hosts()) > 1 else 1
        replacements['SERVICE_REPLICAS'] = 2 if self.get_app_ha_mode() else 1
        if not cmd_args.aws:
            haproxy_advertised_host = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME) \
                        .get_internal_access_host()
            replacements['ADVERTISED_HOST'] = haproxy_advertised_host
            replacements['PORTAL_HOST'] = haproxy_advertised_host
        with open(template_path, 'r') as f:
            template_content = f.read()
        content = string.Template(template_content).safe_substitute(replacements)
        values = yaml.safe_load(content)

        docker_registry = COMPONENT_REGISTRY.require(rdaf.component.dockerregistry.COMPONENT_NAME)
        if docker_registry.get_docker_registry_project() == 'internal':
            for service in values:
                entry = values[service]
                if type(entry) is not dict:
                    continue
                # if 'privileged' in entry:
                #     entry['privileged'] = 'true'
                if 'env' in entry:
                    env = entry['env']
                    if 'RDA_ENABLE_TRACES' in env:
                        env['RDA_ENABLE_TRACES'] = "yes"
        with open(destination_path, 'w') as f:
            yaml.safe_dump(values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

        self.create_cert_configs(config_parser)
        self.provision_load_balancer(config_parser)
        # self.provision_filesystem_id(config_parser)

    def provision_load_balancer(self, config_parser):
        if self.get_deployment_type(config_parser) != 'aws':
            return
        logger.info("Provisioning load balancers and services...")
        template_path = os.path.join(get_templates_dir_root(), 'k8s-aws', "rdaf-services.yaml")
        destination_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'rdaf-services.yaml')

        cert_manager: cert.CertManager = COMPONENT_REGISTRY.require(
            cert.CertManager.COMPONENT_NAME)
        namespace = self.get_namespace(config_parser)
        replacements = dict()
        replacements['CERT_ARN'] = cert_manager.get_cert_arn()
        replacements['NAMESPACE'] = namespace
        with open(template_path, 'r') as f:
            template_content = f.read()
        content = string.Template(template_content).safe_substitute(replacements)
        with open(destination_path, 'w') as f:
            f.write(content)
        # provisioning nlb
        run_command(f'kubectl apply -f {destination_path} -n {namespace}')

    def provision_filesystem_id(self, config_parser):
        if self.get_deployment_type(config_parser) != 'aws':
            return
        logger.info("Provisioning fs id...")
        template_path = os.path.join(get_templates_dir_root(), 'k8s-aws', "efs-pvs.yaml")
        destination_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'efs-pvs.yaml')
        namespace = self.get_namespace(config_parser)
        replacements = dict()
        replacements['FILESYSTEM_ID'] = self.configs[self._option_filesystem_id]
        with open(template_path, 'r') as f:
            template_content = f.read()
        content = string.Template(template_content).safe_substitute(replacements)
        with open(destination_path, 'w') as f:
            f.write(content)
        # provisioning EFS
        run_command(f'kubectl apply -f {destination_path} -n {namespace}')

    def create_cert_configs(self,config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        # create the kubectl secrets
        logger.debug("Creating tls ca cert secret for app services")
        cert_secret = 'kubectl create secret generic -n {} rdaf-ca-certs ' \
                      '--from-file=ca.cert=/opt/rdaf/cert/ca/ca.crt ' \
                      '--from-file=ca.key=/opt/rdaf/cert/ca/ca.key ' \
                      '--from-file=ca.pem=/opt/rdaf/cert/ca/ca.pem ' \
                      '--save-config --dry-run=client -o yaml |  kubectl apply -f -'.format(namespace)
        run_command(cert_secret)

    def do_setup(self, cmd_args, config_parser: configparser.ConfigParser):
        self.docker_compose_check(config_parser)
        known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        # create the install root on each known host and set that dir with the correct
        # ownership and group permissions
        self.setup_install_root_dir_hierarchy(known_hosts, config_parser)
        # move the generated certs (if any) from the temporary location into under
        # the proper install location
        cert_manager: cert.CertManager = COMPONENT_REGISTRY.require(
            cert.CertManager.COMPONENT_NAME)
        tmp_cert_dir = cert_manager.get_temp_cert_dir()
        if os.path.exists(tmp_cert_dir):
            cert_dir = cert_manager.get_cert_root_dir()
            logger.info('Moving generated certs from ' + tmp_cert_dir + ' to ' + cert_dir)
            _copytree(tmp_cert_dir, cert_dir)
            # remove the temporary location
            cert_manager.delete_temp_cert_dir()

            # copy certs to respective hosts
            for host in known_hosts:
                do_potential_scp(host, cert_dir, cert_dir)

        # values.yaml file
        self.copy_values_yaml(config_parser)
        self.copy_logrotation(config_parser)

    def setup_install_root_dir_hierarchy(self, hosts: Union[str, List[str]],
                                         config_parser: configparser.ConfigParser):
        install_root = self.get_rdaf_install_root(config_parser)
        uid = os.getuid()
        gid = os.getgid()
        dirs = [install_root, os.path.join(install_root, 'config'),
                os.path.join(install_root, 'data'),
                os.path.join(install_root, 'logs'),
                os.path.join(install_root, 'deployment-scripts'),
                os.path.join(install_root, 'config', 'runtime')]
        if isinstance(hosts, str):
            hosts = [hosts]
        for host in hosts:
            for path in dirs:
                logger.info('Creating directory ' + path + ' and setting ownership to user '
                            + str(uid) + ' and group to group ' + str(gid) + ' on host ' + host)
                command = 'sudo mkdir -p ' + path + ' && sudo chown -R ' + str(
                    uid) + ' ' + path + ' && sudo chgrp -R ' + str(
                    gid) + ' ' + path
                run_potential_ssh_command(host, command, config_parser)

    def get_hosts(self) -> list:
        hosts = []
        hosts.extend(self.configs[self._option_platform_service_host])
        hosts.extend(self.configs[self._option_service_hosts])
        return hosts
    
    def get_app_ha_mode(self) -> bool:
        return self.configs[self._option_app_deployment_mode]

    def get_platform_ha_mode(self) -> bool:
        return self.configs[self._option_platform_deployment_mode]

    def get_platform_services_hosts(self) -> List[str]:
        hosts = []
        hosts.extend(self.configs[self._option_platform_service_host])
        return hosts

    def get_portal_host_ports(self) -> List[Tuple]:
        host_ports = []
        hosts = self.configs[self._option_platform_service_host]
        # load the values.yaml if any
        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
            if 'portal-frontend' in data['services'] and data['services']['portal-frontend'].get('hosts'):
                hosts = data['services']['portal-frontend'].get('hosts')
        for host in hosts:
            host_ports.append((host, 7780))
        return host_ports

    def get_api_server_exposed_hosts_ports(self) -> List[Tuple]:
        host_ports = []
        hosts = self.configs[self._option_platform_service_host]
        # load the values.yaml if any
        values_file = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        if os.path.exists(values_file):
            with open(values_file) as f:
                data = yaml.safe_load(f)
            if 'rda_api_server' in data['services'] and data['services']['rda_api_server'].get('hosts'):
                hosts = data['services']['rda_api_server'].get('hosts')

        for host in hosts:
            host_ports.append((host, 8807))
        return host_ports

    def get_install_root(self):
        return self._option_install_root

    def status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
               k8s=False) -> List[dict]:
        return []

    def k8s_status(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser) -> List[dict]:
        return []

    def reset(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        # delete the install root on all known hosts
        known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        install_root = self.get_install_root()
        for host in known_hosts:
            try:
                remove_dir_contents(host, install_root, config_parser, use_sudo=True)
            except Exception:
                logger.warning('Failed to delete ' + install_root + ' on host ' + host)
        # for cases where the cli doesn't act as any deployment vm
        if os.path.exists(os.path.join(install_root, 'rdaf.cfg')):
            try:
                remove_dir_contents(socket.gethostname(), install_root, config_parser, use_sudo=True)
            except Exception:
                logger.warning('Failed to delete ' + install_root + ' on local host ')

    def k8s_reset(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.reset(cmd_args, config_parser)
        install_root = self.get_install_root()
        try:
            remove_dir_contents(socket.gethostname(), install_root, config_parser, use_sudo=True)
        except Exception:
            logger.warning('Failed to delete ' + install_root + ' on host ' + socket.gethostname())

    def backup_conf(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_dir_root: os.path):
        component_conf_dir = os.path.join("/opt", "rdaf", "config")
        dest_config_dir = os.path.join(backup_dir_root, "config")
        # copy platform level config
        for entry in ["network_config", "portal"]:
            entry_path = os.path.join(component_conf_dir, entry)
            if not os.path.exists(entry_path):
                continue
            command = 'sudo cp -r --preserve=ownership,timestamps "{}" "{}"'.format(entry_path, dest_config_dir)
            run_command(command)

        # backing up log configurations
        logger.debug("backing up log configurations of platform and services")
        log_path = os.path.join(component_conf_dir, 'log')
        for host in self.get_hosts():
            if not check_potential_remote_file_exists(host, log_path):
                continue
            dest_log_dir = os.path.join(dest_config_dir, host, 'log')
            os.makedirs(dest_log_dir, exist_ok=True)
            do_potential_scp_fetch(host, log_path, dest_log_dir)

    def restore_conf(self, config_parser: configparser.ConfigParser,
                     backup_content_root_dir: os.path):
        self._restore_certs(backup_content_root_dir)
        conf_dir = self.get_conf_dir()
        for host in self.get_platform_services_hosts():
            config_backup_dir = os.path.join(backup_content_root_dir, 'config', 'network_config')
            if os.path.exists(config_backup_dir):
                logger.info('Restoring ' + config_backup_dir + ' to ' + conf_dir + ' on host ' + host)
                if Component.is_local_host(host):
                    os.makedirs(os.path.join(conf_dir, 'network_config'), exist_ok=True)
                do_potential_scp(host, config_backup_dir, os.path.join(conf_dir, 'network_config'), sudo=True)

            portal_backup_dir = os.path.join(backup_content_root_dir, 'config', 'portal')
            if os.path.exists(portal_backup_dir):
                logger.info('Restoring ' + portal_backup_dir + ' to ' + conf_dir + ' on host ' + host)
                if Component.is_local_host(host):
                    os.makedirs(os.path.join(conf_dir, 'portal'), exist_ok=True)
                do_potential_scp(host, portal_backup_dir, os.path.join(conf_dir, 'portal'), sudo=True)

        for host in self.get_hosts():
            log_config_backup = os.path.join(backup_content_root_dir, 'config', host, 'log')
            if os.path.exists(log_config_backup):
                logger.info('Restoring ' + log_config_backup + ' to ' + conf_dir + ' on host ' + host)
                if Component.is_local_host(host):
                    os.makedirs(os.path.join(conf_dir, 'log'), exist_ok=True)
                do_potential_scp(host, log_config_backup, os.path.join(conf_dir, 'log'), sudo=True)

        self._restore_deployment_scripts(config_parser, backup_content_root_dir)

    def add_service_host(self, host: str, config_parser: configparser.ConfigParser):
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        if host not in all_known_hosts:
            self.setup_install_root_dir_hierarchy(host, config_parser)
            self._copy_configs(host, config_parser)
        self.configs[self._option_service_hosts].append(host)
        # update the config parser with changes that have happened to our configs
        self.store_config(config_parser)
    
    def _copy_configs(self, host, config_parser):
        network_config = os.path.join(self.get_rdaf_install_root(), 'config', 'network_config', 'config.json')
        if os.path.exists(network_config) and not Component.is_local_host(host):
            logger.info('Creating directory ' + os.path.dirname(network_config))
            do_potential_scp(host, network_config, network_config)

    def docker_compose_check(self, config_parser: configparser.ConfigParser):
        docker_compose_file = os.path.join(rdaf.get_scripts_dir_root(), 'docker-compose')
        all_known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[comp.dockerregistry.COMPONENT_NAME])
        docker_compose_file_path = os.path.join('/usr', 'local', 'bin', 'docker-compose')
        remote_tmp_path = '/tmp/docker-compose'
        chmod_cmd = 'sudo chmod +x /usr/local/bin/docker-compose'
        copy_command = 'sudo cp ' + docker_compose_file + ' ' + docker_compose_file_path
        ssh_copy_command = 'sudo cp ' + remote_tmp_path + ' ' + docker_compose_file_path

        for host in all_known_hosts:
            if Component.is_local_host(host):
                run_command(copy_command)
                run_command(chmod_cmd)
            else:
                do_potential_scp(host, docker_compose_file, remote_tmp_path, sudo=True)
                run_potential_ssh_command(host, ssh_copy_command, config_parser)
                run_potential_ssh_command(host, chmod_cmd, config_parser)

    def copy_values_yaml(self, config_parser: configparser.ConfigParser):
        template_path = os.path.join(get_templates_dir_root(), 'values.yaml')
        destination_path = os.path.join('/opt', 'rdaf', 'deployment-scripts', 'values.yaml')
        dest_dir = os.path.dirname(destination_path)
        command = 'sudo mkdir -p ' + dest_dir + ' && sudo chown -R ' + str(os.getuid()) \
                  + ' ' + dest_dir + ' && sudo chgrp -R ' + str(os.getgid()) + ' ' + dest_dir
        run_command(command)

        with open(template_path, 'r') as f:
            values = yaml.safe_load(f)

        from rdaf.component.dockerregistry import COMPONENT_NAME
        from rdaf.contextual import COMPONENT_REGISTRY
        docker_registry = COMPONENT_REGISTRY.require(COMPONENT_NAME)
        for entry in values['services']:
            service = values['services'][entry]
            if docker_registry.get_docker_registry_project() == 'internal':
                # if 'privileged' in service:
                #     service['privileged'] = True
                if 'environment' in service:
                    env = service['environment']
                    if 'RDA_ENABLE_TRACES' in env:
                        env['RDA_ENABLE_TRACES'] = "yes"
            haproxy_advertised_host = COMPONENT_REGISTRY.require(haproxy.COMPONENT_NAME) \
                .get_internal_access_host()
            if 'environment' in service:
                env = service['environment']
                if 'ADVERTISED_HOST' in env:
                  env['ADVERTISED_HOST'] = haproxy_advertised_host
                if 'PORTAL_HOST' in env:
                  env['PORTAL_HOST'] = haproxy_advertised_host
        # enabling geodr service deployment
        if config_parser.has_option('rdaf-cli', 'primary'):
            values.setdefault('services', {}).setdefault('rda_geodr_api_server', {})['deployment'] = True

        # updating platform services
        platform_compose = os.path.join(get_docker_compose_scripts_dir(), 'platform.yaml')
        platform_services = set(self.get_involved_services(platform_compose))
        platform_hosts = self.get_platform_hosts(config_parser)
        iterable = itertools.cycle(platform_hosts)
        for service in platform_services:
            if service == 'portal-backend':
                continue
            values['services'][service]['hosts'] = [next(iterable), next(iterable)] \
                if self.get_platform_ha_mode() else [next(iterable)]
        # having portal backend and frontend in same hosts
        portal_hosts = values['services']['portal-frontend']['hosts']
        values['services']['portal-backend']['hosts'] = list(portal_hosts)

        oia_compose = os.path.join(get_docker_compose_scripts_dir(), 'oia.yaml')
        
        services = self.get_involved_services(oia_compose)
        service_hosts = self.get_service_hosts(config_parser)

        iterable = itertools.cycle(service_hosts)
        for service in services:
            values['services'][service]['hosts'] = [next(iterable), next(iterable)] \
                if self.get_app_ha_mode() else [next(iterable)]

        with open(destination_path, 'w') as f:
            yaml.safe_dump(values, f, default_flow_style=False, explicit_start=True,
                           allow_unicode=True, encoding='utf-8', sort_keys=False)

    def copy_logrotation(self, config_parser: configparser.ConfigParser):
        infra = COMPONENT_REGISTRY.get_by_category('infra')
        infra_hosts = set()
        for component in infra:
            infra_hosts.update(component.get_hosts())
        conf = os.path.join('/etc', 'logrotate.d', 'rdaf')
        logrotation = os.path.join(get_templates_dir_root(), 'logrotation')
        with open(logrotation, 'r') as f:
            content = f.read()
        for host in infra_hosts:
            copy_content_to_root_owned_file(host, content, conf, config_parser, append=False)

    @staticmethod
    def _restore_certs(backup_content_root_dir):
        # certs
        known_hosts = COMPONENT_REGISTRY.get_all_known_component_hosts(
            skip_components=[rdaf.component.dockerregistry.COMPONENT_NAME])
        # copy certs to respective hosts
        cert_manager: cert.CertManager = COMPONENT_REGISTRY.require(
            cert.CertManager.COMPONENT_NAME)
        cert_dir = cert_manager.get_cert_root_dir()
        backup_cert_dir = os.path.join(backup_content_root_dir, 'cert')
        logger.info('Moving certs from ' + backup_cert_dir + ' to ' + cert_dir)
        command = 'sudo cp -r --preserve=ownership,timestamps "{}" "{}"'.format(backup_cert_dir, os.path.dirname(cert_dir))
        run_command(command)

        for host in known_hosts:
            do_potential_scp(host, cert_dir, cert_dir)

    def _restore_deployment_scripts(self, config_parser, backup_content_root_dir):
        backup_dir = os.path.join(backup_content_root_dir, 'deployment-scripts')
        deployments_dir = os.path.join('/opt', 'rdaf', 'deployment-scripts')
        # copy all scripts to cli vm
        command = 'sudo cp -r --preserve=ownership,timestamps "{}" "{}"'.format(backup_dir, os.path.join('/opt', 'rdaf'))
        run_command(command)

        hosts = [i for i in os.listdir(os.path.join(deployments_dir))
                 if os.path.isdir(os.path.join(deployments_dir, i))]

        for host in hosts:
            file_path = os.path.join(deployments_dir, host, 'infra.yaml')
            if not os.path.exists(file_path) or Component.is_local_host(host):
                continue
            do_potential_scp(host, file_path, file_path)

        # platform
        platform_file = os.path.join(deployments_dir, 'platform.yaml')
        if os.path.exists(platform_file):
            for host in self.get_platform_services_hosts():
                if Component.is_local_host(host):
                    continue
                do_potential_scp(host, platform_file, platform_file)

        # worker
        worker_yaml = os.path.join(deployments_dir, 'worker.yaml')
        if os.path.exists(worker_yaml):
            for host in self.get_worker_hosts(config_parser):
                if Component.is_local_host(host):
                    continue
                do_potential_scp(host, worker_yaml, worker_yaml)

        # oia
        for host in self.get_service_hosts(config_parser):
            oia_yaml = os.path.join(deployments_dir, host, 'oia.yaml')
            if not os.path.exists(oia_yaml) or Component.is_local_host(host):
                continue
            do_potential_scp(host, oia_yaml, oia_yaml)

    @staticmethod
    def check_kubectl_installation():
        # TODO we expect kubectl to be there.
        # kubectl_file = os.path.join(rdaf.get_scripts_dir_root(), 'kubectl')
        # kubectl_file_path = os.path.join('/usr', 'local', 'bin', 'kubectl')
        #
        # kubectl_copy = 'sudo cp ' + kubectl_file + ' ' + kubectl_file_path \
        #                + ' && sudo chmod +x /usr/local/bin/kubectl'
        # run_command(kubectl_copy)

        ret, stdout, stderr = execute_command('kubectl cluster-info')
        if ret != 0:
            cli_err_exit("Unable to validate kubernetes cluster connectivity. "
                         "Please configure 'Kubectl' correctly")

        # helm installation
        helm_file = os.path.join(rdaf.get_scripts_dir_root(), 'helm')
        helm_file_path = os.path.join('/usr', 'local', 'bin', 'helm')

        helm_copy = 'sudo cp ' + helm_file + ' ' + helm_file_path \
                    + ' && sudo chmod +x /usr/local/bin/helm'
        run_command(helm_copy)

    def _create_network_access(self, config_parser: configparser.ConfigParser):
        namespace = self.get_namespace(config_parser)
        logger.info("Creating internal and external network access to rdaf components.")
        network_service = os.path.join(rdaf.get_k8s_templates_dir_root(),
                                       'rdaf-network-access.yaml')
        destination_path = os.path.join('/opt', 'rdaf', 'rdaf-network-access.yaml')
        replacements = dict()
        replacements['NAMESPACE'] = namespace
        with open(network_service, 'r') as f:
            template_content = f.read()
        content = string.Template(template_content).safe_substitute(replacements)
        with open(destination_path, 'w') as f:
            f.write(content)
        network_access_cmd = f'kubectl apply -f {destination_path} -n {namespace}'
        run_command(network_access_cmd)

    def create_k8s_configs(self, config_parser):
        namespace = self.get_namespace(config_parser)
        logger.info(f"creating {namespace} namespace")
        create_namespace = f'kubectl create namespace {namespace} --dry-run=client -o yaml | kubectl apply -f -'
        run_command(create_namespace)

        if self.get_deployment_type(config_parser) != "aws":
            # create storage policy
            logger.info("applying storage policy")
            sp_path = os.path.join(get_k8s_templates_dir_root(), 'storageclass.yaml')
            sp_cmd = f'kubectl apply -f {sp_path}'
            run_command(sp_cmd)

        # label the nodes
        # <ip, node>
        logger.info("Labelling the nodes...")
        nodes = self.get_k8s_nodes()

        # infra nodes
        infra = COMPONENT_REGISTRY.get_by_category('infra')
        infra_hosts = OrderedDict()
        minio_hosts = OrderedDict()
        nats_hosts = OrderedDict()
        haproxy_hosts = OrderedDict()
        for component in infra:
            if component.get_name() == 'minio':
                for hosts in component.get_hosts():
                    minio_hosts[hosts]=None
            elif component.get_name() == 'nats':
                for hosts in component.get_hosts():
                    nats_hosts[hosts]=None
            elif component.get_name() == 'haproxy':
                for hosts in component.get_hosts():
                    haproxy_hosts[hosts]=None
            else:
                for hosts in component.get_hosts():
                    infra_hosts[hosts]=None
        
        for host in infra_hosts:
            if not nodes[host]:
                rdafutils.cli_err_exit("unknown infra host {}, Please use one of the "
                                       "kubernetes worker nodes".format(host))
            logger.info('applying node label "rdaf_infra_services=allow" for infra host: ' + host)
            run_command('kubectl label nodes {} {} --overwrite'.format(nodes[host], 'rdaf_infra_services=allow'))

        # nats hosts
        for index, host in enumerate(nats_hosts.keys()):
            if not nodes[host]:
                rdafutils.cli_err_exit("unknown nats host {}, Please use one of the "
                                       "kubernetes worker nodes".format(host))
            logger.info('applying node label "rdaf_infra_nats=allow" for nats host: ' + host)
            run_command('kubectl label nodes {} {} --overwrite'.format(nodes[host], 'rdaf_infra_nats=allow'))
            if self.get_deployment_type(config_parser) != "aws":
                run_command('kubectl label nodes {} {} --overwrite'
                            .format(nodes[host], 'rdaf_infra_nats_node=nats-' + str(index)))

        # minio hosts
        for index, host in enumerate(minio_hosts.keys()):
            if not nodes[host]:
                rdafutils.cli_err_exit("unknown minio host {}, Please use one of the "
                                       "kubernetes worker nodes".format(host))
            logger.info('applying node label "rdaf_infra_minio=allow" for minio host: ' + host)
            run_command('kubectl label nodes {} {} --overwrite'.format(nodes[host], 'rdaf_infra_minio=allow'))
            if self.get_deployment_type(config_parser) != "aws":
                run_command('kubectl label nodes {} {} --overwrite'
                            .format(nodes[host], 'rdaf_infra_minio_node=minio-' + str(index)))
        
        # haproxy hosts
        for index, host in enumerate(haproxy_hosts.keys()):
            if not nodes[host]:
                rdafutils.cli_err_exit("unknown haproxy host {}, Please use one of the "
                                       "kubernetes worker nodes".format(host))
            logger.info('applying node label "rdaf_infra_haproxy=allow" for haproxy host: ' + host)
            run_command('kubectl label nodes {} {} --overwrite'.format(nodes[host], 'rdaf_infra_haproxy=allow'))

        # labeling infra nodes for binding PV's
        if self.get_deployment_type(config_parser) != "aws":
            for index, host in enumerate(infra_hosts.keys()):
                logger.info('applying node label "rdaf_infra_node=node-{}" for infra host: {}'.format(index, host))
                run_command('kubectl label nodes {} {} --overwrite'.format(nodes[host], 'rdaf_infra_node=node-'+str(index)))

        # platform hosts
        for host in self.get_platform_services_hosts():
            if not nodes[host]:
                rdafutils.cli_err_exit("unknown host {}, Please use one of the kubernetes worker nodes".format(host))
            logger.info('applying node label "rdaf_platform_services=allow" for platform host: ' + host)
            run_command('kubectl label nodes {} {} --overwrite'.format(nodes[host], 'rdaf_platform_services=allow'))

        # service hosts
        for host in self.configs[self._option_service_hosts]:
            if not nodes[host]:
                rdafutils.cli_err_exit("unknown host {}, Please use one of the kubernetes worker nodes".format(host))
            logger.info('applying node label "rdaf_application_services=allow" for service host: ' + host)
            run_command('kubectl label nodes {} {} --overwrite'.format(nodes[host], 'rdaf_application_services=allow'))

        # worker hosts
        for host in self.get_worker_hosts(config_parser):
            if not nodes[host]:
                rdafutils.cli_err_exit("unknown host {}, Please use one of the kubernetes worker nodes".format(host))
            logger.info('applying node label "rdaf_worker_services=allow" for worker host: ' + host)
            run_command('kubectl label nodes {} {} --overwrite'.format(nodes[host], 'rdaf_worker_services=allow'))

        # event gateway hosts
        event_gateway = COMPONENT_REGISTRY.require(rdaf.component.event_gateway.COMPONENT_NAME)
        for host in event_gateway.get_hosts():
            if not nodes[host]:
                rdafutils.cli_err_exit(
                    "unknown host {}, Please use one of the kubernetes worker nodes".format(host))
            logger.info('applying node label "rdaf_event_gateway=allow" for event gateway host: ' + host)
            run_command('kubectl label nodes {} {} --overwrite'.format(nodes[host], 'rdaf_event_gateway=allow'))

    def k8s_backup_conf(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser,
                    backup_dir_root: os.path):
        component_conf_dir = os.path.join("/opt", "rdaf", "config")
        dest_config_dir = os.path.join(backup_dir_root, "config")
        # copy platform level config
        for entry in ["network_config", "portal"]:
            entry_path = os.path.join(component_conf_dir, entry)
            if not os.path.exists(entry_path):
                continue
            command = 'sudo cp -r --preserve=ownership,timestamps "{}" "{}"'.format(entry_path, dest_config_dir)
            run_command(command)

    def k8s_restore_conf(self, config_parser: configparser.ConfigParser,
                         backup_content_root_dir: os.path):
        namespace = self.get_namespace(config_parser)
        self._restore_certs(backup_content_root_dir)
        conf_dir = self.get_conf_dir()
        config_backup_dir = os.path.join(backup_content_root_dir, 'config', 'network_config')
        if os.path.exists(config_backup_dir):
            logger.info('Restoring ' + config_backup_dir + ' to ' + conf_dir)
            command = 'sudo cp -r --preserve=ownership,timestamps "{}" "{}"'.format(config_backup_dir, conf_dir)
            run_command(command)

        portal_backup_dir = os.path.join(backup_content_root_dir, 'config', 'portal')
        if os.path.exists(portal_backup_dir):
            logger.info('Restoring ' + portal_backup_dir + ' to ' + conf_dir)
            command = 'sudo cp -r --preserve=ownership,timestamps "{}" "{}"'.format(portal_backup_dir, conf_dir)
            run_command(command)

        deployments_backup_dir = os.path.join(backup_content_root_dir, 'deployment-scripts')
        # copy all scripts to cli vm
        command = 'sudo cp -r --preserve=ownership,timestamps "{}" "{}"'.format(deployments_backup_dir, os.path.join('/opt', 'rdaf'))
        run_command(command)

        logger.info("Restoring network configs...")
        network_config = os.path.join(self.get_conf_dir(), 'network_config', 'config.json')
        config = 'kubectl create configmap rda-network-config -n {} ' \
                 '--from-file=config.json={} --save-config --dry-run=client -o yaml ' \
                 '|  kubectl apply -f -'.format(namespace,network_config)
        run_command(config)

        pconfig_yek = os.path.join(self.get_conf_dir(), 'portal', 'pconfig.yek')
        portal_settings = os.path.join(self.get_conf_dir(), 'portal', 'settings.json')
        config = 'kubectl create configmap rda-portal-backend-settings -n {} ' \
                 '--from-file=settings.json={} --save-config --dry-run=client -o yaml ' \
                 '|  kubectl apply -f -'.format(namespace,portal_settings)
        run_command(config)

        config = 'kubectl create configmap rda-portal-backend-pconfig -n {} ' \
                 '--from-file=pconfig.yek={} --save-config --dry-run=client -o yaml ' \
                 '|  kubectl apply -f -'.format(namespace,pconfig_yek)
        run_command(config)

        self.create_cert_configs(config_parser)

