import argparse
import configparser
import json
import logging
import os
import string
import uuid
from typing import Callable, Any

import rdaf
from rdaf.component import (Component, run_potential_ssh_command, do_potential_scp,
                            create_file, run_command)
from rdaf.component import _comma_delimited_to_list
from rdaf.component import _list_to_comma_delimited
from rdaf.rdafutils import cli_err_exit, str_base64_decode, str_base64_encode

COMPONENT_NAME = 'telegraf'
logger = logging.getLogger(__name__)


class Telegraf(Component):
    _option_host = 'host'
    _option_kafka_host = 'kafka_host'
    _option_kafka_user = 'kafka_user'
    _option_kafka_password = 'kafka_password'
    _option_project_id = 'project_id'

    def __init__(self):
        super().__init__(COMPONENT_NAME, 'telegraf', 'telegraf')

    def _get_config_loader(self, config_name: str) -> Callable[[str], Any]:
        if config_name == self._option_host or config_name == self._option_kafka_host:
            return _comma_delimited_to_list
        return None

    def _get_config_storer(self, config_name: str) -> Callable[[Any], str]:
        if config_name == self._option_host or config_name == self._option_kafka_host:
            return _list_to_comma_delimited
        return None

    def _init_default_configs(self):
        default_configs = dict()
        default_configs[self._option_host] = []
        default_configs[self._option_kafka_host] = []
        return default_configs

    @staticmethod
    def get_install_root():
        return os.path.join('/opt', 'rdaf-telegraf')

    def get_conf_dir(self) -> os.path:
        return os.path.join(self.get_install_root(), 'config')

    def gather_setup_inputs(self, cmd_args, config_parser):
        configs = self._init_default_configs()
        default_host_name = Component.get_default_host()
        no_prompt_err_msg = 'No host specified. Use --telegraf-host to specify a telegraf host'
        host_desc = 'What is the "host" on which you want the telegraf to be deployed?'
        hosts = Component._parse_or_prompt_hosts(cmd_args.telegraf_host, default_host_name,
                                                 no_prompt_err_msg, host_desc, 'telegraf host',
                                                 cmd_args.no_prompt)

        configs[self._option_host] = hosts
        project_id = Component._parse_or_prompt_value(cmd_args.project_id, str(uuid.uuid4()),
                                                     '', "Please specify the project id to tag metrics?",
                                                         'Project id', cmd_args.no_prompt)
        configs[self._option_project_id] = project_id
        rdaf_cfg = os.path.join('/opt', 'rdaf', 'rdaf.cfg')
        if os.path.isfile(rdaf_cfg):
            self.cert_dir = '/opt/rdaf/cert'
            rdaf_configs = configparser.ConfigParser(allow_no_value=True)
            with open(rdaf_cfg, 'r') as f:
                rdaf_configs.read_file(f)
            if rdaf_configs.has_section('docker'):
                logger.info(f"Fetching Docker registry details from {rdaf_cfg}")
                config_parser['docker'] = rdaf_configs['docker']

            if rdaf_configs.has_section('kafka'):
                configs[self._option_kafka_host] = _comma_delimited_to_list(rdaf_configs.get('kafka', 'host'))
                configs[self._option_kafka_user] = rdaf_configs.get('kafka', 'external_user')
                configs[self._option_kafka_password] = rdaf_configs.get('kafka', 'external_password')
        else:
            kafka_host = Component._parse_or_prompt_hosts(cmd_args.kafka_host, default_host_name,
                                                     '', "What is the host on which kafka is deployed?",
                                                         'kafka hosts', cmd_args.no_prompt)
            configs[self._option_kafka_host] = kafka_host
            kafka_user = Component._parse_or_prompt_value(cmd_args.kafka_user, '',
                                                     '', "What is the kafka user to be used to send metrics?",
                                                         'kafka user', cmd_args.no_prompt)
            configs[self._option_kafka_user] = kafka_user
            kafka_password = Component._parse_or_prompt_value(cmd_args.kafka_password, '',
                                                     '', "What is the kafka password to be used?",
                                                         'kafka password', cmd_args.no_prompt)
            configs[self._option_kafka_password] = str_base64_encode(kafka_password)

            self.cert_dir = Component._parse_or_prompt_value(cmd_args.cert_dir, '',
                                                     '', "What is the path for certs to be used?",
                                                         'certs dir', cmd_args.no_prompt)
            if not os.path.exists(os.path.join(self.cert_dir)):
                cli_err_exit(f'Cert dir specified {self.cert_dir} does not exists.')

        self._mark_configured(configs, config_parser)

    def get_hosts(self) -> list:
        return self.configs[self._option_host]

    def get_ports(self) -> tuple:
        ports = ['9116']
        hosts = self.get_hosts()
        return hosts, ports

    def do_setup(self, cmd_args, config_parser):
        command = ('sudo mkdir -p /opt/rdaf-telegraf/certs && sudo chown -R '
                       + str(os.getuid()) + ' /opt/rdaf-telegraf/ && sudo chgrp -R ' + str(os.getgid()) + ' /opt/rdaf-telegraf/')
        for host in self.configs[self._option_host]:
            run_potential_ssh_command(host, command, config_parser)
            do_potential_scp(host, os.path.join(self.cert_dir), os.path.join('/opt/rdaf-telegraf/certs'))

        folder_cmd = ('mkdir -p /opt/rdaf-telegraf/config/conf.d /opt/rdaf-telegraf/config/templates '
                      ' /opt/rdaf-telegraf/logs/')
        for host in self.get_hosts():
            run_potential_ssh_command(host, folder_cmd, config_parser)

        # telegraf.conf
        telegraf_conf = os.path.join(rdaf.get_templates_dir_root(), 'telegraf.conf')
        with open(telegraf_conf, 'r') as f:
            template_content = f.read()
        replacements = dict()
        replacements['PROJECT_ID'] =self.configs[self._option_project_id]
        telegraf_conf_content = string.Template(template_content).safe_substitute(replacements)
        dest_telegraf_conf = os.path.join(self.get_conf_dir(), 'telegraf.conf')

        # snmp_metrics.conf
        snmp_conf = os.path.join(rdaf.get_templates_dir_root(), 'snmp_metrics.conf')
        dest_snmp_conf = os.path.join(self.get_conf_dir(), 'conf.d', 'snmp_metrics.conf')
        # telemetry_metrics.conf
        telemetry_conf = os.path.join(rdaf.get_templates_dir_root(), 'telemetry_metrics.conf')
        dest_telemetry_conf = os.path.join(self.get_conf_dir(), 'conf.d', 'telemetry_metrics.conf')
        # telemetry_metrics.conf
        telemetry_tcp_conf = os.path.join(rdaf.get_templates_dir_root(), 'telemetry_metrics_tcp.conf')
        dest_telemetry_tcp_conf = os.path.join(self.get_conf_dir(), 'conf.d', 'telemetry_metrics_tcp.conf')
        # kafka-output.conf
        kafka_conf = os.path.join(rdaf.get_templates_dir_root(), 'kafka-output.conf')
        with open(kafka_conf, 'r') as f:
            template_content = f.read()
        replacements = dict()
        replacements['HOST'] = json.dumps([i+":9093" for i in self.configs[self._option_kafka_host]])
        replacements['USER'] = self.configs[self._option_kafka_user]
        replacements['PASSWORD'] = str_base64_decode(self.configs[self._option_kafka_password])
        kafka_conf_content = string.Template(template_content).safe_substitute(replacements)
        dest_kafka_conf = os.path.join(self.get_conf_dir(), 'conf.d', 'kafka-output.conf')

        # Write telegraf.conf file for each host
        for host in self.get_hosts():
            created_location = create_file(host, telegraf_conf_content.encode(encoding='UTF-8'), dest_telegraf_conf)
            logger.info('Created telegraf configuration at ' + created_location + ' on host ' + host)
            do_potential_scp(host, snmp_conf, dest_snmp_conf)
            do_potential_scp(host, telemetry_conf, dest_telemetry_conf)
            do_potential_scp(host, telemetry_tcp_conf, dest_telemetry_tcp_conf)
            created_location = create_file(host, kafka_conf_content.encode(encoding='UTF-8'), dest_kafka_conf)
            logger.info('Created kafka output configuration at ' + created_location + ' on host ' + host)

    def get_deployment_file_name(self) -> os.path:
        return 'telegraf.yaml'

    def create_compose_file(self, cmd_args: argparse.Namespace, host: str) -> os.path:
        compose_file = os.path.join('/opt', 'rdaf-telegraf', 'deployment-scripts',
                                    self.get_deployment_file_name())
        yaml_path = os.path.join(rdaf.get_docker_compose_scripts_dir(),
                                 self.get_deployment_file_name())

        replacements = self.get_deployment_replacements(cmd_args)

        with open(yaml_path, 'r') as f:
            template_content = f.read()
        original_content = string.Template(template_content).safe_substitute(replacements)
        run_command('mkdir -p ' + os.path.dirname(compose_file))
        with open(compose_file, 'w') as f:
            f.write(original_content)

        if not Component.is_local_host(host):
            do_potential_scp(host, compose_file, compose_file)
        return compose_file

    def install(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        self.open_ports(config_parser)
        for host in self.get_hosts():
            compose_file_path = self.create_compose_file(cmd_args, host)
            command = '/usr/local/bin/docker-compose -f {file} pull {service} && ' \
                      '/usr/local/bin/docker-compose --project-name telegraf -f {file} up -d {service} ' \
                .format(file=compose_file_path, service=self.component_name)
            run_potential_ssh_command(host, command, config_parser)

    def upgrade(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            compose_file_path = self.create_compose_file(cmd_args, host)
            command = '/usr/local/bin/docker-compose -f {file} pull {service} && ' \
                      '/usr/local/bin/docker-compose --project-name telegraf -f {file} up -d {service} ' \
                .format(file=compose_file_path, service=self.component_name)
            run_potential_ssh_command(host, command, config_parser)

    def down(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        # we down in reverse order
        for host in reversed(self.get_hosts()):
            compose_file = os.path.join('/opt', 'rdaf-telegraf', 'deployment-scripts',
                                        self.get_deployment_file_name())
            if not os.path.exists(compose_file):
                continue
            command = '/usr/local/bin/docker-compose --project-name telegraf -f ' + compose_file \
                      + ' rm -fsv ' + self.component_name
            run_potential_ssh_command(host, command, config_parser)

    def up(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            compose_file = os.path.join('/opt', 'rdaf-telegraf', 'deployment-scripts', self.get_deployment_file_name())
            if not os.path.exists(compose_file):
                continue
            if self.component_name not in self.get_involved_services(compose_file):
                continue

            command = '/usr/local/bin/docker-compose --project-name telegraf -f ' + compose_file + ' up -d ' \
                      + self.component_name
            run_potential_ssh_command(host, command, config_parser)

    def stop(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in reversed(self.get_hosts()):
            compose_file = os.path.join('/opt', 'rdaf-telegraf', 'deployment-scripts', self.get_deployment_file_name())
            if not os.path.exists(compose_file):
                return
            if self.component_name not in self.get_involved_services(compose_file):
                continue
            logger.info("Stopping service: {} on host {}".format(self.component_name, host))
            command = '/usr/local/bin/docker-compose --project-name telegraf -f ' + compose_file + \
                      ' stop ' + self.component_name
            run_potential_ssh_command(host, command, config_parser)

    def start(self, cmd_args: argparse.Namespace, config_parser: configparser.ConfigParser):
        for host in self.get_hosts():
            compose_file = os.path.join('/opt', 'rdaf-telegraf', 'deployment-scripts', self.get_deployment_file_name())
            if not os.path.exists(compose_file):
                continue
            if self.component_name not in self.get_involved_services(compose_file):
                continue

            logger.info("Starting service: {} on host {}".format(self.component_name, host))
            command = '/usr/local/bin/docker-compose --project-name telegraf -f ' + compose_file \
                        + ' start ' + self.component_name
            run_potential_ssh_command(host, command, config_parser)
